/*	$OpenBSD: if_rgereg.h,v 1.10 2023/12/22 05:28:14 kevlo Exp $	*/

/*
 * Copyright (c) 2019, 2020 Kevin Lo <kevlo@openbsd.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define RGE_PCI_BAR0		PCI_MAPREG_START
#define RGE_PCI_BAR1		(PCI_MAPREG_START + 4)
#define RGE_PCI_BAR2		(PCI_MAPREG_START + 8)

#define RGE_MAC0		0x0000
#define RGE_MAC4		0x0004
#define RGE_MAR0		0x0008
#define RGE_MAR4		0x000c
#define RGE_TXDESC_ADDR_LO	0x0020
#define RGE_TXDESC_ADDR_HI	0x0024
#define RGE_INT_CFG0		0x0034
#define RGE_CMD			0x0037
#define RGE_IMR			0x0038
#define RGE_ISR			0x003c
#define RGE_TXCFG		0x0040
#define RGE_RXCFG		0x0044
#define RGE_TIMERCNT		0x0048
#define RGE_EECMD		0x0050
#define RGE_CFG0		0x0051
#define RGE_CFG1		0x0052
#define RGE_CFG2		0x0053
#define RGE_CFG3		0x0054
#define RGE_CFG4		0x0055
#define RGE_CFG5		0x0056
#define RGE_TDFNR		0x0057
#define RGE_TIMERINT0		0x0058
#define RGE_TIMERINT1		0x005c
#define RGE_CSIDR		0x0064
#define RGE_CSIAR		0x0068
#define RGE_PHYSTAT		0x006c
#define RGE_PMCH		0x006f
#define RGE_INT_CFG1		0x007a
#define RGE_EPHYAR		0x0080
#define RGE_TIMERINT2		0x008c
#define RGE_TXSTART		0x0090
#define RGE_MACOCP		0x00b0
#define RGE_PHYOCP		0x00b8
#define RGE_DLLPR		0x00d0
#define RGE_TWICMD		0x00d2
#define RGE_MCUCMD		0x00d3
#define RGE_RXMAXSIZE		0x00da
#define RGE_CPLUSCMD		0x00e0
#define RGE_IM			0x00e2
#define RGE_RXDESC_ADDR_LO	0x00e4
#define RGE_RXDESC_ADDR_HI	0x00e8
#define RGE_PPSW		0x00f2
#define RGE_TIMERINT3		0x00f4
#define RGE_INTMITI(i)		(0x0a00 + (i) * 4)
#define RGE_PHYBASE		0x0a40
#define RGE_ADDR0		0x19e0
#define RGE_ADDR1		0x19e4
#define RGE_RSS_CTRL		0x4500
#define RGE_RXQUEUE_CTRL	0x4800
#define RGE_EEE_TXIDLE_TIMER	0x6048

/* Flags for register RGE_INT_CFG0 */
#define RGE_INT_CFG0_EN			0x01
#define RGE_INT_CFG0_TIMEOUT_BYPASS	0x02
#define RGE_INT_CFG0_MITIGATION_BYPASS	0x04

/* Flags for register RGE_CMD */
#define RGE_CMD_RXBUF_EMPTY	0x01
#define RGE_CMD_TXENB		0x04
#define RGE_CMD_RXENB		0x08
#define RGE_CMD_RESET		0x10
#define RGE_CMD_STOPREQ		0x80

/* Flags for register RGE_ISR */
#define RGE_ISR_RX_OK		0x00000001
#define RGE_ISR_RX_ERR		0x00000002
#define RGE_ISR_TX_OK		0x00000004
#define RGE_ISR_TX_ERR		0x00000008
#define RGE_ISR_RX_DESC_UNAVAIL	0x00000010
#define RGE_ISR_LINKCHG		0x00000020
#define RGE_ISR_RX_FIFO_OFLOW	0x00000040
#define RGE_ISR_TX_DESC_UNAVAIL	0x00000080
#define RGE_ISR_SWI		0x00000100
#define RGE_ISR_PCS_TIMEOUT	0x00004000
#define RGE_ISR_SYSTEM_ERR	0x00008000

#define RGE_INTRS		\
	(RGE_ISR_RX_OK | RGE_ISR_RX_ERR | RGE_ISR_TX_OK |		\
	RGE_ISR_TX_ERR | RGE_ISR_LINKCHG | RGE_ISR_TX_DESC_UNAVAIL |	\
	RGE_ISR_PCS_TIMEOUT | RGE_ISR_SYSTEM_ERR)

#define RGE_INTRS_TIMER		\
	(RGE_ISR_RX_ERR | RGE_ISR_TX_ERR | RGE_ISR_PCS_TIMEOUT |	\
	RGE_ISR_SYSTEM_ERR)

/* Flags for register RGE_TXCFG */
#define RGE_TXCFG_HWREV		0x7cf00000

/* Flags for register RGE_RXCFG */
#define RGE_RXCFG_ALLPHYS	0x00000001
#define RGE_RXCFG_INDIV		0x00000002
#define RGE_RXCFG_MULTI		0x00000004
#define RGE_RXCFG_BROAD		0x00000008
#define RGE_RXCFG_RUNT		0x00000010
#define RGE_RXCFG_ERRPKT	0x00000020
#define RGE_RXCFG_VLANSTRIP	0x00c00000

/* Flags for register RGE_EECMD */
#define RGE_EECMD_WRITECFG	0xc0

/* Flags for register RGE_CFG1 */
#define RGE_CFG1_PM_EN		0x01
#define RGE_CFG1_SPEED_DOWN	0x10

/* Flags for register RGE_CFG2 */
#define RGE_CFG2_PMSTS_EN	0x20
#define RGE_CFG2_CLKREQ_EN	0x80

/* Flags for register RGE_CFG3 */
#define RGE_CFG3_RDY_TO_L23	0x02
#define RGE_CFG3_WOL_LINK	0x10
#define RGE_CFG3_WOL_MAGIC	0x20

/* Flags for register RGE_CFG5 */
#define RGE_CFG5_PME_STS	0x01
#define RGE_CFG5_WOL_LANWAKE	0x02
#define RGE_CFG5_WOL_UCAST	0x10
#define RGE_CFG5_WOL_MCAST	0x20
#define RGE_CFG5_WOL_BCAST	0x40

/* Flags for register RGE_CSIAR */
#define RGE_CSIAR_BYTE_EN	0x0000000f
#define RGE_CSIAR_BYTE_EN_SHIFT	12
#define RGE_CSIAR_ADDR_MASK	0x00000fff
#define RGE_CSIAR_BUSY		0x80000000

/* Flags for register RGE_PHYSTAT */
#define RGE_PHYSTAT_FDX		0x0001
#define RGE_PHYSTAT_LINK	0x0002
#define RGE_PHYSTAT_10MBPS	0x0004
#define RGE_PHYSTAT_100MBPS	0x0008
#define RGE_PHYSTAT_1000MBPS	0x0010
#define RGE_PHYSTAT_RXFLOW	0x0020
#define RGE_PHYSTAT_TXFLOW	0x0040
#define RGE_PHYSTAT_2500MBPS	0x0400

/* Flags for register RGE_EPHYAR */
#define RGE_EPHYAR_DATA_MASK	0x0000ffff
#define RGE_EPHYAR_BUSY		0x80000000
#define RGE_EPHYAR_ADDR_MASK	0x0000007f
#define RGE_EPHYAR_ADDR_SHIFT	16

/* Flags for register RGE_TXSTART */
#define RGE_TXSTART_START	0x0001

/* Flags for register RGE_MACOCP */
#define RGE_MACOCP_DATA_MASK	0x0000ffff
#define RGE_MACOCP_BUSY		0x80000000
#define RGE_MACOCP_ADDR_SHIFT	16

/* Flags for register RGE_PHYOCP */
#define RGE_PHYOCP_DATA_MASK	0x0000ffff
#define RGE_PHYOCP_BUSY		0x80000000
#define RGE_PHYOCP_ADDR_SHIFT	16

/* Flags for register RGE_DLLPR. */
#define RGE_DLLPR_PFM_EN	0x40
#define RGE_DLLPR_TX_10M_PS_EN	0x80

/* Flags for register RGE_MCUCMD */
#define RGE_MCUCMD_RXFIFO_EMPTY	0x10
#define RGE_MCUCMD_TXFIFO_EMPTY	0x20
#define RGE_MCUCMD_IS_OOB	0x80

/* Flags for register RGE_CPLUSCMD */
#define RGE_CPLUSCMD_RXCSUM	0x0020

#define RGE_TX_NSEGS		32
#define RGE_TX_LIST_CNT		1024
#define RGE_RX_LIST_CNT		1024
#define RGE_ALIGN		256
#define RGE_TX_LIST_SZ		(sizeof(struct rge_tx_desc) * RGE_TX_LIST_CNT)
#define RGE_RX_LIST_SZ		(sizeof(struct rge_rx_desc) * RGE_RX_LIST_CNT)
#define RGE_NEXT_TX_DESC(x)	(((x) + 1) % RGE_TX_LIST_CNT)
#define RGE_NEXT_RX_DESC(x)	(((x) + 1) % RGE_RX_LIST_CNT)
#define RGE_ADDR_LO(y)		((uint64_t) (y) & 0xffffffff)
#define RGE_ADDR_HI(y)		((uint64_t) (y) >> 32)

#define RGE_ADV_2500TFDX	0x0080

/* Tx descriptor */
struct rge_tx_desc {
	uint32_t		rge_cmdsts;
	uint32_t		rge_extsts;
	uint32_t		rge_addrlo;
	uint32_t		rge_addrhi;
	uint32_t		reserved[4];
};

#define RGE_TDCMDSTS_COLL	0x000f0000
#define RGE_TDCMDSTS_EXCESSCOLL	0x00100000
#define RGE_TDCMDSTS_TXERR	0x00800000
#define RGE_TDCMDSTS_EOF	0x10000000
#define RGE_TDCMDSTS_SOF	0x20000000
#define RGE_TDCMDSTS_EOR	0x40000000
#define RGE_TDCMDSTS_OWN	0x80000000

#define RGE_TDEXTSTS_VTAG	0x00020000
#define RGE_TDEXTSTS_IPCSUM	0x20000000
#define RGE_TDEXTSTS_TCPCSUM	0x40000000
#define RGE_TDEXTSTS_UDPCSUM	0x80000000

/* Rx descriptor */
struct rge_rx_desc {
	union {
		struct {
			uint32_t	rsvd0;
			uint32_t	rsvd1;
		} rx_qword0;
	} lo_qword0;

	union {
		struct {
			uint32_t	rss;
			uint16_t	length;
			uint16_t	hdr_info;
		} rx_qword1;

		struct {
			uint32_t	rsvd2;
			uint32_t	rsvd3;
		} rx_qword2;
	} lo_qword1;

	union {
		uint64_t		rge_addr;

		struct {
			uint64_t	timestamp;
		} rx_timestamp;

		struct {
			uint32_t	rsvd4;
			uint32_t	rsvd5;
		} rx_qword3;
	} hi_qword0;

	union {
		struct {
			uint32_t	rge_extsts;
			uint32_t	rge_cmdsts;
		} rx_qword4;

		struct {
			uint16_t	rsvd6;
			uint16_t	rsvd7;
			uint32_t	rsvd8;
		} rx_ptp;
	} hi_qword1;
};

#define RGE_RDCMDSTS_RXERRSUM	0x00100000
#define RGE_RDCMDSTS_EOF	0x01000000
#define RGE_RDCMDSTS_SOF	0x02000000
#define RGE_RDCMDSTS_EOR	0x40000000
#define RGE_RDCMDSTS_OWN	0x80000000
#define RGE_RDCMDSTS_FRAGLEN	0x00003fff

#define RGE_RDEXTSTS_VTAG	0x00010000
#define RGE_RDEXTSTS_VLAN_MASK	0x0000ffff
#define RGE_RDEXTSTS_TCPCSUMERR	0x01000000
#define RGE_RDEXTSTS_UDPCSUMERR	0x02000000
#define RGE_RDEXTSTS_IPCSUMERR	0x04000000
#define RGE_RDEXTSTS_TCPPKT	0x10000000
#define RGE_RDEXTSTS_UDPPKT	0x20000000
#define RGE_RDEXTSTS_IPV4	0x40000000
#define RGE_RDEXTSTS_IPV6	0x80000000

 /*
  * Statistics counter structure
  */
struct rge_stats {
	uint64_t		rge_tx_ok;
	uint64_t		rge_rx_ok;
	uint64_t		rge_tx_er;
	uint32_t		rge_rx_er;
	uint16_t		rge_miss_pkt;
	uint16_t		rge_fae;
	uint32_t		rge_tx_1col;
	uint32_t		rge_tx_mcol;
	uint64_t		rge_rx_ok_phy;
	uint64_t		rge_rx_ok_brd;
	uint32_t		rge_rx_ok_mul;
	uint16_t		rge_tx_abt;
	uint16_t		rge_tx_undrn;
} __packed __aligned(sizeof(uint64_t));

#define RGE_STATS_ALIGNMENT	64

struct rge_txq {
	struct mbuf		*txq_mbuf;
	bus_dmamap_t		txq_dmamap;
	int			txq_descidx;
};

struct rge_rxq {
	struct mbuf		*rxq_mbuf;
	bus_dmamap_t		rxq_dmamap;
};

struct rge_tx {
	struct rge_txq		rge_txq[RGE_TX_LIST_CNT];
	int			rge_txq_prodidx;
	int			rge_txq_considx;

	bus_dma_segment_t	rge_tx_listseg;
	int			rge_tx_listnseg;
	bus_dmamap_t		rge_tx_list_map;
	struct rge_tx_desc	*rge_tx_list;
};

struct rge_rx {
	struct rge_rxq		rge_rxq[RGE_RX_LIST_CNT];
	int			rge_rxq_prodidx;
	int			rge_rxq_considx;

	struct if_rxring	rge_rx_ring;
	bus_dma_segment_t	rge_rx_listseg;
	int			rge_rx_listnseg;
	bus_dmamap_t		rge_rx_list_map;
	struct rge_rx_desc	*rge_rx_list;

	struct mbuf		*rge_head;
	struct mbuf		*rge_tail;
};

struct rge_queues {
	struct rge_softc	*q_sc;
	void			*q_ihc;
	int			q_index;
	char			q_name[16];
	pci_intr_handle_t	q_ih;
	struct rge_tx		q_tx;
	struct rge_rx		q_rx;
};

/* Microcode version */
#define RGE_MAC_CFG3_MCODE_VER	0x0b33
#define RGE_MAC_CFG5_MCODE_VER	0x0b74

enum rge_mac_type {
	MAC_CFG_UNKNOWN = 1,
	MAC_CFG2,
	MAC_CFG3,
	MAC_CFG4,
	MAC_CFG5
};

#define RGE_TIMEOUT		100

#define RGE_JUMBO_FRAMELEN	9216
#define RGE_JUMBO_MTU							\
	(RGE_JUMBO_FRAMELEN - ETHER_HDR_LEN - ETHER_CRC_LEN - 		\
	ETHER_VLAN_ENCAP_LEN)

#define RGE_TXCFG_CONFIG	0x03000700
#define RGE_RXCFG_CONFIG	0x41c00700
#define RGE_RXCFG_CONFIG_8125B	0x41c00f00

struct kstat;

struct rge_softc {
	struct device		sc_dev;
	struct arpcom		sc_arpcom;	/* Ethernet common data */
	void			*sc_ih;		/* interrupt vectoring */
	bus_space_handle_t	rge_bhandle;	/* bus space handle */
	bus_space_tag_t		rge_btag;	/* bus space tag */
	bus_size_t		rge_bsize;
	bus_dma_tag_t		sc_dmat;
	pci_chipset_tag_t	sc_pc;
	pcitag_t		sc_tag;
	struct ifmedia		sc_media;	/* media info */
	enum rge_mac_type	rge_type;

	struct rge_queues	*sc_queues;
	unsigned int		sc_nqueues;

	struct task		sc_task;

	struct timeout		sc_timeout;	/* tick timeout */

	uint16_t		rge_mcodever;
	uint32_t		rge_flags;
#define RGE_FLAG_MSI		0x00000001

	uint32_t		rge_intrs;
	int			rge_timerintr;
#define RGE_IMTYPE_NONE		0
#define RGE_IMTYPE_SIM		1

	struct kstat		*sc_kstat;
};

/*
 * Register space access macros.
 */
#define RGE_WRITE_4(sc, reg, val)	\
	bus_space_write_4(sc->rge_btag, sc->rge_bhandle, reg, val)
#define RGE_WRITE_2(sc, reg, val)	\
	bus_space_write_2(sc->rge_btag, sc->rge_bhandle, reg, val)
#define RGE_WRITE_1(sc, reg, val)	\
	bus_space_write_1(sc->rge_btag, sc->rge_bhandle, reg, val)

#define RGE_READ_4(sc, reg)		\
	bus_space_read_4(sc->rge_btag, sc->rge_bhandle, reg)
#define RGE_READ_2(sc, reg)		\
	bus_space_read_2(sc->rge_btag, sc->rge_bhandle, reg)
#define RGE_READ_1(sc, reg)		\
	bus_space_read_1(sc->rge_btag, sc->rge_bhandle, reg)

#define RGE_SETBIT_4(sc, reg, val)	\
	RGE_WRITE_4(sc, reg, RGE_READ_4(sc, reg) | (val))
#define RGE_SETBIT_2(sc, reg, val)	\
	RGE_WRITE_2(sc, reg, RGE_READ_2(sc, reg) | (val))
#define RGE_SETBIT_1(sc, reg, val)	\
	RGE_WRITE_1(sc, reg, RGE_READ_1(sc, reg) | (val))

#define RGE_CLRBIT_4(sc, reg, val)	\
	RGE_WRITE_4(sc, reg, RGE_READ_4(sc, reg) & ~(val))
#define RGE_CLRBIT_2(sc, reg, val)	\
	RGE_WRITE_2(sc, reg, RGE_READ_2(sc, reg) & ~(val))
#define RGE_CLRBIT_1(sc, reg, val)	\
	RGE_WRITE_1(sc, reg, RGE_READ_1(sc, reg) & ~(val))

#define RGE_EPHY_SETBIT(sc, reg, val)	\
	rge_write_ephy(sc, reg, rge_read_ephy(sc, reg) | (val))

#define RGE_EPHY_CLRBIT(sc, reg, val)	\
	rge_write_ephy(sc, reg, rge_read_ephy(sc, reg) & ~(val))

#define RGE_PHY_SETBIT(sc, reg, val)	\
	rge_write_phy_ocp(sc, reg, rge_read_phy_ocp(sc, reg) | (val))

#define RGE_PHY_CLRBIT(sc, reg, val)	\
	rge_write_phy_ocp(sc, reg, rge_read_phy_ocp(sc, reg) & ~(val))

#define RGE_MAC_SETBIT(sc, reg, val)	\
	rge_write_mac_ocp(sc, reg, rge_read_mac_ocp(sc, reg) | (val))

#define RGE_MAC_CLRBIT(sc, reg, val)	\
	rge_write_mac_ocp(sc, reg, rge_read_mac_ocp(sc, reg) & ~(val))

static const struct {
	uint16_t	reg;
	uint16_t	val;
} rtl8125_mac_bps[] = {
	{ 0xf800, 0xe010 }, { 0xf802, 0xe012 }, { 0xf804, 0xe022 },
	{ 0xf806, 0xe024 }, { 0xf808, 0xe029 }, { 0xf80a, 0xe02b },
	{ 0xf80c, 0xe094 }, { 0xf80e, 0xe09d }, { 0xf810, 0xe09f },
	{ 0xf812, 0xe0aa }, { 0xf814, 0xe0b5 }, { 0xf816, 0xe0c6 },
	{ 0xf818, 0xe0cc }, { 0xf81a, 0xe0d1 }, { 0xf81c, 0xe0d6 },
	{ 0xf81e, 0xe0d8 }, { 0xf820, 0xc602 }, { 0xf822, 0xbe00 },
	{ 0xf824, 0x0000 }, { 0xf826, 0xc60f }, { 0xf828, 0x73c4 },
	{ 0xf82a, 0x49b3 }, { 0xf82c, 0xf106 }, { 0xf82e, 0x73c2 },
	{ 0xf830, 0xc608 }, { 0xf832, 0xb406 }, { 0xf834, 0xc609 },
	{ 0xf836, 0xff80 }, { 0xf838, 0xc605 }, { 0xf83a, 0xb406 },
	{ 0xf83c, 0xc605 }, { 0xf83e, 0xff80 }, { 0xf840, 0x0544 },
	{ 0xf842, 0x0568 }, { 0xf844, 0xe906 }, { 0xf846, 0xcde8 },
	{ 0xf848, 0xc602 }, { 0xf84a, 0xbe00 }, { 0xf84c, 0x0000 },
	{ 0xf84e, 0x48c1 }, { 0xf850, 0x48c2 }, { 0xf852, 0x9c46 },
	{ 0xf854, 0xc402 }, { 0xf856, 0xbc00 }, { 0xf858, 0x0a12 },
	{ 0xf85a, 0xc602 }, { 0xf85c, 0xbe00 }, { 0xf85e, 0x0eba },
	{ 0xf860, 0x1501 }, { 0xf862, 0xf02a }, { 0xf864, 0x1500 },
	{ 0xf866, 0xf15d }, { 0xf868, 0xc661 }, { 0xf86a, 0x75c8 },
	{ 0xf86c, 0x49d5 }, { 0xf86e, 0xf00a }, { 0xf870, 0x49d6 },
	{ 0xf872, 0xf008 }, { 0xf874, 0x49d7 }, { 0xf876, 0xf006 },
	{ 0xf878, 0x49d8 }, { 0xf87a, 0xf004 }, { 0xf87c, 0x75d2 },
	{ 0xf87e, 0x49d9 }, { 0xf880, 0xf150 }, { 0xf882, 0xc553 },
	{ 0xf884, 0x77a0 }, { 0xf886, 0x75c8 }, { 0xf888, 0x4855 },
	{ 0xf88a, 0x4856 }, { 0xf88c, 0x4857 }, { 0xf88e, 0x4858 },
	{ 0xf890, 0x48da }, { 0xf892, 0x48db }, { 0xf894, 0x49fe },
	{ 0xf896, 0xf002 }, { 0xf898, 0x485a }, { 0xf89a, 0x49ff },
	{ 0xf89c, 0xf002 }, { 0xf89e, 0x485b }, { 0xf8a0, 0x9dc8 },
	{ 0xf8a2, 0x75d2 }, { 0xf8a4, 0x4859 }, { 0xf8a6, 0x9dd2 },
	{ 0xf8a8, 0xc643 }, { 0xf8aa, 0x75c0 }, { 0xf8ac, 0x49d4 },
	{ 0xf8ae, 0xf033 }, { 0xf8b0, 0x49d0 }, { 0xf8b2, 0xf137 },
	{ 0xf8b4, 0xe030 }, { 0xf8b6, 0xc63a }, { 0xf8b8, 0x75c8 },
	{ 0xf8ba, 0x49d5 }, { 0xf8bc, 0xf00e }, { 0xf8be, 0x49d6 },
	{ 0xf8c0, 0xf00c }, { 0xf8c2, 0x49d7 }, { 0xf8c4, 0xf00a },
	{ 0xf8c6, 0x49d8 }, { 0xf8c8, 0xf008 }, { 0xf8ca, 0x75d2 },
	{ 0xf8cc, 0x49d9 }, { 0xf8ce, 0xf005 }, { 0xf8d0, 0xc62e },
	{ 0xf8d2, 0x75c0 }, { 0xf8d4, 0x49d7 }, { 0xf8d6, 0xf125 },
	{ 0xf8d8, 0xc528 }, { 0xf8da, 0x77a0 }, { 0xf8dc, 0xc627 },
	{ 0xf8de, 0x75c8 }, { 0xf8e0, 0x4855 }, { 0xf8e2, 0x4856 },
	{ 0xf8e4, 0x4857 }, { 0xf8e6, 0x4858 }, { 0xf8e8, 0x48da },
	{ 0xf8ea, 0x48db }, { 0xf8ec, 0x49fe }, { 0xf8ee, 0xf002 },
	{ 0xf8f0, 0x485a }, { 0xf8f2, 0x49ff }, { 0xf8f4, 0xf002 },
	{ 0xf8f6, 0x485b }, { 0xf8f8, 0x9dc8 }, { 0xf8fa, 0x75d2 },
	{ 0xf8fc, 0x4859 }, { 0xf8fe, 0x9dd2 }, { 0xf900, 0xc616 },
	{ 0xf902, 0x75c0 }, { 0xf904, 0x4857 }, { 0xf906, 0x9dc0 },
	{ 0xf908, 0xc613 }, { 0xf90a, 0x75c0 }, { 0xf90c, 0x49da },
	{ 0xf90e, 0xf003 }, { 0xf910, 0x49d0 }, { 0xf912, 0xf107 },
	{ 0xf914, 0xc60b }, { 0xf916, 0xc50e }, { 0xf918, 0x48d9 },
	{ 0xf91a, 0x9dc0 }, { 0xf91c, 0x4859 }, { 0xf91e, 0x9dc0 },
	{ 0xf920, 0xc608 }, { 0xf922, 0xc702 }, { 0xf924, 0xbf00 },
	{ 0xf926, 0x3ae0 }, { 0xf928, 0xe860 }, { 0xf92a, 0xb400 },
	{ 0xf92c, 0xb5d4 }, { 0xf92e, 0xe908 }, { 0xf930, 0xe86c },
	{ 0xf932, 0x1200 }, { 0xf934, 0xc409 }, { 0xf936, 0x6780 },
	{ 0xf938, 0x48f1 }, { 0xf93a, 0x8f80 }, { 0xf93c, 0xc404 },
	{ 0xf93e, 0xc602 }, { 0xf940, 0xbe00 }, { 0xf942, 0x10aa },
	{ 0xf944, 0xc010 }, { 0xf946, 0xea7c }, { 0xf948, 0xc602 },
	{ 0xf94a, 0xbe00 }, { 0xf94c, 0x0000 }, { 0xf94e, 0x740a },
	{ 0xf950, 0x4846 }, { 0xf952, 0x4847 }, { 0xf954, 0x9c0a },
	{ 0xf956, 0xc607 }, { 0xf958, 0x74c0 }, { 0xf95a, 0x48c6 },
	{ 0xf95c, 0x9cc0 }, { 0xf95e, 0xc602 }, { 0xf960, 0xbe00 },
	{ 0xf962, 0x13fe }, { 0xf964, 0xe054 }, { 0xf966, 0x72ca },
	{ 0xf968, 0x4826 }, { 0xf96a, 0x4827 }, { 0xf96c, 0x9aca },
	{ 0xf96e, 0xc607 }, { 0xf970, 0x72c0 }, { 0xf972, 0x48a6 },
	{ 0xf974, 0x9ac0 }, { 0xf976, 0xc602 }, { 0xf978, 0xbe00 },
	{ 0xf97a, 0x07dc }, { 0xf97c, 0xe054 }, { 0xf97e, 0xc60f },
	{ 0xf980, 0x74c4 }, { 0xf982, 0x49cc }, { 0xf984, 0xf109 },
	{ 0xf986, 0xc60c }, { 0xf988, 0x74ca }, { 0xf98a, 0x48c7 },
	{ 0xf98c, 0x9cca }, { 0xf98e, 0xc609 }, { 0xf990, 0x74c0 },
	{ 0xf992, 0x4846 }, { 0xf994, 0x9cc0 }, { 0xf996, 0xc602 },
	{ 0xf998, 0xbe00 }, { 0xf99a, 0x2480 }, { 0xf99c, 0xe092 },
	{ 0xf99e, 0xe0c0 }, { 0xf9a0, 0xe054 }, { 0xf9a2, 0x7420 },
	{ 0xf9a4, 0x48c0 }, { 0xf9a6, 0x9c20 }, { 0xf9a8, 0x7444 },
	{ 0xf9aa, 0xc602 }, { 0xf9ac, 0xbe00 }, { 0xf9ae, 0x12f8 },
	{ 0xf9b0, 0x1bff }, { 0xf9b2, 0x46eb }, { 0xf9b4, 0x1bff },
	{ 0xf9b6, 0xc102 }, { 0xf9b8, 0xb900 }, { 0xf9ba, 0x0d5a },
	{ 0xf9bc, 0x1bff }, { 0xf9be, 0x46eb }, { 0xf9c0, 0x1bff },
	{ 0xf9c2, 0xc102 }, { 0xf9c4, 0xb900 }, { 0xf9c6, 0x0e2a },
	{ 0xf9c8, 0xc602 }, { 0xf9ca, 0xbe00 }, { 0xf9cc, 0x0000 },
	{ 0xf9ce, 0xc602 }, { 0xf9d0, 0xbe00 }, { 0xf9d2, 0x0000 },
	{ 0xf9d4, 0x0000 }, { 0xf9d6, 0x0000 }, { 0xf9d8, 0x0000 },
	{ 0xf9da, 0x0000 }, { 0xf9dc, 0x0000 }, { 0xf9de, 0x0000 },
	{ 0xf9e0, 0x0000 }, { 0xf9e2, 0x0000 }, { 0xf9e4, 0x0000 },
	{ 0xf9e6, 0x0000 }, { 0xf9e8, 0x0000 }, { 0xf9ea, 0x0000 },
	{ 0xf9ec, 0x0000 }, { 0xf9ee, 0x0000 }, { 0xf9f0, 0x0000 },
	{ 0xf9f2, 0x0000 }, { 0xf9f4, 0x0000 }, { 0xf9f6, 0x0000 },
	{ 0xf9f8, 0x0000 }, { 0xf9fa, 0x0000 }, { 0xf9fc, 0x0000 },
	{ 0xf9fe, 0x0000 }
}, rtl8125b_mac_bps[] = {
	{ 0xf800, 0xe010 }, { 0xf802, 0xe01b }, { 0xf804, 0xe026 },
	{ 0xf806, 0xe037 }, { 0xf808, 0xe03d }, { 0xf80a, 0xe057 },
	{ 0xf80c, 0xe05b }, { 0xf80e, 0xe060 }, { 0xf810, 0xe062 },
	{ 0xf812, 0xe064 }, { 0xf814, 0xe066 }, { 0xf816, 0xe068 },
	{ 0xf818, 0xe06a }, { 0xf81a, 0xe06c }, { 0xf81c, 0xe06e },
	{ 0xf81e, 0xe070 }, { 0xf820, 0x740a }, { 0xf822, 0x4846 },
	{ 0xf824, 0x4847 }, { 0xf826, 0x9c0a }, { 0xf828, 0xc607 },
	{ 0xf82a, 0x74c0 }, { 0xf82c, 0x48c6 }, { 0xf82e, 0x9cc0 },
	{ 0xf830, 0xc602 }, { 0xf832, 0xbe00 }, { 0xf834, 0x13f0 },
	{ 0xf836, 0xe054 }, { 0xf838, 0x72ca }, { 0xf83a, 0x4826 },
	{ 0xf83c, 0x4827 }, { 0xf83e, 0x9aca }, { 0xf840, 0xc607 },
	{ 0xf842, 0x72c0 }, { 0xf844, 0x48a6 }, { 0xf846, 0x9ac0 },
	{ 0xf848, 0xc602 }, { 0xf84a, 0xbe00 }, { 0xf84c, 0x081c },
	{ 0xf84e, 0xe054 }, { 0xf850, 0xc60f }, { 0xf852, 0x74c4 },
	{ 0xf854, 0x49cc }, { 0xf856, 0xf109 }, { 0xf858, 0xc60c },
	{ 0xf85a, 0x74ca }, { 0xf85c, 0x48c7 }, { 0xf85e, 0x9cca },
	{ 0xf860, 0xc609 }, { 0xf862, 0x74c0 }, { 0xf864, 0x4846 },
	{ 0xf866, 0x9cc0 }, { 0xf868, 0xc602 }, { 0xf86a, 0xbe00 },
	{ 0xf86c, 0x2494 }, { 0xf86e, 0xe092 }, { 0xf870, 0xe0c0 },
	{ 0xf872, 0xe054 }, { 0xf874, 0x7420 }, { 0xf876, 0x48c0 },
	{ 0xf878, 0x9c20 }, { 0xf87a, 0x7444 }, { 0xf87c, 0xc602 },
	{ 0xf87e, 0xbe00 }, { 0xf880, 0x12dc }, { 0xf882, 0x733a },
	{ 0xf884, 0x21b5 }, { 0xf886, 0x25bc }, { 0xf888, 0x1304 },
	{ 0xf88a, 0xf111 }, { 0xf88c, 0x1b12 }, { 0xf88e, 0x1d2a },
	{ 0xf890, 0x3168 }, { 0xf892, 0x3ada }, { 0xf894, 0x31ab },
	{ 0xf896, 0x1a00 }, { 0xf898, 0x9ac0 }, { 0xf89a, 0x1300 },
	{ 0xf89c, 0xf1fb }, { 0xf89e, 0x7620 }, { 0xf8a0, 0x236e },
	{ 0xf8a2, 0x276f }, { 0xf8a4, 0x1a3c }, { 0xf8a6, 0x22a1 },
	{ 0xf8a8, 0x41b5 }, { 0xf8aa, 0x9ee2 }, { 0xf8ac, 0x76e4 },
	{ 0xf8ae, 0x486f }, { 0xf8b0, 0x9ee4 }, { 0xf8b2, 0xc602 },
	{ 0xf8b4, 0xbe00 }, { 0xf8b6, 0x4a26 }, { 0xf8b8, 0x733a },
	{ 0xf8ba, 0x49bb }, { 0xf8bc, 0xc602 }, { 0xf8be, 0xbe00 },
	{ 0xf8c0, 0x47a2 }, { 0xf8c2, 0x48c1 }, { 0xf8c4, 0x48c2 },
	{ 0xf8c6, 0x9c46 }, { 0xf8c8, 0xc402 }, { 0xf8ca, 0xbc00 },
	{ 0xf8cc, 0x0a52 }, { 0xf8ce, 0xc602 }, { 0xf8d0, 0xbe00 },
	{ 0xf8d2, 0x0000 }, { 0xf8d4, 0xc602 }, { 0xf8d6, 0xbe00 },
	{ 0xf8d8, 0x0000 }, { 0xf8da, 0xc602 }, { 0xf8dc, 0xbe00 },
	{ 0xf8de, 0x0000 }, { 0xf8e0, 0xc602 }, { 0xf8e2, 0xbe00 },
	{ 0xf8e4, 0x0000 }, { 0xf8e6, 0xc602 }, { 0xf8e8, 0xbe00 },
	{ 0xf8ea, 0x0000 }, { 0xf8ec, 0xc602 }, { 0xf8ee, 0xbe00 },
	{ 0xf8f0, 0x0000 }, { 0xf8f2, 0xc602 }, { 0xf8f4, 0xbe00 },
	{ 0xf8f6, 0x0000 }, { 0xf8f8, 0xc602 }, { 0xf8fa, 0xbe00 },
	{ 0xf8fc, 0x0000 }, { 0xf8fe, 0xc602 }, { 0xf900, 0xbe00 },
	{ 0xf902, 0x0000 }, { 0xfc26, 0x8000 }, { 0xfc28, 0x13e6 },
	{ 0xfc2a, 0x0812 }, { 0xfc2c, 0x248c }, { 0xfc2e, 0x12da },
	{ 0xfc30, 0x4a20 }, { 0xfc32, 0x47a0 }, { 0xfc48, 0x003f }
};

static const struct {
	uint16_t	reg;
	uint16_t	val;
} rtl8125_mac_cfg3_ephy[] = {
	{ 0x0004, 0xd000 }, { 0x000a, 0x8653 }, { 0x0023, 0xab66 },
	{ 0x0020, 0x9455 }, { 0x0021, 0x99ff }, { 0x0029, 0xfe04 },
	{ 0x0044, 0xd000 }, { 0x004a, 0x8653 }, { 0x0063, 0xab66 },
	{ 0x0060, 0x9455 }, { 0x0061, 0x99ff }, { 0x0069, 0xfe04 }
}, rtl8125_mac_cfg5_ephy[] = {
	{ 0x000b, 0xa908 }, { 0x001e, 0x20eb }, { 0x0022, 0x0023 },
	{ 0x0002, 0x60c2 }, { 0x0029, 0xff00 }, { 0x004b, 0xa908 },
	{ 0x005e, 0x28eb }, { 0x0062, 0x0023 }, { 0x0042, 0x60c2 },
	{ 0x0069, 0xff00 }
};

#define RTL8125_MAC_CFG3_MCU	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x808f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8093 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8097 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x809d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80a1 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80aa },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x607b },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0xf00e },	\
	{ 0xa438, 0x42da },	\
	{ 0xa438, 0xf01e },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x615b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf01c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf024 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf02c },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14a4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14bc },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5f2e },	\
	{ 0xa438, 0xf034 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac11 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa410 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4779 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf034 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac22 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa420 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4559 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf023 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac44 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa440 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4339 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf012 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4118 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac88 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa480 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x4119 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xac0f },	\
	{ 0xa438, 0xae01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1444 },	\
	{ 0xa438, 0xf001 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1456 },	\
	{ 0xa438, 0xd718 },	\
	{ 0xa438, 0x5fac },	\
	{ 0xa438, 0xc48f },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x141b },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x121a },	\
	{ 0xa438, 0xd0b4 },	\
	{ 0xa438, 0xd1bb },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0898 },	\
	{ 0xa438, 0xd0b4 },	\
	{ 0xa438, 0xd1bb },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a0e },	\
	{ 0xa438, 0xd064 },	\
	{ 0xa438, 0xd18a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0b7e },	\
	{ 0xa438, 0x401c },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xa804 },	\
	{ 0xa438, 0x8804 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x053b },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0648 },	\
	{ 0xa438, 0xc520 },	\
	{ 0xa438, 0xa201 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x252d },	\
	{ 0xa438, 0x1646 },	\
	{ 0xa438, 0xd708 },	\
	{ 0xa438, 0x4006 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1646 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0308 },	\
	{ 0xa436, 0xa026 },	\
	{ 0xa438, 0x0307 },	\
	{ 0xa436, 0xa024 },	\
	{ 0xa438, 0x1645 },	\
	{ 0xa436, 0xa022 },	\
	{ 0xa438, 0x0647 },	\
	{ 0xa436, 0xa020 },	\
	{ 0xa438, 0x053a },	\
	{ 0xa436, 0xa006 },	\
	{ 0xa438, 0x0b7c },	\
	{ 0xa436, 0xa004 },	\
	{ 0xa438, 0x0a0c },	\
	{ 0xa436, 0xa002 },	\
	{ 0xa438, 0x0896 },	\
	{ 0xa436, 0xa000 },	\
	{ 0xa438, 0x11a1 },	\
	{ 0xa436, 0xa008 },	\
	{ 0xa438, 0xff00 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0010 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8015 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0xad02 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x02d7 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00ed },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0509 },	\
	{ 0xa438, 0xc100 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x008f },	\
	{ 0xa436, 0xa08e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa088 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa086 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa084 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa082 },	\
	{ 0xa438, 0x008d },	\
	{ 0xa436, 0xa080 },	\
	{ 0xa438, 0x00eb },	\
	{ 0xa436, 0xa090 },	\
	{ 0xa438, 0x0103 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8018 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8024 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8051 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8055 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8072 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80dc },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0xfffd },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0xfffd },	\
	{ 0xa438, 0x8301 },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0xa70c },	\
	{ 0xa438, 0x9402 },	\
	{ 0xa438, 0x890c },	\
	{ 0xa438, 0x8840 },	\
	{ 0xa438, 0xa380 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xcb91 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd139 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd140 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xb404 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e0 },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xa110 },	\
	{ 0xa438, 0xa2a0 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4085 },	\
	{ 0xa438, 0xa180 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x8280 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07f0 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5f74 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb6 },	\
	{ 0xa438, 0x8190 },	\
	{ 0xa438, 0x82a0 },	\
	{ 0xa438, 0x8404 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x0c0f },	\
	{ 0xa438, 0x0d01 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e0 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xd158 },	\
	{ 0xa438, 0xd04d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x03d4 },	\
	{ 0xa438, 0x94bc },	\
	{ 0xa438, 0x870c },	\
	{ 0xa438, 0x8380 },	\
	{ 0xa438, 0xd10d },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07c4 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xa190 },	\
	{ 0xa438, 0xa00a },	\
	{ 0xa438, 0xa280 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0xa220 },	\
	{ 0xa438, 0xd130 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07c4 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0xbb80 },	\
	{ 0xa438, 0xd1c4 },	\
	{ 0xa438, 0xd074 },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x604b },	\
	{ 0xa438, 0xa90c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0556 },	\
	{ 0xa438, 0xcb92 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x4063 },	\
	{ 0xa438, 0xd116 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd119 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x60a0 },	\
	{ 0xa438, 0x6241 },	\
	{ 0xa438, 0x63e2 },	\
	{ 0xa438, 0x6583 },	\
	{ 0xa438, 0xf054 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d10 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf02f },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d50 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf02a },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d20 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf021 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d60 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf01c },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d30 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf013 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d70 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0xf00e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x611e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x40da },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d40 },	\
	{ 0xa438, 0xa010 },	\
	{ 0xa438, 0x8740 },	\
	{ 0xa438, 0xf005 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d80 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e8 },	\
	{ 0xa438, 0xa610 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x405d },	\
	{ 0xa438, 0xa720 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5ff4 },	\
	{ 0xa438, 0xa008 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x4046 },	\
	{ 0xa438, 0xa002 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07fb },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x7f6f },	\
	{ 0xa438, 0x7f4e },	\
	{ 0xa438, 0x7f2d },	\
	{ 0xa438, 0x7f0c },	\
	{ 0xa438, 0x800a },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0d00 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x07e8 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0xa740 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0743 },	\
	{ 0xa438, 0xd702 },	\
	{ 0xa438, 0x7fb5 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x3ad4 },	\
	{ 0xa438, 0x0556 },	\
	{ 0xa438, 0x8610 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x066e },	\
	{ 0xa438, 0xd1f5 },	\
	{ 0xa438, 0xd049 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x01ec },	\
	{ 0xa436, 0xa10e },	\
	{ 0xa438, 0x01ea },	\
	{ 0xa436, 0xa10c },	\
	{ 0xa438, 0x06a9 },	\
	{ 0xa436, 0xa10a },	\
	{ 0xa438, 0x078a },	\
	{ 0xa436, 0xa108 },	\
	{ 0xa438, 0x03d2 },	\
	{ 0xa436, 0xa106 },	\
	{ 0xa438, 0x067f },	\
	{ 0xa436, 0xa104 },	\
	{ 0xa438, 0x0665 },	\
	{ 0xa436, 0xa102 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa100 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa110 },	\
	{ 0xa438, 0x00fc },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8530 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x3caf },	\
	{ 0xa438, 0x8545 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x45af },	\
	{ 0xa438, 0x8545 },	\
	{ 0xa438, 0xee82 },	\
	{ 0xa438, 0xf900 },	\
	{ 0xa438, 0x0103 },	\
	{ 0xa438, 0xaf03 },	\
	{ 0xa438, 0xb7f8 },	\
	{ 0xa438, 0xe0a6 },	\
	{ 0xa438, 0x00e1 },	\
	{ 0xa438, 0xa601 },	\
	{ 0xa438, 0xef01 },	\
	{ 0xa438, 0x58f0 },	\
	{ 0xa438, 0xa080 },	\
	{ 0xa438, 0x37a1 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0xae16 },	\
	{ 0xa438, 0xa185 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x11a1 },	\
	{ 0xa438, 0x8702 },	\
	{ 0xa438, 0xae0c },	\
	{ 0xa438, 0xa188 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x07a1 },	\
	{ 0xa438, 0x8902 },	\
	{ 0xa438, 0xae02 },	\
	{ 0xa438, 0xae1c },	\
	{ 0xa438, 0xe0b4 },	\
	{ 0xa438, 0x62e1 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0x6901 },	\
	{ 0xa438, 0xe4b4 },	\
	{ 0xa438, 0x62e5 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0xe0b4 },	\
	{ 0xa438, 0x62e1 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0x6901 },	\
	{ 0xa438, 0xe4b4 },	\
	{ 0xa438, 0x62e5 },	\
	{ 0xa438, 0xb463 },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa436, 0xb85e },	\
	{ 0xa438, 0x03b3 },	\
	{ 0xa436, 0xb860 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb862 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb864 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb878 },	\
	{ 0xa438, 0x0001 }

#define RTL8125_MAC_CFG5_MCU	\
	{ 0xa436, 0x8024 },	\
	{ 0xa438, 0x3701 },	\
	{ 0xa436, 0xb82e },	\
	{ 0xa438, 0x0001 },	\
	{ 0xb820, 0x0090 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803f },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8045 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8067 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x806d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8071 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x80b1 },	\
	{ 0xa438, 0xd093 },	\
	{ 0xa438, 0xd1c4 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x5fbc },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0xc9f1 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0fc9 },	\
	{ 0xa438, 0xbb50 },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0xa202 },	\
	{ 0xa438, 0xd504 },	\
	{ 0xa438, 0x8c0f },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x1519 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x5fae },	\
	{ 0xa438, 0x9b50 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x7fae },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd707 },	\
	{ 0xa438, 0x40a7 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x4071 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1557 },	\
	{ 0xa438, 0xd719 },	\
	{ 0xa438, 0x2f70 },	\
	{ 0xa438, 0x803b },	\
	{ 0xa438, 0x2f73 },	\
	{ 0xa438, 0x156a },	\
	{ 0xa438, 0x5e70 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x155d },	\
	{ 0xa438, 0xd505 },	\
	{ 0xa438, 0xa202 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xffed },	\
	{ 0xa438, 0xd709 },	\
	{ 0xa438, 0x4054 },	\
	{ 0xa438, 0xa788 },	\
	{ 0xa438, 0xd70b },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x172a },	\
	{ 0xa438, 0xc0c1 },	\
	{ 0xa438, 0xc0c0 },	\
	{ 0xa438, 0xd05a },	\
	{ 0xa438, 0xd1ba },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x2529 },	\
	{ 0xa438, 0x022a },	\
	{ 0xa438, 0xd0a7 },	\
	{ 0xa438, 0xd1b9 },	\
	{ 0xa438, 0xa208 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x080e },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x408b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a65 },	\
	{ 0xa438, 0xf003 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0a6b },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0920 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0915 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0909 },	\
	{ 0xa438, 0x228f },	\
	{ 0xa438, 0x804e },	\
	{ 0xa438, 0x9801 },	\
	{ 0xa438, 0xd71e },	\
	{ 0xa438, 0x5d61 },	\
	{ 0xa438, 0xd701 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x022a },	\
	{ 0xa438, 0x2005 },	\
	{ 0xa438, 0x091a },	\
	{ 0xa438, 0x3bd9 },	\
	{ 0xa438, 0x0919 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0916 },	\
	{ 0xa438, 0xd090 },	\
	{ 0xa438, 0xd1c9 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1064 },	\
	{ 0xa438, 0xd096 },	\
	{ 0xa438, 0xd1a9 },	\
	{ 0xa438, 0xd503 },	\
	{ 0xa438, 0xa104 },	\
	{ 0xa438, 0x0c07 },	\
	{ 0xa438, 0x0902 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xbc10 },	\
	{ 0xa438, 0xd501 },	\
	{ 0xa438, 0xce01 },	\
	{ 0xa438, 0xa201 },	\
	{ 0xa438, 0x8201 },	\
	{ 0xa438, 0xce00 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xc484 },	\
	{ 0xa438, 0xd503 },	\
	{ 0xa438, 0xcc02 },	\
	{ 0xa438, 0xcd0d },	\
	{ 0xa438, 0xaf01 },	\
	{ 0xa438, 0xd500 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x4371 },	\
	{ 0xa438, 0xbd08 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x5fb3 },	\
	{ 0xa438, 0xd503 },	\
	{ 0xa438, 0xd0f5 },	\
	{ 0xa438, 0xd1c6 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0e50 },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x401c },	\
	{ 0xa438, 0xd0f5 },	\
	{ 0xa438, 0xd1c6 },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0ea0 },	\
	{ 0xa438, 0x401c },	\
	{ 0xa438, 0xd07b },	\
	{ 0xa438, 0xd1c5 },	\
	{ 0xa438, 0x8ef0 },	\
	{ 0xa438, 0x401c },	\
	{ 0xa438, 0x9d08 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x7fb3 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd75e },	\
	{ 0xa438, 0x5fad },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x14c5 },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x3181 },	\
	{ 0xa438, 0x80af },	\
	{ 0xa438, 0x60ad },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x135c },	\
	{ 0xa438, 0xd703 },	\
	{ 0xa438, 0x5fba },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0cc7 },	\
	{ 0xa438, 0xa802 },	\
	{ 0xa438, 0xa301 },	\
	{ 0xa438, 0xa801 },	\
	{ 0xa438, 0xc004 },	\
	{ 0xa438, 0xd710 },	\
	{ 0xa438, 0x4000 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x1e79 },	\
	{ 0xa436, 0xa026 },	\
	{ 0xa438, 0x1e78 },	\
	{ 0xa436, 0xa024 },	\
	{ 0xa438, 0x0c93 },	\
	{ 0xa436, 0xa022 },	\
	{ 0xa438, 0x1062 },	\
	{ 0xa436, 0xa020 },	\
	{ 0xa438, 0x0915 },	\
	{ 0xa436, 0xa006 },	\
	{ 0xa438, 0x020a },	\
	{ 0xa436, 0xa004 },	\
	{ 0xa438, 0x1726 },	\
	{ 0xa436, 0xa002 },	\
	{ 0xa438, 0x1542 },	\
	{ 0xa436, 0xa000 },	\
	{ 0xa438, 0x0fc7 },	\
	{ 0xa436, 0xa008 },	\
	{ 0xa438, 0xff00 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0010 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x801d },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802c },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6090 },	\
	{ 0xa438, 0x60d1 },	\
	{ 0xa438, 0xc95c },	\
	{ 0xa438, 0xf007 },	\
	{ 0xa438, 0x60b1 },	\
	{ 0xa438, 0xc95a },	\
	{ 0xa438, 0xf004 },	\
	{ 0xa438, 0xc956 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xc94e },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x00cd },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6090 },	\
	{ 0xa438, 0x60d1 },	\
	{ 0xa438, 0xc95c },	\
	{ 0xa438, 0xf007 },	\
	{ 0xa438, 0x60b1 },	\
	{ 0xa438, 0xc95a },	\
	{ 0xa438, 0xf004 },	\
	{ 0xa438, 0xc956 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xc94e },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x022a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0132 },	\
	{ 0xa436, 0xa08e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa08a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa088 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa086 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa084 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa082 },	\
	{ 0xa438, 0x012f },	\
	{ 0xa436, 0xa080 },	\
	{ 0xa438, 0x00cc },	\
	{ 0xa436, 0xa090 },	\
	{ 0xa438, 0x0103 },	\
	{ 0xa436, 0xa016 },	\
	{ 0xa438, 0x0020 },	\
	{ 0xa436, 0xa012 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xa014 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8010 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8020 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x802a },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x8035 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803c },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x803c },	\
	{ 0xa438, 0xd107 },	\
	{ 0xa438, 0xd042 },	\
	{ 0xa438, 0xa404 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x09df },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x8280 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x6065 },	\
	{ 0xa438, 0xd125 },	\
	{ 0xa438, 0xf002 },	\
	{ 0xa438, 0xd12b },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x077f },	\
	{ 0xa438, 0x0cf0 },	\
	{ 0xa438, 0x0c50 },	\
	{ 0xa438, 0xd104 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0aa8 },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x0a2e },	\
	{ 0xa438, 0xcb9b },	\
	{ 0xa438, 0xd110 },	\
	{ 0xa438, 0xd040 },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x0b7b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x09df },	\
	{ 0xa438, 0xd700 },	\
	{ 0xa438, 0x5fb4 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x081b },	\
	{ 0xa438, 0x1000 },	\
	{ 0xa438, 0x09df },	\
	{ 0xa438, 0xd704 },	\
	{ 0xa438, 0x7fb8 },	\
	{ 0xa438, 0xa718 },	\
	{ 0xa438, 0x1800 },	\
	{ 0xa438, 0x074e },	\
	{ 0xa436, 0xa10e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa10c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa10a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa108 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xa106 },	\
	{ 0xa438, 0x074d },	\
	{ 0xa436, 0xa104 },	\
	{ 0xa438, 0x0818 },	\
	{ 0xa436, 0xa102 },	\
	{ 0xa438, 0x0a2c },	\
	{ 0xa436, 0xa100 },	\
	{ 0xa438, 0x077e },	\
	{ 0xa436, 0xa110 },	\
	{ 0xa438, 0x000f },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8625 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0x3daf },	\
	{ 0xa438, 0x8689 },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x69af },	\
	{ 0xa438, 0x8887 },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x9caf },	\
	{ 0xa438, 0x889c },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x9caf },	\
	{ 0xa438, 0x889c },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0x49d7 },	\
	{ 0xa438, 0x0040 },	\
	{ 0xa438, 0x0277 },	\
	{ 0xa438, 0x7daf },	\
	{ 0xa438, 0x2727 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7205 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7208 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x71f3 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x71f6 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7229 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x722c },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7217 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x721a },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x721d },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7211 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7220 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7214 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x722f },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7223 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7232 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa438, 0x7226 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfae0 },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x3802 },	\
	{ 0xa438, 0xad27 },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x8830 },	\
	{ 0xa438, 0x1f66 },	\
	{ 0xa438, 0xef65 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x1f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00da },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x2f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00db },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4202 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1b03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4502 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1a03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4802 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x3f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00da },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xbfc2 },	\
	{ 0xa438, 0x4f1a },	\
	{ 0xa438, 0x96f7 },	\
	{ 0xa438, 0x05ee },	\
	{ 0xa438, 0xffd2 },	\
	{ 0xa438, 0x00db },	\
	{ 0xa438, 0xf605 },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4b02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1b03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x4e02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef02 },	\
	{ 0xa438, 0x1a03 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0x0d42 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5102 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xef56 },	\
	{ 0xa438, 0xd020 },	\
	{ 0xa438, 0x1f11 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5402 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5702 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5a02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0xa0ef },	\
	{ 0xa438, 0x0348 },	\
	{ 0xa438, 0x0a28 },	\
	{ 0xa438, 0x05ef },	\
	{ 0xa438, 0x201b },	\
	{ 0xa438, 0x01ad },	\
	{ 0xa438, 0x2735 },	\
	{ 0xa438, 0x1f44 },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x88e1 },	\
	{ 0xa438, 0x8589 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5d02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x8ee1 },	\
	{ 0xa438, 0x858f },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6002 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x94e1 },	\
	{ 0xa438, 0x8595 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6302 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe085 },	\
	{ 0xa438, 0x9ae1 },	\
	{ 0xa438, 0x859b },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6602 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x3cbf },	\
	{ 0xa438, 0x883f },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x9cad },	\
	{ 0xa438, 0x2835 },	\
	{ 0xa438, 0x1f44 },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xf8e1 },	\
	{ 0xa438, 0x8ff9 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x5d02 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xfae1 },	\
	{ 0xa438, 0x8ffb },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6002 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xfce1 },	\
	{ 0xa438, 0x8ffd },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6302 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xfee1 },	\
	{ 0xa438, 0x8fff },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x6602 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf88 },	\
	{ 0xa438, 0x3ce1 },	\
	{ 0xa438, 0x85a1 },	\
	{ 0xa438, 0x1b21 },	\
	{ 0xa438, 0xad37 },	\
	{ 0xa438, 0x341f },	\
	{ 0xa438, 0x44e0 },	\
	{ 0xa438, 0x858a },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x8bbf },	\
	{ 0xa438, 0x885d },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8590 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x91bf },	\
	{ 0xa438, 0x8860 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8596 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x97bf },	\
	{ 0xa438, 0x8863 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x859c },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x9dbf },	\
	{ 0xa438, 0x8866 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7dae },	\
	{ 0xa438, 0x401f },	\
	{ 0xa438, 0x44e0 },	\
	{ 0xa438, 0x858c },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x8dbf },	\
	{ 0xa438, 0x885d },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8592 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x93bf },	\
	{ 0xa438, 0x8860 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x8598 },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x99bf },	\
	{ 0xa438, 0x8863 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7de0 },	\
	{ 0xa438, 0x859e },	\
	{ 0xa438, 0xe185 },	\
	{ 0xa438, 0x9fbf },	\
	{ 0xa438, 0x8866 },	\
	{ 0xa438, 0x026e },	\
	{ 0xa438, 0x7dae },	\
	{ 0xa438, 0x0ce1 },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x3904 },	\
	{ 0xa438, 0xac2f },	\
	{ 0xa438, 0x04ee },	\
	{ 0xa438, 0x85b3 },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x39d9 },	\
	{ 0xa438, 0x22ac },	\
	{ 0xa438, 0xeaf0 },	\
	{ 0xa438, 0xacf6 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xfaf0 },	\
	{ 0xa438, 0xacf8 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xfcf0 },	\
	{ 0xa438, 0xad00 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xfef0 },	\
	{ 0xa438, 0xacf0 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xf4f0 },	\
	{ 0xa438, 0xacf2 },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xb0f0 },	\
	{ 0xa438, 0xacae },	\
	{ 0xa438, 0xf0ac },	\
	{ 0xa438, 0xacf0 },	\
	{ 0xa438, 0xacaa },	\
	{ 0xa438, 0xa100 },	\
	{ 0xa438, 0x0ce1 },	\
	{ 0xa438, 0x8ff7 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf26 },	\
	{ 0xa438, 0xe9e1 },	\
	{ 0xa438, 0x8ff6 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0x6e7d },	\
	{ 0xa438, 0xaf26 },	\
	{ 0xa438, 0xf520 },	\
	{ 0xa438, 0xac86 },	\
	{ 0xa438, 0xbf88 },	\
	{ 0xa438, 0x3f02 },	\
	{ 0xa438, 0x6e9c },	\
	{ 0xa438, 0xad28 },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x3324 },	\
	{ 0xa438, 0xad38 },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x32e6 },	\
	{ 0xa438, 0xaf32 },	\
	{ 0xa438, 0xfb00 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ff6 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x0705 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ff8 },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x19cc },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ffa },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x28e3 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ffc },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x1047 },	\
	{ 0xa436, 0xb87c },	\
	{ 0xa438, 0x8ffe },	\
	{ 0xa436, 0xb87e },	\
	{ 0xa438, 0x0a45 },	\
	{ 0xa436, 0xb85e },	\
	{ 0xa438, 0x271e },	\
	{ 0xa436, 0xb860 },	\
	{ 0xa438, 0x3846 },	\
	{ 0xa436, 0xb862 },	\
	{ 0xa438, 0x26e6 },	\
	{ 0xa436, 0xb864 },	\
	{ 0xa438, 0x32e3 },	\
	{ 0xa436, 0xb886 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb888 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb88a },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb88c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb838 },	\
	{ 0xa438, 0x000f },	\
	{ 0xb820, 0x0010 },	\
	{ 0xa436, 0x846e },	\
	{ 0xa438, 0xaf84 },	\
	{ 0xa438, 0x86af },	\
	{ 0xa438, 0x8690 },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0xa4af },	\
	{ 0xa438, 0x86a4 },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0xa4af },	\
	{ 0xa438, 0x86a4 },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0xa4af },	\
	{ 0xa438, 0x86a4 },	\
	{ 0xa438, 0xee82 },	\
	{ 0xa438, 0x5f00 },	\
	{ 0xa438, 0x0284 },	\
	{ 0xa438, 0x90af },	\
	{ 0xa438, 0x0441 },	\
	{ 0xa438, 0xf8e0 },	\
	{ 0xa438, 0x8ff3 },	\
	{ 0xa438, 0xa000 },	\
	{ 0xa438, 0x0502 },	\
	{ 0xa438, 0x84a4 },	\
	{ 0xa438, 0xae06 },	\
	{ 0xa438, 0xa001 },	\
	{ 0xa438, 0x0302 },	\
	{ 0xa438, 0x84c8 },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xef59 },	\
	{ 0xa438, 0xe080 },	\
	{ 0xa438, 0x15ad },	\
	{ 0xa438, 0x2702 },	\
	{ 0xa438, 0xae03 },	\
	{ 0xa438, 0xaf84 },	\
	{ 0xa438, 0xc3bf },	\
	{ 0xa438, 0x53ca },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8ad },	\
	{ 0xa438, 0x2807 },	\
	{ 0xa438, 0x0285 },	\
	{ 0xa438, 0x2cee },	\
	{ 0xa438, 0x8ff3 },	\
	{ 0xa438, 0x01ef },	\
	{ 0xa438, 0x95fd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xfaef },	\
	{ 0xa438, 0x69bf },	\
	{ 0xa438, 0x53ca },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8ac },	\
	{ 0xa438, 0x2822 },	\
	{ 0xa438, 0xd480 },	\
	{ 0xa438, 0x00bf },	\
	{ 0xa438, 0x8684 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x8687 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x868a },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x868d },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9ee },	\
	{ 0xa438, 0x8ff3 },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x8526 },	\
	{ 0xa438, 0xe08f },	\
	{ 0xa438, 0xf4e1 },	\
	{ 0xa438, 0x8ff5 },	\
	{ 0xa438, 0xe28f },	\
	{ 0xa438, 0xf6e3 },	\
	{ 0xa438, 0x8ff7 },	\
	{ 0xa438, 0x1b45 },	\
	{ 0xa438, 0xac27 },	\
	{ 0xa438, 0x0eee },	\
	{ 0xa438, 0x8ff4 },	\
	{ 0xa438, 0x00ee },	\
	{ 0xa438, 0x8ff5 },	\
	{ 0xa438, 0x0002 },	\
	{ 0xa438, 0x852c },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x26e0 },	\
	{ 0xa438, 0x8ff4 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf52c },	\
	{ 0xa438, 0x0001 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xf4e5 },	\
	{ 0xa438, 0x8ff5 },	\
	{ 0xa438, 0xef96 },	\
	{ 0xa438, 0xfefd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf8f9 },	\
	{ 0xa438, 0xef59 },	\
	{ 0xa438, 0xbf53 },	\
	{ 0xa438, 0x2202 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xa18b },	\
	{ 0xa438, 0x02ae },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x85da },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x7202 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xf8e5 },	\
	{ 0xa438, 0x8ff9 },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x7502 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfae5 },	\
	{ 0xa438, 0x8ffb },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x7802 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfce5 },	\
	{ 0xa438, 0x8ffd },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x7b02 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfee5 },	\
	{ 0xa438, 0x8fff },	\
	{ 0xa438, 0xbf57 },	\
	{ 0xa438, 0x6c02 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xa102 },	\
	{ 0xa438, 0x13ee },	\
	{ 0xa438, 0x8ffc },	\
	{ 0xa438, 0x80ee },	\
	{ 0xa438, 0x8ffd },	\
	{ 0xa438, 0x00ee },	\
	{ 0xa438, 0x8ffe },	\
	{ 0xa438, 0x80ee },	\
	{ 0xa438, 0x8fff },	\
	{ 0xa438, 0x00af },	\
	{ 0xa438, 0x8599 },	\
	{ 0xa438, 0xa101 },	\
	{ 0xa438, 0x0cbf },	\
	{ 0xa438, 0x534c },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8a1 },	\
	{ 0xa438, 0x0303 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0x77bf },	\
	{ 0xa438, 0x5322 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8a1 },	\
	{ 0xa438, 0x8b02 },	\
	{ 0xa438, 0xae03 },	\
	{ 0xa438, 0xaf86 },	\
	{ 0xa438, 0x64e0 },	\
	{ 0xa438, 0x8ff8 },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xf9bf },	\
	{ 0xa438, 0x8684 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9e0 },	\
	{ 0xa438, 0x8ffa },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x8687 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9e0 },	\
	{ 0xa438, 0x8ffc },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xfdbf },	\
	{ 0xa438, 0x868a },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9e0 },	\
	{ 0xa438, 0x8ffe },	\
	{ 0xa438, 0xe18f },	\
	{ 0xa438, 0xffbf },	\
	{ 0xa438, 0x868d },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9af },	\
	{ 0xa438, 0x867f },	\
	{ 0xa438, 0xbf53 },	\
	{ 0xa438, 0x2202 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xa144 },	\
	{ 0xa438, 0x3cbf },	\
	{ 0xa438, 0x547b },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8e4 },	\
	{ 0xa438, 0x8ff8 },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xf9bf },	\
	{ 0xa438, 0x547e },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8e4 },	\
	{ 0xa438, 0x8ffa },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xfbbf },	\
	{ 0xa438, 0x5481 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8e4 },	\
	{ 0xa438, 0x8ffc },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xfdbf },	\
	{ 0xa438, 0x5484 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8e4 },	\
	{ 0xa438, 0x8ffe },	\
	{ 0xa438, 0xe58f },	\
	{ 0xa438, 0xffbf },	\
	{ 0xa438, 0x5322 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8a1 },	\
	{ 0xa438, 0x4448 },	\
	{ 0xa438, 0xaf85 },	\
	{ 0xa438, 0xa7bf },	\
	{ 0xa438, 0x5322 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xc8a1 },	\
	{ 0xa438, 0x313c },	\
	{ 0xa438, 0xbf54 },	\
	{ 0xa438, 0x7b02 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xf8e5 },	\
	{ 0xa438, 0x8ff9 },	\
	{ 0xa438, 0xbf54 },	\
	{ 0xa438, 0x7e02 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfae5 },	\
	{ 0xa438, 0x8ffb },	\
	{ 0xa438, 0xbf54 },	\
	{ 0xa438, 0x8102 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfce5 },	\
	{ 0xa438, 0x8ffd },	\
	{ 0xa438, 0xbf54 },	\
	{ 0xa438, 0x8402 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xe48f },	\
	{ 0xa438, 0xfee5 },	\
	{ 0xa438, 0x8fff },	\
	{ 0xa438, 0xbf53 },	\
	{ 0xa438, 0x2202 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xa131 },	\
	{ 0xa438, 0x03af },	\
	{ 0xa438, 0x85a7 },	\
	{ 0xa438, 0xd480 },	\
	{ 0xa438, 0x00bf },	\
	{ 0xa438, 0x8684 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x8687 },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x868a },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9bf },	\
	{ 0xa438, 0x868d },	\
	{ 0xa438, 0x0252 },	\
	{ 0xa438, 0xa9ef },	\
	{ 0xa438, 0x95fd },	\
	{ 0xa438, 0xfc04 },	\
	{ 0xa438, 0xf0d1 },	\
	{ 0xa438, 0x2af0 },	\
	{ 0xa438, 0xd12c },	\
	{ 0xa438, 0xf0d1 },	\
	{ 0xa438, 0x44f0 },	\
	{ 0xa438, 0xd146 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xa102 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xbf86 },	\
	{ 0xa438, 0xa102 },	\
	{ 0xa438, 0x52c8 },	\
	{ 0xa438, 0xd101 },	\
	{ 0xa438, 0xaf06 },	\
	{ 0xa438, 0xa570 },	\
	{ 0xa438, 0xce42 },	\
	{ 0xa436, 0xb818 },	\
	{ 0xa438, 0x043d },	\
	{ 0xa436, 0xb81a },	\
	{ 0xa438, 0x06a3 },	\
	{ 0xa436, 0xb81c },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb81e },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb850 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb852 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb878 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb884 },	\
	{ 0xa438, 0xffff },	\
	{ 0xa436, 0xb832 },	\
	{ 0xa438, 0x0003 },	\
	{ 0xa436, 0x0000 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0xb82e },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0x8024 },	\
	{ 0xa438, 0x0000 },	\
	{ 0xa436, 0x801e },	\
	{ 0xa438, 0x0021 },	\
	{ 0xb820, 0x0000 }
