/*
 * Copyright (c) 2023, Oracle and/or its affiliates.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "scope.h"
#include "x11.h"
#include "damagescope.h"

/* Print the portion of the damage request header common to all requests */
static inline void
DamageRequestHeader(FD fd, const unsigned char *buf)
{
    PrintField(buf, 0, 1, REQUEST, REQUESTHEADER); /* DamageRequest */
    PrintField(buf, 1, 1, DAMAGEREQUEST, DAMAGEREQUESTHEADER); /* minor op */
    if (Verbose > 1)
        PrintField(SBf, 0, 4, CARD32, "sequence number");
}

void
DamageQueryVersion(FD fd, const unsigned char *buf)
{
    DamageRequestHeader(fd, buf);
    if (Verbose < 1)
        return;

    printreqlen(buf, fd, CONST2(2));
    PrintField(buf, 4, 4, CARD32, "major-version");
    PrintField(buf, 8, 4, CARD32, "minor-version");
}

void
DamageQueryVersionReply(FD fd, const unsigned char *buf)
{
    PrintField(RBf, 0, 1, REPLY, REPLYHEADER); /* DamageRequest reply */
    PrintField(RBf, 1, 1, DAMAGEREPLY,
               DAMAGEREPLYHEADER); /* DamageQueryVersion reply */
    if (Verbose < 1)
        return;
    printfield(buf, 2, 2, CARD16, "sequence number");
    printfield(buf, 4, 4, DVALUE4(0), "reply length");
    PrintField(buf, 8, 4, CARD32, "major-version");
    PrintField(buf, 12, 4, CARD32, "minor-version");
}

void
DamageCreate(FD fd, const unsigned char *buf)
{
    DamageRequestHeader(fd, buf);
    if (Verbose < 1)
        return;

    printreqlen(buf, fd, CONST2(3));
    PrintField(buf, 4, 4, DRAWABLE, "drawable");
    PrintField(buf, 8, 4, DAMAGE, "damage");
    PrintField(buf, 12, 1, DAMAGEREPORTLEVEL, "level");        
}

void
DamageDestroy(FD fd, const unsigned char *buf)
{
    DamageRequestHeader(fd, buf);
    if (Verbose < 1)
        return;

    printreqlen(buf, fd, CONST2(1));
    PrintField(buf, 4, 4, DAMAGE, "damage");
}

void
DamageSubtract(FD fd, const unsigned char *buf)
{
    DamageRequestHeader(fd, buf);
    if (Verbose < 1)
        return;

    printreqlen(buf, fd, CONST2(3));
    PrintField(buf, 4, 4, DAMAGE, "damage");
    PrintField(buf, 8, 4, REGION, "repair");
    PrintField(buf, 12, 4, REGION, "parts");
}

void
DamageAdd(FD fd, const unsigned char *buf)
{
    DamageRequestHeader(fd, buf);
    if (Verbose < 1)
        return;

    printreqlen(buf, fd, CONST2(2));
    PrintField(buf, 4, 4, DRAWABLE, "drawable");
    PrintField(buf, 8, 4, REGION, "region");
}

void
DamageNotifyEvent(const unsigned char *buf)
{
    PrintField(buf, 0, 1, EVENT, EVENTHEADER); /* DamageNotify */
    if (Verbose < 1)
        return;
    PrintField(buf, 1, 1, DAMAGEREPORTLEVEL, "level");
    printfield(buf, 2, 2, CARD16, "sequence number");
    PrintField(buf, 4, 4, DRAWABLE, "drawable");
    PrintField(buf, 8, 4, DAMAGE, "damage");
    PrintField(buf, 12, 4, TIMESTAMP, "timestamp");
    PrintField(buf, 16, 8, RECTANGLE, "area");
    PrintField(buf, 24, 8, RECTANGLE, "geometry");
}

void
DamageBadDamageError(FD fd, const unsigned char *buf)
{
    printErrorWithValue(buf, DAMAGE, "bad damage");
}
