/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <boost/shared_ptr.hpp>

#include <libepubgen/libepubgen.h>

#include <libetonyek/libetonyek.h>

#include "EpubPackage.hxx"

using libetonyek::EtonyekDocument;

class EpubOutputFileHelper
{
public:
	EpubOutputFileHelper(const char *const outFileName) :
		m_fileName(outFileName)
	{
	}

	bool convertDocument(librevenge::RVNGInputStream &input)
	{
		EpubPackage package(m_fileName);
		libepubgen::EPUBPresentationGenerator collector(&package);

		return EtonyekDocument::parse(&input, &collector);
	}

	EtonyekDocument::Confidence isSupportedFormat(librevenge::RVNGInputStream &input, EtonyekDocument::Type *type = 0)
	{
		return EtonyekDocument::isSupported(&input, type);
	}

private:
	std::string m_fileName;
};

int printUsage(char *name)
{
	fprintf(stderr, "USAGE : %s [--stdout] <infile> [outfile]\n", name);
	fprintf(stderr, "USAGE : Where <infile> is the etonyek source document\n");
	fprintf(stderr, "USAGE : and [outfile] is the ePub target document.\n");
	return 1;
}

int main(int argc, char *argv[])
{
	if ((argc < 2) || (argc > 3))
		return printUsage(argv[0]);

	char *szInputFile = argv[1];
	char *szOutFile = 0;

	if (argc > 2)
		szOutFile = argv[2];

	if (!szInputFile)
		return printUsage(argv[0]);

	boost::shared_ptr<librevenge::RVNGInputStream> input;
	if (librevenge::RVNGDirectoryStream::isDirectory(szInputFile))
		input.reset(new librevenge::RVNGDirectoryStream(szInputFile));
	else
		input.reset(new librevenge::RVNGFileStream(szInputFile));

	EpubOutputFileHelper helper(szOutFile);

	EtonyekDocument::Type type = EtonyekDocument::TYPE_UNKNOWN;
	const EtonyekDocument::Confidence confidence = helper.isSupportedFormat(*input, &type);
	if ((EtonyekDocument::CONFIDENCE_NONE == confidence) || (EtonyekDocument::TYPE_KEYNOTE != type))
	{
		fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Pages Document.\n");
		return 1;
	}

	if (EtonyekDocument::CONFIDENCE_SUPPORTED_PART == confidence)
	{
		input.reset(librevenge::RVNGDirectoryStream::createForParent(szInputFile));

		if (EtonyekDocument::CONFIDENCE_EXCELLENT != helper.isSupportedFormat(*input))
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Pages Document.\n");
			return 1;
		}
	}

	if (!helper.convertDocument(*input))
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
