/*
 * DMI.CPP - Contains Desktop Management Interface functions.
 * Copyright (C) 1998, 1999 Prashant TR
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING.TR for more details.
*/

// ID for this file.
#define _DMI_CC_

#include "dmi.h"

FILE *fp;
char cmdline[20];
int errflag = 0;
unsigned char buf1[100], buf2[100];
char output[256];

// Write any string to the file and check for successfulness.
void writestring(const char *string)
{
 if (fprintf(fp, "%s", string) == EOF) {
    errflag = 2;
    checkerrors();
 }
}

PNP *get_pnp_dmi_entry_point(const char *string)
{
	unsigned short segment, offset, f;
	char flag;

	// Search for $PnP/$DMI.
	segment = 0xf000;
	offset = flag = 0;

	do {
		if (!strncmp((char *)MK_FP(segment, offset), string, 4))
		flag++;
		else offset += 0x10;
	} while ((offset) && (!flag));

	// Check if actually PnP if we are searching for $PnP.
	if ((flag) && (!strncmp(string, "$PnP", 4))) {
		flag = 0;
		for(f = 0; f <= 0x20; f++)
		flag += *(char *)MK_FP(segment, offset + f);
		flag = (!flag) ? 1 : 0;
	}
	return ((offset) ? (PNP *)MK_FP(segment, offset) : NULL);
}

char *get_string_number(char *str_ptr, unsigned short index)
{
	if ((!index) || (index == 0xff)) return (NULL);
	index--;
	while (index)
	{
		if (!*(int *)str_ptr) break;
		while (*str_ptr) *str_ptr++;
		if (!*(int *)str_ptr) break;
		*str_ptr++;
		index--;
	}
	if (!*(int *)str_ptr) return (NULL);
	if (!index) return (str_ptr);
	return (NULL);
}

void dmi_slots_info(DMI_INFO *dmi_info, DMI_SLOT *slot_info)
{
	char *str_ptr = (char *)MK_FP(FP_SEG(dmi_info),
					FP_OFF(dmi_info) +
					dmi_info -> strings_offset);
	char *str_result;

	writestring("\nSLOTS INFORMATION (DMI) :\n\n");
	sprintf(output, "\tDMI Handle                    : %04Xh\n",
			dmi_info -> handle);
	writestring(output);

	str_result = get_string_number(str_ptr,
			slot_info -> index_to_socket_designation);
	sprintf(output, "\tSocket Designation            : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	sprintf(output, "\tSlot type                     : %s\n",
			(slot_info -> type == 1) ? "(Other)" :
			(slot_info -> type == 2) ? "(Unknown)" :
			(slot_info -> type == 3) ? "ISA" :
			(slot_info -> type == 4) ? "MCA" :
			(slot_info -> type == 5) ? "EISA" :
			(slot_info -> type == 6) ? "PCI" :
			(slot_info -> type == 7) ? "PCMCIA" :
			(slot_info -> type == 8) ? "VL-VESA" :
			(slot_info -> type == 9) ? "Proprietary" :
			(slot_info -> type == 10) ? "Processor Card" :
			(slot_info -> type == 11) ? "Proprietary Memory Card" :
			(slot_info -> type == 12) ? "I/O Riser Card Slot" :
			(slot_info -> type == 13) ? "NuBus" :
			(slot_info -> type == 14) ? "PCI - 66MHz capable" :
			(slot_info -> type == 160) ? "PC-98/C20" :
			(slot_info -> type == 161) ? "PC-8/C24" :
			(slot_info -> type == 162) ? "PC-98/E" :
			(slot_info -> type == 163) ? "PC-98/Local Bus" :
			(slot_info -> type == 164) ? "PC-98/Card" :
			"(Unknown)");
	writestring(output);

	sprintf(output, "\tSlot Width                    : %s\n",
			(slot_info -> width == 1) ? "(Other)" :
			(slot_info -> width == 2) ? "(Unknown)" :
			(slot_info -> width == 3) ? "8-bit" :
			(slot_info -> width == 4) ? "16-bit" :
			(slot_info -> width == 5) ? "32-bit" :
			(slot_info -> width == 6) ? "64-bit" :
			(slot_info -> width == 7) ? "128-bit" :
			"(Unknown)");
	writestring(output);

	sprintf(output, "\tSlot Current Usage            : %s\n",
			(slot_info -> usage == 1) ? "(Other)" :
			(slot_info -> usage == 2) ? "(Unknown)" :
			(slot_info -> usage == 3) ? "Available" :
			(slot_info -> usage == 4) ? "In use" :
			"(Unknown)");
	writestring(output);

	sprintf(output, "\tSlot Length                   : %s\n",
			(slot_info -> usage == 1) ? "(Other)" :
			(slot_info -> usage == 2) ? "(Unknown)" :
			(slot_info -> usage == 3) ? "Half Length" :
			(slot_info -> usage == 4) ? "Full Length" :
			"(Unknown)");
	writestring(output);

	sprintf(output, "\tSlot Characteristics available: %s\n",
			(!(slot_info -> flags & 1)) ? "Yes" : "No");
	writestring(output);

	if (slot_info -> flags & 1) return;

	sprintf(output, "\tSlot Provides 5.0V            : %s\n",
			(slot_info -> flags & 2) ? "Yes" : "No");
	writestring(output);

	sprintf(output, "\tSlot Provides 3.3V            : %s\n",
			(slot_info -> flags & 4) ? "Yes" : "No");
	writestring(output);

	sprintf(output, "\tSlot opening shared           : %s\n",
			(slot_info -> flags & 8) ? "Yes" : "No");
	writestring(output);
}

void dmi_cache_info(DMI_INFO *dmi_info, DMI_CACHE *cache_info)
{
	char *str_ptr = (char *)MK_FP(FP_SEG(dmi_info),
					FP_OFF(dmi_info) +
					dmi_info -> strings_offset);
	char *str_result;

	writestring("\nCACHE INFORMATION (DMI) :\n\n");
	sprintf(output, "\tDMI Handle                    : %04Xh\n",
			dmi_info -> handle);
	writestring(output);

	str_result = get_string_number(str_ptr,
			cache_info -> index_to_socket_designation);
	sprintf(output, "\tSocket Designation            : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	sprintf(output, "\tCache level                   : Level %d\n",
			cache_info -> flags % 8 + 1);
	writestring(output);

	sprintf(output, "\tCache Socketed                : %s\n",
			(cache_info -> flags & 8) ? "Yes" : "No");
	writestring(output);

	sprintf(output, "\tCache Location                : %s\n",
			(!(cache_info -> flags & 0x60)) ? "Internal" :
			((cache_info -> flags & 0x60) == 0x20) ? "External" :
			((cache_info -> flags & 0x60) == 0x40) ? "(Unknown)" :
			((cache_info -> flags & 0x60) == 0x60) ? "-NA-" :
			"(Unknown)");
	writestring(output);

	sprintf(output, "\tCache Status at Boot time     : %s\n",
			(!(cache_info -> flags & 0x80)) ? "Disabled" :
			"Enabled");
	writestring(output);

	sprintf(output, "\tOperation Mode                : %s\n",
			(!(cache_info -> flags & 0x300)) ? "Write Through" :
			((cache_info -> flags & 0x300) == 0x100) ? "Write Back" :
			((cache_info -> flags & 0x300) == 0x200) ? "Varies with"
							" Memory Adderess" :
			"(Unknown)");
	writestring(output);

	sprintf(output, "\tMaximum Cache size            : %dK\n",
			cache_info -> maximum_size);
	writestring(output);

	sprintf(output, "\tInstalled Cache size          : %dK\n",
			cache_info -> installed_size);
	writestring(output);

	sprintf(output, "\tSupported SRAM type           : ");
	strcat(output, (cache_info -> type & 1) ? "(Other), " : "");
	strcat(output, (cache_info -> type & 2) ? "(Unknown), " : "");
	strcat(output, (cache_info -> type & 4) ? "Non-Burst, " : "");
	strcat(output, (cache_info -> type & 8) ? "Burst, " : "");
	strcat(output, (cache_info -> type & 0x10) ? "Pipelined Burst, " : "");
	strcat(output, (cache_info -> type & 0x20) ? "Synchronous, " : "");
	strcat(output, (cache_info -> type & 0x40) ? "Asynchronous, " : "");
	strcat(output, (!(cache_info -> type & 0x4f)) ? "(Unknown), " : "");

	// Cut off comma.
	output[strlen(output) - 2] = 0;
	strcat(output, "\n");
	writestring(output);

	sprintf(output, "\tCurrent SRAM type             : ");
	strcat(output, (cache_info -> type & 1) ? "(Other), " : "");
	strcat(output, (cache_info -> type & 2) ? "(Unknown), " : "");
	strcat(output, (cache_info -> type & 4) ? "Non-Burst, " : "");
	strcat(output, (cache_info -> type & 8) ? "Burst, " : "");
	strcat(output, (cache_info -> type & 0x10) ? "Pipelined Burst, " : "");
	strcat(output, (cache_info -> type & 0x20) ? "Synchronous, " : "");
	strcat(output, (cache_info -> type & 0x40) ? "Asynchronous, " : "");
	strcat(output, (!(cache_info -> type & 0x7f)) ? "(Unknown), " : "");

	// Cut off comma.
	output[strlen(output) - 2] = 0;
	strcat(output, "\n");
	writestring(output);
}

void dmi_processor_info(DMI_INFO *dmi_info, DMI_PROC *proc_info)
{
	char *str_ptr = (char *)MK_FP(FP_SEG(dmi_info),
					FP_OFF(dmi_info) +
					dmi_info -> strings_offset);
	char *str_result;

	writestring("\nPROCESSOR INFORMATION (DMI) :\n\n");
	sprintf(output, "\tDMI Handle                    : %04Xh\n",
			dmi_info -> handle);
	writestring(output);

	str_result = get_string_number(str_ptr,
			proc_info -> index_to_socket_designation);
	sprintf(output, "\tSocket Designation            : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	sprintf(output, "\tProcessor Type                : %s\n",
			(proc_info -> type == 1) ? "(Other)" :
			(proc_info -> type == 2) ? "(Unknown)" :
			(proc_info -> type == 3) ? "Central Processor" :
			(proc_info -> type == 4) ? "Math Processor" :
			(proc_info -> type == 5) ? "DSP Processor" :
			(proc_info -> type == 6) ? "Video Processor" :
			"(Unknown)");
	writestring(output);

	sprintf(output, "\tProcessor Family              : %s\n",
			(proc_info -> family == 1) ? "(Other)" :
			(proc_info -> family == 2) ? "(Unknown)" :
			(proc_info -> family == 3) ? "8086" :
			(proc_info -> family == 4) ? "80286" :
			(proc_info -> family == 5) ? "80386" :
			(proc_info -> family == 6) ? "80486" :
			(proc_info -> family == 7) ? "8087" :
			(proc_info -> family == 8) ? "80287" :
			(proc_info -> family == 9) ? "80387" :
			(proc_info -> family == 0xa) ? "80487" :
			(proc_info -> family == 0xb) ? "Pentium Family" :
			(proc_info -> family == 0xc) ? "Pentium Pro" :
			(proc_info -> family == 0xd) ? "Pentium II" :
			(proc_info -> family == 0x12) ? "M1 Family" :
			(proc_info -> family == 0x30) ? "Pentium Pro Family" :
			(proc_info -> family == 0xa0) ? "V30 Family" :
			"(Unknown)");
	writestring(output);

	str_result = get_string_number(str_ptr,
			proc_info -> index_to_manufacturer);
	sprintf(output, "\tProcessor Manufacturer        : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	sprintf(output, "\tProcessor ID                  : %08lX%08lX\n",
			proc_info -> id_hi, proc_info -> id_low);
	writestring(output);

	str_result = get_string_number(str_ptr,
			proc_info -> index_to_version);
	sprintf(output, "\tProcessor Version             : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	sprintf(output, "\t5V Capability                 : %s\n",
			(proc_info -> voltage_capability & 1) ? "Yes" : "No");
	writestring(output);

	sprintf(output, "\t3.3V Capability               : %s\n",
			(proc_info -> voltage_capability & 2) ? "Yes" : "No");
	writestring(output);

	sprintf(output, "\t2.9V Capability               : %s\n",
			(proc_info -> voltage_capability & 4) ? "Yes" : "No");
	writestring(output);

	sprintf(output, "\tExternal Clock Frequency      : %uMHz\n",
			proc_info -> external_clock);
	writestring(output);

	sprintf(output, "\tMaximum Clock Frequency       : %uMHz\n",
			proc_info -> maximum_speed);
	writestring(output);

	sprintf(output, "\tCurrent Clock Frequency       : %uMHz\n",
			proc_info -> current_speed);
	writestring(output);

	sprintf(output, "\tCPU Socket Populated          : %s\n",
			(proc_info -> status & 0x40) ? "Yes" : "No");
	writestring(output);

	sprintf(output, "\tCPU Status                    : %s\n",
			(!(proc_info -> status & 7)) ? "(Unknown)" :
			((proc_info -> status & 7) == 1) ? "Enabled" :
			((proc_info -> status & 7) == 2) ? "Disabled by User"
							" via BIOS Setup" :
			((proc_info -> status & 7) == 3) ? "Disabled by System"
							" BIOS (POST Error)" :
			((proc_info -> status & 7) == 4) ? "CPU is Idle (wait"
							"ing to be Enabled)" :
			((proc_info -> status & 7) == 5) ? "(Reserved Setting"
							" by the BIOS)" :
			((proc_info -> status & 7) == 6) ? "(Reserved Setting"
							" by the BIOS)" :
			((proc_info -> status & 7) == 7) ? "(Other)" :
			"(Unknown)");
	writestring(output);

	if (proc_info -> upgrade > 8) {
		str_result = get_string_number(str_ptr,
				proc_info -> upgrade);
		sprintf(output, "\tProcessor Version             : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
		writestring(output);
	}
	else {
		sprintf(output, "\tProcessor Upgrade             : %s\n",
			((proc_info -> upgrade & 7) == 1) ? "(Other)" :
			((proc_info -> upgrade & 7) == 2) ? "(Unknown)" :
			((proc_info -> upgrade & 7) == 3) ? "Daughter Board" :
			((proc_info -> upgrade & 7) == 4) ? "ZIF Socket" :
			((proc_info -> upgrade & 7) == 5) ? "Replacable Piggy Back" :
			((proc_info -> upgrade & 7) == 6) ? "None" :
			((proc_info -> upgrade & 7) == 7) ? "LIF Socket" :
			"(Unknown)");
		writestring(output);
	}
}

void dmi_memory_module_info(DMI_INFO *dmi_info, DMI_MEMM *memm_info)
{
	char *str_ptr = (char *)MK_FP(FP_SEG(dmi_info),
					FP_OFF(dmi_info) +
					dmi_info -> strings_offset);
	char str[10];
	char *str_result;

	writestring("\nMEMORY MODULE(S) INFORMATION (DMI) :\n\n");
	sprintf(output, "\tDMI Handle                    : %04Xh\n",
			dmi_info -> handle);
	writestring(output);

	str_result = get_string_number(str_ptr,
			memm_info -> index_to_socket_designation);
	sprintf(output, "\tSocket Designation            : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	sprintf(output, "\tBank Connection               : ");
	if ((memm_info -> bank_connections >> 4) &&
		(memm_info -> bank_connections >> 4) != 0xf) {
		strcat(output, itoa(memm_info -> bank_connections >> 4,
				str, 10));
		if ((memm_info -> bank_connections & 0xf) != 0xf) {
			strcat(output, " ");
			strcat(output, itoa(memm_info -> bank_connections
							& 0xf,
							str, 10));
		}
	}
	else strcat(output, "None");
	strcat(output, "\n");
	writestring(output);

	sprintf(output, "\tCurrent Speed                 : %uns\n",
				memm_info -> current_speed);
	writestring(output);

	sprintf(output, "\tCurrent Memory Type           : ");
	if (memm_info -> memory_types & 1) strcat(output, "(Other), ");
	if (memm_info -> memory_types & 2) strcat(output, "(Unknown), ");
	if (memm_info -> memory_types & 4) strcat(output, "Standard, ");
	if (memm_info -> memory_types & 8) strcat(output, "Fast Page Mode "
							"(FPM), ");
	if (memm_info -> memory_types & 0x10) strcat(output, "Extended Data"
							" Output (EDO), ");
	if (memm_info -> memory_types & 0x20) strcat(output, "Parity, ");
	if (memm_info -> memory_types & 0x40) strcat(output, "ECC, ");
	if (memm_info -> memory_types & 0x80) strcat(output, "SIMM, ");
	if (memm_info -> memory_types & 0x100) strcat(output, "DIMM, ");
	if (memm_info -> memory_types & 0x200) strcat(output, "Burst EDO, ");
	if (memm_info -> memory_types & 0x400) strcat(output, "Synchronous "
						"Data RAM (SDRAM), ");
	if (memm_info -> memory_types & 0x800) strcat(output, "Reserved Setting"
							" by the BIOS, ");
	if (memm_info -> memory_types & 0x1000) strcat(output, "Reserved Setting"
							" by the BIOS, ");
	if (memm_info -> memory_types & 0x2000) strcat(output, "Reserved Setting"
							" by the BIOS, ");
	if (memm_info -> memory_types & 0x4000) strcat(output, "Reserved Setting"
							" by the BIOS, ");
	if (memm_info -> memory_types & 0x8000) strcat(output, "Reserved Setting"
							" by the BIOS, ");
	if (!memm_info -> memory_types) strcat(output, "(Unknown), ");
	output[strlen(output) - 2] = '\0';
	strcat(output, "\n");
	writestring(output);

	switch (memm_info -> installed_size & 0x7f) {
		case 0x7d:
			strcpy(output, "\tInstalled Size                : "
					"(Not Determinable)\n");
			break;

		case 0x7e:
			strcpy(output, "\tInstalled Size                : "
					"(No Memory Enabled)\n");
			break;

		case 0x7f:
			strcpy(output, "\tInstalled Size                : "
					"(No Memory Installed)\n");
			break;

		default:
			sprintf(output, "\tInstalled Size                : "
					"%luMB, %s\n",
					((unsigned long)2) <<
					(memm_info -> installed_size & 0x7f) / 2,
					(memm_info -> installed_size & 0x10) ?
					"Double-bank" : "Single-bank");
	}
	writestring(output);

	switch (memm_info -> enabled_size & 0x7f) {
		case 0x7d:
			strcpy(output, "\tEnabled Memory Size           : "
					"(Not Determinable)\n");
			break;

		case 0x7e:
			strcpy(output, "\tEnabled Memory Size           : "
					"(No Memory Enabled)\n");
			break;

		case 0x7f:
			strcpy(output, "\tEnabled Memory Size           : "
					"(No Memory Installed)\n");
			break;

		default:
			sprintf(output, "\tEnabled Memory Size           : "
					"%luMB, %s\n",
					((unsigned long)2) <<
					(memm_info -> installed_size & 0x7f) / 2,
					(memm_info -> installed_size & 0x10) ?
					"Double-bank" : "Single-bank");
	}
	writestring(output);

	sprintf(output, "\tError Status                  : ");
	if (memm_info -> error_status & 1) strcat(output, "Uncorrectable errors, ");
	if (memm_info -> error_status & 2) strcat(output, "Correctable errors, ");
	if (!(memm_info -> error_status & 3)) strcat(output, "None, ");
	output[strlen(output) - 2] = '\0';
	strcat(output, "\n");
	writestring(output);
}

void dmi_memory_controller_info(DMI_INFO *dmi_info, DMI_MEMC *memc_info)
{
	writestring("\nMEMORY CONTROLLER INFORMATION (DMI) :\n\n");
	sprintf(output, "\tDMI Handle                    : %04Xh\n",
			dmi_info -> handle);
	writestring(output);

	sprintf(output, "\tError Detecting method        : %s\n",
		(memc_info -> error_detecting_method == 1) ? "(Other)" :
		(memc_info -> error_detecting_method == 2) ? "(Unknown)" :
		(memc_info -> error_detecting_method == 3) ? "None" :
		(memc_info -> error_detecting_method == 4) ? "8-bit Parity" :
		(memc_info -> error_detecting_method == 5) ? "32-bit ECC" :
		(memc_info -> error_detecting_method == 6) ? "64-bit ECC" :
		(memc_info -> error_detecting_method == 7) ? "128-bit ECC" :
		"(Unknown)");
	writestring(output);

	sprintf(output, "\tError Correcting capability   : ");
	strcat(output, (memc_info -> error_correcting_method & 1) ?
				"(Other), " : "");
	strcat(output, (memc_info -> error_correcting_method & 2) ?
				"(Unknown), " : "");
	strcat(output, (memc_info -> error_correcting_method & 8) ?
				"Single Bit Error Correcting, " : "");
	strcat(output, (memc_info -> error_correcting_method & 0x10) ?
				"Double Bit Error Correcting, " : "");
	strcat(output, (memc_info -> error_correcting_method & 0x20) ?
				"Error Scrubbing, " : "");
	if (memc_info -> error_correcting_method & 0x3f)
			strcat(output, "(Unknown), ");
	output[strlen(output) - 2] = '\0';
	strcat(output, "\n");
	writestring(output);

	sprintf(output, "\tSupported Interleave          : %s\n",
		(memc_info -> supported_interleave == 1) ? "(Other)" :
		(memc_info -> supported_interleave == 2) ? "(Unknown)" :
		(memc_info -> supported_interleave == 3) ? "One way" :
		(memc_info -> supported_interleave == 4) ? "Two way" :
		(memc_info -> supported_interleave == 5) ? "Four way" :
		(memc_info -> supported_interleave == 6) ? "Eight way" :
		(memc_info -> supported_interleave == 7) ? "Sixteen way" :
		"(Unknown)");
	writestring(output);

	sprintf(output, "\tCurrent Interleave            : %s\n",
		(memc_info -> current_interleave == 1) ? "(Other)" :
		(memc_info -> current_interleave == 2) ? "(Unknown)" :
		(memc_info -> current_interleave == 3) ? "One way" :
		(memc_info -> current_interleave == 4) ? "Two way" :
		(memc_info -> current_interleave == 5) ? "Four way" :
		(memc_info -> current_interleave == 6) ? "Eight way" :
		(memc_info -> current_interleave == 7) ? "Sixteen way" :
		"(Unknown)");
	writestring(output);

	sprintf(output, "\tMaximum Memory Module Size    : %luMB\n",
		((unsigned long)2) << memc_info -> max_module_size / 2);
	writestring(output);

	sprintf(output, "\tSupported Speed(s)            : ");
	if (memc_info -> supported_speeds & 1) strcat(output, "(Other), ");
	if (memc_info -> supported_speeds & 2) strcat(output, "(Unknown), ");
	if (memc_info -> supported_speeds & 4) strcat(output, "70ns, ");
	if (memc_info -> supported_speeds & 8) strcat(output, "60ns, ");
	if (memc_info -> supported_speeds & 0x10) strcat(output, "50ns, ");
	if (memc_info -> supported_speeds & 0x20) strcat(output, "Reserved "
						"setting by the BIOS, ");
	if (memc_info -> supported_speeds & 0x40) strcat(output, "Reserved "
						"setting by the BIOS, ");
	if (memc_info -> supported_speeds & 0x80) strcat(output, "Reserved "
						"setting by the BIOS, ");
	if (!memc_info -> supported_speeds) strcat(output, "(Unknown), ");
	output[strlen(output) - 2] = '\0';
	strcat(output, "\n");
	writestring(output);

	sprintf(output, "\tSupported Memory Type(s)      : ");
	if (memc_info -> memory_types & 1) strcat(output, "(Other), ");
	if (memc_info -> memory_types & 2) strcat(output, "(Unknown), ");
	if (memc_info -> memory_types & 4) strcat(output, "Standard, ");
	if (memc_info -> memory_types & 8) strcat(output, "Fast Page Mode "
							"(FPM), ");
	if (memc_info -> memory_types & 0x10) strcat(output, "Extended Data"
							" Output (EDO), ");
	if (memc_info -> memory_types & 0x20) strcat(output, "Parity, ");
	if (memc_info -> memory_types & 0x40) strcat(output, "ECC, ");
	if (memc_info -> memory_types & 0x80) strcat(output, "SIMM, ");
	if (memc_info -> memory_types & 0x100) strcat(output, "DIMM, ");
	if (memc_info -> memory_types & 0x200) strcat(output, "Burst EDO, ");
	if (memc_info -> memory_types & 0x400) strcat(output, "Synchronous "
						"Data RAM (SDRAM), ");
	if (memc_info -> memory_types & 0x800) strcat(output, "Reserved Setting"
							" by the BIOS, ");
	if (memc_info -> memory_types & 0x1000) strcat(output, "Reserved Setting"
							" by the BIOS, ");
	if (memc_info -> memory_types & 0x2000) strcat(output, "Reserved Setting"
							" by the BIOS, ");
	if (memc_info -> memory_types & 0x4000) strcat(output, "Reserved Setting"
							" by the BIOS, ");
	if (memc_info -> memory_types & 0x8000) strcat(output, "Reserved Setting"
							" by the BIOS, ");
	if (!memc_info -> memory_types) strcat(output, "(Unknown), ");
	output[strlen(output) - 2] = '\0';
	strcat(output, "\n");
	writestring(output);

	sprintf(output, "\tMemory Module Voltage(s)      : ");
	if (memc_info -> voltage & 1) strcat(output, "5.0V, ");
	if (memc_info -> voltage & 2) strcat(output, "3.3V, ");
	if (memc_info -> voltage & 4) strcat(output, "2.9V, ");
	if (!(memc_info -> voltage & 7)) strcat(output, "(Unknown), ");
	output[strlen(output) - 2] = '\0';
	strcat(output, "\n");
	writestring(output);
}

void dmi_chassis_info(DMI_INFO *dmi_info, DMI_CHASSIS *chassis_info)
{
	char *str_ptr = (char *)MK_FP(FP_SEG(dmi_info),
					FP_OFF(dmi_info) +
					dmi_info -> strings_offset);
	char *str_result;

	writestring("\nCHASSIS (SYSTEM ENCLOSURE) INFORMATION (DMI) :\n\n");
	sprintf(output, "\tDMI Handle                    : %04Xh\n",
			dmi_info -> handle);
	writestring(output);

	str_result = get_string_number(str_ptr,
			chassis_info -> index_to_manufacturer);
	sprintf(output, "\tManufacturer                  : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	sprintf(output, "\tLock Present                  : %s\n",
		(chassis_info -> flags & 0x80) ? "Present" :
		"Not Present / Unknown");
	writestring(output);

	sprintf(output, "\tSystem Enclosure              : %s\n",
		((chassis_info -> flags & 0x7f) == 1) ? "(Other)" :
		((chassis_info -> flags & 0x7f) == 2) ? "(Unknown)" :
		((chassis_info -> flags & 0x7f) == 3) ? "Desktop" :
		((chassis_info -> flags & 0x7f) == 4) ? "Low Profile"
							"Desktop" :
		((chassis_info -> flags & 0x7f) == 5) ? "Pizza Box" :
		((chassis_info -> flags & 0x7f) == 6) ? "Mini Tower" :
		((chassis_info -> flags & 0x7f) == 7) ? "Tower" :
		((chassis_info -> flags & 0x7f) == 8) ? "Portable" :
		((chassis_info -> flags & 0x7f) == 9) ? "LapTop" :
		((chassis_info -> flags & 0x7f) == 10) ? "Notebook" :
		((chassis_info -> flags & 0x7f) == 11) ? "Hand Held" :
		((chassis_info -> flags & 0x7f) == 12) ? "Docking"
							"Station" :
		((chassis_info -> flags & 0x7f) == 13) ? "All in One" :
		((chassis_info -> flags & 0x7f) == 14) ? "Sub Notebook" :
		((chassis_info -> flags & 0x7f) == 15) ? "Space-saving" :
		((chassis_info -> flags & 0x7f) == 16) ? "Lunch Box" :
		((chassis_info -> flags & 0x7f) == 17) ? "Mini Server"
							"Chassis" :
		((chassis_info -> flags & 0x7f) == 18) ? "Expansion"
							"Chassis" :
		((chassis_info -> flags & 0x7f) == 19) ? "SubChassis" :
		((chassis_info -> flags & 0x7f) == 20) ? "Bus Expansion"
							"Chassis" :
		((chassis_info -> flags & 0x7f) == 21) ? "Peripheral"
							"Chassis" :
		((chassis_info -> flags & 0x7f) == 22) ? "RAID Chassis" :
		((chassis_info -> flags & 0x7f) == 23) ? "Rack Mount"
							"Chassis" :
		"(Other)");
	writestring(output);

	str_result = get_string_number(str_ptr,
			chassis_info -> index_to_version);
	sprintf(output, "\tVersion                       : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	str_result = get_string_number(str_ptr,
			chassis_info -> index_to_serial_no);
	sprintf(output, "\tSerial Number                 : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	str_result = get_string_number(str_ptr,
			chassis_info -> index_to_asset_tag_no);
	sprintf(output, "\tAsset Tag Number              : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);
}

void dmi_mother_board_info(DMI_INFO *dmi_info, DMI_BOARD *board_info)
{
	char *str_ptr = (char *)MK_FP(FP_SEG(dmi_info),
					FP_OFF(dmi_info) +
					dmi_info -> strings_offset);
	char *str_result;

	writestring("\nMOTHER BOARD (BASE BOARD) INFORMATION (DMI) :\n\n");
	sprintf(output, "\tDMI Handle                    : %04Xh\n",
			dmi_info -> handle);
	writestring(output);

	str_result = get_string_number(str_ptr,
			board_info -> index_to_manufacturer);
	sprintf(output, "\tManufacturer                  : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	str_result = get_string_number(str_ptr,
			board_info -> index_to_product_name);
	sprintf(output, "\tProduct Name                  : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	str_result = get_string_number(str_ptr,
			board_info -> index_to_version);
	sprintf(output, "\tVersion                       : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	str_result = get_string_number(str_ptr,
			board_info -> index_to_serial_no);
	sprintf(output, "\tSerial Number                 : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);
}

void dmi_system_info(DMI_INFO *dmi_info, DMI_SYSINFO *sys_info)
{
	char *str_ptr = (char *)MK_FP(FP_SEG(dmi_info),
					FP_OFF(dmi_info) +
					dmi_info -> strings_offset);
	char *str_result;

	writestring("\nSYSTEM INFORMATION (DMI) :\n\n");
	sprintf(output, "\tDMI Handle                    : %04Xh\n",
			dmi_info -> handle);
	writestring(output);

	str_result = get_string_number(str_ptr,
			sys_info -> index_to_manufacturer);
	sprintf(output, "\tManufacturer                  : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	str_result = get_string_number(str_ptr,
			sys_info -> index_to_product_name);
	sprintf(output, "\tProduct Name                  : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	str_result = get_string_number(str_ptr,
			sys_info -> index_to_version);
	sprintf(output, "\tVersion                       : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	str_result = get_string_number(str_ptr,
			sys_info -> index_to_serial_no);
	sprintf(output, "\tSerial Number                 : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);
}

void dmi_bios_info(DMI_INFO *dmi_info, DMI_BIOS *bios_info)
{
	char *str_ptr = (char *)MK_FP(FP_SEG(dmi_info),
					FP_OFF(dmi_info) +
					dmi_info -> strings_offset);
	char *str_result;

	writestring("\nBIOS INFORMATION (DMI) :\n\n");
	sprintf(output, "\tDMI Handle                    : %04Xh\n",
			dmi_info -> handle);
	writestring(output);

	str_result = get_string_number(str_ptr,
			bios_info -> index_to_vendor);
	sprintf(output, "\tBIOS Vendor                   : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	str_result = get_string_number(str_ptr,
			bios_info -> index_to_version);
	sprintf(output, "\tBIOS Version                  : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	sprintf(output, "\tROM Starting Segment Address  : %04Xh\n",
			bios_info -> rom_segment);
	writestring(output);

	str_result = get_string_number(str_ptr,
			bios_info -> index_to_date);
	sprintf(output, "\tBIOS Release Date             : %s\n",
			(str_result == NULL) ? "(Unknown)" : str_result);
	writestring(output);

	sprintf(output, "\tROM Size                      : %dK\n",
			(bios_info -> rom_size_multiplier + 1) * 64);
	writestring(output);

	// See if BIOS Characteristics are supported.
	if (!(bios_info -> bios_features & 8)) {
		sprintf(output, "\tISA Supported                 : %s\n",
				(bios_info -> bios_features & 0x10) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tMCA Supported                 : %s\n",
				(bios_info -> bios_features & 0x20) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tEISA Supported                : %s\n",
				(bios_info -> bios_features & 0x40) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tPCI Supported                 : %s\n",
				(bios_info -> bios_features & 0x80) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tPCMCIA Supported              : %s\n",
				(bios_info -> bios_features & 0x100) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tPlug and Play Supported       : %s\n",
				(bios_info -> bios_features & 0x200) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tAPM Supported                 : %s\n",
				(bios_info -> bios_features & 0x400) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tBIOS can be Flashed (upgrade) : %s\n",
				(bios_info -> bios_features & 0x800) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tBIOS Shadowing Supported      : %s\n",
				(bios_info -> bios_features & 0x1000) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tVL-VESA Supported             : %s\n",
				(bios_info -> bios_features & 0x2000) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tESCD Supported                : %s\n",
				(bios_info -> bios_features & 0x4000) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tBoot from CDROM Supported     : %s\n",
				(bios_info -> bios_features & 0x8000) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tSelectable Boot Supported     : %s\n",
				(bios_info -> bios_features & 0x10000) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tBIOS ROM Socketed             : %s\n",
				(bios_info -> bios_features & 0x20000) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tBoot from PCMCIA Supported    : %s\n",
				(bios_info -> bios_features & 0x40000) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tEDD Specification Support     : %s\n",
				(bios_info -> bios_features & 0x80000) ?
				"Yes" : "No");
		writestring(output);
		sprintf(output, "\tJapanese Floppy (NEC 9800)1.2M: %s\n",
				(bios_info -> bios_features & 0x100000) ?
				"Supported" : "Not Supported");
		writestring(output);
		sprintf(output, "\tJapanese Floppy (Toshiba) 1.2M: %s\n",
				(bios_info -> bios_features & 0x200000) ?
				"Supported" : "Not Supported");
		writestring(output);
		sprintf(output, "\t360K Floppy Disk Services     : %s\n",
				(bios_info -> bios_features & 0x400000) ?
				"Supported" : "Not Supported");
		writestring(output);
		sprintf(output, "\t1.2M Floppy Disk Services     : %s\n",
				(bios_info -> bios_features & 0x800000) ?
				"Supported" : "Not Supported");
		writestring(output);
		sprintf(output, "\t720K Floppy Disk Services     : %s\n",
				(bios_info -> bios_features & 0x1000000) ?
				"Supported" : "Not Supported");
		writestring(output);
		sprintf(output, "\t2.88M Floppy Disk Services    : %s\n",
				(bios_info -> bios_features & 0x2000000) ?
				"Supported" : "Not Supported");
		writestring(output);
		sprintf(output, "\tPrint Screen Services         : %s\n",
				(bios_info -> bios_features & 0x4000000) ?
				"Supported" : "Not Supported");
		writestring(output);
		sprintf(output, "\t8042 Keyboard Services        : %s\n",
				(bios_info -> bios_features & 0x8000000) ?
				"Supported" : "Not Supported");
		writestring(output);
		sprintf(output, "\tSerial Port Services          : %s\n",
				(bios_info -> bios_features & 0x10000000) ?
				"Supported" : "Not Supported");
		writestring(output);
		sprintf(output, "\tPrinter Services              : %s\n",
				(bios_info -> bios_features & 0x20000000) ?
				"Supported" : "Not Supported");
		writestring(output);
		sprintf(output, "\tCGA Video Services            : %s\n",
				(bios_info -> bios_features & 0x4000000) ?
				"Supported" : "Not Supported");
		writestring(output);
/*		sprintf(output, "\tPC-98                         : %s\n",
				(bios_info -> bios_features & 0x8000000) ?
				"Yes" : "No");
		writestring(output);*/
	}
}

int dmi_version()
{
 int result;
 PNP *header;
 DMI_INFO *dmi_header;

 // Create output file.
 if ((fp = fopen("temp.$$$", "w")) == NULL) {
    errflag = 1;
    checkerrors();
 }

 // Search for $DMI.
 header = get_pnp_dmi_entry_point("$DMI");
 if (header == NULL) header = get_pnp_dmi_entry_point("$PnP");
 else {
	// Only $DMI. Cannot get information.
	fprintf(fp, "%c", 0);
	fclose(fp);
	return 0;
 }

 if (header == NULL) {
	fprintf(fp, "%c", 0);
	fclose(fp);
	return 0;
 }

 result = header -> pnp_entry_point(0x50,
				buf1,
				&buf1[1],
				&buf1[3],
				&buf1[5],
				&buf1[9],
				header -> bios_selector);

 if ((result) || (buf1[0] == 0xff) || (!buf1[0])) {
	fprintf(fp, "%c", 0);
	fclose(fp);
	return 0;
 }

 // We have a DMI 2.0 or better.
 fprintf(fp, "%c", buf1[0]);
 return 0;
}

int sysinfo()
{
 int result;
 PNP *header;
 DMI_INFO *dmi_header;

 // Create output file.
 if ((fp = fopen("dmi.txt", "w")) == NULL) {
    errflag = 1;
    checkerrors();
 }

 writestring("\nDESKTOP MANAGEMENT INTERFACE (DMI) :\n\n");

 // Search for $DMI.
 header = get_pnp_dmi_entry_point("$DMI");
 if (header == NULL) header = get_pnp_dmi_entry_point("$PnP");
 else {
	// Only $DMI. Cannot get information.
	writestring("\tDMI BIOS Present              : Yes\n");
	writestring("\tNo additional information available\n\n");
	fclose(fp);
	return 0;
 }

 if (header == NULL) {
	writestring("\tDMI BIOS not present.\n");
	fclose(fp);
	return 0;
 }

 result = header -> pnp_entry_point(0x50,
				buf1,
				&buf1[1],
				&buf1[3],
				&buf1[5],
				&buf1[9],
				header -> bios_selector);

 if ((result) || (buf1[0] == 0xff) || (!buf1[0])) {
	writestring("\tDMI BIOS not present.\n");
	writestring("\tNo additional information available.\n\n");
	fclose(fp);
	return 0;
 }

 // We have a DMI 2.0 or better.
 writestring("\tDMI BIOS Present              : Yes\n");
 sprintf(output, "\tDMI BIOS Version              : %X.%X\n",
			buf1[0] >> 4,
			buf1[0] & 0xf);
 writestring(output);
 if (buf1[0] < 0x20) {
	writestring("\tNo additional information available.\n\n");
	fclose(fp);
	return 0;
 }

 // Reset buffers.
 memset(buf1, 0, 100);
 memset(buf2, 0, 100);

 // Get DMI information.
 do {
	result = header -> pnp_entry_point(0x51,
					buf1,
					buf2,
					header -> bios_selector);
	if (result) {
		fprintf(stderr, "Error: DMI error code %04Xh\r\n", result);
		fclose(fp);
		return 0;
	}

	dmi_header = (DMI_INFO *)buf2;
	switch (dmi_header -> info_type_specifier) {
		case 0:
			dmi_bios_info(dmi_header, (DMI_BIOS *)&buf2[4]);
			break;

		case 1:
			dmi_system_info(dmi_header, (DMI_SYSINFO *)&buf2[4]);
			break;

		case 2:
			dmi_mother_board_info(dmi_header, (DMI_BOARD *)
								&buf2[4]);
			break;

		case 3:
			dmi_chassis_info(dmi_header, (DMI_CHASSIS *)
								&buf2[4]);
			break;

		case 4:
			dmi_processor_info(dmi_header, (DMI_PROC *)&buf2[4]);
			break;

		case 5:
			dmi_memory_controller_info(dmi_header,
						(DMI_MEMC *)&buf2[4]);
			break;

		case 6:
			dmi_memory_module_info(dmi_header,
						(DMI_MEMM *)&buf2[4]);
			break;

		case 7:
			dmi_cache_info(dmi_header, (DMI_CACHE *)&buf2[4]);
			break;

		case 9:
			dmi_slots_info(dmi_header, (DMI_SLOT *)&buf2[4]);
			break;

		default:
			break;
	}

 } while (*(int *)&buf1[0]  != -1);

 fclose(fp);
 return 0;
}

void open_stderr()
{
	fclose(stdout);
	fclose(&_streams[2]);
	if (fopen("nul", "wb") == NULL) exit(0x7f);
	if (fopen("nul", "wb") == NULL) exit(0x7f);
	if ((stderr = fopen("errors.$$$", "ab")) == NULL) exit(0x7f);
}

void get_cmdline()
{
 if ((fp = fopen("cmdline.$$$", "rb")) == NULL) exit (0x7f);

 if (fscanf(fp, "%s", cmdline) != 1) {
		fclose(fp);
		exit (0x7f);
 }

 fclose(fp);
 unlink("cmdline.$$$");
}

#pragma argsused

// The main function.
int main(int argc, char **argv)
{
 open_stderr();
 get_cmdline();

 if (!strcmp(cmdline, "sysinfo")) return (sysinfo());
 if (!strcmp(cmdline, "dmi_version")) return (dmi_version());

 return 0;
}
