//////////////////////////////////////////////////////////////////////////////
// ASCII format Model implementation
//////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "mdlib.h"
//#include "ascmdl.h"


// load a .ASC model file
ascmdl_t ASCLoadFile(char *filename)
{
    FILE *mdl_file;
    V3D_f v1, v2, v3;
    ascmdl_t model;
    long i, vertex, trig;

    mdl_file = fopen( filename, "rt" );

    fscanf( mdl_file, "Named Object: \"%[^\n\"]s", model.name );
    fscanf( mdl_file, "\" Tri-mesh, Vertices: %d Faces: %d ", &model.numverts, &model.numtris );

    // allocate memory for the vertices
    model.pos = (V3D_f *)malloc( model.numverts*sizeof(V3D_f) );
    model.rpos = (V3D_f *)malloc( model.numverts*sizeof(V3D_f) );
    model.triangles = (asctrig_t *)malloc( model.numtris*sizeof(asctrig_t) );

    // read the vertice list
    fscanf( mdl_file, "Vertex List: ");
    for(i=0; i<model.numverts; i++)
    {
        fscanf( mdl_file, "Vertex %d: ", &vertex);
        if(vertex > model.numverts) printf("ASCLoadFile(): Error reading vertex in %s", filename);
        fscanf( mdl_file, "X: %f Y: %f Z: %f ", &model.pos[vertex].x, &model.pos[vertex].y, &model.pos[vertex].z);
    }

    // read the trig list
    fscanf( mdl_file, "Face list:");
    for(i=0; i<model.numtris; i++)
    {
        fscanf( mdl_file, "Face %d: ", &trig);
        if(trig > model.numtris) printf("ASCLoadFile(): Error reading trig in %s", filename);
        fscanf( mdl_file, "A:%d B:%d C:%d AB:1 BC:1 CA:1 ", &model.triangles[trig].vertices[0], &model.triangles[trig].vertices[1], &model.triangles[trig].vertices[2]);
        if(model.triangles[trig].vertices[0] > model.numtris ||
           model.triangles[trig].vertices[1] > model.numtris ||
           model.triangles[trig].vertices[2] > model.numtris) printf("ASCLoadFile(): Error reading trig list in %s", filename);
        fscanf( mdl_file, "Smoothing: 1 ");
    }


    fclose( mdl_file );

    model.dist2 = (float *)malloc( model.numtris*sizeof(float) );
    model.trigorder = (long *)malloc( model.numtris*sizeof(long) );
    model.drawtrig = (char *)malloc( model.numtris*sizeof(char) );


    return model;
}

// show info on the model
void ASCDebug(mdl_t *model)
{
    long i;


    printf("Name: %s\n", model->ascmdl->name);
    printf("Vertices: %i\n", model->ascmdl->numverts);
    printf("Triangles: %i\n", model->ascmdl->numtris);

    for(i=0; i<model->ascmdl->numverts; i++)
        printf("Vertex %i: %f %f %f\n", i, model->ascmdl->pos[i].x, model->ascmdl->pos[i].y, model->ascmdl->pos[i].z);

    for(i=0; i<model->ascmdl->numtris; i++)
        printf("Trig %i: %i %i %i\n", i, model->ascmdl->triangles[i].vertices[0], model->ascmdl->triangles[i].vertices[1], model->ascmdl->triangles[i].vertices[2]);


}

// draw only the points
void ASCDrawPoints(BITMAP *dbuf, mdl_t *model)
{
   long i;

   // draw dots
   for(i=0; i<model->ascmdl->numverts; i++)
   {
       circle(dbuf, (int)model->ascmdl->rpos[i].x, (int)model->ascmdl->rpos[i].y, 1, 208);
//       putpixel(dbuf, (int)model->ascmdl->rpos[i].x, (int)model->ascmdl->rpos[i].y, 110);
       putpixel(dbuf, (int)model->ascmdl->rpos[i].x, (int)model->ascmdl->rpos[i].y, 254);
   }
}

// draw wireframe triangles
void ASCDrawWireframe(BITMAP *dbuf, mdl_t *model)
{
    V3D_f v1, v2, v3;
    long i, trig;

    // draw every triangle
    for(i=model->ascmdl->numtris-1; i>=0; i--)
    {
        trig = model->ascmdl->trigorder[i];
        v1 = model->ascmdl->rpos[model->ascmdl->triangles[trig].vertices[0]];
        v2 = model->ascmdl->rpos[model->ascmdl->triangles[trig].vertices[1]];
        v3 = model->ascmdl->rpos[model->ascmdl->triangles[trig].vertices[2]];

//        if(model->ascmdl->drawtrig[i] && v1.z > Z_CLIP && v2.z > Z_CLIP && v3.z > Z_CLIP)
        {
            line(dbuf, v1.x, v1.y, v2.x, v2.y, 6);
            line(dbuf, v1.x, v1.y, v3.x, v3.y, 6);
            line(dbuf, v3.x, v3.y, v2.x, v2.y, 6);
        }
    }
}

// Draw flat shaded triangles
void ASCDrawFlatShade(BITMAP *dbuf, mdl_t *model)
{
    long i, j;
    long trig;
    V3D_f v1, v2, v3;        // tmp variables

    // draw every triangle
    for(i=model->ascmdl->numtris-1; i>=0; i--)
    {
        trig = model->ascmdl->trigorder[i];
        v1 = model->ascmdl->rpos[model->ascmdl->triangles[trig].vertices[0]];
        v2 = model->ascmdl->rpos[model->ascmdl->triangles[trig].vertices[1]];
        v3 = model->ascmdl->rpos[model->ascmdl->triangles[trig].vertices[2]];

//        if(model->ascmdl->drawtrig[i] && v1.z > Z_CLIP && v2.z > Z_CLIP && v3.z > Z_CLIP)
            triangle3d_f(dbuf, POLYTYPE_FLAT, NULL, &v1, &v2, &v3);
   }
}

void ASCUpdateView(camera_t cam, mdl_t *model)
{
    float xfront, yfront, zfront;
    float xup, yup, zup;
    float dot;
    long i, j;
    MATRIX_f roller, camview;
    V3D_f v1, v2, v3, v4, vc, v4r, vn, vnr;   // tmp variables


    // calculate the in-front vector
    xfront = sin(cam.head) * cos(cam.pitch);
    yfront = sin(cam.pitch);
    zfront = cos(cam.head) * cos(cam.pitch);

    // rotate the up vector around the in-front vector by the roll angle
//    get_vector_rotation_matrix_f(&roller, xfront, yfront, zfront, cam.roll*128.0/M_PI);
//    apply_matrix_f(&roller, 0, -1, 0, &xup, &yup, &zup);

    get_transformation_matrix_f(&camview, 1, cam.head, cam.roll, cam.pitch, cam.x, cam.y, cam.z);
/*
    // build the camera matrix
    get_camera_matrix_f(&camview,
                        cam.x, cam.y, cam.z,     // camera position
                        xfront, yfront, zfront,  // in-front vector
                        xup, yup, zup,           // up vector
                        48,                      // field of view
                        1.);                     // aspect ratio
*/

    // rotate the points and
    for(i=0; i<model->ascmdl->numverts; i++)
        apply_matrix_f(&camview, model->ascmdl->pos[i].x, model->ascmdl->pos[i].y, model->ascmdl->pos[i].z,
                       &model->ascmdl->rpos[i].x, &model->ascmdl->rpos[i].y, &model->ascmdl->rpos[i].z);

    // this might do the work for now!
    for(i=0; i<model->ascmdl->numtris; i++)
    {
        v1 = model->ascmdl->rpos[model->ascmdl->triangles[i].vertices[0]];
        v2 = model->ascmdl->rpos[model->ascmdl->triangles[i].vertices[2]];
        v3 = model->ascmdl->rpos[model->ascmdl->triangles[i].vertices[1]];

        // average of the triangle
        v4.x = (v3.x + v2.x + v1.x)/3.;
        v4.y = (v3.y + v2.y + v1.y)/3.;
        v4.z = (v3.z + v2.z + v1.z)/3.;

        // calculate distance from view to average triangle point
//         model->ascmdl->dist2[i] = (v4.x-xpos)*(v4.x-xpos) + (v4.y-ypos)*(v4.y-ypos) + (v4.z-zpos)*(v4.z-zpos);
        model->ascmdl->dist2[i] = (v4.x)*(v4.x) + (v4.y)*(v4.y) + (v4.z)*(v4.z);
        model->ascmdl->trigorder[i] = i;

        cross_product_f(v2.x-v1.x, v2.y-v1.y, v2.z-v1.z, v3.x-v1.x, v3.y-v1.y, v3.z-v1.z, &vc.x, &vc.y, &vc.z);
        dot = dot_product_f(vc.x, vc.y, vc.z, v4.x, v4.y, v4.z);

//        if(dot > 0) model->ascmdl->drawtrig[i] = 1;
//        else model->ascmdl->drawtrig[i] = 0;
        model->ascmdl->drawtrig[i] = 1;
    }

    // calculate perspective
    for(i=0; i<model->ascmdl->numverts; i++)
    persp_project_f(model->ascmdl->rpos[i].x, model->ascmdl->rpos[i].y, model->ascmdl->rpos[i].z, &model->ascmdl->rpos[i].x, &model->ascmdl->rpos[i].y);

    // sort the triangles
    Mysort(model->ascmdl->dist2, model->ascmdl->trigorder, 0, model->ascmdl->numtris-1);


}
