/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains different functions for changing filenames.

*/
#ifndef s_fname_c
#define s_fname_c

#include "s_fname.h"

#include "s_common.h"
#include "s_string.h"
#include "s_char.h"
#include "s_math.h"
#include "s_conv.h"
#include "s_alloc.h"
#include "s_global.h"

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <dir.h>
#include <allegro.h>
#include <sys/stat.h>



/********************
****             ****
**** is_path_sep ****
****             ****
*********************
Returns `TRUE' if the character is a valid path separator.
*/
int is_path_sep(uchar ascii)
{
return _is_path_sep(ascii);
}


/********************
****             ****
**** getfilename ****
****             ****
*********************
Returns the filename in a path.*/
int getfilename_pos(uchar *path)
{
int i;
int pos=-1;
for(i=0;path[i];i++)
	if(_is_path_sep(path[i]))
		pos=i;
return pos+1;
}

uchar *getfilename_d(uchar *path, uchar *de)
{
int pos = getfilename_pos(path);
int length = strlen(path+pos);
_ifstrxmalloc(de, length-pos);
return memcopy(de, path+pos, length-pos);
}

uchar *getfilename_s(uchar *path, int allocmode)
{
int pos = getfilename_pos(path);
int length = strlen(path+pos);
memmove(path, path+pos, length);
return ifstrxrealloc(path, allocmode, length);
}

/***********************
****                ****
**** changefilename ****
****                ****
************************
Replaces the filename in a full path with the parameter.
*/
uchar *changefilename_d(uchar *so, uchar *de, uchar *filename)
{
int start=getfilename_pos(so);
int length=strlen(filename);
_ifstrxmalloc(de,start+length);
memcopy(de+start, filename, length);
return memcopy(de,so,start);
}
uchar *changefilename_s(uchar *path, int allocmode, uchar *filename)
{
int start=getfilename_pos(path);
int length=strlen(filename);
_ifstrxrealloc(path, allocmode, start+length);
memcopy(path+start, filename, length);
return path;
}
/*********************
****              ****
**** getextension ****
****              ****
**********************
Returns the position of the extension in a filename (with or without path).
*/
int getextension_pos(uchar *filename)
{
int i;
int pointpos=-1;
for(i=0;filename[i];i++){
	if(filename[i]=='.')
		pointpos=i;
	else if(_is_path_sep(filename[i]))
		pointpos=-1;
}
if(pointpos!=-1)
	return pointpos+1;
else
	return i;
}

uchar *getextension_d(uchar *path, uchar *de)
{
int pos = getextension_pos(path);
int length = strlen(path+pos);
_ifstrxmalloc(de, length-pos);
return memcopy(de, path+pos, length-pos);
}

uchar *getextension_s(uchar *path, int allocmode)
{
int pos = getextension_pos(path);
int length = strlen(path+pos);
memmove(path, path+pos, length);
return ifstrxrealloc(path, allocmode, length);
}

/****************
****         ****
**** getpath ****
****         ****
*****************
Returns a only the path, WITH the last backslash.*/
uchar *getpath_s(uchar *filename,int allocmode)
{
int i,backslashpos=0;
//Search for the backslash.
for(i=0;filename[i];i++)
	if(is_path_sep(filename[i]))
		backslashpos=i;
_ifstrxrealloc(filename,allocmode,backslashpos+1);
return filename;
}
uchar *getpath_d(uchar *so, uchar *de)
{
int i,backslashpos=0;
//Search for the backslash.
for(i=0;so[i];i++)
	if(is_path_sep(so[i]))
		backslashpos=i;
_ifstrxmalloc(de,backslashpos+1);
memcopy(de,so,backslashpos+1);
return de;
}



/*********************
****              ****
**** findfile_new ****
****              ****
**********************
Returns a full path where the filename was found using access() function.
This means that the string that this function returns can be used as filename
to all standard file functions. If something got wrong (for example if malloc
failed), the program will exit. If the file couldn't be found, the function
returns NULL.
The first parameter is the filename. If the second parameter is not NULL, the
function will use that as filename extension and add it to the filename (if
the file with the extension is not found, the function looks for the file
without the extension). Include the point in it if you want to; you don't need
to do that if you don't want to. If you specify more parameters, those mean
additional paths to check. Current directory will always be checked first.
The path names may or may not end with a backslash (or slash). The function
will find out wether there is one. THE PARAMETER LIST MUST END WITH A NULL.
Otherwise the program will of course die.*/
uchar *findfile_new(uchar *filenameandpaths,uchar *extension, ...)
{
uchar **paths=(&extension)+1;
//Set to `NULL' to avoid uninitialized warning.
uchar *filename=filenameandpaths,*filename2=NULL;
uchar *path;
int filenamelength,filename2length=0,pathlength,extensionlength;//Set to 0 to avoid uninitialized warning.
int i;

filenamelength=strlen(filename);
//Adds the extension if !NULL.
if(extension){
	extensionlength=strlen(extension);
	if(extension[0]=='.'){
		filename2length=filenamelength+extensionlength;
		_strxmalloc(filename2,filename2length);
		memcopy(filename2,filename,filenamelength);
		memcopy(filename2+filenamelength,extension,extensionlength);
	}else{
		filename2length=filenamelength+1+extensionlength;
		_strxmalloc(filename2,filename2length);
		memcopy(filename2,filename,filenamelength);
		filename2[filenamelength]='.';
		memcopy(filename2+filenamelength+1,extension,extensionlength);
	}
	//Checking only the filename+extension.
	if(!access(filename2,F_OK))
		return strdup(filename2);
}

//Checking only the filename.
if(!access(filename,F_OK))
	return strdup(filename);

/* I can't figure out why current directory doesn't work.

//Current directory.
path=getcd_new();
pathlength=strlen(path);
path=strxrealloc(path,pathlength+filenamelength);
memcopy(path+pathlength,filename,filenamelength);
if(!access(path,F_OK))
	return path;

free(path);
*/

for(i=0;paths[i];i++){
	pathlength=strlen(paths[i]);
	//Checking parameters+filename+extension.
	if(extension){
		//First we check without backslash if path end with a path separator.
		if(is_path_sep(paths[i][pathlength-1])){
			_strxmalloc(path,pathlength+filename2length);
			memcopy(path,paths[i],pathlength);
			memcopy(path+pathlength,filename2,filename2length);
			if(!access(path,F_OK))
				return path;
			free(path);
		}
		//If there is no backslash at the end of the string, we try with one.
		if((paths[i][pathlength-1]!='\\') && (paths[i][pathlength-1]!='/')){
			_strxmalloc(path,pathlength+1+filename2length);
			memcopy(path,paths[i],pathlength);
			path[pathlength]='\\';
			memcopy(path+pathlength+1,filename2,filename2length);
			if(!access(path,F_OK))
				return path;
			free(path);
		}
	}
	//Checking parameters+filename.
	//First we check without backslash if path end with a path separator.
	if(is_path_sep(paths[i][pathlength-1])){
		path=strxmalloc(pathlength+filenamelength);
		memcopy(path,paths[i],pathlength);
		memcopy(path+pathlength,filename,filenamelength);
		if(!access(path,F_OK))
			return path;
		free(path);
	}
	//If there is no backslash at the end of the path, we try with one.
	if((paths[i][pathlength-1]!='\\') && (paths[i][pathlength-1]!='/')){
		path=strxmalloc(pathlength+1+filenamelength);
		memcopy(path,paths[i],pathlength);
		path[pathlength]='\\';
		memcopy(path+pathlength+1,filename,filenamelength);
		if(!access(path,F_OK))
			return path;
		free(path);
	}
}
return NULL;
}
/**************
****       ****
**** getcd ****
****       ****
***************
Returns a newly allocated string containing current directory, always with
the ending backslash. (The standard function puts a backslash only if you're
on the root, which is rather uncomfortable.) Exit on error.
WARNING!!! THIS FUNCTION FAILS VERY OFTEN!.*/
uchar *getcd_d(uchar *buf)
{
int length;
int allocmode=!!buf;

//There's something strange here. I have to reprogram this later: this
//function seems to contain a lot of debug lines.
getcwd(buf,32767);
if(!buf){
	buf=realloc(buf,1);
//Remove this and make a separate `xgetcd()' function.
// exitfail(
//  "In function `getcd_d()': getcwd failed (returned `NULL'). Couldn't find current\n\r"
//  "directory, or string length of current directory is more than 32767. This may be"
//  "Allegro's fault. The programmer shouldn't call `getcd_d()' if Allegro is\n\r"
//  "present.");
}
length=strlen(buf);
if((buf[length-1]=='\\') || (buf[length-1]=='/'))
	return buf;
else{
	_ifstrxrealloc(buf,allocmode,length+1);
	buf[length]='\\';
	return buf;
}
}



/*************************
****                  ****
**** filename2datname ****
****                  ****
**************************
Converts a filename to the corresponding NAME property in a datafile.*/
uchar *filename2datname_d(uchar *so,uchar *de)
{
int i;
int length=strlen(so);
_ifstrxmalloc(de,length);
for(i=0;i<length;i++){
	if(so[i]=='.')
		de[i]='_';
	else
		de[i]=chartoupper(so[i]);
}
return de;
}
uchar *filename2datname_change(uchar *filename)
{
int i;
strupr(filename);
for(i=0;filename[i];i++){
	if(filename[i]=='.')
		filename[i]='_';
}
return filename;
}



/*********************
****              ****
**** addextension ****
****              ****
**********************
Adds the extension to the filename. It doesn't matter wether there is a point
in the end of the filename or wether there is a point in the beginning of the
extension, this function fixes that.*/
uchar *addextension_s(uchar *filename,int allocmode,uchar *extension)
{
int filenamelength=strlen(filename),extensionlength=strlen(extension);
if(extension[0]=='.'){
	if(filename[filenamelength-1]=='.'){
		_ifstrxrealloc(filename,allocmode,filenamelength+extensionlength-1);
		memcopy(filename+filenamelength,extension+1,extensionlength-1);
	}else{
		_ifstrxrealloc(filename,allocmode,filenamelength+extensionlength);
		memcopy(filename+filenamelength,extension,extensionlength);
	}
}else{
	if(filename[filenamelength-1]=='.'){
		_ifstrxrealloc(filename,allocmode,filenamelength+extensionlength);
		memcopy(filename+filenamelength,extension,extensionlength);
	}else{
		_ifstrxrealloc(filename,allocmode,filenamelength+extensionlength+1);
		filename[filenamelength]='.';
		memcopy(filename+filenamelength+1,extension,extensionlength);
	}
}
return filename;
}
uchar *addextension_d(uchar *so,uchar *de,uchar *extension)
{
int filenamelength=strlen(so),extensionlength=strlen(extension);
if(extension[0]=='.'){
	if(so[filenamelength-1]=='.'){
		_ifstrxmalloc(de,filenamelength+extensionlength-1);
		memcopy(de,so,filenamelength);
		memcopy(de+filenamelength,extension+1,extensionlength-1);
	}else{
		_ifstrxmalloc(de,filenamelength+extensionlength);
		memcopy(de,so,filenamelength);
		memcopy(de+filenamelength,extension,extensionlength);
	}
}else{
	if(so[filenamelength-1]=='.'){
		_ifstrxmalloc(de,filenamelength+extensionlength);
		memcopy(de,so,filenamelength);
		memcopy(de+filenamelength,extension,extensionlength);
	}else{
		_ifstrxmalloc(de,filenamelength+extensionlength+1);
		memcopy(de,so,filenamelength);
		de[filenamelength]='.';
		memcopy(de+filenamelength+1,extension,extensionlength);
	}
}
return de;
}
/********************
****             ****
**** addfilename ****
****             ****
*********************
Adds the filename to the path. It doesn't matter wether there is a backslash
in the end of the path or wether there is a backslash in the beginning of the
filename, this function fixes that.*/
uchar *addfilename_s(uchar *path,int allocmode,uchar *filename)
{
int pathlength=strlen(path),filenamelength=strlen(filename);
if((filename[0]=='\\')||(filename[0]=='/')){
	if((path[pathlength-1]=='\\')||(path[pathlength-1]=='/')){
		_ifstrxrealloc(path,allocmode,pathlength+filenamelength-1);
		memcopy(path+pathlength,filename+1,filenamelength-1);
	}else{
		_ifstrxrealloc(path,allocmode,pathlength+filenamelength);
		memcopy(path+pathlength,filename,filenamelength);
	}
}else{
	if((path[pathlength-1]=='\\')||(path[pathlength]=='/')){
		_ifstrxrealloc(path,allocmode,pathlength+filenamelength);
		memcopy(path+pathlength,filename,filenamelength);
	}else{
		_ifstrxrealloc(path,allocmode,pathlength+filenamelength+1);
		path[pathlength]='\\';
		memcopy(path+pathlength+1,filename,filenamelength);
	}
}
return path;
}
uchar *addfilename_d(uchar *so,uchar *de,uchar *filename)
{
int pathlength=strlen(so),filenamelength;
filenamelength=strlen(filename);
if((filename[0]=='\\')||(filename[0]=='/')){
	if((so[pathlength-1]=='\\')||(so[pathlength-1]=='/')){
		_ifstrxmalloc(de,pathlength+filenamelength-1);
		memcopy(de,so,pathlength);
		memcopy(de+pathlength,filename+1,filenamelength-1);
	}else{
		_ifstrxmalloc(de,pathlength+filenamelength);
		memcopy(de,so,pathlength);
		memcopy(de+pathlength,filename,filenamelength);
	}
}else{
	if((so[pathlength-1]=='\\')||(so[pathlength-1]=='/')){
		_ifstrxmalloc(de,pathlength+filenamelength);
		memcopy(de,so,pathlength);
		memcopy(de+pathlength,filename,filenamelength);
	}else{
		_ifstrxmalloc(de,pathlength+filenamelength+1);
		memcopy(de,so,pathlength);
		de[pathlength]='\\';
		memcopy(de+pathlength+1,filename,filenamelength);
	}
}
return de;
}

/************************
****                 ****
**** removeextension ****
****                 ****
*************************
Removes the extension from the filename.
*/
uchar *removeextension_s(uchar *filename,int allocmode)
{
int length=strlen(filename),i;
for(i=length-1; (i>=0) && !is_path_sep(filename[i]); i--)
	if(filename[i]=='.'){
		_ifstrxrealloc(filename,allocmode,i);
		return filename;
	}
return filename;
}
uchar *removeextension_d(uchar *so,uchar *de)
{
int length = strlen(so),i;
for(i = length - 1; (i >= 0) && !is_path_sep(so[i]); i--)
	if(so[i] == '.'){
		_ifstrxmalloc(de,i);
		return memcopy(de,so,i);
	}
_ifstrxmalloc(de,length);
return memcopy(de,so,length);
}

/************************
****                 ****
**** changeextension ****
****                 ****
*************************
Changes the extension for a filename to another.
*/
uchar *changeextension_d(uchar *so,uchar *de,uchar *extension)
{
int solength,namelength=0,extensionlength=0;
//Find total length of filename and length to the point.
for(solength=0;so[solength];solength++)
	if(so[solength]=='.')
		namelength=solength;
//If no point was found, name continues to the end of so.
if(namelength==0)
	namelength=solength;
//Find length of extension.
while(extension[extensionlength]){
	if(extension[extensionlength]=='.'){
		extension+=extensionlength+1;
		extensionlength=0;
	}else
		extensionlength++;
}
_ifstrxmalloc(de,namelength+1+extensionlength);//Malloc
memcopy(de,so,namelength);//Filename.
de[namelength]='.';//Point.
memcopy(de+namelength+1,extension,extensionlength);//Extension.
return de;
}
uchar *changeextension_s(uchar *filename,int allocmode,uchar *extension)
{
int filenamelength,namelength=0,extensionlength=0;
//Find length of filename and length of filename up to the last point in it.
for(filenamelength=0;filename[filenamelength];filenamelength++)
	if(filename[filenamelength]=='.')
		namelength=filenamelength;
//If no point was found, name continues to the end of so.
if(namelength==0)
	namelength=filenamelength;
//Find length of extension.
while(extension[extensionlength]){
	if(extension[extensionlength]=='.'){
		extension+=extensionlength+1;
		extensionlength=0;
	}else
		extensionlength++;
}
_ifstrxrealloc(filename,allocmode,namelength+1+extensionlength);//Realloc.
filename[namelength]='.';//Point.
memcopy(filename+namelength+1,extension,extensionlength);//Extension.
return filename;
}



/************************
****                 ****
**** anotherfilename ****
****                 ****
*************************
Returns a filename that doesn't exist, and that looks like the parameter,
but with a digit added at the end. (For example "foo.dat" would be
"foo0.dat".) This function checks for existing filenames. Also, returns only
short 8.3-filenames.
*/
uchar *anothername_d(uchar *so, uchar *de,
 int (*_check_exists)(uchar *filename))
{
uchar *extension;
int filenamestart;
int length,i;
int mindigitsinmax,max=10,num;
int allocmode=!!de;
int extensionlength,extensionpos;
int numberlength;

if((!so) || (so[0]==0))
	return NULL;
//Get extension and calculate its length. If there isn't a point before the
//extension, that means the file has no extension, and in that case it
//shouldn't be restored with an extension either.
extensionpos=getextension_pos(so)-1;
if(so[extensionpos]!='.'){
	extension=NULL;
	extensionlength=0;
}else{
	extensionlength=strlen(so+extensionpos);
	extension=strxmalloc(extensionlength);
	memcopy(extension, so+extensionpos, extensionlength);
}
//Initialize `de' to a copy of `so' without the extension.
de=removeextension_d(so,de);
//Get beginning of filename in the string (after the path).
filenamestart=getfilename_pos(de);
//Get number of characters in filename.
length=strlen(de+filenamestart);
//Realloc de to the biggest possible. `realloc()' instead of `malloc()' since
//`de' already is malloced (in `removeextension_d()').
de=ifstrxrealloc(de,allocmode,filenamestart+8+extensionlength);

//Length should be the number of characters in filename before the digits.
if(length>7)
	length=7;
mindigitsinmax=8-length;
for(i=1;i<mindigitsinmax;i++)
	cm10(max);
while(length>=0){
	numberlength=1;
	for(num=0;num<max;num++){
		//Add number.
		ul2str(num,de+filenamestart+length);
		//Increase distance to point if number of digits in the number has
		//increased (which is when the number is 10, 100, 1000 etc).
		if(de[filenamestart+length+numberlength]!=0)
			numberlength++;
		//Add extension.
		if(extension)
			memcopy(de+filenamestart+length+numberlength,
			 extension,extensionlength+1);
		//Check if the file exists.
		if(!_check_exists(de)){
			free(extension);
			return ifstrxrealloc(de,allocmode,strlen(de));
		}
	}
	length--;
	cm10(max);
}
return NULL;//Maybe this line is rather useless: to come here, the function
//must have tried about 100 millions of filenames, and all of them must have
//existed. However, this avoids a warning.
}
uchar *anothername_s(uchar *filename, int allocmode,
 int (*_check_exists)(uchar *filename))
{
uchar *extension;
int filenamestart;
int length,i;
int mindigitsinmax, max=10, num;
int extensionlength,extensionpos;
int numberlength;

if((!filename) || (filename[0]==0))
	return NULL;
//Get extension and calculate its length. If there isn't a point before the
//extension, that means the file has no extension, and in that case it
//shouldn't be restored with an extension either.
extensionpos=getextension_pos(filename)-1;
if(filename[extensionpos]!='.'){
	extension=NULL;
	extensionlength=0;
}else{
	extensionlength=strlen(filename+extensionpos);
	extension=strxmalloc(extensionlength);
	memcopy(extension, filename+extensionpos, extensionlength);
}
//Remove the extension from filename.
filename=removeextension_s(filename,allocmode);
//Get beginning of filename in the string (after the path).
filenamestart=getfilename_pos(filename);
//Get number of characters in filename.
length=strlen(filename+filenamestart);
//Realloc filename to the biggest possible.
filename=ifstrxrealloc(filename,allocmode,filenamestart+8+extensionlength);
//Length should be the number of characters in filename before the digits.
if(length>7)
	length=7;
mindigitsinmax=8-length;
for(i=1;i<mindigitsinmax;i++)
	cm10(max);
while(length>=0){
	numberlength=1;
	for(num=0;num<max;num++){
		//Add number.
		ul2str(num,filename+filenamestart+length);
		//Increase distance to point if number of digits in the number has
		//increased (which is when the number is 10, 100, 1000 etc).
		if(filename[filenamestart+length+numberlength]!=0)
			numberlength++;
		//Add extension.
		if(extension)
			memcopy(filename+filenamestart+length+numberlength,
			 extension,extensionlength+1);
		if(!_check_exists(filename)){
			if(extension)
				free(extension);
			return ifstrxrealloc(filename,allocmode,strlen(filename));
		}
	}
	length--;
	cm10(max);
}
return NULL;//Maybe this line is rather useless: to come here, the function
//must have tried about 100 millions of filenames, and all of them must have
//existed. However, we avoid a warning when doing this.
}

//The same, but changes the filename only if it doesn't exist.
uchar *maybe_anothername_s(uchar *filename, int allocmode,
 int (*_check_exists)(uchar *filename))
{
if(!_check_exists(filename))
	return filename;
return anothername_s(filename, allocmode, _check_exists);
}

//The same, but changes the filename only if it doesn't exist.
uchar *maybe_anothername_d(uchar *so, uchar *de,
 int (*_check_exists)(uchar *filename))
{
if(!_check_exists(so)){
	int solen=strlen(so);
	de=ifstrxmalloc(de, solen);
	memcpy(de, so, solen);
	return de;
}
return anothername_d(so, de, _check_exists);
}



/*******************
****            ****
**** search_dir ****
****            ****
********************
Searches a directory for the specified file, and puts all matches in the
string list. This function supports the same wildcards as findfirst().
*/
int search_dir(uchar *name, TSTRLIST *sl)
{
struct ffblk f;
int done;
int pathlength;
int ret=0;
if(!(name && sl))
	return -1;
done = findfirst(name, &f, FA_ARCH);
pathlength = getfilename_pos(name);
while (!done){
	//Creating new filename.
	int namelength=strlen(f.ff_name);
	uchar *new_file=strxmalloc(pathlength+strlen(f.ff_name));
	memcopy(new_file,name,pathlength);
	memcopy(new_file+pathlength,f.ff_name,namelength);
	//Add filename.
	add_strlist_item(sl,new_file);
	ret++;
	//Search next.
	done = findnext(&f);
}
return ret;
}
/***********************
****                ****
**** search_subdirs ****
****                ****
************************
Searches a directory and all its subdirectories for the specified file, and
puts all matches in the string list. This function supports the same
wildcards as findfirst().
*/
int search_subdirs(uchar *name, TSTRLIST *sl)
{
struct ffblk f;
int done;
uchar *wildname;//The name with filename replaced with *.*.
int namepos;
int namelength;
int ret=0;
if(!(name && sl))
	return -1;
namepos=getfilename_pos(name);
namelength=strlen(name)-namepos;
search_dir(name,sl);
wildname=changefilename_new(name,"*.*");
done=findfirst(wildname, &f, FA_DIREC);//|FA_HIDDEN|FA_RDONLY|FA_SYSTEM|FA_ARCH);
while (!done){
	if((f.ff_attrib&FA_DIREC) &&
	 strcomp(f.ff_name,".") && strcomp(f.ff_name,"..")){
		//Create the subdir name.
		int subdirlength=strlen(f.ff_name);
		uchar *subdirname=strxmalloc(namepos+subdirlength+1+namelength);
		memcopy(subdirname, name, namepos);
		memcopy(subdirname+namepos, f.ff_name, subdirlength);
		subdirname[namepos+subdirlength]='\\';
		memcopy(subdirname+namepos+subdirlength+1, name+namepos, namelength);
		//Recurse.
		ret+=search_subdirs(subdirname, sl);
		free(subdirname);
	}
	done = findnext(&f);
}
free(wildname);
return ret;
}


/*void make_subdirlist_callback(uchar *fullpath,int attr, dat
void make_subdirlist(uchar *name, TSTRLIST *sl)
{
}*/



/*********************
****              ****
**** drive_exists ****
****              ****
**********************
Returns `TRUE' if the drive exists. Pass the name of the drive, like 'c' as
parameter. Note that this function is extremely slow for a: and b: (DOS'
fault), so you may think it has hung even though it has not.
*/
int drive_exists(uchar drive)
{
union REGS r;
r.x.ax=0x0;
r.h.ah=0x36;
r.h.dl=chartolower(drive)-'a'+1;
int86(0x21,&r,&r);
if(r.x.ax==0xFFFF)
	return FALSE;
return TRUE;
}

/********************
****             ****
**** path_exists ****
****             ****
*********************
Returns `TRUE' if the path exists.
*/
int path_exists(uchar *path)
{
struct ffblk f;
if(!findfirst(path, &f,
 FA_RDONLY | FA_HIDDEN | FA_SYSTEM | FA_DIREC | FA_ARCH))
	if(f.ff_attrib&FA_DIREC)
		return TRUE;
//Check if the path contains only a drive letter and a colon, which means
//findfirst won't detect it as valid.
if(path[0] && path[1]==':' && !path[2])
	if(drive_exists(path[0]))
		return TRUE;
return FALSE;
}
/*******************
****            ****
**** any_exists ****
****            ****
********************
Returns `TRUE' if the path/filename/drive exists.
*/
int any_exists(uchar *path)
{
struct ffblk f;
if(!findfirst(path, &f,
 FA_RDONLY | FA_HIDDEN | FA_SYSTEM | FA_DIREC | FA_ARCH))
	return TRUE;
//Check if the path contains only a drive letter and a colon, which means
//findfirst won't detect it as valid.
if(path[0] && path[1]==':' && !path[2])
	if(drive_exists(path[0]))
		return TRUE;
return FALSE;
}

/*******************
****            ****
**** force_path ****
****            ****
********************
Creates the path if it doesn't exist. The path may include several
non-existing directories, that all will be created. Returns the number of
directories created (could be 0), or the negative value of
[number of paths + 1] on error.
*/
int force_path(uchar *path)
{
int i;
uchar ascii;
int numcreated=0;
//Loops until it finds nonexisting paths, and creates them.
for(i=1; ; i++){
	if(((path[i]=='\\') || (path[i]=='/') || !path[i])){
		//Ugly thing: stores what was in the path, then sets it to end of
		//string.
		ascii=path[i];
		path[i]=0;
		//If we're at end of string, we shouldn't do anything if last character
		//was a (back)slash. Also, we should only do things if the path doesn't
		//exist, which is automatically true if numcreated>0.
		if((path[i] || ((path[i-1]!='\\') && (path[i-1]!='/'))) &&
		 (numcreated || !path_exists(path))){
			numcreated++;
			if(mkdir(path,0)){
				//Error.
				path[i]=ascii;
				return numcreated ? -numcreated : min_slong;
			}
			path[i]=ascii;
		}
		//Restores the path and checks for end of string.
		if(!(path[i]=ascii))
			return numcreated;
	}
}
}

int xforce_path(uchar *path)
{
int ret;
if((ret=force_path(path))<0)
	exitfail(mergestrs_new(
	 "In function xforce_path: force_path() failed (Returned ",
	 l2str(ret,NULL),").",NULL));
return ret;
}



#endif
