// Copyright (C) 1997 Keith Whitwell.
// This file may only be copied under the terms of the GNU Library General
// Public License - see the file COPYING in the lib3d distribution.

// This file is contributed by
// Markus F.X.J. Oberhumer <markus.oberhumer@jk.uni-linz.ac.at>

#include <Lib3d/Viewport.H>
#include <Lib3d/Model.H>
#include <Lib3d/ModelBuilder.H>
#include <Lib3d/PlgReader.H>
#include <Lib3d/World.H>
#include <Lib3d/Camera.H>
#include <Lib3d/Light.H>
#include <Lib3d/SmoothPipeline.H>
#include <Lib3d/WirePipeline.H>
#include <Lib3d/internals/HrTimer.H>
#include <iostream.h>
#include <fstream.h>


main(int argc, char *argv[])
{
    Device *device = Device::create(320,200,8);
    Viewport *viewport = Viewport::create(device);

    if (!viewport) {
	cout << "Failed to create viewport" << endl;
	exit(1);
    }

    World world;
    PLGReader ob;

    const char *name = "../models/plant.plg";
    if (argc > 1)
        name = argv[1];
    ifstream in(name);
    ob.startModel();
    // ob.setColourfulMode(.2);	// Beachball mode
    if (in.bad() || !ob.readPLG( in )) {
	cerr << "Couldn't read " << name << endl;
	exit(-1);
    }

    Model *model = ob.endModel();
    world.adopt(model);

    cout << "Created object:\n" << *model << endl;

    Camera *camera = new Camera(world);
    camera->setParameters(3,100,15,1);
    world.setActiveCamera( *camera );

    Vector3 colour(1,1,1);	// A white light
    Vector3 direction(0,1,.5);
    Light *light = new Light(world);
    light->setParameters(colour, colour, direction);
    world.registerLight( *light );

    for (int n = 0 ; n < 3 ; n++) {

	if ( n == 1 ) {
	    Pipeline *pipeline = new SmoothPipeline;
	    model->usePipeline(*pipeline);
	    cout << "Smooth pipeline:" << endl;
	    // continue;
	} else if ( n == 2 ) {
	    Pipeline *pipeline = new WirePipeline;
	    model->usePipeline(*pipeline);
	    cout << "Wire pipeline:" << endl;
	    // continue;
	} else {
	    cout << "Flat pipeline:" << endl;
	    // continue;
	}

	Matrix34 transform;
	transform.setIdentity();
	model->setTransform(transform);
    
	Matrix34 tmp;
	Matrix34 scale;
	scale.setIdentity();

	HrTimer   clip("    Clipping");
	HrTimer noclip("Not clipping");
	HrTimer pageflip("Page flips");

	// Real close, lots of clipping

	// transform.setRotation(-90, 1,0,0);
	transform.setRotation(0, 1,0,0);
	scale.setRotation((4.56), 0, 1, 0);
	// scale.setRotation((4.56), 1, 1, 1);

	tmp.setTranslation(0,0,-10);
	camera->setTransform(tmp);

	int nrFrames = 512;
	cout << "Rendering " << nrFrames << " frames: " << endl;
	clip.start();
	for ( int i = 0 ; i < nrFrames ; i++ ) {
	    device->processPendingEvents();
	    transform.premul(scale);
	    model->setTransform(transform);
	    world.renderHierarchy( *viewport );
	    viewport->swapBuffers();
	}
	clip.end();
	
	cout << clip << endl;
	cout << "\t" << (float(nrFrames)/clip.getElapsedSeconds()) 
	     << " frames per second" << endl;
	cout << setprecision(8) << "\t" 
	     << (float(nrFrames * model->getNrPolygons())/
		 clip.getElapsedSeconds()) 
	     << " polygons per second" << endl;


	// From a dignified distance,  no bounding box intersections.
	tmp.setTranslation(0,0,-16);    
	camera->setTransform(tmp);
	// transform.setRotation(-90, 1,0,0);
	transform.setRotation(0, 1,0,0);
	scale.setRotation((4.56), 0, 1, 0);
	
	nrFrames = 1023;
	cout << "Rendering " << nrFrames << " frames: " << endl;
	noclip.start();
	for ( int j = 0 ; j < nrFrames ; j++ ) {
	    device->processPendingEvents();
	    transform.premul(scale);
	    model->setTransform(transform);
	    world.renderHierarchy( *viewport );
	    pageflip.start();
	    viewport->swapBuffers();
	    pageflip.end();
	}
	noclip.end();
	
	cout << pageflip << endl;
	cout << noclip << endl;
	cout << "\t" << (float(nrFrames)/noclip.getElapsedSeconds()) 
	     << " frames per second" << endl;
        cout << setprecision(8) << "\t" 
	     << (float(nrFrames*model->getNrPolygons())/
		 noclip.getElapsedSeconds()) 
	     << " polygons per second" << endl;

    }

    delete viewport;
}


