/* Copyright (C) 2008 DJ Delorie, see COPYING.DJ for details */
#include <stdlib.h>
#include <string.h>
#include <stddef.h>

#include "libsupp.h"


error_t
libsupp_argz_insert(char **argz, size_t *argz_len, char *before, const char *entry)
{
  /*
   *  Insert ENTRY into ARGZ of length ARGZ_LEN before BEFORE, which should be an
   *  existing entry in ARGZ; if BEFORE is NULL, ENTRY is appended to the end.
   *  Since ARGZ's first entry is the same as ARGZ, argz_insert(ARGZ, ARGZ_LEN,
   *  ARGZ, ENTRY) will insert ENTRY at the beginning of ARGZ.  If BEFORE is not
   *  in ARGZ, EINVAL is returned, else if memory can't be allocated for the new
   *  ARGZ, ENOMEM is returned, else 0.
   */

  ptrdiff_t first_part_len;
  size_t entry_len;


  if (before == NULL)
    return libsupp_argz_add(argz, argz_len, entry);

  if (before < *argz || before >= *argz + *argz_len)
    return EINVAL;

  for (; before != *argz && before[-1]; before--)
    ;  /*  Make sure before is actually the beginning of an entry.  */

  first_part_len = before - *argz;
  entry_len = strlen(entry) + 1;
  *argz = realloc(*argz, *argz_len + entry_len);
  if (*argz == NULL)
    return ENOMEM;

  memmove(*argz + first_part_len + entry_len, *argz + first_part_len,  *argz_len - first_part_len);
  memcpy(*argz + first_part_len, entry, entry_len);
  *argz_len += entry_len;

  return 0;
}
