/*----------------------------------------------------------------------
 *
 *  wgetstr.c
 *
 *  copyright (c) 1987,88,89,90 J. Alan Eldridge
 *
 *  The Berkeley Curses function is:
 *
 *  wgetstr(win,buf)
 *
 *  The extended functions are:
 *
 *  wegetstr(win,buf)
 *  weditstr(win,buf,max)
 *  weditfld(win,buf,len,term,ins,skip,fix,zap)
 *  wedittxt(win,buf,len,term,ins,skip,fix,filter)
 *
 *----------------------------------------------------------------------
 */

#include "curses.h"

/*----------------------------------------------------------------------
 *
 * WEDITSTR
 *
 * edit buf with maximum length max, output to win, end on newline
 * the newline is not echoed and is not in the buffer
 *
 *----------------------------------------------------------------------
 */

static int
okfunc()
{
    return 1;
}

int
weditstr(win, buf, cnt)
WINDOW  *win;
char    *buf;
int     cnt;
{
    static int nl_esc[] = { K_NL, K_ESC, K_ILLEGAL };

    return wedittxt(win, buf, cnt, nl_esc, 1, 0, -1, 0, okfunc);
}

/*----------------------------------------------------------------------
 *
 * WGETSTR
 *
 * the standard Curses string input function
 *
 * gets a string, ends on newline, output to win, newline is echoed
 *
 *----------------------------------------------------------------------
 */

int
wgetstr(win, buf)
WINDOW  *win;
char    *buf;
{
    *buf = 0;
    return wegetstr(win,buf);
}

/*----------------------------------------------------------------------
 * 
 * WEGETSTR
 *
 * same as wgetstr (which calls it), but the buffer is not emptied
 *
 *----------------------------------------------------------------------
 */

int
wegetstr(win, buf)
WINDOW  *win;
char    *buf;
{
    static int nl[] = { K_NL, K_ILLEGAL };

    int c;
    char localbuf[512];

    strcpy(localbuf,buf);
    c = weditfld(win, localbuf, 511, nl, 1, 0, -1, 0);
    strcpy(buf, localbuf);
    if (c == '\n') {
        waddch(win, c);
        wrefresh(win);
    }

    return c;
}

/*----------------------------------------------------------------------
 *
 *  WEDITTXT & WEDITFLD -- main editing functions
 *
 *  wedittxt(win, buf, len, term, ins, skip, fix, zap, filter)
 *  WINDOW  *win;
 *  char    *buf;
 *  int     len;
 *  int     *term;
 *  int     ins,
 *          skip,
 *          fix,
 *          zap;
 *  int     (*filter)();
 *
 *  edit the field whose initial contents are contained in buf
 *  screen output is to window win
 *  the maximum length of the field contents is len
 *  term is string of ints which, when received from the
 *      keyboard, will terminate the edit
 *  ins tells whether insert mode can be turned on
 *  skip tells whether auto-field skipping should occur:
 *      (this is for data entry forms, and only works if fix,
 *      explained next, is set > 0)
 *      if you're at the end of the buffer and press space or
 *      you're at the beginning of the buffer and cursor left
 *      the edit terminates, returning TAB or BACKTAB
 *  fix tells how to handle trailing spaces:
 *      < 0 means truncate trailing spaces
 *      0 means leave them alone
 *      > 0 add spaces and keep field length at its maximum
 *  zap tells whether to erase the field if the first character typed
 *      is a printing character (useful for data entry forms)    
 *  filter is a function that takes 1 argument and returns non-zero
 *      if that character is ok to put into the text and zero if the
 *      character is illegal. the character passed will be a printing
 *      character.
 *
 *  weditfld() takes all arguments that wedittxt() does except
 *  for the filter; no filtering is done.
 *
 *----------------------------------------------------------------------
 */

int
weditfld(win, buf, len, term, ins, skip, fix, zap)
WINDOW  *win;
char    *buf;
int     len;
int *term;
int     ins,
        skip,
        fix,
        zap;
{
    return wedittxt(win, buf, len, term, ins, skip, fix, zap, okfunc);
}

int
wedittxt(win, buf, len, term, ins, skip, fix, zap, filter)
WINDOW  *win;
char    *buf;
int     len;
int     *term;
int     ins,
        skip,
        fix,
        zap;
int     (*filter)();
{
    int c, 
        l = strlen(buf), 
        insmode = 0, 
        pos = 0, 
        y, x;

    char *cp = buf + l;

    if (fix > 0 && l < len) {
        while (l < len) {
            l++;
            *cp++ = ' ';
        }
        *cp = 0;
    } else if (fix < 0) {
        while (l > 0 && *(cp-1) == ' ') {
            l--;
            *cp-- = 0;
        }
    }

    getyx(win,y,x);
    waddstr(win,buf);
    wmove(win,y,x);
    wrefresh(win);

    while (kb_matchkey(term, c=wgetch(win)) == -1) {
        if (skip && fix > 0)
            if ((c == ' ' || c == K_RIGHT) && pos == len)
                return '\t';
            else if ((c == '\b' || c == K_LEFT) && pos == 0)
                return K_BACKTAB;
        if (isascii(c) && isprint(c) && (*filter)(c)) {
            if (zap) {
                int y,x,f;

                while (pos > 0) {
                    pos--;
                    waddch(win, '\b');
                }
                f = fix > 0 ? ' ' : 0;
                getyx(win,y,x);
                while (l > 0) {
                    buf[--l] = f;
                    waddch(win,' ');
                }
                if (fix > 0)
                    l = len;
                wmove(win,y,x);
                zap = 0;
            }
            if (pos >= l)
                if (l < len) {
                    buf[l++]=c;
                    buf[++pos] = 0;
                    waddch(win, c);
                } else
                    beep();
            else if (insmode)
                if (l < len || (l == len && buf[l-1] == ' ')) {
                    int n, y, x;
                    for (n=l-(l>=len);n>pos;n--)
                        buf[n] = buf[n-1];
                    waddch(win, buf[pos]=c);
                    if (l < len)
                        buf[++l] = 0;
                    getyx(win, y, x);
                    waddstr(win, buf+(++pos));
                    wmove(win, y, x);
                } else
                    beep();
            else
                waddch(win, buf[pos++]=c);
            wrefresh(win);
            zap = 0;
            continue;
        }
        switch (c) {
        case K_LEFT:
            if (pos > 0) {
                pos--;
                waddch(win, '\b');
            } else
                beep();
            break;
        case K_RIGHT:
            if (pos < l) {
                pos++;
                waddch(win, winch(win));
            } else
                beep();
            break;
        case K_HOME:
            while (pos > 0) {
                pos--;
                waddch(win, '\b');
            }
            break;
        case K_END:
            while (pos < l) {
                pos++;
                waddch(win, winch(win));
            }
            break;
        case K_INS:
            if (ins)
                insmode = !insmode;
            else
                beep();
            break;
        case K_CTL_X:
            /* ^x cancels line */
            {
                int y,x,f;

                while (pos > 0) {
                    pos--;
                    waddch(win, '\b');
                }
                f = fix > 0 ? ' ' : 0;
                getyx(win,y,x);
                while (l > 0) {
                    buf[--l] = f;
                    waddch(win,' ');
                }
                if (fix > 0)
                    l = len;
                wmove(win,y,x);
            }
            break;
        case '\b':
        case K_DEL:
            if (c == '\b')
                if (pos > 0) {
                    pos--;
                    waddch(win, '\b');
                } else {
                    beep();
                    break;
                }
            if (pos < l && l > 0) {
                int p,y,x;
                for (p = pos; p < l-1; p++)
                    buf[p] = buf[p+1];
                buf[p] = ' ';
                getyx(win,y,x);
                waddstr(win,buf+pos);
                wmove(win,y,x);
                if (fix <= 0) {
                    buf[p] = 0;
                    l--;
                }
            } else
                beep();
            break;
        default:
            beep();
            break;
        }
        wrefresh(win);
        zap = 0;
    }

    if (c == '\n') 
        while (pos++ < l) 
            waddch(win, winch(win));

    return c;
}

