/* Project SWORD
   V2.0

   SubSystem : ToolBox for 3D graphics
   File      : LibSrc/ToolBox/Graph3D/Transf.CC
   Author    : Eric NICOLAS
   Overview  : Object TTransform : A geometrical transformation matrix
   UpDate    : Mar 19, 1995

** Copyright (C) 1993,1995 The SWORD Group
**
** This file is distributed under the terms listed in the document
** "copying.en". A copy of "copying.en" should accompany this file.
** if not, a copy should be available from where this file was obtained.
** This file may not be distributed without a verbatim copy of "copying.en".
**
** This file is distributed WITHOUT ANY WARRANTY; without even the implied
** warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

#include <math.h>

#include "Common/Common.H"
#include "Mecanism/Collec.H"
#include "ToolBox/Graph3D/Point3D.H"
#include "ToolBox/Graph3D/Transf.H"

// Local variables

static TTransform TTransformTemp;
static TPoint3D   TPoint3DTemp;

// Constructors

TTransform::TTransform()
{ Coefs[0][0]=1; Coefs[0][1]=0; Coefs[0][2]=0; Coefs[0][3]=0;
  Coefs[1][0]=0; Coefs[1][1]=1; Coefs[1][2]=0; Coefs[1][3]=0;
  Coefs[2][0]=0; Coefs[2][1]=0; Coefs[2][2]=1; Coefs[2][3]=0;
  Coefs[3][0]=0; Coefs[3][1]=0; Coefs[3][2]=0; Coefs[3][3]=1;
}

TTransform::TTransform(TTransform& T)
{ int i,j;
  for(j=0;j<4;j++)
    for(i=0;i<4;i++)
      Coefs[j][i]=T.Coefs[j][i];
}

// Operations on Transormation matrix

#define CalcElement(i,j)                                          \
  TTransformTemp.Coefs[j][i] = Coefs[j][0]*T.Coefs[0][i] +        \
                               Coefs[j][1]*T.Coefs[1][i] +        \
                               Coefs[j][2]*T.Coefs[2][i]

#define CalcLigne(j)                                              \
  CalcElement(0,j);                                               \
  CalcElement(1,j);                                               \
  CalcElement(2,j);                                               \
  CalcElement(3,j) + Coefs[j][3]

TTransform TTransform::operator*(TTransform& T)
{ CalcLigne(0);
  CalcLigne(1);
  CalcLigne(2);
  return TTransformTemp;
}

float TTransform::Det22(int j1, int j2, int i1, int i2)
{ return Coefs[j1][i1]*Coefs[j2][i2]-Coefs[j2][i1]*Coefs[j1][i2];
}

float TTransform::Det33(int j1, int j2, int j3, int i1, int i2, int i3)
{ return +Coefs[j1][i1]*Det22(j2,j3,i2,i3)
         -Coefs[j1][i2]*Det22(j2,j3,i1,i3)
         +Coefs[j1][i3]*Det22(j2,j3,i1,i2);
}

float TTransform::Det()
{ return Det33(0,1,2,0,1,2);
}

TTransform Inv(TTransform& T)
{ float D=T.Det();
  // Lgn 1
  TTransformTemp.Coefs[0][0] = +1/D*T.Det22(1,2,1,2);       // Coff(0,0)
  TTransformTemp.Coefs[0][1] = -1/D*T.Det22(0,2,1,2);       // Coff(1,0)
  TTransformTemp.Coefs[0][2] = +1/D*T.Det22(0,1,1,2);       // Coff(2,0)
  TTransformTemp.Coefs[0][3] = -1/D*T.Det33(0,1,2,1,2,3);   // Coff(3,0)
  // Lgn 2
  TTransformTemp.Coefs[1][0] = -1/D*T.Det22(1,2,0,2);       // Coff(0,1)
  TTransformTemp.Coefs[1][1] = +1/D*T.Det22(0,2,0,2);       // Coff(1,1)
  TTransformTemp.Coefs[1][2] = -1/D*T.Det22(0,1,0,2);       // Coff(2,1)
  TTransformTemp.Coefs[1][3] = +1/D*T.Det33(0,1,2,0,2,3);   // Coff(3,1)
  // Lgn 2
  TTransformTemp.Coefs[2][0] = +1/D*T.Det22(1,2,0,1);       // Coff(0,2)
  TTransformTemp.Coefs[2][1] = -1/D*T.Det22(0,2,0,1);       // Coff(1,2)
  TTransformTemp.Coefs[2][2] = +1/D*T.Det22(0,1,0,1);       // Coff(2,2)
  TTransformTemp.Coefs[2][3] = -1/D*T.Det33(0,1,2,0,1,3);   // Coff(3,2)
  // Lgn 3
  TTransformTemp.Coefs[3][0] = 0;
  TTransformTemp.Coefs[3][1] = 0;
  TTransformTemp.Coefs[3][2] = 0;
  TTransformTemp.Coefs[3][3] = 1;
  return TTransformTemp;
}

// Using Transformation matrix on points

TPoint3D TTransform::operator*(TPoint3D& P)
{ TPoint3DTemp.X()= Coefs[0][0]*P.X() + Coefs[0][1]*P.Y() +
                    Coefs[0][2]*P.Z() + Coefs[0][3];
  TPoint3DTemp.Y()= Coefs[1][0]*P.X() + Coefs[1][1]*P.Y() +
                    Coefs[1][2]*P.Z() + Coefs[1][3];
  TPoint3DTemp.Z()= Coefs[2][0]*P.X() + Coefs[2][1]*P.Y() +
                    Coefs[2][2]*P.Z() + Coefs[2][3];
  return TPoint3DTemp;
}

// Geometrical transformations

TTransform Homothetie(TPoint3D& Coefs)
{ TTransformTemp=TTransform();
  TTransformTemp(0,0)=Coefs.X();
  TTransformTemp(1,1)=Coefs.Y();
  TTransformTemp(2,2)=Coefs.Z();
  return TTransformTemp;
}

TTransform Translation(TPoint3D& V)
{ TTransformTemp=TTransform();
  TTransformTemp(0,3)=V.X();
  TTransformTemp(1,3)=V.Y();
  TTransformTemp(2,3)=V.Z();
  return TTransformTemp;
}

TTransform RotateOXY(float A)
{ float SA=sin(A);
  float CA=cos(A);
  TTransformTemp=TTransform();
  TTransformTemp(0,0)= CA;
  TTransformTemp(0,1)=-SA;
  TTransformTemp(1,0)= SA;
  TTransformTemp(1,1)= CA;
  return TTransformTemp;
}

TTransform RotateOYZ(float A)
{ float SA=sin(A);
  float CA=cos(A);
  TTransformTemp=TTransform();
  TTransformTemp(1,1)= CA;
  TTransformTemp(1,2)=-SA;
  TTransformTemp(2,1)= SA;
  TTransformTemp(2,2)= CA;
  return TTransformTemp;
}

TTransform RotateOXZ(float A)
{ float SA=sin(A);
  float CA=cos(A);
  TTransformTemp=TTransform();
  TTransformTemp(0,0)= CA;
  TTransformTemp(0,2)=-SA;
  TTransformTemp(2,0)= SA;
  TTransformTemp(2,2)= CA;
  return TTransformTemp;
}
