/*
 *  djp.c -- DJGPP V2 executable file compressor with the LZO algorithm
 *
 *  Copyright (C) 1996-1998 Laszlo Molnar, see lzo/COPYING for details
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <limits.h>
#include <errno.h>
#include <sys/stat.h>
#include <coff.h>
#include <netinet/in.h>
#if defined(__MSDOS__) || defined(__EMX__)
#  include <dos.h>
#endif

#ifdef __DJGPP__
#  define rename _rename
#endif

#include <lzoconf.h>
#if (LZO_VERSION < 0x1040)
#  error "need LZO v1.04 or above"
#endif
#include <lzo1x.h>
#include <lzo1f.h>
#include <lzo1.h>
#ifdef NEED_LZO1C
#include <lzo1c.h>               /* for files compessed with djp 1.04 b4 */
#endif

#define SIMAGIC "go32stub"

#define EOFLEN 3                 /* for LZO1C & LZO1F & LZO1X */

/* options */
#define OPTD 0x01                /* decompress */
#define OPTS 0x02                /* use stub */
#define OPTT 0x04                /* test */
#define OPTB 0x08                /* keep backup */
#define OPTQ 0x10                /* quiet */
#define OPTC 0x20                /* coff output */
#define OPT0 0x40                /* ignore default options */
#define OPTH 0x80                /* usage */
#define OPT1 0x100               /* fast compressor */
#define OPTL 0x200               /* list compression */
#define OPTI 0x400               /* display software license */

static FILE *fi,*fo;             /* input,output */

static char wrkmem[LZO1X_999_MEM_COMPRESS];  /* for lzo1x_999 compression*/
static /*const*/ unsigned char stub[]={    /* "stubify" stub */
#include "stub.h"
};
/* these are not const because the checksum is stored */
static unsigned char djpstub[]={       /* stubbed decompressor */
#include "djpstub.h"
};
static unsigned char decomph[]={       /* stubless decompressor */
#include "startup.h"
};

static const char *packerr;      /* error message */
static char     tmp_file[512];
static char     line[512];       /* line buffer */
static char     *chksumpos;      /* checksum position */
static int      options=0;
/* exetype = (compr: 0-coff,1-exe) (decompr: 0-coff,1-stubless,2-stubbed) */
static int      exetype;
static int      isatty_stdout=0;
static char     *argv0=NULL;
static unsigned total_files=0;
static unsigned total_size=0;
static unsigned total_packedsize=0;

static unsigned get_ratio (unsigned size,unsigned packedsize)
{
    if (size <= 0 || packedsize >= size)
        return 100 * 100;
    if (packedsize <= UINT_MAX / 10000)
        return ((packedsize * 10000) / size) + 5;
    if (packedsize <= UINT_MAX / 1000)
        return ((packedsize * 1000) / (size/10)) + 5;
    return ((packedsize * 100) / (size/100)) + 5;
}

static char *findbytes (void *b,int blen,const char *what,int wlen)
{
    int ic,firstc=what[0];
    char *base=(char*)b;

    for (ic=0; ic<=blen-wlen; ic++)
        if (base[ic]==firstc && memcmp (base+ic,what,wlen)==0)
            return base+ic;

    return NULL;
}

static void printline (const char *s,int pos,unsigned mlen)
{
    if (strlen (s) > mlen)
        s+=mlen-strlen (s);
    sprintf (line+pos,"%-*.*s",79-pos,79-pos,s);
    if (isatty_stdout && (!(options & OPTQ)))
    {
        printf ("\r%s",line);
        fflush (stdout);
    }
}

#ifndef __DJGPP__
static int __file_exists (const char *fn)
{
    struct stat s;
    int olderr,r;

    olderr = errno;
    r = stat (fn,&s);
    errno = olderr;
    return r==0;
}
#endif

/* this code is from stubify.c */
static void maketempname (char *ofilename,const char *ifilename,const char *ext,int force)
{
    char *ofext = NULL,*ofname;
    int  ofile;
    strcpy (ofilename,ifilename);

    for (ofname=ofilename; *ofname; ofname++)
    {
        if (strchr("/\\:", *ofname))
            ofext = NULL;
        if (*ofname == '.')
            ofext = ofname;
    }
    if (ofext == NULL)
        ofext = ofilename + strlen(ofilename);
    strcpy(ofext, ext);
    if (force==0 && __file_exists (ofilename))
        for (ofile=0; ofile<999; ofile++)
        {
            sprintf(ofext, ".%03d", ofile);
            if (!__file_exists (ofilename))
                break;
        }
}

static int patch_decomp (char *s,long oldw,long neww)
{
    char *p;
    if ((p=findbytes (s,sizeof(decomph),(char*)&oldw,4))==NULL)
        return 1;
    *(long*) p = neww;
    return 0;
}

static unsigned inpsize;

static int djpcb_perc=-1;
static void djpcb (unsigned texts,unsigned codes)
{
    int  ic;
    char temp[40];

    if (inpsize > 0 && (ic=(texts*100)/inpsize) != djpcb_perc)
    {
        unsigned ratio = get_ratio (texts,codes);
        sprintf (temp,"%02d%%       ratio:%3d.%01d%%",djpcb_perc=ic,
                 ratio/100,(ratio%100)/10);
        printline (temp,51,28);
    }
}

static int pack_sections (char *in,char *out,unsigned foffset,unsigned size,unsigned tpos,unsigned hdrsize)
{
    unsigned packedsize,new_size;
    unsigned adler1,adler2,ratio;
    char buf[80];
    int  r;

    packerr=NULL;
    if (in==NULL || out==NULL)
        return packerr="out of memory",-1;
    if (fseek (fi,foffset,0) || fread (in,hdrsize,1,fi)!=1 ||
        fseek (fi,foffset+tpos,0) || fread (in+(tpos&0x1ff),size,1,fi)!=1)
        return packerr="read error",-1;         /* ^^ this 0x1FF can be wrong */

    size+=(tpos&0x1ff);

    /* compute checksum of uncompressed data */
    adler1 = lzo_adler32 (0,NULL,0);
    adler1 = lzo_adler32 (adler1,in,size);

    packedsize=0;
    if (options & OPT1)
        r=lzo1x_1_compress (in,inpsize=size,out,&packedsize,wrkmem);
    else
        r=lzo1x_999_compress_level (in,inpsize=size,out,&packedsize,wrkmem,NULL,0,djpcb,8);
    if (r != LZO_E_OK || size <= packedsize)
        return packerr="compression failed",-1;

    ratio = get_ratio (size,packedsize);
    sprintf (buf,"%8d -> %8d ratio:%3d.%01d%%",size,packedsize,ratio/100,(ratio%100)/10);
    printline (buf,40,38);

    /* optimize compressed data */
    new_size=size;
    if (lzo1x_optimize (out,packedsize,in,&new_size,wrkmem) != LZO_E_OK || new_size != size)
        return printf ("\n%d\n",size),packerr="optimization failed",-1;

    /* decompress and verify */
    new_size=size; /* safe decompression */
    if (lzo1x_decompress_safe (out,packedsize,in,&new_size,wrkmem) != LZO_E_OK || new_size != size)
        return printf ("\n%d\n",size),packerr="decompression failed",-1;

    /* compute checksum of decompressed data */
    adler2 = lzo_adler32 (0,NULL,0);
    adler2 = lzo_adler32 (adler2,in,new_size);
    if (adler1 != adler2)
        return packerr="decompression data error",-1;

    /* put both checksums into the stub */
    adler2 = lzo_adler32 (0,NULL,0);
    adler2 = lzo_adler32 (adler2,out,packedsize);
    if (chksumpos!=NULL)
        if (!(options & OPTS))
        {
            *(long*) chksumpos=adler2;
            *(long*) (chksumpos+4)=adler1;
        }
        else
        {
            sprintf (buf,"%08X%08X",adler2,adler1);
            memcpy (chksumpos,buf,16);
        }
        
    total_files++;
    total_size+=size;
    total_packedsize+=packedsize;
    return packedsize;
}

/*
    It seems, I can't write a general overlay copying function,
    because I can't easily find the end of the debug info, when
    the overlay is appended after the debug info.
*/
static const char *handle_allegropak (void)
{
    unsigned long buf[2],ic;
    if (fseek (fi,-8,2) || fread (&buf,8,1,fi)!=1 || buf[0]!=*(unsigned long*)"slh+")
         return 0;
    buf[1]=ntohl (buf[1]);
    if (fseek (fi,-buf[1],2) || fseek (fo,0,2))
         return "can't copy Allegro pakfile";
    while (buf[1])
    {
        ic=buf[1]>sizeof (wrkmem) ? sizeof (wrkmem) : buf[1];
        if (fread (wrkmem,ic,1,fi)!=1 || fwrite (wrkmem,ic,1,fo)!=1)
             return "can't copy Allegro pakfile";
        buf[1]-=ic;
    }
    return NULL;
}

static const char *dorenames (const char *program,char *tmp_file)
{
    int nodel;
#ifdef __DJGPP__
    struct ftime t;
    getftime (fileno (fi),&t);
#endif

    handle_allegropak ();
    fclose (fi);
    fclose (fo);
    fi=fo=NULL;

    if ((nodel=((exetype!=0) ^ (!(options & OPTC)))))
        strcpy (wrkmem,program);
    else
    {
        maketempname (wrkmem,program,".~",0);   /* backup file name */

        if (rename (program,wrkmem))
            return "can't rename";
    }

    maketempname (wrkmem+1000,program,(options & OPTC) ? ".cof" : ".exe",1);
    if (rename (tmp_file,wrkmem+1000))
    {
        rename (wrkmem,program);                /* undo rename */
        return packerr="can't rename .~ file";
    }

    if (!(options & OPTB) && !nodel)
        remove (wrkmem);

    if ((fo=fopen (wrkmem+1000,"rb"))!=NULL)    /* copy timestamp */
#ifdef __DJGPP__
        setftime (fileno (fo),&t);
#else
        ((void)0);
#endif
    return NULL;
}

static void stripdbg (unsigned long *coffhdr)
{
    /* "strip" debug info */
    coffhdr[12+6]=coffhdr[12+7]=coffhdr[12+8]=coffhdr[2]=coffhdr[3]=0;
    coffhdr[4]=(coffhdr[4]&0xFFFF)|(0x010F0000);
}

static int is_dlm (void)
{
    long off;
    return !(fseek (fi,-4,2) || fread (&off,4,1,fi)!=1 ||
             off < 0 || off > ftell (fi) ||
             fseek (fi,off,0) || fread (&off,4,1,fi)!=1 ||
             off!=*(long*)"DLMF");
}

#define textfpos coffhdr[12+5]
#define textsize coffhdr[12+4]
#define textvirt coffhdr[12+3]
#define datafpos coffhdr[22+5]
#define datasize coffhdr[22+4]
#define datavirt coffhdr[22+3]
#define bsssize  coffhdr[32+4]
#define bssvirt  coffhdr[32+3]
#define bssphys  coffhdr[32+2]

static const char *v2packimage (const char *program)
{
    unsigned short header[0x1c/2];
    char     magic[8],*in,*out;

    unsigned coff_offset=0,size;
    unsigned long coffhdr[42];

    exetype=0;
    printline (program,0,38);
    printline ("packing...",40,30);
    if ((fi = fopen (program,"rb"))==NULL)
        return "can't open";
    if (fread (header,sizeof(header),1,fi)!=1)
        return "read error";
    if (header[0] == 0x5a4d)                    /* MZ exe signature, stubbed? */
    {
        coff_offset = (long)header[2]*512L;
        if (header[1])
            coff_offset += (long)header[1] - 512L;
        if (fseek (fi, 512, 0) || fread (magic,8,1,fi)!=1) /* Position of V2 stubinfo */
            return "not V2 image";              /* Not V2 image, show failure */
        if (memcmp (SIMAGIC,magic,8) != 0)
            return "not V2 image";              /* Not V2 image, show failure */
        exetype=1;

        if (fseek (fi,0,0) || fread (wrkmem,coff_offset,1,fi)!=1)
            return "can't read stub";
        if (is_dlm ())
            return "can't pack DLMs";
    }
    /* Check for real image, (for now assume imbedded) */
    if (fseek (fi,coff_offset,0) || fread (coffhdr,0x0a8,1,fi)!=1)
        return "can't read COFF header";
    if ((short)coffhdr[0] != 0x014c)            /* COFF? */
        return "not V2 image";                  /* Not V2 image, show failure */
    if ((coffhdr[4] & 0x00020000)==0)
        return "COFF not executable";           /* Not executable */

    if (textsize+datasize != coffhdr[6]+coffhdr[7])
        return "already packed";
    if (textvirt+textsize != datavirt || datavirt+datasize != bssvirt)
    {
        if (textvirt+textsize < datavirt &&
            datavirt-textvirt == datafpos-textfpos)
        {
            coffhdr[6]=textsize=datavirt-textvirt;
            /* This needs to compress Quake! */
        }
        else
            return "already packed";
    }

    maketempname (tmp_file,program,".DjP",0);

    if ((fo=fopen (tmp_file,"wb"))==NULL)
        return "can't open temp file";

    /* keep original stub */
    if (!(options & (OPTS|OPTC)) && exetype && fwrite (wrkmem,coff_offset,1,fo)!=1)
        return "write error";

    stripdbg (coffhdr);

    size=textsize+datasize;
    in=(char*) malloc (size+1024);
    out=(char*) malloc (size+size/64+1024);
    textsize=datasize=pack_sections (in,out,coff_offset,size,textfpos,sizeof(coffhdr)+SCNHSZ*((coffhdr[0]>>16)-3));

    if (in!=NULL)
        free (in);

    if (packerr!=NULL)
    {
        if (out!=NULL)
            free (out);
        return packerr;
    }

    textfpos=0xa8;
    if (!(options & OPTS))
    {
        unsigned char decomp2[sizeof(decomph)];
        unsigned      t1;

        memcpy (decomp2,decomph,sizeof(decomph));
        if (bsssize < 0x200)
            bsssize = 0x200;            /* give it a .bss */

        textsize=sizeof(decomph);       /* new size of .text */
        while (1)
        {
            datafpos=textfpos+textsize; /* offset of raw data */
            if ((t1=((datafpos+datasize)&0x1ff)) >= EOFLEN && t1 < 0x1fc)
                break;
            textsize+=4;
        }

        datavirt=bssvirt+t1-datasize;
        if (patch_decomp (decomp2,0xdeadbee0,((size+0x1ff)&~0x1ff)+(textvirt&~0x1ff)) ||
            patch_decomp (decomp2,0xdeadbee1,textvirt&~0x1ff) ||
            patch_decomp (decomp2,0xdeadbee2,datasize+datavirt) ||
            patch_decomp (decomp2,0xdeadbee3,datavirt) ||
            patch_decomp (decomp2,0xdeadbee4,bssvirt) ||
            patch_decomp (decomp2,0xdeadbee5,0x200/4) ||
            patch_decomp (decomp2,0xdeadbee6,coffhdr[9]))
        {
            printline ("bad decompressor file",40,30);
            exit (1);
        }
        textvirt=coffhdr[9]=0xa8;
        bssvirt+=0x200;
        bsssize-=0x200;

        /* because of a feature (bug?) in stub.asm we need some padding */
        t1=4-(datasize & 3);
        memset (out+datasize,t1,t1);
        datasize+=t1;

        if (options & OPT1)
            (decomp2+(chksumpos-(char*)decomph))[-1]='x';

        if ((!(options & OPTC) && exetype==0 && fwrite (stub,sizeof (stub),1,fo)!=1) ||
            fwrite (coffhdr,sizeof (coffhdr),1,fo)!=1 ||
            fwrite (decomp2,textsize,1,fo)!=1 ||
            fwrite (out,datasize,1,fo)!=1)
        {
            free (out);
            return "write error";
        }
    }
    else
    {
        findbytes (djpstub,sizeof (djpstub),"LZO1",4)[4]=(options & OPT1) ? 'x' : 'X';
        datafpos=datasize=0;
        textvirt=((((bssvirt+bsssize)+0xFFFF)&~0xFFFF)-textsize-512)&~511;
        if (fwrite (djpstub,sizeof (djpstub),1,fo)!=1 ||
            fwrite (coffhdr,sizeof (coffhdr),1,fo)!=1 ||
            fwrite (out,textsize,1,fo)!=1)
        {
            free (out);
            return "write error";
        }
    }
    free (out);

    return dorenames (program,tmp_file);
}

static int checksumgood (char *in,unsigned size)
{
    char     buf [20];
    unsigned adler1;

    if (chksumpos==NULL)
    {
        if (options & OPTT)
            printline ("no checksum",40,30);
        return 1;
    }
    adler1 = lzo_adler32 (0,NULL,0);
    adler1 = lzo_adler32 (adler1,in,size);
    sprintf (buf,"%08X",adler1);
    if (memcmp (buf,chksumpos,8)==0)
        return 1;
    packerr="bad checksum";
    return 0;
}

static int unpack_section (char *in,char *out,unsigned foffset,
                           unsigned isize,unsigned osize,int pmethod)
{
    unsigned unpackedsize;
    char buf[80],method=pmethod;

    packerr=NULL;
    if (in==NULL || out==NULL)
        return packerr="out of memory",-1;

    if (options & OPTL)
    {
        if (fseek (fi,foffset+isize-1,0) || fread (&in[isize-1],1,1,fi)!=1)
            return packerr="read error",-1;
    }
    else
    {
        if (fseek (fi,foffset,0) || fread (in,isize,1,fi)!=1)
            return packerr="read error",-1;
    }

    if (exetype!=2)                     /* remove padding */
    {
        if (in[isize-1]<1 || in[isize-1]>4)
            return packerr="read error",-1;
        isize-=in[isize-1];
    }
    if (!(options & OPTL) && !checksumgood (in,isize))
        return -1;

    if (method>='a' && method<='z')
        method+='A'-'a';
#ifdef NEED_LZO1C
    if (!(method=='/' || method=='F' || method=='X' || method=='C'))
#else
    if (!(method=='/' || method=='F' || method=='X' ||
         (method=='C' && (options & OPTL))))
#endif
        return packerr="unknown compression algorithm",-1;
    unpackedsize=osize; /* for safe decompression */

    if (options & OPTL)
    {
        unsigned ratio = get_ratio (osize,isize);
        total_files++;
        total_size+=osize;
        total_packedsize+=isize;

        sprintf (buf,"%8d -> %8d ratio:%3d.%01d%%",osize,isize,ratio/100,(ratio%100)/10);
        printline (buf,40,38);
        strcpy (buf,"LZO1 ");
        if (method!='/')
            buf[4]=pmethod;
        printline (buf,74,5);
        return 0;
    }

    /* Always use the safe decompressors */
    if ((method=='F' && lzo1f_decompress_safe (in,isize,out,&unpackedsize,NULL)) ||
#ifdef NEED_LZO1C
        (method=='C' && lzo1c_decompress_safe (in,isize,out,&unpackedsize,NULL)) ||
#endif
        (method=='X' && lzo1x_decompress_safe (in,isize,out,&unpackedsize,NULL)) ||
        (method=='/' && lzo1_decompress (in,isize,out,&unpackedsize,NULL)) ||
         unpackedsize!=osize)
    {
        return packerr="decompression failed",-1;
    }

    total_files++;
    total_size+=osize;
    total_packedsize+=isize;

    if (options & OPTT)
    {
        if (chksumpos!=NULL)
            printline ("ok!",40,30);
        return 0;
    }

    sprintf (buf,"%8d -> %8d",isize,unpackedsize);
    printline (buf,40,30);

    if (fwrite (out,unpackedsize,1,fo)!=1)
        return packerr="write error",-1;
    return unpackedsize;
}

static const char *v2unpackimage (const char *program)
{
    char     magic[20],*p1,*in,*out;
    unsigned coff_offset=0,size,osize,beginaddr;
    unsigned long coffhdr[42];
    unsigned short header[512/2];
    int      method='/';                        /* for LZO1/5-99 */

    exetype=0;
    printline (program,0,38);
    if ((fi = fopen (program,"rb"))==NULL)
        return "can't open";
    if (fread (header,sizeof(header),1,fi)!=1)
        return "not V2 image";                  /* Not V2 image, show failure */
    if (header[0] == 0x5a4d)                    /* MZ exe signature, stubbed? */
    {
        if (header[0x1a/2] == 0x6a64)           /* 'dj' */
            exetype++;
        coff_offset = (long)header[2]*512L;
        if (header[1])
            coff_offset += (long)header[1] - 512L;
        if (fseek (fi, 512, 0) || fread (magic,8,1,fi)!=1) /* Position of V2 stubinfo */
            return "read error";
        if (memcmp (SIMAGIC,magic,8) != 0)
            return "not V2 image";              /* Not V2 image, show failure */
        exetype++;
        if (fseek (fi,0,0) || fread (wrkmem,coff_offset,1,fi)!=1)
            return "read error";
    }

    if (fseek (fi,coff_offset,0) || fread (coffhdr,0x0a8,1,fi)!=1)
        return "can't find COFF header";
    if ((short)coffhdr[0] != 0x014c)            /* COFF? */
        return "not V2 image";                  /* Not V2 image, show failure */
    if ((coffhdr[4] & 0x00020000)==0)
        return "COFF not executable";           /* Not executable */

    maketempname (tmp_file,program,".DjP",0);
    if ((fo=fopen ((options & OPTD) ? tmp_file : "/dev/null","wb"))==NULL)
        return "can't open temp file";

    if (exetype==2)                             /* stubbed decompressor */
    {
        if ((chksumpos=findbytes (header,sizeof (header),"Checksum: ",10))!=NULL)
            chksumpos+=10;
        if ((p1=findbytes (header,sizeof (header),"LZO1",4))!=NULL)
            method=p1[4];                           /* 'X' for LZO1X */
        if (!(options & OPTC) && fwrite (stub,sizeof (stub),1,fo)!=1) /* stubify */
            return "write error";
    }
    else
    {
        if (fread (wrkmem+coff_offset,1024,1,fi)!=1)
            return "read error";
        if ((p1=findbytes (wrkmem+coff_offset,1024,"DJP ",4))==NULL)
            return "not compressed with DJP";
        if ((p1=(char*) memchr (p1,0,10))==NULL)
            return "not DJP header";
        method=*(p1+1);                         /* decompression algorithm */
        sprintf (chksumpos=magic,"%08X%08X",*(unsigned*)(p1+2),*(unsigned*)(p1+6)); /* checksums */

        if ((exetype==0 && !(options & OPTC) &&
            fwrite (stub,sizeof (stub),1,fo)!=1) || /* stubify */
            (exetype && !(options & OPTC) &&
            fwrite (wrkmem,coff_offset,1,fo)!=1)) /* copy the stub */
        {
            return "write error";
        }
    }

    size=textsize > datasize ? textsize : datasize;

    beginaddr=coffhdr[10]&~0x1ff;
    osize=bssphys-beginaddr;
    in=(char*)malloc (size+512);
    out=(char*)malloc (coffhdr[6]+coffhdr[7]+512); /* original size from the a.out header */

    if (exetype==2)
    {
        unsigned char tempch[sizeof (coffhdr)];
        if (unpack_section (in,out,coff_offset+0xa8,textsize,datasize ? coffhdr[6]+0xa8 : osize,method)==-1 ||
            memcpy (tempch,out,sizeof (coffhdr))==NULL ||
            (datasize && unpack_section (in,out,coff_offset+0xa8+textsize,datasize,coffhdr[7],method)==-1))
        {}
        memcpy (coffhdr,tempch,sizeof (coffhdr));
    }
    else
    {
        unpack_section (in,out,coff_offset+datafpos,datasize,osize,method);
        memcpy (coffhdr,out,sizeof (coffhdr)); /* restore original coff header */
    }
    stripdbg (coffhdr);
    textfpos&=0x1ff; /* this can be wrong */
    datafpos=textfpos+textsize;

    if (chksumpos)
        chksumpos+=8;
    if ((method|=0x20)==('X'|0x20) && !(options & OPTL) && !checksumgood (out,osize))
        return packerr;

    if (in!=NULL)
        free (in);
    if (out!=NULL)
        free (out);
    in = out = NULL;
    if (packerr!=NULL)
        return packerr;

    if (options & (OPTT|OPTL))
        return packerr;                         /* test or list */
    if (fseek (fo,(options & OPTC) ? 0 : exetype!=1 ? sizeof (stub) : coff_offset,0) ||
        fwrite (coffhdr,sizeof(coffhdr),1,fo)!=1)
    {
        return "write error";
    }

    return dorenames (program,tmp_file);
}

#ifdef __DJGPP__
/* Don't link this function from libc.a ==> -1600 bytes */
int _is_executable (const char *p,int i,const char *q)
{
    return 0;
}
#endif

static void license (void)
{
    printf (
"   DJP and the LZO library are free software; you can redistribute them\n"
"   and/or modify them under the terms of the GNU General Public License as\n"
"   published by the Free Software Foundation; either version 2 of\n"
"   the License, or (at your option) any later version.\n"
"\n"
"   This program is distributed in the hope that it will be useful,\n"
"   but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"   GNU General Public License for more details.\n"
"\n"
"   You should have received a copy of the GNU General Public License\n"
"   along with this program; see the file COPYING.\n"
"   If not, write to the Free Software Foundation, Inc.,\n"
"   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.\n"
"\n"
"   For more information visit:\n"
"   http://cdata.tvnet.hu/~ml/\n"
"   http://wildsau.idv.uni-linz.ac.at/mfx/lzo.html\n"
    );
    exit(0);
}

#define BAD_OPTION 0x40000000
static int checkoption (int ch)
{
    static const char *onames="dstbqc0h1li";
    const char *p;

    if ((p=strchr (onames,ch|0x20))==NULL)
        return BAD_OPTION;
    return 1 << (p-onames);
}

int main (int argc,char **argv)
{
    int waserror=0,ic,jc,opt2=0;
    const char *p;
    char  buf[80];

#ifdef __EMX__
    _response(&argc,&argv);
    _wildcard(&argc,&argv);
#endif

    argv0 = argv[0];
    isatty_stdout = isatty(fileno(stdout));

    if (lzo_init()!=LZO_E_OK)
        return 3;

    if ((p=getenv ("DJP_OPTIONS"))!=NULL)   /* get the default options */
        for (ic=0; ic<(int)strlen(p); ic++) /* from the environment */
            opt2 |= checkoption (p[ic]);
    opt2 &= OPTS|OPTC|OPTB|OPTQ;            /* use only these options */

    for (ic=1; ic<argc && argv[ic][0]=='-'; ic++)
        for (jc=1; jc<(int)strlen(argv[ic]); jc++)
            if ((options |= checkoption (argv[ic][jc])) >= BAD_OPTION)
            {
                printf ("%s: unknown option '-%c' !\n",argv0,argv[ic][jc]);
                return 1;
            }

    if ((options & OPTI) || !(options & OPTQ))
        printf ("\nDJP 1.07 executable file compressor for DJGPP V2 programs.\n"
                "(C) 1996-1998 Laszlo Molnar <ml1050@cdata.tvnet.hu>\n"
                "LZO data compression (C) 1996-1998 Markus Franz Xaver Johannes Oberhumer\n\n");

    if (options & OPTI)
        license();

    if (ic>=argc || (options & OPTH))
    {
        if (options & OPTQ)
            printf ("\n-qh ?? ok, here is the help in quiet mode :-)\n");
        else
            printf ("usage: %s [options] file...\n"
                    "options:\n"
		    "  -d decompress\t\t\t"
		    	"-l list compression ratio\n"
                    "  -t integrity test\n"
                    "  -b %s backup files  \t"
                    	"-1 fast compression\n"
                    "  -c %s output   \t\t"
                    	"-s put the decompression code into %s\n"
		    "  -q %s\t\t\t"
                    	"-0 ignore default options from DJP_OPTIONS\n"
                    "  -h this help\t\t\t"
		    	"-i display software license\n"
                    "file can be .exe or COFF\n\n",
		    argv0,
                    (opt2 & OPTB) ? "delete" : "keep",
                    (opt2 & OPTC) ? "EXE" : "COFF",
                    (opt2 & OPTS) ? ".text" : "the stub",
                    (opt2 & OPTQ) ? "verbose" : "quiet"
                    );
        return 1;
    }

    if (!(options & OPT0))
        options ^= opt2;   /* apply the default options */
    if ((options & (OPTC|OPTS|OPTT|OPTD))==(OPTC|OPTS))
    {
        printf ("%s: -sc ?? sorry, this won't work.\n",argv0);
        return 1;
    }

    if (!(options & OPTS))
        chksumpos=findbytes (decomph,sizeof (decomph),"Checksum",8);
    else if ((chksumpos=findbytes (djpstub,sizeof (djpstub),"Checksum: ",10))!=NULL)
        chksumpos+=10;

    for (; ic<argc; ic++)
    {
        djpcb_perc=-1;
        fi=fo=NULL;
        tmp_file[0]=0;
        if (options & (OPTD|OPTT|OPTL))
            p=v2unpackimage (argv[ic]);
        else
            p=v2packimage (argv[ic]);
        if (p!=NULL)
        {
            sprintf (buf,"%s!",p);
            printline (buf,40,30);
            waserror++;
            if (options & OPTQ)         /* print error in quiet mode */
                printf ("%s\n",line);
        }
        if (!(options & OPTQ))
        {
            if (!isatty_stdout)         /* print full line if !isatty(1) */
                printf ("%s",line);
            printf ("\n");
        }
        if (fi!=NULL)
            fclose (fi);
        if (fo!=NULL)
            fclose (fo);
        if (tmp_file[0])
            remove (tmp_file);
    }

    if (total_files>1 && !(options & OPTT) && !(options & OPTQ))
    {
        unsigned sw,ratio = get_ratio (total_size,total_packedsize);
        if (options & OPTD)
            sw=total_size,total_size=total_packedsize,total_packedsize=sw;
        printf ("--------------------------------------------------------------------------\n");
        sprintf (buf,"  %d files",total_files);
        printf ("%-40s%8d -> %8d ratio:%3d.%01d%%\n",buf,total_size,
                 total_packedsize,ratio/100,(ratio%100)/10);
    }

    return waserror ? 2 : 0;
}

/*
Todo:

-  keep stubinfo with -s

*/


