/****************************************************************************/
/* TOBJECT                                                                  */
/*--------------------------------------------------------------------------*/
/* Objet TObject (type gnrique pour les lments de l'interface JPTUI)    */
/*--------------------------------------------------------------------------*/
/* Auteur     : DELPRAT Jean-Pierre                                         */
/* Cr le    : 10/01/95                                                    */
/****************************************************************************/

#include <conio.h>
#include <stdlib.h>
#include <values.h>

#include "Const.h"
#include "JPDebug.h"

#include "JPAppli.h"

#include "Callback.h"
#include "Errors.h"
#include "Strings.h"

#include "TObject.h"
#include "TApplica.h"
#include "TWindow.h"

/*ͻ*/
/*                           METHODES PUBLIQUES                           */
/*ͼ*/

/****************************************************************************/
/* Constructeur                                                             */
/*--------------------------------------------------------------------------*/
/* parent           : Objet auquel appartient l'objet                       */
/* type             : Type de l'objet                                       */
/* rel_x,rel_y      : Coordonnes de l'objet p/r au groupe                  */
/* width,height     : Dimensions de l'objet                                 */
/* background       : Couleur de fond                                       */
/* caption          : Lgende de l'objet (hot-key prcd de ~)             */
/* enabled          : ENABLED si l'objet est activable (DISABLED sinon)     */
/* focus_depending_                                                         */
/*   aspect         : Indique si l'aspect de l'objet varie quand il prend ou*/
/*                    perd le focus                                         */
/* can_be_enabled   : Indique si l'objet peut devenir activable             */
/* simple           : Indique si l'objet est simple (non compos d'autres)  */
/****************************************************************************/

TObject::TObject(PObject parent,
		 int type,
		 int rel_x,int rel_y,
		 int width,int height,
		 unsigned background,
		 const char *caption,
		 boolean enabled,
		 boolean focus_depending_aspect,
                 boolean can_be_enabled,
		 boolean simple)
 : focus_lost_callback(NULL), focus_taken_callback(NULL)
{
  if (width<0)
    width=0;

  if (height<0)
    height=0;

  if ((parent==NULL) && (type!=OBJ_WINDOW))
    {
      parent=new TWindow(background,
			 1,1,
			 width,height,
			 "",
			 NO_INFO_BAR,
			 MODAL,
			 NOT_MOVABLE);
      f_dynamic_window=TRUE;
      rel_x=rel_y=0;
    }
  else
    f_dynamic_window=FALSE;


  // Fentre de l'objet

  if (parent!=NULL)
    {
      f_window=parent->f_window;

      // Impossible de rajouter un objet  une fentre ouverte

      JPDEBUG_TEST(DEBUG_ERROR_1,!f_window->m_is_open());
    }
  else
    {
      // Pas de parent, c'est donc une fentre

      f_window=(PWindow)this;
    }

  // Parent de l'objet

  f_parent=parent;

  // Type de l'objet
  // -> pas forcment redfini pour les objets drivs

  f_type=type;

  // Coordonnes de l'objet p/r au groupe le contenant
  // et ses dimensions

  if (f_dynamic_window)
    {
      f_rel_x=0;
      f_rel_y=0;
    }
  else
    {
      f_rel_x =rel_x;
      f_rel_y =rel_y;
    }

  f_width =width;
  f_height=height;

  // Couleur de fond

  f_background=background;

  // Lgende de l'objet

  if (caption==NULL)
    caption="";

  f_caption=new char [strlen(caption)+1];
  strcpy(f_caption,caption);

  // Hot-Key de l'objet (0 si aucune)

  f_hot_key=HotKey(caption);

  // Etat de l'objet

  f_open=FALSE;
  f_focused=FALSE;


  // L'objet est simple

  f_simple=simple;

  // Elments de l'objet
  // (Objet compos)

  f_nb_elements=0;
  f_element_list=NULL;
  f_last_element=NULL;

  // Si le groupe n'est pas activable, l'objet non plus

  if (parent!=NULL)
    {
      if (!parent->f_enabled)
	enabled=FALSE;
    }
  f_enabled=enabled;

  // L'objet peut-il devenir activable ?

  f_can_normally_be_enabled=can_be_enabled;
  f_can_be_enabled=can_be_enabled;

  if (!can_be_enabled)
    f_enabled=FALSE;

  // L'aspect de l'objet varie-t-il quand il prend ou perd le focus ?

  f_focus_depending_aspect=focus_depending_aspect;

  // L'affichage des objets contenus dans le groupe est autoris

  f_elements_display_enabled=TRUE;

  // Callbacks

  InitCallback(f_focus_lost_action,f_focus_lost_argument);
  InitCallback(f_focus_taken_action,f_focus_taken_argument);
  InitCallback(f_opened_action,f_opened_argument);
  InitCallback(f_closed_action,f_closed_argument);

  // Numro de l'objet dans la fentre

  if (parent!=NULL)
    f_number=(parent->m_add_element(this));
  else
    f_number=1;

}

/****************************************************************************/
/* Destructeur                                                              */
/*--------------------------------------------------------------------------*/
/****************************************************************************/

TObject::~TObject()
{
  // La fentre doit tre ferme avant de dtruire l'objet

  JPDEBUG_TEST(DEBUG_ERROR_4,!f_window->m_is_open());

  // Pointeur des autres objets sur l'objet detruit

  if (f_window->f_previous_focused_object==this)
    f_window->f_previous_focused_object=NULL;

  // Suppression de l'lment dans son parent

  if (f_parent!=NULL)
    f_parent->m_del_element(f_number);

  // Destruction des variables dynamiques

  DestroyCallback(f_closed_action,f_closed_argument);
  DestroyCallback(f_opened_action,f_opened_argument);
  DestroyCallback(f_focus_taken_action,f_focus_taken_argument);
  DestroyCallback(f_focus_lost_action,f_focus_lost_argument);

  if (f_caption!=NULL)
    {
      delete [] f_caption;
      f_caption=NULL;
    }

  // Tous les lments doivent avoir t supprims

  JPDEBUG_TEST(DEBUG_ERROR_2,f_element_list==NULL);

  // Suppression de la fentre si elle a t alloue par l'objet

  if (f_dynamic_window)
    delete f_window;
}

/****************************************************************************/
/* m_get_x_in_window                                                        */
/*--------------------------------------------------------------------------*/
/* Retourne l'abscisse de l'objet p/r  la fentre qui le contient          */
/****************************************************************************/

int TObject::m_get_x_in_window()
{
  if (f_parent!=NULL)
    return (f_parent->m_get_x_in_window()+f_rel_x);

  return (0);
}

/****************************************************************************/
/* m_get_y_in_window                                                        */
/*--------------------------------------------------------------------------*/
/* Retourne l'ordonne de l'objet p/r  la fentre qui le contient          */
/****************************************************************************/

int TObject::m_get_y_in_window()
{
  if (f_parent!=NULL)
    return (f_parent->m_get_y_in_window()+f_rel_y);

  return (0);
}

/****************************************************************************/
/* m_get_x                                                                  */
/*--------------------------------------------------------------------------*/
/* Retourne l'abscisse de l'objet dans l'cran                              */
/****************************************************************************/

int TObject::m_get_x()
{
  return(m_get_x_in_window()+f_window->m_get_x());
}

/****************************************************************************/
/* m_get_y                                                                  */
/*--------------------------------------------------------------------------*/
/* Retourne l'ordonne de l'objet dans l'cran                              */
/****************************************************************************/

int TObject::m_get_y()
{
  return(m_get_y_in_window()+f_window->m_get_y());
}

/****************************************************************************/
/* m_set_size                                                               */
/*--------------------------------------------------------------------------*/
/* Modifie la taille de l'objet (s'il est ferm).                           */
/****************************************************************************/

void TObject::m_set_size(int width,int height)
{
  if (f_open)
    return;

  f_width=width;
  f_height=height;
}

/****************************************************************************/
/* m_set_caption                                                            */
/*--------------------------------------------------------------------------*/
/* Permet de modifier la lgende d'un objet (s'il en a dj une)            */
/****************************************************************************/

void TObject::m_set_caption(const char *caption)
{
  DEBUG_TEST(caption!=NULL);

  // Ne peut tre appel que si l'objet  dj une lgende,

  if (f_caption[0]==0)
    return;

  // On ne peut alors lui attribuer qu'une lgende non nulle

  if (caption[0]==0)
    caption=" ";

  delete []f_caption;

  f_caption=new char [strlen(caption)+1];
  strcpy(f_caption,caption);

  // Hot-Key de l'objet (0 si aucune)

  f_hot_key=HotKey(caption);

  // On raffiche la lgende

  m_display_caption();
}

/****************************************************************************/
/* m_enable                                                                 */
/*--------------------------------------------------------------------------*/
/* Rend l'objet activable                                                   */
/****************************************************************************/

void TObject::m_enable()
{
  PObjectNode node;

  if (!f_can_be_enabled)
    return;

  // Pour rendre un objet activable, il faut que son parent
  // le soit aussi

  if (f_parent!=NULL)
    {
      if (!f_parent->f_enabled)
        f_parent->m_enable();
    }

  if (!f_enabled)         // L'objet peut dj avoir t rendu activable
    {		          // avec l'appel  f_group->m_enable()
      f_enabled=TRUE;
      f_elements_display_enabled=FALSE;
      m_display();
      f_elements_display_enabled=TRUE;
    }

  // Rend activable les composants ventuels

  node=f_element_list;
  while (node!=NULL)
    {
      node->object->m_enable();  // Affichage si besoin
      node=node->next;
    }
}

/****************************************************************************/
/* m_disable                                                                */
/*--------------------------------------------------------------------------*/
/* Rend l'objet inactivable                                                 */
/****************************************************************************/

void TObject::m_disable()
{
  PObjectNode node;

  if (f_enabled)
    {
      f_enabled=FALSE;
      f_elements_display_enabled=FALSE;
      m_display();
      f_elements_display_enabled=TRUE;
    }

  // Rend inactivable les composants ventuels

  node=f_element_list;
  while (node!=NULL)
    {
      node->object->m_disable();  // Affichage si besoin
      node=node->next;
    }

  // Si on dsactive l'objet qui a le focus

  if (f_focused)
    {
      // L'objet TWindow doit toujours retourner vrai avec la mthode m_set_focus
      if (!f_window->m_set_focus())
	JPDEBUG_STOP(DEBUG_ERROR_15);
    }
}

/****************************************************************************/
/* m_set_can_be_enabled                                                     */
/*--------------------------------------------------------------------------*/
/* Pour un objet qui peut normalement tre activable, change cette proprit*/
/* (inefficace si l'objet ne peut par dfinition pas l'tre)                */
/****************************************************************************/

void TObject::m_set_can_be_enabled(boolean can_be_enabled)
{
  if (!f_can_normally_be_enabled)
    return;

  if ((!can_be_enabled) && (f_enabled))
    m_disable();

  f_can_be_enabled=can_be_enabled;
}


/****************************************************************************/
/* m_can_lose_focus                                                         */
/*--------------------------------------------------------------------------*/
/* Indique si l'objet peut perdre le focus                                  */
/****************************************************************************/

boolean TObject::m_can_lose_focus()
{
  PObjectNode node;
  PObject element;

  boolean success;

  if (!f_focused)
    return(TRUE);

  success=TRUE;
  node=f_element_list;

  while ((node!=NULL) && (success))
    {
      element=node->object;
      if (element->f_focused)
        success=element->m_can_lose_focus();
      node=node->next;
    }
  return(success);
}

/****************************************************************************/
/* m_set_focus                                                              */
/*--------------------------------------------------------------------------*/
/* Donne le focus  l'objet. Retourne TRUE en cas de russite               */
/****************************************************************************/

boolean TObject::m_set_focus()
{
  if (!f_enabled)
    return(FALSE);

  if (f_focused)
    return(TRUE);

  if (f_parent!=NULL)
    return (f_parent->m_set_focus_to_element(f_number));

  m_take_focus();

  return(TRUE);
}

/****************************************************************************/
/* m_set_focus_to_last_element                                              */
/*--------------------------------------------------------------------------*/
/* Pour un groupe, cette mthode donne le focus au dernier objet qu'il      */
/* contient.                                                                */
/* Pour un objet simple, se contente de donner le focus  l'objet.          */
/* Retourne TRUE en cas de russite                                         */
/****************************************************************************/

boolean TObject::m_set_focus_to_last_element()
{
  return (m_set_focus());
}

/****************************************************************************/
/* m_set_focus_lost_callback                                                */
/*--------------------------------------------------------------------------*/
/* Dfinition du callback associ  la perte de focus de l'objet            */
/****************************************************************************/

void TObject::m_set_focus_lost_callback(void (*focus_lost_action)(PObject, const char *),
					const char *focus_lost_argument)
{
  SetCallback(f_focus_lost_action,f_focus_lost_argument,
	      focus_lost_action,focus_lost_argument);
}

void TObject::set_focus_lost_callback(JPTUI::base_callback_t* a_callback,
	 			      const char* a_callback_argument)
{
  focus_lost_callback = a_callback;
  focus_lost_callback_argument = a_callback_argument;
}

/****************************************************************************/
/* m_set_focus_taken_callback                                               */
/*--------------------------------------------------------------------------*/
/* Dfinition du callback associ  la prise de focus de l'objet            */
/****************************************************************************/

void TObject::m_set_focus_taken_callback(void (*focus_taken_action)(PObject, const char *),
					 const char *focus_taken_argument)
{
  SetCallback(f_focus_taken_action,f_focus_taken_argument,
	      focus_taken_action,focus_taken_argument);
}

void TObject::set_focus_taken_callback(JPTUI::base_callback_t* a_callback,
	 		  	       const char* a_callback_argument)
{
  focus_taken_callback = a_callback;
  focus_taken_callback_argument = a_callback_argument;
}

/****************************************************************************/
/* m_set_opened_callback                                                    */
/*--------------------------------------------------------------------------*/
/* Defines the function called when this object is opened                   */
/****************************************************************************/

void TObject::m_set_opened_callback(void (*opened_action)(PObject, const char *),
			            const char *opened_argument)
{
  SetCallback(f_opened_action,f_opened_argument,
	      opened_action,opened_argument);
}

/****************************************************************************/
/* m_set_closed_callback                                                    */
/*--------------------------------------------------------------------------*/
/* Defines the function called when this object is closed                   */
/****************************************************************************/

void TObject::m_set_closed_callback(void (*closed_action)(PObject, const char *),
				    const char *closed_argument)
{
  SetCallback(f_closed_action,f_closed_argument,
	      closed_action,closed_argument);
}

/*ͻ*/
/*                           METHODES PROTEGEES                           */
/*ͼ*/

/****************************************************************************/
/* m_add_element                                                            */
/*--------------------------------------------------------------------------*/
/* Ajoute un lment  l'objet. Retourne le numro qui lui est attribu     */
/****************************************************************************/

int TObject::m_add_element(PObject object)
{
  PObjectNode new_node;
  int object_number;

  // L'objet ne doit pas tre ouvert
  // L'objet doit tre compos

  JPDEBUG_TEST(DEBUG_ERROR_3,(!f_simple) && (!f_open));

  f_nb_elements++;

  new_node=new TObjectNode;
  new_node->object=object;
  new_node->next=NULL;
  new_node->last=f_last_element; // Peut tre NULL

  if (f_element_list==NULL)
    {
      object_number=1;
      f_element_list=new_node;
    }
  else
    {
      object_number=f_last_element->object->f_number+1;
      f_last_element->next=new_node;
    }

  f_last_element=new_node;

  return (object_number);
};

/****************************************/
/* m_del_element : Supprime un lment  */
/* -------------   de l'objet           */
/****************************************/

void TObject::m_del_element(int object_number)
{
  PObjectNode node,node_before,node_after;
  PObject object;

  // Suppression du noeud

  node=m_object_number_to_element_node(object_number);
  object=node->object;

  node_before=node->last;
  node_after=node->next;

  if (f_last_element==node)
    f_last_element=node_before;

  if (node_before!=NULL)
    node_before->next=node_after;
  else
    f_element_list=node_after;

  if (node_after!=NULL)
    node_after->last=node_before;

  delete node;

  f_nb_elements--;

  // Si l'lment supprim a le focus

  if (object->f_focused)
    {
      // L'objet TWindow doit toujours retourner vrai avec la mthode m_set_focus
      if (!f_window->m_set_focus())
	JPDEBUG_STOP(DEBUG_ERROR_15);
    }
};


/***********************************************************/
/* m_object_number_to_element : Retourne un objet  partir */
/* -------------------------   de son numro (1...)        */
/***********************************************************/

PObject TObject::m_object_number_to_element(int object_number)
{
  return(m_object_number_to_element_node(object_number)->object);
}

/***************************************************************************/
/* m_object_number_to_element_node : Retourne un pointeur sur l'lment de */
/* ------------------------------   la liste des lments de l'objet      */
/*                                  partir de son numro dans le groupe.   */
/***************************************************************************/

PObjectNode TObject::m_object_number_to_element_node(int object_number)
{
  register PObjectNode node;

  node=f_element_list;
  while (node!=NULL)
    {
      if (node->object->f_number==object_number)
	break;
      node=node->next;
    }

  JPDEBUG_TEST(DEBUG_ERROR_30,node!=NULL);

  return(node);
}

/**************************/
/* m_display : Affichage  */
/* ---------   de l'objet */
/**************************/

void TObject::m_display()
{
  PObjectNode node;

  if (!f_open)
    return;

  m_display_caption();

  if (f_elements_display_enabled)
    {

      node=f_element_list;
      while (node!=NULL)
        {
	  node->object->m_display();
          node=node->next;
        }
    }

}

/****************************************************************************/
/* m_display_focus_depending_part                                           */
/*--------------------------------------------------------------------------*/
/* Affichage de la partie de l'objet dont l'aspect dpend du focus          */
/****************************************************************************/

void TObject::m_display_focus_depending_part()
{
  // Penser  tester que l'objet est ouvert
}

/****************************************************************************/
/* m_display_caption                                                        */
/*--------------------------------------------------------------------------*/
/* Affichage de la lgende de l'objet                                       */
/****************************************************************************/

void TObject::m_display_caption()
{
  // Penser  tester que l'objet est ouvert
}

/****************************************************************************/
/* m_set_focus_to_element                                                   */
/*--------------------------------------------------------------------------*/
/* Attribution du focus  un lment de l'objet. Retourne TRUE si russite. */
/****************************************************************************/

boolean TObject::m_set_focus_to_element(int /*object_number*/)
{
  return(m_set_focus());
}


/****************************************/
/* m_lose_focus : Appele quand l'objet */
/* ------------   perd le focus         */
/****************************************/

void TObject::m_lose_focus()
{
  PObjectNode node;
  PObject element;

  if (f_focused)
    {
      node=f_element_list;

      while (node!=NULL)
        {
          element=node->object;
          if (element->f_focused)
            element->m_lose_focus();
	  node=node->next;
	}

      f_focused=FALSE;
      m_set_previous_focused_object();

      if ((f_focus_depending_aspect) && (f_window->m_is_active()))
	m_display_focus_depending_part();

      CallCallback(this,f_focus_lost_action,f_focus_lost_argument);
      if (focus_lost_callback != NULL)
        (*focus_lost_callback)(this, focus_lost_callback_argument.c_str());
    }
}

/****************************************/
/* m_take_focus : Appele quand l'objet */
/* ------------   prend le focus        */
/****************************************/

void TObject::m_take_focus()
{
  PObjectNode node;

  if ((!f_focused) && (f_enabled))
    {
      node=f_element_list;
      while (node!=NULL)
	{
	  node->object->m_take_focus();
	  node=node->next;
	}

      f_focused=TRUE;
      if ((f_focus_depending_aspect) && (f_window->m_is_active()))
	m_display_focus_depending_part();

      CallCallback(this,f_focus_taken_action,f_focus_taken_argument);
      if (focus_taken_callback != NULL)
        (*focus_taken_callback)(this, focus_taken_callback_argument.c_str());
    }
}


/*********************************************/
/* m_set_open : Ouvre/Ferme l'objet          */
/* ----------   (ne modifie pas l'affichage) */
/*********************************************/

void TObject::m_set_open(boolean open)
{
  PObjectNode node=f_element_list;

  if (open!=f_open)
    {
      while (node!=NULL)
	{
	  node->object->m_set_open(open);
          node=node->next;
	}
      f_open=open;

      if (open)
        m_opened_callback();
      else
        m_closed_callback();
    }
}

/***************************************************/
/* m_set_active : L'objet devient/n'est plus actif */
/* ------------   (ne modifie pas l'affichage)     */
/***************************************************/

void TObject::m_set_active(boolean active)
{
  PObjectNode node=f_element_list;

  while (node!=NULL)
    {
      node->object->m_set_active(active);
      node=node->next;
    }
}

/****************************************************************************/
/* m_opened_callback                                                        */
/*--------------------------------------------------------------------------*/
/* Called when the object is opened                                         */
/****************************************************************************/

void TObject::m_opened_callback()
{
  CallCallback(this,f_opened_action,f_opened_argument);
}

/****************************************************************************/
/* m_closed_callback                                                        */
/*--------------------------------------------------------------------------*/
/* Called when the object is opened                                         */
/****************************************************************************/

void TObject::m_closed_callback()
{
 CallCallback(this,f_closed_action,f_closed_argument);
}

/*********************************************************************/
/* m_left_button_pressed_event : L'utilisateur a cliqu dans l'objet */
/* ---------------------------   avec le bouton gauche               */
/*                               (l'objet tant activable).          */
/*                               Retourne TRUE si l'objet est        */
/*                               intress par cet vnement.        */
/*********************************************************************/

boolean TObject::m_left_button_pressed_event(int x,int y)
{
  PObjectNode node;
  PObject element;

  int  x1_element,y1_element;
  int  x2_element,y2_element;

  int  x1=m_get_x();
  int  y1=m_get_y();

  boolean event=FALSE;


  // Voir les vnements sur l'objet lui-mme,
  // pas sur un de ses composants
  // (Pour les objets drivs)

  // Clic sur un des objets composant l'objet

  node=f_element_list;

  while ((node!=NULL) && (!event))
    {
      element=node->object;
      x1_element=x1         + element->f_rel_x;
      x2_element=x1_element + element->f_width-1;
      y1_element=y1         + element->f_rel_y;
      y2_element=y1_element + element->f_height-1;

      if (element->f_enabled)
        {
	  if (   (x >= x1_element)
	      && (x <= x2_element)
	      && (y >= y1_element)
	      && (y <= y2_element)
	     )
	    event=element->m_left_button_pressed_event(x,y);
        }
      node=node->next;
    }

  return(event);
}

/**************************************************************************/
/* m_left_button_double_click_event : L'utilisateur a double-cliqu dans  */
/* --------------------------------   l'objet avec le bouton gauche       */
/*                                    (l'objet tant activable et         */
/*                                    ayant dj subi l'vnement         */
/*                                    m_left_button_pressed_event)        */
/*                                    Retourne TRUE si l'objet est        */
/*                                    intress par cet vnement.        */
/**************************************************************************/

boolean TObject::m_left_button_double_click_event(int x,int y)
{
  PObjectNode node;
  PObject element;

  int  x1_element,y1_element;
  int  x2_element,y2_element;

  int  x1=m_get_x();
  int  y1=m_get_y();

  boolean event=FALSE;

  // Voir les vnements sur l'objet lui-mme,
  // pas sur un de ses composants
  // (Pour les objets drivs)

  // Double-Clic sur un des objets composant l'objet

  node=f_element_list;

  while ((node!=NULL) && (!event))
    {
      element=node->object;
      x1_element=x1         + element->f_rel_x;
      x2_element=x1_element + element->f_width-1;
      y1_element=y1         + element->f_rel_y;
      y2_element=y1_element + element->f_height-1;

      if (element->f_enabled)
        {
	  if (   (x >= x1_element)
	      && (x <= x2_element)
	      && (y >= y1_element)
	      && (y <= y2_element)
	     )
	    event=element->m_left_button_double_click_event(x,y);
        }
      node=node->next;
    }

  return(event);
}

/****************************************************************************/
/* m_key_pressed_event                                                      */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur une touche qui est propose  l'objet (qui    */
/* est activable). Retourne TRUE si l'objet est intress par cette touche. */
/****************************************************************************/

boolean TObject::m_key_pressed_event(TKey key)
{
  PObjectNode node;
  PObject element;

  boolean event=FALSE;

  // Voir les vnements sur l'objet lui-mme,
  // pas sur un de ses composants
  // (Pour les objets drivs)

  // Hot-ket de l'objet

  if (key.hot_character==f_hot_key)
    {
      if (!f_focused)
        {
          if (m_set_focus())
	    return(TRUE);
        }

      // On cherche un autre objet qui a la mme hotkey
      return(FALSE);
    }

  // On propose l'vnement aux composants

  node=f_element_list;

  while ((node!=NULL) && (!event))
    {
      element=node->object;
      if (element->f_enabled)
        event=element->m_key_pressed_event(key);
      node=node->next;
    }

  return(event);

}

/****************************************************************************/
/* m_short_cut_pressed_event                                                */
/*--------------------------------------------------------------------------*/
/* Appele quand l'objet gre les acclrateurs de sa fentre.              */
/* Retourne TRUE si l'acclrateur intresse l'objet                        */
/****************************************************************************/

boolean TObject::m_short_cut_pressed_event(int /*short_cut*/)
{
  return(FALSE);
}

/*ͻ*/
/*                            METHODES PRIVEES                            */
/*ͼ*/

void TObject::m_set_previous_focused_object()
{
  JPDEBUG_TEST(DEBUG_ERROR_5,f_window!=NULL);
  f_window->f_previous_focused_object=this;
}
