/****************************************************************************/
/* JPHC                                                                     */
/*--------------------------------------------------------------------------*/
/* JPTUI Help Compiler                                                      */
/*--------------------------------------------------------------------------*/
/* Huffman routines and the first ideas from DFLAT 2.0                      */
/* (Al Stevens/Dr Dobb's Journal - 07/95)       			    */
/*--------------------------------------------------------------------------*/
/* Author      : DELPRAT Jean-Pierre / Al Stevens 			    */
/* Created on  : 01-May-95                                                  */
/* Modified on : 17-May-95                                                  */
/****************************************************************************/

#include <conio.h>
#include <ctype.h>
#include <iostream.h>
#include <new.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "..\..\Compat.h"
#include "..\..\Version.h"

/*ͻ*/
/*                                 CONSTANTS                              */
/*ͼ*/

#define MAX_PATH   260

#define MAX_TAGGED_LINE_LENGTH 2001    // Avec les tags  (avec \n)


#define MAX(a,b)    (((a) >= (b)) ? (a) : (b))

#define MAX_LINE_LENGTH        133    // Sans les tags  (avec \n)

#define DEFAULT_WIDTH          72     // (sans \n)
#define MAX_WIDTH             ((MAX_LINE_LENGTH)-1)  // (sans \n)

#define MAX_TOPIC_LENGTH       80

#define TAB_SIZE	       8

/*ͻ*/
/*                                   TYPES                                */
/*ͼ*/

typedef enum {FALSE=0,TRUE} boolean;


// Huffman tree for encoding
// (empty node : -1)

typedef struct
{ unsigned short count;   // Frequency
  short		 parent;
  short          right;
  short          left; } TCompressionHuffmanTree,*PCompressionHuffmanTree;

/*
 *     i | count |  p  l  r
 *     --+-------+---------                            D(16)
 *     0 | a  2  |  6 -1 -1                           /     \
 *     1 | b  2  |  7 -1 -1                      C(8)        f(8)
 *     2 | c  1  |  6 -1 -1                   /       \
 *     3 | d  0  | -1 -1 -1             A(3)             B(5)
 *     4 | e  3  |  7 -1 -1            /    \           /    \
 *     5 | f  8  |  9 -1 -1        c(1)      a(2)   b(2)      e(3)
 *     --+-------+---------
 *     6 | A  3  |  8  2  0
 *     7 | B  5  |  8  1  4
 *     8 | C  8  |  9  6  7
 *     9 | D 16  | -1  8  5
 */

// Huffman tree in compressed file (for decoding)

typedef struct { short right,left; } THuffmanTree,*PHuffmanTree;


// Index item

#define NO_TOPIC       -1
#define INVALID_TOPIC  -2

typedef struct index_topic
{
  short int    topic_nb;
  short int    topic_pos;
  char         *topic;
  char         *title;
  char         *previous_topic;
  short int    previous_topic_nb;
  char         *next_topic;
  short int    next_topic_nb;
  long int     byte_pos;             // Index (in bytes) in the help file
  short int    bit_pos;              // Index (in bits) in the help file
  int	       hwidth;
  int          hheight;

  struct index_topic *next;
} TIndexTopic,*PIndexTopic,*PTopicList;

typedef struct index_keyword
{
  char         *keyword;
  PIndexTopic  topic_node;
  struct index_keyword *next;
} TIndexKeyword,*PIndexKeyword,*PKeywordList;

// Local index

typedef struct local_index_topic
{
  PIndexTopic  topic_node;
  struct local_index_topic *next;
} TLocalIndexTopic,*PLocalIndexTopic,*PLocalIndex;

// Suspicious lines

typedef struct suspicious_line
{
  char *filename;
  int  line_nb;
  struct suspicious_line *next;
} TSuspiciousLine,*PSuspiciousLine;

/*ͻ*/
/*                             GLOBAL VARIABLES                           */
/*ͼ*/

// Help File Signature

char                    *Signature="JPHLP39";

// To build the TopicList

PTopicList    		TopicList=NULL;
PKeywordList            KeywordList=NULL;
int                     IndexWidth=0;

PIndexTopic	        *PosToNode=NULL;

// Suspicious lines

PSuspiciousLine         SuspiciousLines=NULL;
PSuspiciousLine         LastSuspiciousLine=NULL;

// To encode the help file

PCompressionHuffmanTree CompressionHelpTree;

char                    Out8;

// To decode the help file

PHuffmanTree            HelpTree;

int                     In8;
FILE                    *File;




char                    Line[MAX_TAGGED_LINE_LENGTH+1];
char                    TempLine1[MAX_TAGGED_LINE_LENGTH+1];
char                    TempLine2[MAX_TAGGED_LINE_LENGTH+1];


// Misc

unsigned short          TreeCount; // Number of node in the huffman tree
short int               Root;      // Offset in the huffman tree/array
int                     Ct8;

// To check the help file

boolean InvalidLinks=FALSE;
char    *CurrentTopic=NULL;
short int   HelpCount = 0;

// To generate a text file

PLocalIndex LocalIndex=NULL;
PLocalIndexTopic LastLocalIndexTopic=NULL;

/*ͻ*/
/*                      SUSPICIOUS LINES FUNCTIONS                        */
/*ͼ*/

void FreeSuspiciousLines()
{
  PSuspiciousLine node,next_node;

  node=SuspiciousLines;
  while (node!=NULL)
    {
      delete []node->filename;
      next_node=node->next;
      delete node;

      node=next_node;
    }

  SuspiciousLines=NULL;
  LastSuspiciousLine=NULL;
}

void AddSuspiciousLine(char *filename,int line_nb)
{
  PSuspiciousLine new_node;
  new_node= new TSuspiciousLine;
  new_node->line_nb=line_nb;
  new_node->filename=new char[strlen(filename)+1];
  strcpy(new_node->filename,filename);
  strupr(new_node->filename);
  new_node->next=NULL;

  if (SuspiciousLines!=NULL)
    {
      LastSuspiciousLine->next=new_node;
    }
  else
    {
      SuspiciousLines=new_node;
    }
  LastSuspiciousLine=new_node;
}

void ShowSuspiciousLines()
{
  char *file="";
  PSuspiciousLine node=SuspiciousLines;

  if (node==NULL)
    {
      cout << "Ok\n";
      return;
    }

  cout << "Not Ok\n";

  while (node!=NULL)
    {
      if (strcmp(node->filename,file))
        {
          cout << "    . In '" << node->filename << "' :\n";
          file=node->filename;
        }
      cout << "       - Line " << node->line_nb <<"\n";
      node=node->next;
    }
}
/*ͻ*/
/*                        INDEX BUILDING FUNCTIONS                        */
/*ͼ*/

// Functions to sort in alphabetical order, not in ASCII order
char NormalizeChar(char character)
{
  unsigned char u_character;

  u_character=character;

  if ((u_character >=(unsigned char)'A') && (u_character <=(unsigned char)'Z'))
    return(character-'A'+'a');

  if ((u_character < (unsigned char)'') || (u_character > (unsigned char)''))
    return(character);

  switch(character)
    {
      case '' :
      case '' : return('c');

      case '' :
      case '' :
      case '' :
      case '' :
      case '' : return('u');

      case '' :
      case '' :
      case '' :
      case '' :
      case '' : return('e');

      case '' :
      case '' :
      case '' :
      case '' :
      case '' :
      case '' :
      case '' :
      case '' :
      case '' : return('a');

      case '' :
      case '' :
      case '' :
      case '' : return('i');

      case '' :
      case '' :
      case '' :
      case '' :
      case '' : return('o');

      case '' : return('y');

      case '' :
      case '' : return('n');
    }
  return(character);


}

int alpha_strcmp(const char *s1, const char *s2)
{
  char *ptr;

  strcpy(TempLine1,s1);
  strcpy(TempLine2,s2);

  ptr=TempLine1;
  while ((*ptr)!=0)
    {
      (*ptr)=NormalizeChar(*ptr);
      ptr++;
    }

  ptr=TempLine2;
  while ((*ptr)!=0)
    {
      (*ptr)=NormalizeChar(*ptr);
      ptr++;
    }

  return(strcmp(TempLine1,TempLine2));
}

PIndexTopic AddTopicToIndex(char *topic,char *title)
{
  char *ptr;
  PIndexTopic node,previous_node,new_node;
  int comp;

  while (((*topic)==' ') || ((*topic)=='\t'))
    topic++;
  ptr=strchr(topic,0)-1;
  while ((ptr>=topic) && (  ((*ptr)==' ') || ((*ptr)=='\t')))
    {
      (*ptr)=0;
      ptr--;
    }

  while (((*title)==' ') || ((*title)=='\t'))
    title++;
  ptr=strchr(title,0)-1;
  while ((ptr>=title) && (  ((*ptr)==' ') || ((*ptr)=='\t')))
    {
      (*ptr)=0;
      ptr--;
    }


  node=TopicList;
  previous_node=NULL;

  while (node!=NULL)
    {
      comp=stricmp(node->topic,topic);

      if (comp>=0)
	break;


      previous_node=node;
      node=node->next;
    }


  new_node=new TIndexTopic;
  new_node->topic_nb=0;
  new_node->topic_pos=HelpCount;
  new_node->topic=new char [strlen(topic)+1];
  strcpy(new_node->topic,topic);
  new_node->title=new char [strlen(title)+1];
  new_node->previous_topic_nb=NO_TOPIC;
  new_node->previous_topic=NULL;
  new_node->next_topic_nb=NO_TOPIC;
  new_node->next_topic=NULL;
  new_node->byte_pos=0;
  new_node->bit_pos=0;
  new_node->hwidth=0;
  new_node->hheight=0;


  strcpy(new_node->title,title);

  if (previous_node==NULL)
    {
      new_node->next=TopicList;
      TopicList=new_node;
    }
  else
    {
      new_node->next=node;
      previous_node->next=new_node;
    }

  HelpCount++;
  return(new_node);
}

void SetPreviousTopic(PIndexTopic node,char *previous_topic)
{
  char *ptr;

  while (((*previous_topic)==' ') || ((*previous_topic)=='\t'))
    previous_topic++;
  ptr=strchr(previous_topic,0)-1;
  while ((ptr>=previous_topic) && (  ((*ptr)==' ') || ((*ptr)=='\t')))
    {
      (*ptr)=0;
      ptr--;
    }

  node->previous_topic=new char [strlen(previous_topic)+1];
  strcpy(node->previous_topic,previous_topic);
}

void SetNextTopic(PIndexTopic node,char *next_topic)
{
  char *ptr;

  while (((*next_topic)==' ') || ((*next_topic)=='\t'))
    next_topic++;
  ptr=strchr(next_topic,0)-1;
  while ((ptr>=next_topic) && (  ((*ptr)==' ') || ((*ptr)=='\t')))
    {
      (*ptr)=0;
      ptr--;
    }

  node->next_topic=new char [strlen(next_topic)+1];
  strcpy(node->next_topic,next_topic);
}

void AddKeywordToIndex(char *keyword,PIndexTopic topic_node)
{
  char *ptr;
  int keyword_length;
  PIndexKeyword node,previous_node,new_node;
  int comp;

  while (((*keyword)==' ') || ((*keyword)=='\t'))
    keyword++;
  ptr=strchr(keyword,0)-1;
  while ((ptr>=keyword) && (  ((*ptr)==' ') || ((*ptr)=='\t')))
    {
      (*ptr)=0;
      ptr--;
    }

  keyword_length=strlen(keyword);
  if (keyword_length==0)
    return;

  if (keyword_length>IndexWidth)
    IndexWidth=keyword_length;
  node=KeywordList;
  previous_node=NULL;

  while (node!=NULL)
    {
      comp=alpha_strcmp(node->keyword,keyword);
      if (comp==0)
	comp=strcmp(node->keyword,keyword);

      if (comp>=0)
	break;

      previous_node=node;
      node=node->next;
    }

  new_node=new TIndexKeyword;
  new_node->keyword=new char [keyword_length+1];
  strcpy(new_node->keyword,keyword);
  new_node->topic_node=topic_node;

  if (previous_node==NULL)
    {
      new_node->next=KeywordList;
      KeywordList=new_node;
    }
  else
    {
      new_node->next=node;
      previous_node->next=new_node;
    }
}

void FreeIndex()
{
  PIndexTopic   node,next_node;
  PIndexKeyword node2,next_node2;

  node=TopicList;
  while (node!=NULL)
  {
    delete [](node->title);
    delete [](node->topic);
    if (node->previous_topic!=NULL)
      delete []node->previous_topic;
    if (node->next_topic!=NULL)
      delete []node->next_topic;
    next_node=node->next;
    delete node;
    node=next_node;
  }

  node2=KeywordList;
  while (node2!=NULL)
  {
    delete [](node2->keyword);
    next_node2=node2->next;
    delete node2;
    node2=next_node2;
  }

  if (PosToNode!=NULL)
    delete []PosToNode;
}

boolean TopicInIndex(char *topic,PIndexTopic &ptrNode)
{
  boolean found=FALSE;
  PIndexTopic node=TopicList;

  ptrNode=NULL;

  while ((node!=NULL) && (!found))
    {
      if (!stricmp(node->topic,topic))
	{
	  found=TRUE;
	  ptrNode=node;
	}

      node=node->next;
    }
 return(found);
}

boolean BuildIndex(FILE *fi,int &phase,boolean for_included_file)
{
  static PIndexTopic s_node=NULL;

  short int i;
  char *ptr;
  char *ptr2;
  char *title;
  char *keyword;
  PIndexTopic link_node;
  boolean success;

  FILE *included_file;
  char filename[MAX_PATH];

  boolean include;
  char topic[MAX_TOPIC_LENGTH+1];

  // Create topic list

  while (fgets(Line, MAX_TAGGED_LINE_LENGTH+1, fi))
    {
      if (Line[0]==';')
        continue;
      include=FALSE;

      if (!strnicmp(Line,"[@include",sizeof("[@include")-1))
	{
	  ptr=strchr(Line,'(');
	  if (ptr!=NULL)
	    {
	      ptr++;
	      ptr2=strchr(ptr,')');
	      if (ptr2!=NULL)
		{
		  include=TRUE;
		  (*ptr2)=0;
		  strncpy(filename,ptr,MAX_PATH-1);
                  filename[MAX_PATH-1]=0;
		  if ((included_file = fopen(filename,"rt")) == NULL)
		    {
		      cout << "Failed\n";
		      cout << "    . Cannot open included file '" << filename << "'\n";
		      return(FALSE);
		    }

		  success=BuildIndex(included_file,phase,TRUE);
		  fclose(included_file);
		  if (!success)
		    return(FALSE);
		}
	    }
	}

      if (!include)
	{
	  switch (phase)
	    {
	      case 2 : // Recherche des mots cls et des prcdents/suivants

		if (s_node==NULL)
		  {
		    cout << "Failed\n";
		    cout << "    . Abnormal program termination\n";
		    return(FALSE);
		  }

		if (Line[0]=='[')
		  {
		    if (!strnicmp("kw]",Line+1,3))
		      {
			keyword=Line+4;
			ptr=strchr(keyword,'\n');
			if (ptr!=NULL)
			  (*ptr)=0;


			AddKeywordToIndex(keyword,s_node);
			break;
		      }

		    if (!strnicmp("<<]",Line+1,3))
		      {
			ptr=strchr(Line+3,'<');
			if (ptr!=NULL)
			  {
			    ptr++;
			    ptr2=strchr(ptr,'>');
			    if (ptr2!=NULL)
			      {
				(*ptr2)=0;
				SetPreviousTopic(s_node,ptr);
				break;
			      }
			  }
			cout << "Failed\n";
			cout << "    . Invalid [<<] declaration in topic <" << topic << ">\n";
			return(FALSE);
		      }

		    if (!strnicmp(">>]",Line+1,3))
		      {
			ptr=strchr(Line+3,'<');
			if (ptr!=NULL)
			  {
			    ptr++;
			    ptr2=strchr(ptr,'>');
			    if (ptr2!=NULL)
			      {
				(*ptr2)=0;
				SetNextTopic(s_node,ptr);
				break;
			      }
			  }
			cout << "Failed\n";
			cout << "    . Invalid [>>] declaration in topic <" << topic << ">\n";
			return(FALSE);
		      }

		    break;


		  }

		phase=1;

		// Si Line ne commence pas par [, on continue sur les
		// traitements de la phase 1

	      case 1 : // Recherche du topic

		s_node=NULL;

		if (Line[0]=='<')
		  {

		    if ((ptr = strchr(Line, '>')) != NULL)
		      {
			(*ptr)=0;

			if ((!stricmp("#",Line+1)) || (!stricmp("Index",Line+1)))
			  {
			    cout << "Failed\n";
			    cout << "    . Topic name <" << (Line+1) << "> is reserved\n";
			    return(FALSE);
			  }

			if ((*(Line+1))==0)
			  {
			    cout << "Failed\n";
			    cout << "    . Topic name <> is not valid\n";
			    return(FALSE);
			  }

			strcpy(topic,Line+1);
			title=ptr+1;
			ptr=strchr(title,'\n');
			if (ptr!=NULL)
			  (*ptr)=0;
			s_node=AddTopicToIndex(topic,title);

			phase=2;
		      }
		  }
		break;
	    }
	}
    }

  if (!feof(fi))
    {
      cout << "Failed\n";
      cout << "    . Error while reading file\n";
      return(FALSE);
    }

  if (for_included_file)
    return(TRUE);

  if (TopicList==NULL)
    {
      cout << "Failed\n";
      cout << "    . No topic in source file\n";
      return(FALSE);
    }

  if (KeywordList!=NULL)
    s_node=AddTopicToIndex("Index","");

  PosToNode=new PIndexTopic[HelpCount];

  s_node=TopicList;
  i=0;
  while (s_node!=NULL)
    {
      s_node->topic_nb=i;
      PosToNode[s_node->topic_pos]=s_node;
      i++;
      s_node=s_node->next;
    }



  s_node=TopicList;
  while (s_node!=NULL)
    {
      if (s_node->previous_topic!=NULL)
	{
          if ((s_node->previous_topic)[0]==0)
            {
              if (s_node->topic_pos>0)
                s_node->previous_topic_nb=PosToNode[s_node->topic_pos-1]->topic_nb;
              else
                s_node->previous_topic_nb=INVALID_TOPIC;
            }
          else
            {
	      if (TopicInIndex(s_node->previous_topic,link_node))
	        s_node->previous_topic_nb=link_node->topic_nb;
	      else
	        s_node->previous_topic_nb=INVALID_TOPIC;
            }
	  delete []s_node->previous_topic;
	  s_node->previous_topic=NULL;
	}

      if (s_node->next_topic!=NULL)
	{
          if ((s_node->next_topic)[0]==0)
            {
              if (s_node->topic_pos<(HelpCount-1))
                s_node->next_topic_nb=PosToNode[s_node->topic_pos+1]->topic_nb;
              else
                s_node->next_topic_nb=INVALID_TOPIC;
            }
          else
            {
	      if (TopicInIndex(s_node->next_topic,link_node))
	        s_node->next_topic_nb=link_node->topic_nb;
	      else
	        s_node->next_topic_nb=INVALID_TOPIC;
            }
	  delete []s_node->next_topic;
	  s_node->next_topic=NULL;
	}

      s_node=s_node->next;
    }

  return(TRUE);
}

boolean WriteIndexTopic(FILE *fi)
{
  char *separator;
  PIndexKeyword node;
  char old_letter,letter;

  if (KeywordList!=NULL)
    {
      if (IndexWidth<5)
	IndexWidth=5;
      IndexWidth+=3;
      separator=new char [IndexWidth+1];
      memcpy(separator," %c ",6);
      memset(separator+6,'',IndexWidth-6-2);
      strcpy(separator+IndexWidth-2,"\n\n");

      fprintf(fi,"\n<Index>\n\n");

      node=KeywordList;
      old_letter=0;

      while (node!=NULL)
	{
	   letter=toupper(NormalizeChar(*(node->keyword)));
	   if (letter!=0)
	     {
	       if (letter!=old_letter)
		 {
		    if (old_letter!=0)
		      fprintf(fi,"\n");
		    fprintf(fi,separator,letter);
		    old_letter=letter;
		 }
	       fprintf(fi,"[..%s]<%s>\n",node->keyword,node->topic_node->topic);
	     }

	   node=node->next;
	}

      delete []separator;
    }

  fputs("\n<#>\n",fi);

  cout << "Done\n";
  return(TRUE);

}

/*ͻ*/
/*                        INDEX CHECKING FUNCTIONS                        */
/*ͼ*/

void CheckIndexForDuplicateTopics()
{
  PIndexTopic node1,node2;
  boolean duplicates=FALSE;

  node1=TopicList;

  while (node1!=NULL)
    {
      node2=node1->next;
      if (node2!=NULL)
	{
	  if (!stricmp(node1->topic,node2->topic))
	    {
	      if (!duplicates)
		cout << "Not Ok\n";
	      cout << "    . <" << node1->topic << ">\n";
	      duplicates=TRUE;
	    }
	}
      node1=node2;
    }

  if (!duplicates)
    cout << "Ok\n";
}

void CheckTopicLength()
{
  boolean too_long=FALSE;

  PIndexTopic node=TopicList;

  while (node!=NULL)
    {
      if (strlen(node->topic)>MAX_TOPIC_LENGTH)
	{
	  if (!too_long)
	    {
	      cout << "Not Ok\n";
	      cout << "    . Too long (max : " << MAX_TOPIC_LENGTH << " characters)\n";
	      too_long=TRUE;
	    }
	  cout << "       - <" << node->topic << ">\n";

	}
      node=node->next;
    }
  if (!too_long)
    cout << "Ok\n";

}

void CheckSpecialTopics()
{
  char *topic="";
  boolean topic_missing=FALSE;
  PIndexTopic node;
  int i;

  for (i=1;i<=1;i++)
    {
      switch (i)
	{
	  case 1 : topic="Contents";
		   break;
	}

      if (!TopicInIndex(topic,node))
	{
	  if (!topic_missing)
	    {
	      cout << "Not Ok\n";
	      cout << "    . <" << topic << "> is missing\n";
	      topic_missing=TRUE;
	    }
	}
    }

  if (!topic_missing)
    cout << "Ok\n";
}

void CheckHelpWindowSize(int max_width)
{
  boolean too_large=FALSE;
  PIndexTopic node=TopicList;

  while (node!=NULL)
    {
      if (node->hwidth>max_width)
	{
	  if (!too_large)
	    {
	      cout << "Not Ok\n";
	      cout << "    . Too large (max : " << max_width << " characters)\n";
	      too_large=TRUE;
	    }
	  cout << "       - <" << node->topic << "> (" << node->hwidth << ")\n";
	}
      node=node->next;
    }
  if (!too_large)
    cout << "Ok\n";
}

void CheckInvalidPrevNextTopics()
{
  boolean invalid=FALSE;
  boolean invalid_previous=FALSE;
  boolean invalid_next=FALSE;

  PIndexTopic node;

  node=TopicList;

  while (node!=NULL)
    {
      if (node->previous_topic_nb==INVALID_TOPIC)
	{
	  node->previous_topic_nb=NO_TOPIC;
	  if (!invalid)
	    {
	      cout << "Not Ok\n";
	      invalid=TRUE;
	    }

	  if (!invalid_previous)
	    {
	      cout << "    . Invalid previous topic for : \n";
	      invalid_previous=TRUE;
	    }
	  cout << "       - <" << node->topic << ">\n";

	}
      node=node->next;
    }

  node=TopicList;

  while (node!=NULL)
    {
      if (node->next_topic_nb==INVALID_TOPIC)
	{
	  node->next_topic_nb=NO_TOPIC;
	  if (!invalid)
	    {
	      cout << "Not Ok\n";
	      invalid=TRUE;
	    }
	  if (!invalid_next)
	    {
	      cout << "    . Invalid next topic for : \n";
	      invalid_next=TRUE;
	    }
	  cout << "       - <" << node->topic << ">\n";

	}
      node=node->next;
    }


  if (!invalid)
    cout << "Ok\n";
}

/*ͻ*/
/*                        PREPROCESSING FUNCTIONS                         */
/*ͼ*/

boolean Preprocess(FILE *in,char *in_name,FILE *out)
{
  boolean macro_ok;
  boolean success;
  PIndexTopic node;
  static boolean s_before_topics=TRUE;

  char *ptr;
  char *ptr2;
  char *ptr3;
  char *ptr_test;
  char *title;
  char *end_ptr;
  char filename[MAX_PATH];
  FILE *included_file;
  boolean suspicious=FALSE;

  int len;
  int line_nb=0;
  int tab_size;

  while (fgets(Line, MAX_TAGGED_LINE_LENGTH+1, in))
    {
      if (!strchr(Line,'\n'))
	strcat(Line,"\n");

      ptr=strchr(Line,'\t');
      while (ptr!=NULL)
	{
	  tab_size=TAB_SIZE-(((int)(ptr-Line))%TAB_SIZE);
	  memmove(ptr+tab_size,ptr+1,strlen(ptr+1)+1);
	  memset(ptr,' ',tab_size);
	  ptr+=tab_size;

	  ptr=strchr(ptr,'\t');
	}

      line_nb++;
      if (!strnicmp(Line,"[@",sizeof("[@")-1))
	{
	  macro_ok=FALSE;
	  if (!strnicmp(Line,"[@include",sizeof("[@include")-1))
	    {
	      ptr=strchr(Line,'(');
	      if (ptr!=NULL)
		{
		  ptr++;
		  ptr2=strchr(ptr,')');
		  if (ptr2!=NULL)
		    {
		      macro_ok=TRUE;
		      (*ptr2)=0;
		      strncpy(filename,ptr,MAX_PATH-1);
                      filename[MAX_PATH-1]=0;
		      if ((included_file = fopen(filename,"rt")) == NULL)
			{
			  cout << "Failed\n";
			  cout << "    . Cannot open included file '" << filename << "'\n";
			  return(FALSE);
			}

		      success=Preprocess(included_file,filename,out);
		      fclose(included_file);
		      if (!success)
			return(FALSE);
		    }
		}
	    }
	  if (!macro_ok)
	    {
	      cout << "Failed\n";
	      cout << "    . Invalid macro : " << Line ;
	      return(FALSE);
	    }
	}
      else
	{
	  if (Line[0]!=';')
	    {
	      // Index + titre

	      if (Line[0]=='<')
		{
		  ptr=strchr(Line,'>');
		  if (ptr!=NULL)
		    {
		      s_before_topics=FALSE;

		      ptr3=Line+1;
		      while (((*ptr3)==' ') || ((*ptr3)=='\t'))
			ptr3++;
		      memmove(Line+1,ptr3,strlen(ptr3)+1);
		      ptr-=(int)(ptr3-(Line+1));
		      ptr3=ptr-1;
		      while (((*ptr3)==' ') || ((*ptr3)=='\t'))
			ptr3--;
		      memmove(ptr3+1,ptr,strlen(ptr)+1);
		      ptr=ptr3+1;
		      ptr++;
		      ptr2=ptr;
		      while (((*ptr2)==' ') || ((*ptr2)=='\t'))
			ptr2++;
		      memmove(ptr+1,ptr2,strlen(ptr2)+1);
		      (*ptr)='\n';
		      ptr++;
		      ptr2=strchr(ptr,0)-2; // (\n en fin de ligne)
		      while ((ptr2>=ptr) && (  ((*ptr2)==' ') || ((*ptr2)=='\t')))
			{
			  (*ptr2)='\n';
			  (*(ptr2+1))=0;
			  ptr2--;
			}
		    }
		}

	      // Infos non utilisee par la suite

	      if (Line[0]=='[')
		{
		  if (  (!strnicmp("kw",Line+1,2))
                      ||(!strnicmp("<<",Line+1,2))
		      ||(!strnicmp(">>",Line+1,2)))
                    {
                      if (Line[3]!=']')
                        AddSuspiciousLine(in_name,line_nb);
                      else
                        continue;
                    }
		}


	      // Liens sans mots cles

	      ptr=Line;

	      while (ptr != NULL)
		{
		  if ((ptr=strstr(ptr,"[.."))!=NULL)
                    {
                      suspicious=TRUE;
                      ptr_test=strchr(ptr+3,']');
                      if (ptr_test!=NULL)
                        {
                          ptr_test=strchr(ptr_test+1,'<');
                          if (ptr_test!=NULL)
                            {
                              ptr_test=strchr(ptr_test+1,'>');
                              if (ptr_test!=NULL)
                                suspicious=FALSE;
                            }
                        }
                      if (suspicious)
                        AddSuspiciousLine(in_name,line_nb);

                      ptr+=3;
                      if ((*ptr)==']')
                        {
                          ptr++;
		          if ((ptr2 = strchr(ptr,'<')) != NULL)
			    {
			      ptr3 = strchr(ptr2,'>');

			      if (ptr3 != NULL)
			        {
			          (*ptr3)=0;

			          if (TopicInIndex(ptr2+1,node))
				    title=node->title;
			          else
				    title="";

			          end_ptr=strchr(title,0)-1;
			          while (   (end_ptr>=title)
				         && (((*end_ptr)==' ') || ((*end_ptr)=='\t')))
				    {
				      (*end_ptr)=0;
				      end_ptr--;
				    }

			          (*ptr3)='>';
			          len=strlen(title);
			          ptr--;
			          memmove(ptr+len,ptr,strlen(ptr)+1);
			          memcpy(ptr,title,len);
                                  ptr=ptr3+len+1;
			        }
			    }
		        }
                    }
		}
	      if (!s_before_topics)
		fputs(Line,out);
	    }
	}
    }

  if (!feof(in))
    {
      cout << "Failed\n";
      cout << "    . Error while reading file\n";
      return(FALSE);
    }
  return(TRUE);
}

void ResolveLineLinks(char *line)
{
  char *cp=line;
  char *ncp;
  char *cp1;
  PIndexTopic node;
  boolean link_valid;
  unsigned int pos;

  char topic_ref[4+1];

  while (cp != NULL)
    {
      if ((cp = strchr(cp,'[')) != NULL)
	{
	  /* ----- hit a new key word ----- */

	  cp++;
	  if ((*cp) != '.')
	    continue;

	  cp++;
	  if ((*cp) != '.')
	    continue;

	  cp++;

	  if ((ncp = strchr(cp,']')) != NULL)
	    {
	      ncp++;

	      if ((ncp = strchr(ncp,'<')) != NULL)
		{
		  cp1 = strchr(ncp,'>');

		  if (cp1 != NULL)
		    {
		      (*cp1)=0;

		      link_valid=TopicInIndex(ncp+1,node);

		      (*cp1)='>';
		      memmove(ncp+5,cp1,strlen(cp1)+1);

                      pos=link_valid?node->topic_nb:0xFFFF;
		      sprintf(topic_ref,"%04X",pos);
		      memcpy(ncp+1,topic_ref,4);
		      cp=ncp+6;
		    }
		}

	    }
	}
    }
}

boolean ResolveLinks(FILE *in, FILE *out)
{
  char topic[MAX_TOPIC_LENGTH+1];
  char *ptr;
  PIndexTopic node;

  while (fgets(Line, MAX_TAGGED_LINE_LENGTH+1, in))
    {
      ptr=strchr(Line,'>');
      if ((Line[0]=='<') && (ptr!=NULL))
	{
	  if (strnicmp(Line,"<#>",sizeof("<#>")-1))
	    {
	      (*ptr)=0;
	      strcpy(topic,Line+1);
	      if (!TopicInIndex(topic,node))
		{

		  cout << "Failed\n";
		  cout << "    . Abnormal program termination\n";
		  return(FALSE);
		}
	      fputs("<>\n",out);
	    }
	  else
	    {

	      fputs(Line,out);
	    }
	}
      else
	{
	  if (node==NULL)
	    {
	      cout << "Failed\n";
	      cout << "    . Abnormal program termination\n";
	      return(FALSE);
	    }

	  ResolveLineLinks(Line);
	  fputs(Line,out);
	}
    }

  if (!feof(in))
    {
      cout << "Failed\n";
      cout << "    . Error while reading file\n";
      return(FALSE);
    }

  cout << "Done\n";
  return(TRUE);
}

/*ͻ*/
/*                       TEXT FILE CREATING FUNCTIONS                      */
/*ͼ*/

void FreeLocalIndex()
{
  PLocalIndexTopic node,next_node;

  node=LocalIndex;
  while (node!=NULL)
    {
      next_node=node->next;
      delete node;

      node=next_node;
    }

  LocalIndex=NULL;
  LastLocalIndexTopic=NULL;

}

int AddItemToLocalIndex(char *topic)
{
  PIndexTopic topic_node;
  PLocalIndexTopic node;
  PLocalIndexTopic new_node;
  boolean found;

  int i;

  if (!TopicInIndex(topic,topic_node))
    return(0);

  node=LocalIndex;
  found=FALSE;
  i=1;

  while (node!=NULL)
    {
      if (node->topic_node==topic_node)
	{
	  found=TRUE;
	  break;
	}
      node=node->next;
      i++;
    }

  if (found)
    return(i);

  new_node= new TLocalIndexTopic;
  new_node->topic_node=topic_node;
  new_node->next=NULL;

  if (LocalIndex!=NULL)
    {
      LastLocalIndexTopic->next=new_node;
    }
  else
    {
      LocalIndex=new_node;
    }
  LastLocalIndexTopic=new_node;
  return(i);
}

boolean SourceFileToTextFile(FILE *in,FILE *out,int width)
{
  int off1,off2;
  int phase;
  char *ptr;
  int len;
  int i;
  char *up_separator;
  char *down_separator;
  char *cp,*cp1,*pt1,*ncp,*pt;
  char *ptr_ref;
  int ref_nb;
  int max_ref_nb=0;

  char reference[20];
  int ref_len;
  PLocalIndexTopic node;

  char topic[MAX_TOPIC_LENGTH+1];


  up_separator = new char [width+1+1];
  memset(up_separator,'',width);
  ptr=up_separator+width;
  (*ptr++)='\n';
  (*ptr)=0;
  down_separator = new char [width+1+1];
  memset(down_separator,'',width);
  ptr=down_separator+width;
  (*ptr++)='\n';
  (*ptr)=0;

  phase=1;

  while (fgets(Line, MAX_TAGGED_LINE_LENGTH+1, in))
    {
      switch (phase)
	{
	  case 2 : // Recherche du titre
	    FreeLocalIndex();
	    ptr=strchr(Line,'\n');
	    if (ptr!=NULL)
	      (*ptr)=0;

	    fputs("\n",out);
	    fputs(up_separator,out);
	    len=strlen(Line);
	    fprintf(out," %s\n",(len==0)?topic:Line);
	    fputs(down_separator,out);
	    fputs("\n",out);
	    phase = 3;
	    break;

	  case 3 : // Contenu du sujet

	    if (Line[0]=='<')
	      {
		if ((ptr = strchr(Line, '>')) != NULL)
		  phase=1;
	      }

	    if (phase!=1)
	      {
		cp=Line;
		while (cp!=NULL)
		  {
		    if ((cp = strchr(cp,'[')) != NULL)
		      {
			/* ----- hit a new key word ----- */

			cp++;
			if ((*cp) != '.')
			  continue;

			cp++;
			if ((*cp) != '.')
			  continue;

			cp++;
			ptr=cp-3;
			off1 = (int)(cp - Line)-2;

			pt1=cp;
			if ((ncp = strchr(cp,']')) != NULL)
			  {
			    pt=ncp;

			    if ((ncp = strchr(ncp,'<')) != NULL)
			      {
				cp1 = strchr(ncp,'>');

				if (cp1 != NULL)
				  {
				    off2 = off1 + (int)(pt - pt1) -1;

				    (*cp1)=0;

				    memmove(ptr,ptr+3,(int)(pt-pt1));
				    ptr+=(int)(pt-pt1);

				    if (off2<off1)
				      reference[0]=0;
				    else
				      {
					ref_nb=AddItemToLocalIndex(ncp+1);

					if (ref_nb!=0)
					  {
					    sprintf(reference," (%02d)",ref_nb);
					    if (ref_nb>max_ref_nb)
					      max_ref_nb=ref_nb;
					  }
					else
					  reference[0]=0;
				      }

				    ref_len=strlen(reference);
				    ptr_ref=ptr;
				    ptr+=ref_len;
				    memmove(ptr,cp1 + 1,strlen(cp1+1)+1);
				    memcpy(ptr_ref,reference,ref_len);
				    cp=ptr;
				  }
			      }
			  }
		      }
		  }

		fputs(Line,out);
		break;
	      }

	    // Si on a trouv un nouveau sujet, on fait les traitements
	    // de la phase 1

	  case 1 : // Recherche du topic

	    if (Line[0]=='<')
	      {

		if ((ptr = strchr(Line, '>')) != NULL)
		  {
		    if (max_ref_nb>0)
		      {
			fputs("\n\n",out);
			node=LocalIndex;
			for (i=1;i<=max_ref_nb;i++)
			  {
			    fprintf(out," (%02d) \t-> %s\n",i,
				    (node->topic_node->title[0]==0)?
				    (node->topic_node->topic):
				    (node->topic_node->title));
			    node=node->next;
			  }
			fputs("\n",out);
			max_ref_nb=0;
		      }

		    (*ptr)=0;
		    strncpy(topic,Line+1,MAX_TOPIC_LENGTH);
		    topic[MAX_TOPIC_LENGTH]=0;

		    if (!strnicmp(Line,"<#>",sizeof("<#>")-1))
		      {
			cout << "Done\n";
			delete []up_separator;
			delete []down_separator;
			return(TRUE);
		      }

		    phase=2;
		  }
	      }
	    break;


	}
    }

  delete []up_separator;
  delete []down_separator;

  if (!feof(in))
    {
      cout << "Failed\n";
      cout << "    . Error while reading file\n";
      return(FALSE);
    }
  cout << "Done\n";
  return(TRUE);
}

/*ͻ*/
/*                     HUFFMAN COMPRESSION FUNCTIONS                      */
/*ͼ*/

void BuildTree()
{
  int h1,h2;
  PCompressionHuffmanTree htt;
  int i;

  TreeCount = 256;

  for (i = 0; i < TreeCount; i++)
    {
      CompressionHelpTree[i].parent = -1;
      CompressionHelpTree[i].right  = -1;
      CompressionHelpTree[i].left   = -1;
    }

  while (1)
    {
      h1 = -1;
      h2 = -1;

      for (i = 0; i < TreeCount; i++)
	{
	  if (i != h1)
	    {
	      htt = CompressionHelpTree + i;

	      if (htt->count > 0 && htt->parent == -1)
		{

		  if (h1 == -1 || htt->count < CompressionHelpTree[h1].count)
		    {
		      if (h2 == -1 || CompressionHelpTree[h1].count < CompressionHelpTree[h2].count) h2 = h1;
		      h1 = i;
		    }
		  else
		    if (h2 == -1 || htt->count < CompressionHelpTree[h2].count)
		      h2 = i;
		}
	    }
	}

      if (h2 == -1)
	{
	  Root = h1;
	  break;
	}

      CompressionHelpTree[h1].parent     = TreeCount;
      CompressionHelpTree[h2].parent     = TreeCount;

      CompressionHelpTree = (PCompressionHuffmanTree)realloc(CompressionHelpTree,(TreeCount + 1) * sizeof(TCompressionHuffmanTree));
      if (CompressionHelpTree == NULL) break;

      CompressionHelpTree[TreeCount].count    = CompressionHelpTree[h1].count + CompressionHelpTree[h2].count;

      CompressionHelpTree[TreeCount].right  = h1;
      CompressionHelpTree[TreeCount].left   = h2;

      CompressionHelpTree[TreeCount].parent = -1;

      TreeCount++;

    }
}



void outbit(FILE *fo,int bit)
{
  if (Ct8 == 8 || bit == -1)
    {
      while (Ct8 < 8)
	{
	  Out8 <<= 1;
	  Ct8++;
	}
      fputc(Out8,fo);
      Ct8 = 0;
    }
  Out8 = (Out8 << 1) | bit;
  Ct8++;
}

void compress(FILE *fo,int h,int child)
{
  if (child==0)
    {
      h++;
      h--;
    }

  if (CompressionHelpTree[h].parent != -1)
    compress(fo,CompressionHelpTree[h].parent,h);

  if (child)
    {
      if (child == CompressionHelpTree[h].right)
	outbit(fo,0);
      else
	if (child == CompressionHelpTree[h].left)
	  outbit(fo,1);
    }
}



/*ͻ*/
/*                          HELP FIXING FUNCTIONS                         */
/*ͼ*/

FILE *OpenHelpFile(const char *fn,const char *md)
  {
    unsigned short    TreeCount;
    int            i;
    char           helpname[65];
    unsigned short    bytectr ;


    strcpy(helpname,fn);

    if ((File = fopen(helpname,md)) == NULL) return (NULL);


    if (HelpTree == NULL)
      {
	fread(Signature,strlen(Signature),1,File);
	fread(&bytectr, sizeof(bytectr),1,File);
	fread(&TreeCount , sizeof(TreeCount) ,1,File);
	fread(&Root   , sizeof(Root)   ,1,File);

	HelpTree = new THuffmanTree [TreeCount - 256];

	if (HelpTree != NULL)
	  {
	    for (i = 0; i < TreeCount - 256; i++)
	      {
		fread(&HelpTree[i].left, sizeof(short),1,File);
		fread(&HelpTree[i].right,sizeof(short),1,File);
	      }
	  }
      }

    return (File);
  }


void HelpFilePosition(long *offset,short *bit)
  {
    *offset = ftell(File);
    if (Ct8 < 8) --*offset;
    *bit = Ct8;
  }


void *GetHelpLine(char *line)
  {
    int h;                                            /* Index in HelpTree */

    *line = '\0';
    while(TRUE)
      {
	h = Root;
	while (h > 255)
	  {
	    if (Ct8 == 8)
	      {
		if ((In8 = fgetc(File)) == EOF)
		  {
		    *line = '\0';
		    return (NULL);
		  }
		Ct8 = 0;
	      }

	    if (In8 & 0x80)
	      h = HelpTree[h - 256].left;
	    else
	      h = HelpTree[h - 256].right;

	    In8 <<= 1;
	    Ct8++;
	  }

	*line++ = h;
	if (h == '\n') break;
      }

    *line = '\0';
    return (line);
  }



static void WriteText(char *text)
  {
    char           *np  = text ? text : (char*)"";
    short    len = strlen(np);

    fwrite(&len,sizeof(len),1,File);
    if (len) fwrite(np,len + 1,1,File);
  }


void CheckLineInvalidLinks(char *line,int &width)
{
  char *cp=line;
  char *ncp;
  char *cp1;
  char *ptr;
  int  len_keyword;
  boolean link_valid;
  char *keyword;
  unsigned int pos;

  int line_width;

  ptr=strchr(line,'\n');
  if (ptr!=NULL)
    (*ptr)=0;
  line_width=strlen(line);


  while (cp != NULL)
    {
      if ((cp = strchr(cp,'[')) != NULL)
	{
	  /* ----- hit a new key word ----- */

	  cp++;
	  if ((*cp) != '.')
	    continue;

	  cp++;
	  if ((*cp) != '.')
	    continue;

	  cp++;

	  if ((ncp = strchr(cp,']')) != NULL)
	    {
	      keyword=cp;
	      (*ncp)=0;
	      len_keyword=(int)(ncp-cp);

	      ncp++;

	      if ((ncp = strchr(ncp,'<')) != NULL)
		{
		  cp1 = strchr(ncp,'>');

		  if (cp1 == ncp+5)
		    {
		      line_width-=4;
		      line_width-=(int)(cp1-ncp+1);

		      (*cp1)=0;

		      sscanf(ncp+1,"%04X",&pos);
		      if (pos==0xFFFF)
			link_valid=FALSE;
		      else
			link_valid=TRUE;

		      if ((len_keyword==0) || (!link_valid))
			{
			  if (!InvalidLinks)
			    cout << "Not Ok\n";

			  if (CurrentTopic!=NULL)
			    {
			      cout << "    . In <" << CurrentTopic << ">\n";
			      CurrentTopic=NULL;
			    }

			  if (!link_valid)
			    cout << "       - unknown topic for key word '" << keyword << "'\n";
			  else
			    {
			      if (len_keyword==0)
				cout << "       - link without key word to a help window without title\n";
			    }
			  InvalidLinks=TRUE;
			}
		      cp=cp1+1;

		    }
		}

	    }
	}
    }

  if (line_width>width)
    width=line_width;

}


boolean CheckInvalidLinks()
  {
    PIndexTopic node=NULL;
    short int i=-1;

    *Line = '\0';
    Ct8=8;

    if (GetHelpLine(Line) == NULL)
      {
	cout << "Failed\n";
	cout << "    . Invalid help file\n";
	return (FALSE);
      }

    if (Line[0] != '<')
      {
	cout << "Failed\n";
	cout << "    . Invalid help file\n";
	return (FALSE);
      }

    while (*Line == '<')
      {
	if (strnicmp(Line,"<#>",sizeof("<#>")-1) == 0) break;

	if ((strchr(Line, '>')) != NULL)
	  {
	    i++;
	    node=PosToNode[i];
	    CurrentTopic=node->topic;
	    HelpFilePosition(&node->byte_pos,&node->bit_pos);

	    // Titre
	    if (GetHelpLine(Line) == NULL)
	      {
		cout << "Failed\n";
		cout << "    . Abnormal program termination\n";
		return(FALSE);
	      }

	    node->hheight  = 0;
	    node->hwidth   = 0;
	  }

	if (GetHelpLine(Line) == NULL)
	  {
	    cout << "Failed\n";
	    cout << "    . Abnormal program termination\n";
	    return(FALSE);
	  }

	while (*Line != '<')
	  {
	    if (node!=NULL)
	      {
		node->hheight++;
		CheckLineInvalidLinks(Line,node->hwidth);
	      }
	    if (GetHelpLine(Line) == NULL)
	      {
		cout << "Failed\n";
		cout << "    . Abnormal program termination\n";
		return(FALSE);
	      }
	  }

      }

    CurrentTopic=NULL;
    if (!InvalidLinks)
      cout << "Ok\n";


    return (TRUE);
  }


boolean SaveIndex(FILE *help_file)
{
  PIndexTopic node;
  long where;

  fseek(help_file,0L,SEEK_END);
  where = ftell(help_file);
  fwrite(&HelpCount,sizeof(HelpCount),1,help_file);

  node=TopicList;
  while (node != NULL)
    {
      WriteText(node->topic);

      fwrite(&node->byte_pos   ,sizeof(node->byte_pos)   ,1,help_file);
      fwrite(&node->bit_pos   ,sizeof(node->bit_pos)    ,1,help_file);
      fwrite(&node->next_topic_nb,sizeof(node->next_topic_nb),1,help_file);
      fwrite(&node->previous_topic_nb,sizeof(node->previous_topic_nb),1,help_file);

      node = node->next;
    }

  fwrite(&where,sizeof(where),1,help_file);
  cout << "Done\n";

  return(TRUE);
}

void NewHandler()
{
  cerr << "\nNot enough memory!\n";
  exit(EXIT_FAILURE);
}

int main(int argc,char *argv[])
{
  int 		key;
  int           index_phase;
  char 	        *ptr;
  int 		i;
  char          *preproc_file_name=NULL;
  char          *preproc2_file_name=NULL;
  char          *text_file_name=NULL;
  boolean       success=TRUE;
  FILE          *fi=NULL;
  FILE          *fo=NULL;
  FILE          *fpreproc=NULL;
  FILE          *fpreproc2=NULL;
  FILE          *ftext=NULL;
  int            c;
  boolean       unlink_preproc_file=FALSE;
  boolean       unlink_preproc2_file=FALSE;
  boolean       debug_mode=FALSE;

  char          *default_ext=".HLP";
  char          *source_file=NULL;
  char          *help_file=NULL;

  int           max_width=DEFAULT_WIDTH;

  unsigned short    bytectr = 0;         /* #Zeichen unkomprimiert in Datei */
      short lf;
      short rt;

  boolean help_wanted=FALSE;

  // Pour grer les problmes d'allocation

  set_new_handler(NewHandler);

  // Aide demande explicitement

  for (i=1;i<argc;i++)
    {
      if (argv[i][0]!='/')
	{
	  if (source_file==NULL)
	    source_file=argv[i];
	  else
	    {
	      if (help_file==NULL)
		help_file=argv[i];
	      else
		{
		   help_wanted=TRUE;
		}
	    }
	}
      else
	{
	  if (!stricmp(argv[i],"/?"))
	    help_wanted=TRUE;

	  else if (!strnicmp(argv[i],"/WIDTH=",sizeof("/WIDTH=")-1))
	    {
	      max_width=atoi(argv[i]+sizeof("/WIDTH=")-1);
	      if (max_width>MAX_WIDTH)
		max_width=MAX_WIDTH;
	    }
	  else if (!strnicmp(argv[i],"/DEBUG",sizeof("/DEBUG")-1))
	    {
	      debug_mode=TRUE;
	    }
	  else if (!strnicmp(argv[i],"/TEXT=",sizeof("/TEXT=")-1))
	    {
	      text_file_name=argv[i]+sizeof("/TEXT=")-1;
	    }
	  else
	    help_wanted=TRUE;

	}
    }


  if ((argc<2) || (source_file==NULL))
    help_wanted=TRUE;

  clrscr();
  cout << "\nJPTUI Help Compiler for JPTUI " << JPTUI_VERSION_STRING;
  cout << "\nBy Jean-Pierre Delprat, " << JPTUI_DATE_STRING ;
  cout << "\nGenerates an hypertext help file for use with JPTUI.\n\n";


  if (help_wanted)
    {
      cout << "Usage : JPHC <Source File> [Help File] [/TEXT=File] [/WIDTH=nn] [/DEBUG] [/?]\n\n";
      cout << "        Source File  : Text file describing the different help windows\n";
      cout << "        Help File    : Generated hypertext help file\n";
      cout << "                       (default : <Source File>.HLP)\n";
      cout << "        /TEXT=File   : Generates also the text help file 'File'\n";
      cout << "        /WIDTH=nn    : Checks if help windows are no larger than nn characters\n";
      cout << "                       (default : " << DEFAULT_WIDTH << ", max : " << MAX_WIDTH << ")\n";
      cout << "        /DEBUG       : Keeps intermediate files\n";
      cout << "        /?           : Displays this help\n\n";
      cout << "\nHit any key to continue (<ESC> to quit) ...\n";
      key=getch();
      if (key==27)
        return(EXIT_SUCCESS);

      clrscr();
      cout << "In the source file, each help window is of form :\n\n";
      cout << " (1st column of each line)\n";
      cout << "\n";
      cout << ";Anything with a semicolon in the 1st column is a comment\n";
      cout << "<help topic>Help window title (name and title of the help window)\n";
      cout << "[<<]<help topic>              (name of the previous window in chain)\n";
      cout << "[>>]<help topic>              (name of the next window in chain)\n";
      cout << "[kw]key word 1                (key words of the window : appear in the index)\n";
      cout << "...\n";
      cout << "[kw]key word n\n";
      cout << "\nHelp text follows until the following help window.\n\n";
      cout << "Hypertext reference : [..key word]<help topic> embedded in the text\n";
      cout << "                      (if there is no key word, the title of the corresponding\n";
      cout << "                      help window is used instead).\n\n";
      cout << "Any line of the source file can be one of the following macros :\n";
      cout << "[@include(File)]    : Includes the file 'File'\n\n";
      cout << "Notes :\n";
      cout << "- <Contents> is the default help window\n";
      cout << "- [<<]<> instead of [<<]<help topic> means the previous topic in the source file";
      cout << "- [>>]<> instead of [>>]<help topic> means the next topic in the source file\n";

      return(EXIT_SUCCESS);
    }

  if (help_file==NULL)
    {
      help_file=new char [strlen(source_file)+strlen(default_ext)+1];
      strcpy(help_file,source_file);
      ptr=strchr(help_file,'.');
      if (ptr==NULL)
	ptr=strchr(help_file,0);
      strcpy(ptr,default_ext);
    }
  else
    {
      ptr=help_file;
      help_file=new char [strlen(help_file)+1];
      strcpy(help_file,ptr);
    }

  if (!stricmp(source_file,help_file))
    {
      cout <<     "    . Source file and hypertext help file must have different file names\n";
      success=FALSE;
    }

  if ((success) && (text_file_name!=NULL))
    {
      if (!stricmp(text_file_name,source_file))
	{
	  cout << "    . Source file and text help file must have different file names\n";
	  success=FALSE;
	}
    }

  if ((success) && (text_file_name!=NULL))
    {
      if (!stricmp(text_file_name,help_file))
	{
	  cout << "    . Hypertext and text help files must have different file names\n";
	  success=FALSE;
	}
    }

  CompressionHelpTree=NULL;

  if (success)
    {
      if ((fi = fopen(source_file,"rt")) == NULL)
	{
	  cout << "    . Cannot open '" << source_file << "'\n";
	  success=FALSE;
	}
    }

  if (success)
    {
      ptr=tmpnam(NULL);
      preproc_file_name=new char [strlen(ptr)+1];
      strcpy(preproc_file_name,ptr);

      if ((fpreproc = fopen(preproc_file_name,"w+t")) == NULL)
	{
	  cout << "    . Cannot open '" << preproc_file_name << "'\n";
	  success=FALSE;
	}
    }

  // Preprocess the file

  if (success)
    {
      unlink_preproc_file=TRUE;
      cout << " Preprocessing source file ............. In progress\n";
      cout << "   Building index ...................... ";
      cout.flush();
      index_phase=1;
      success=BuildIndex(fi,index_phase,FALSE);
      if (success)
	cout << "Done\n";
    }

  if (success)
    {
      cout << "   Expanding macros .................... ";
      cout.flush();
      fseek(fi,0,SEEK_SET);
      success=Preprocess(fi,source_file,fpreproc);
      if (success)
	cout << "Done\n";
    }

  if (fi!=NULL)
    {
      fclose(fi);
      fi=NULL;
    }

  /* Add the <topic> index */

  if (success)
    {
      cout << "   Adding <Index> topic ................ ";
      cout.flush();
      fseek(fpreproc,0,SEEK_END);
      success=WriteIndexTopic(fpreproc);
    }

  if (success)
    {
      cout << "   Resolving links ..................... ";
      cout.flush();

      ptr=tmpnam(NULL);
      preproc2_file_name=new char [strlen(ptr)+1];
      strcpy(preproc2_file_name,ptr);

      if ((fpreproc2 = fopen(preproc2_file_name,"w+t")) == NULL)
	{
	  cout << "Failed\n";
	  cout << "    . Cannot open '" << preproc2_file_name << "'\n";
	  success=FALSE;
	}

      if (success)
	{
	  unlink_preproc2_file=TRUE;


	  fseek(fpreproc,0,SEEK_SET);
	  success=ResolveLinks(fpreproc,fpreproc2);
	}
    }

  if (success)
    {
      cout << "  Preprocessing source file ............. Done\n";

    }

  /* Generate a text file equivalent to the help file */

  if (success)
    {
      if (text_file_name!=NULL)
	{
	  cout << " Creating text help file ............... ";
	  cout.flush();

	  if ((ftext = fopen(text_file_name,"wt")) == NULL)
	    {
	      cout << "Failed\n";
	      cout << "    . Cannot open '" << text_file_name << "'\n";
	      success=FALSE;
	    }

	  if (success)
	    {
	      fseek(fpreproc,0,SEEK_SET);
	      success=SourceFileToTextFile(fpreproc,ftext,max_width);
	    }
	}
    }

  if (fpreproc!=NULL)
    {
      fclose(fpreproc);
      fpreproc=NULL;
    }

  if (ftext!=NULL)
    {
      fclose(ftext);
      ftext=NULL;
    }


  if (success)
    {
      cout << " Compressing hypertext help file ....... ";
      cout.flush();

      if (success)
	{
	  if ((fo = fopen(help_file,"wb")) == NULL)
	    {
	      cout << "Failed\n";
	      cout << "    . Cannot open '" << help_file << "'\n";
	      success=FALSE;
	    }
	}

      if (success)
	{
	  fseek(fpreproc2,0,SEEK_SET);
	  CompressionHelpTree = (PCompressionHuffmanTree)calloc(256,sizeof(TCompressionHuffmanTree));

	  while ((c = fgetc(fpreproc2)) != EOF)
	    {
	      c &= 0xFF;
	      CompressionHelpTree[c].count++;
	      bytectr++;
	    }

	  BuildTree();

	  // Save the Huffman tree

	  fwrite(Signature,strlen(Signature),1,fo);
	  fwrite(&bytectr,sizeof(bytectr),1,fo);
	  fwrite(&TreeCount,sizeof(TreeCount),1,fo);
	  fwrite(&Root,sizeof(Root),1,fo);

	  for (c = 256; c < TreeCount; c++)
	    {
	      lf = CompressionHelpTree[c].left;
	      rt = CompressionHelpTree[c].right;

	      fwrite(&lf,sizeof(lf),1,fo);
	      fwrite(&rt,sizeof(rt),1,fo);
	    }

	  fseek(fpreproc2, 0L, SEEK_SET);

	  while ((c = fgetc(fpreproc2)) != EOF)
	    compress(fo,c & 0xFF,0);
	  outbit(fo,-1);

	  cout << "Done\n";
	}

    }

  if (fpreproc2!=NULL)
    {
      fclose(fpreproc2);
      fpreproc2=NULL;
    }

  if (!debug_mode)
    {
      if (unlink_preproc_file)
	unlink(preproc_file_name);
      if (unlink_preproc2_file)
	unlink(preproc2_file_name);
    }


  if (fo!=NULL)
    {
      fclose(fo);
      fo=NULL;
    }

  if (CompressionHelpTree!=NULL)
    {
      free(CompressionHelpTree);
      CompressionHelpTree=NULL;
    }

  if (success)
    {
      cout << " Checking hypertext help file .......... In progress\n";

      cout << "   Suspicious source file lines ........ ";
      cout.flush();
      ShowSuspiciousLines();

      cout << "   Topic name length ................... ";
      cout.flush();

      CheckTopicLength();

      cout << "   Special topics ...................... ";
      cout.flush();

      CheckSpecialTopics();
      cout << "   Duplicate topics .................... ";
      cout.flush();

      CheckIndexForDuplicateTopics();
    }

  if (success)
    {
      if ((fi = OpenHelpFile(help_file,"r+b")) == NULL)
	{
	  cout << "    . Cannot open '" << help_file << "'\n";
	  success=FALSE;
	}
    }

  if (success)
    {
      cout << "   Invalid previous/next topics ........ ";
      cout.flush();
      CheckInvalidPrevNextTopics();
    }

  if (success)
    {
      cout << "   Invalid links ....................... ";
      cout.flush();
      success=CheckInvalidLinks();
    }

  if (success)
    {
      cout << "   Help window size .................... ";
      cout.flush();
      CheckHelpWindowSize(max_width);

      cout << "  Checking hypertext help file .......... Done\n";
      cout << " Indexing hypertext help file .......... ";
      cout.flush();

      success=SaveIndex(fi);
    }

  delete []HelpTree;

  if (fi!=NULL)
    {
      fclose(fi);
      fi=NULL;
    }

  FreeSuspiciousLines();
  FreeIndex();

  if (success)
    {
      cout << "\nHypertext help file '" << help_file << "' generated...\n";

      if (text_file_name!=NULL)
	cout <<   "     Text help file '" << text_file_name << "' generated...\n";
    }
  else
    {
      cout << "\nHypertext help file '" << help_file << "' NOT generated...\n";
      unlink(help_file);
      if (text_file_name!=NULL)
	{
	  cout <<   "     Text help file '" << text_file_name << "' NOT generated...\n";
	  unlink(text_file_name);
	}
    }

  delete [] help_file;

  if (debug_mode)
    {
      if (unlink_preproc_file)
	{
	  cout << "\nIntermediate files : \n";
	  cout << preproc_file_name << "\n";
	}
      if (unlink_preproc2_file)
	{
	  cout << preproc2_file_name << "\n";
	}
    }

  if (preproc_file_name!=NULL)
    delete []preproc_file_name;
  if (preproc2_file_name!=NULL)
    delete []preproc2_file_name;

  return(success?EXIT_SUCCESS:EXIT_FAILURE);
}
