/****************************************************************************/
/* TTEXTZN                                                                  */
/*--------------------------------------------------------------------------*/
/* Objet TTextZone (Zone d'dition de texte)                                */
/*--------------------------------------------------------------------------*/
/* Auteur     : DELPRAT Jean-Pierre                                         */
/* Cr le    : 05/07/97                                                    */
/****************************************************************************/

#include <conio.h>
#include <dos.h>
#include <io.h>
#include <fcntl.h>
#include <limits.h>
#include <new.h>
#include <share.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>

#include "Values.h"
#include "Settings.h"

#include "Const.h"
#include "JPDebug.h"

#include "JPAppli.h"

#include "Compat.h"

#include "Callback.h"
#include "Cursor.h"
#include "Mouse.h"
#include "Sound.h"
#include "SpChars.h"
#include "Strings.h"


#include "Screen.h"
#include "TScrBar.h"
#include "TWindow.h"

#include "TTextZn.h"

/*ͻ*/
/*                   PROPRIETES DES LIGNES DE LA ZONE                     */
/*ͼ*/

TLineProperties::TLineProperties()
{
}
TLineProperties::~TLineProperties()
{
}

/*ͻ*/
/*                               CONSTANTES                               */
/*ͼ*/

#define  DEFAULT_TAB_SIZE       8
#define  MAX_NB_LINES  		(MAXINT-1)

/*ͻ*/
/*                 INITIALISATION DES VARIABLES STATIQUES                 */
/*ͼ*/

int TTextZone::f_cursor_style=INSERT_CURSOR;

/*ͻ*/
/*                           METHODES PUBLIQUES                           */
/*ͼ*/

/****************************************************************************/
/* Constructeur                                                             */
/*--------------------------------------------------------------------------*/
/* parent           : Objet auquel appartient l'objet                       */
/* rel_x,rel_y      : Coordonnes de la zone texte p/r au groupe            */
/* width,height     : Dimensions de la zone                                 */
/* caption          : Titre de la zone (hot-key prcd de ~)               */
/* nb_cols          : Nombre de colonnes maximal du texte                   */
/* scrollbars       : Scrollbars de l'objet                                 */
/* enabled          : ENABLED si la zone est activable (DISABLED sinon)     */
/****************************************************************************/

TTextZone::TTextZone(PObject parent,
		     int rel_x,int rel_y,
                     int width,int height,
                     const char *caption,
		     int nb_cols,
		     TScrollbarChoice scrollbars,
		     boolean enabled)
	  :TObject(parent,
		   OBJ_TEXT_ZONE,
                   rel_x,rel_y,
		   width,height,
		   WHITE,
		   caption,
                   enabled,
		   (boolean)(caption[0]!=0), // FOCUS_DEPENDING_ASPECT
		   TRUE,  // CAN_BE_ENABLED
		   FALSE) // NOT_SIMPLE
{
  boolean vert,horiz;

  switch (scrollbars)
    {
      case SBC_HORIZONTAL : vert=FALSE;horiz=TRUE;break;
      case SBC_VERTICAL   : vert=TRUE; horiz=FALSE;break;
      case SBC_BOTH       : vert=TRUE; horiz=TRUE; break;
      default             : vert=FALSE;horiz=FALSE;break;
    }

  // Position /Dimensions de la zone de texte elle-mme p/r  l'objet

  f_text_rel_x=1;
  f_text_width=width-((vert)?3:2);

  if (caption[0]==0)
    {
      f_text_rel_y=0;
      f_text_height=height-((horiz)?1:0);
    }
  else
    {
      f_text_rel_y=1;
      f_text_height=height-((horiz)?2:1);
    }

  // Nombre maximal de colonnes du texte
  // Nombre de lignes actuel du texte

  if (nb_cols<1)
    nb_cols=1;

  f_nb_cols=nb_cols;
  f_nb_lines=0;

  // Tab size (when using m_insert_line, m_set_line...)

  f_tab_size=DEFAULT_TAB_SIZE;

  // Modification du texte autorise

  f_modification_enabled=TRUE;

  // Texte contenue dans la TextBox

  f_text=NULL;

  // Elments spciaux du texte

  f_first_visible_string=NULL;
  f_last_string=NULL;
  f_current_string=NULL;
  f_current_string_line=0;

  // Premire colonne,ligne visible

  f_first_visible_col=0;
  f_first_visible_line=0;

  // Zone slectionne

  f_zone_selected=FALSE;
  f_min_selected_line=0;
  f_min_selected_col=0;
  f_max_selected_line=0;
  f_max_selected_col=0;

  // Donnes en cours de slection

  f_first_selected_col=0;
  f_last_selected_col=0;
  f_first_selected_line=0;
  f_last_selected_line=0;

  // Position du cursor

  f_cursor_col=0;
  f_cursor_line=0;

  // Ascenseur

  if (vert)
    {
      f_vertical_scrollbar  =new TScrollBar(this,
					SB_VERTICAL,
					width-2,
					f_text_rel_y,
					f_text_height,
					DISABLED);

      f_vertical_scrollbar->m_set_can_be_enabled(FALSE);
      f_vertical_scrollbar->m_set_min_max_values(1L,1L);
      f_vertical_scrollbar->m_set_little_change(1L);
      f_vertical_scrollbar->m_set_big_change((long)(f_text_height));
    }

  else
    f_vertical_scrollbar=NULL;

  if (horiz)
    {
      f_horizontal_scrollbar=new TScrollBar(this,
					SB_HORIZONTAL,
					0,
					height-1,
					f_text_width+((vert)?1:2),
					(boolean)((f_nb_cols+1)>f_text_width)); // ENABLED/DISABLED

      f_horizontal_scrollbar->m_set_can_be_enabled((boolean)((f_nb_cols+1)>f_text_width));
      f_horizontal_scrollbar->m_set_min_max_values(1L,f_nb_cols-f_text_width+2);
      f_horizontal_scrollbar->m_set_little_change(1L);
      f_horizontal_scrollbar->m_set_big_change((long)(f_text_width));
    }
  else
    f_horizontal_scrollbar=NULL;

  // Chane retourne par m_get_line

  f_return_string=NULL;

  // Buffer pour la zone retourne par m_get_zone()

  f_zone_buffer=NULL;

  // Callbacks

  InitCallback(f_nb_lines_changed_action,f_nb_lines_changed_argument);
  InitCallback(f_text_changed_action,f_text_changed_argument);
  InitCallback(f_cursor_line_changed_action,f_cursor_line_changed_argument);
  InitCallback(f_cursor_col_changed_action,f_cursor_col_changed_argument);
  InitCallback(f_cursor_pos_changed_action,f_cursor_pos_changed_argument);
  InitCallback(f_visible_lines_changed_action,f_visible_lines_changed_argument);
  InitCallback(f_visible_cols_changed_action,f_visible_cols_changed_argument);
  InitCallback(f_selection_status_changed_action,f_selection_status_changed_argument);

  // Callbacks de l'ascenseur

  if (vert)
    f_vertical_scrollbar->m_set_value_changed_callback(m_vert_scrollbar_value_changed_call,"");

  if (horiz)
    f_horizontal_scrollbar->m_set_value_changed_callback(m_horiz_scrollbar_value_changed_call,"");

  // Cration d'une ligne

  m_insert_line(1,"",NULL);
}

/****************************************************************************/
/* Destructeur                                                              */
/*--------------------------------------------------------------------------*/
/****************************************************************************/

TTextZone::~TTextZone()
{
  PStringNode next_node,node;

  // Destruction des variables dynamiques

  DestroyCallback(f_selection_status_changed_action,f_selection_status_changed_argument);
  DestroyCallback(f_visible_cols_changed_action,f_visible_cols_changed_argument);
  DestroyCallback(f_visible_lines_changed_action,f_visible_lines_changed_argument);
  DestroyCallback(f_cursor_pos_changed_action,f_cursor_pos_changed_argument);
  DestroyCallback(f_cursor_col_changed_action,f_cursor_col_changed_argument);
  DestroyCallback(f_cursor_line_changed_action,f_cursor_line_changed_argument);
  DestroyCallback(f_text_changed_action,f_text_changed_argument);
  DestroyCallback(f_nb_lines_changed_action,f_nb_lines_changed_argument);


  m_free_zone_buffer();

  if (f_return_string!=NULL)
    {
      delete []f_return_string;
      f_return_string=NULL;
    }

  delete f_horizontal_scrollbar;
  delete f_vertical_scrollbar;

  // Effacement du texte (bien dsactiver les callbacks avant)

  node=f_text;

  while (node!=NULL)
    {
      delete [](node->string);
      if (node->properties!=NULL)
	delete node->properties;
      next_node=node->next;
      delete node;
      node=next_node;
    }
}

/****************************************************************************/
/* m_set_cursor_at                                                          */
/*--------------------------------------------------------------------------*/
/* Place le curseur a la position indique                                  */
/****************************************************************************/

void TTextZone::m_set_cursor_at(int col,int line,boolean make_visible)
{
  m_unselect_zone();
  m_set_cursor_position(col,line,make_visible);
}


/****************************************************************************/
/* m_get_cursor_pos                                                         */
/*--------------------------------------------------------------------------*/
/* Retourne la position du curseur                                          */
/****************************************************************************/

void TTextZone::m_get_cursor_pos(int &col,int &line)
{
  col=f_cursor_col;
  line=f_cursor_line;
}

/****************************************************************************/
/* m_set_tab_size                                                           */
/*--------------------------------------------------------------------------*/
/* Sets the size of the tab                                                 */
/****************************************************************************/

void TTextZone::m_set_tab_size(int tab_size)
{
  if (tab_size<1)
    tab_size=1;

  f_tab_size=tab_size;
}


/****************************************************************************/
/* m_set_first_visible_col                                                  */
/*--------------------------------------------------------------------------*/
/* Modifie la premire colonne visible du texte                             */
/****************************************************************************/

void TTextZone::m_set_first_visible_col(int col)
{
  if (col+f_text_width-2>f_nb_cols)
    col=f_nb_cols-f_text_width+2;

  if (col<1)
    col=1;

  if (col!=f_first_visible_col)
    {
      f_first_visible_col=col;
      m_display_text();
      m_set_cursor_position(f_cursor_col,f_cursor_line,FALSE);
      m_visible_cols_changed_callback();
    }
}

/****************************************************************************/
/* m_set_first_visible_line                                                 */
/*--------------------------------------------------------------------------*/
/* Modifie la premire ligne visible du texte                               */
/****************************************************************************/

void TTextZone::m_set_first_visible_line(int line)
{
  if (line+f_text_height-1>f_nb_lines)
    line=f_nb_lines+1-f_text_height;

  if (line<1)
    line=1;

  if (line!=f_first_visible_line)
    {
      f_first_visible_string=m_line_to_string(line);
      f_first_visible_line=line;
      m_display_text();
      m_set_cursor_position(f_cursor_col,f_cursor_line,FALSE);
      m_visible_lines_changed_callback();
    }
}

/****************************************************************************/
/* m_line_length                                                            */
/*--------------------------------------------------------------------------*/
/* Retourne la longueur d'une ligne du texte (sans les espaces de fin)      */
/****************************************************************************/

int TTextZone::m_line_length(int line_nb)
{
  if ((line_nb<1) || (line_nb>f_nb_lines))
    return(0);

  return(m_useful_length(m_line_to_string(line_nb)->string));
}

/****************************************************************************/
/* m_clear_text                                                             */
/*--------------------------------------------------------------------------*/
/* Efface tout le texte                                                     */
/****************************************************************************/

void TTextZone::m_clear_text()
{
  PStringNode next_node;
  PStringNode node;

  if (!f_modification_enabled)
    return;

  // Dselection

  m_unselect_zone();

  node=f_text;

  while (node!=NULL)
    {
      delete [](node->string);
      if (node->properties!=NULL)
	delete node->properties;
      next_node=node->next;
      delete node;
      node=next_node;
    }

  f_nb_lines=0;
  f_text=NULL;

  f_first_visible_line=0;
  f_first_visible_col=0;
  f_cursor_col=0;
  f_cursor_line=0;

  f_first_visible_string=NULL;
  f_last_string=NULL;

  f_current_string=NULL;
  f_current_string_line=0;

  m_insert_line(1,"",NULL);
}

/****************************************************************************/
/* m_delete_line                                                            */
/*--------------------------------------------------------------------------*/
/* Suppression d'une ligne du texte                                         */
/****************************************************************************/

void TTextZone::m_delete_line(int line_nb)
{
  PStringNode node_before,node_after;
  PStringNode node;

  if (!f_modification_enabled)
    return;

  // Validit des arguments

  if ((line_nb<1) || (line_nb>f_nb_lines))
    return;

  // Dselection

  m_unselect_zone();


  node=m_line_to_string(line_nb);

  // S'il n'y a qu'une ligne,
  // on la remplace en fait par une ligne vide

  if (f_nb_lines==1)
    {
      m_clear_text();
      return;
    }

  // Il restera au moins une ligne aprs suppression
  // Suppression du noeud

  node_before=node->last;
  node_after=node->next;

  if (f_last_string==node)
    f_last_string=node_before;

  if (node_before!=NULL)
    node_before->next=node_after;
  else
    f_text=node_after;

  if (node_after!=NULL)
    node_after->last=node_before;

  delete [](node->string);
  if (node->properties!=NULL)
    delete node->properties;

  delete node;

  // Mise  jour des champs

  f_nb_lines--;

  if (f_current_string_line>line_nb)
    f_current_string_line--;
  else
    {
      if (f_current_string_line==line_nb)
	{
	  f_current_string=node_before; // Peut tre NULL (line_nb==1)
          f_current_string_line--;      // Vaut alors 0
        }
    }

  if (f_first_visible_line>line_nb)
     f_first_visible_line--;
  else
    {
      if (f_first_visible_line==line_nb)
        {
          if (node_after!=NULL)
            f_first_visible_string=node_after;
          else
            {
              f_first_visible_string=node_before;
              f_first_visible_line--;
            }
        }

    }

  if (f_cursor_line>line_nb)
    {
      f_cursor_line--;
      m_set_cursor_position(f_cursor_col,f_cursor_line,FALSE);
    }
  else
    {
      if (f_cursor_line==line_nb)
        {
	  f_cursor_line=0;
          if (node_after!=NULL)
	    m_set_cursor_position(f_cursor_col,line_nb,FALSE);
	  else
	    m_set_cursor_position(f_cursor_col,line_nb-1,FALSE);
        }
    }

  // Pour tre sr que la premire ligne est correcte

  m_set_first_visible_line(f_first_visible_line);
  m_display_text();
  m_set_cursor_position(f_cursor_col,f_cursor_line,TRUE);

  m_nb_lines_changed_callback();
  m_text_changed_callback();
}

/****************************************************************************/
/* m_clear_zone                                                             */
/*--------------------------------------------------------------------------*/
/* Efface une zone de texte                                                 */
/* - si first_col vaut f_nb_cols+1, seul le saut de ligne de first_line est */
/*   pris en compte                                                         */
/* - si last_col vaut 0, la zone va jusqu'au saut de ligne de la ligne      */
/*   (last_line-1)                                                          */
/****************************************************************************/

void TTextZone::m_clear_zone(int first_col,int first_line,int last_col,int last_line)
{
  boolean was_open;
  int tmp;
  int size;
  PStringNode string_node;
  char *string;
  char *ptr;
  int line_length;

  char *end_of_last_string;
  int  end_of_last_string_length;

  int new_cursor_col,
      new_cursor_line;

  register int i;

  if (!f_modification_enabled)
    return;

  // Dselection

  m_unselect_zone();

  // Validit des arguments

  if (!m_correct_zone_limits(first_col,first_line,last_col,last_line))
    return;

  new_cursor_col=f_cursor_col;
  new_cursor_line=f_cursor_line;

  // Une partie d'une seule ligne  effacer

  if (last_line==first_line)
    {
      string_node=m_line_to_string(first_line);
      line_length=m_useful_length(string_node->string);

      if (f_cursor_line==first_line)
        {
          if (f_cursor_col>last_col)
	    new_cursor_col=f_cursor_col-(last_col-first_col+1);
          else
	    new_cursor_col=first_col;
	}

      if (first_col<=line_length)
	{
	  if (last_col>line_length)
	    last_col=line_length;

	  if ((!f_focused) || (f_cursor_line!=first_line))
	    m_expand_line(first_line);

	  string=string_node->string;

	  tmp=last_col-first_col+1;
	  size=line_length-last_col;
	  ptr=string+first_col-1;
	  memmove(ptr,string+last_col,size);
	  memset(ptr+size,' ',tmp);

	  if ((!f_focused) || (f_cursor_line!=first_line))
            m_reduce_line(first_line);

	  m_display_lines(f_cursor_line,f_cursor_line);
        }
    }

  // Partie  supprimer sur plusieurs lignes

  else
    {
      new_cursor_col=f_cursor_col;
      new_cursor_line=f_cursor_line;

      if (f_cursor_line>=first_line)
        {
	  if (f_cursor_line==first_line)
            {
              if (f_cursor_col>first_col)
		new_cursor_col=first_col;
            }
          else
            {
              if (    (f_cursor_line<last_line)
                  || ((f_cursor_line==last_line) && (f_cursor_col<=last_col)))
                {
                  new_cursor_col=first_col;
                  new_cursor_line=first_line;
                }
              else
                {
		  new_cursor_line-=(last_line-first_line);
                  new_cursor_col=first_col+(f_cursor_col-(last_col+1));
                }
            }
        }

      was_open=f_open;
      f_open=FALSE;   // Pour ne pas afficher plusieurs fois la mme chose

      // Rcupration de la fin de la dernire ligne

      string=m_line_to_string(last_line)->string;
      end_of_last_string_length=m_useful_length(string);

      if (last_col<end_of_last_string_length)
        {
          string+=last_col;
          end_of_last_string_length-=last_col;
        }
      else
	end_of_last_string_length=0;

      end_of_last_string=new char [end_of_last_string_length+1];
      memcpy(end_of_last_string,string,end_of_last_string_length);
      end_of_last_string[end_of_last_string_length]=0;

      // Effacement des lignes intermdiaires

      for (i=last_line;i>first_line;i--)
	m_delete_line(i);

      // Remplacement de la fin de la premire ligne par la fin de la dernire

      if ((!f_focused) || (f_cursor_line!=first_line))
        m_expand_line(first_line);

      string=m_line_to_string(first_line)->string;
      line_length=m_useful_length(string);

      tmp=f_nb_cols+1-first_col;
      if (tmp<end_of_last_string_length)
        end_of_last_string_length=tmp;

      ptr=string+first_col-1;
      memcpy(ptr,end_of_last_string,end_of_last_string_length);
      tmp=line_length-(first_col+end_of_last_string_length-1);
      if (tmp>0)
	memset(ptr+end_of_last_string_length,' ',tmp);

      if ((!f_focused) || (f_cursor_line!=first_line))
	m_reduce_line(first_line);

      delete []end_of_last_string;

      f_open=was_open;
      m_display_text();
    }

  m_set_cursor_position(new_cursor_col,new_cursor_line,TRUE);
  m_text_changed_callback();
}

/****************************************************************************/
/* m_insert_text                                                            */
/*--------------------------------------------------------------------------*/
/* Insre du texte  la position du curseur (lignes spares par des \n)    */
/****************************************************************************/

void TTextZone::m_insert_text(const char *text)
{
  char        *string;
  char        *new_string;

  int         line_length,
	      start_of_1st_string_length=0,
	      end_of_1st_string_length=0;

  char        *end_of_1st_string=NULL;

  char        *insertion_place;

  int         tmp;

  boolean     was_open;

  char *end_of_line;
  char        end_character=0;
  int         y;

  char        *ptr;


  if (!f_modification_enabled)
    return;

  // Effacement de la zone slectionne

  m_clear_selection();

  // Parcours sur toutes les lignes  afficher

  if (text[0]==0)
    return;


  // Une seule ligne  insrer

  if (strchr(text,'\n')==NULL)
    {
      if (!f_focused)
	m_expand_line(f_cursor_line);

      string=m_line_to_string(f_cursor_line)->string;
      line_length=strlen(text);

      tmp=f_nb_cols+line_length;
      new_string=new char[tmp+1];
      ptr=new_string;
      memcpy(ptr,string,f_cursor_col-1);
      ptr+=f_cursor_col-1;
      memcpy(ptr,text,line_length);
      ptr+=line_length;
      memcpy(ptr,string+f_cursor_col-1,f_nb_cols-(f_cursor_col-1));
      new_string[tmp]=0;

      m_set_line(f_cursor_line,new_string);
      delete []new_string;

      m_display_lines(f_cursor_line,f_cursor_line);
    }

  // Plusieurs lignes  insrer

  else
    {
      was_open=f_open;
      f_open=FALSE;   // Pour ne pas afficher plusieurs fois la mme chose
      y=f_cursor_line;
      end_of_line=(char*)text;

      // Parcours des diffrentes lignes  ajouter

      while (end_of_line!=NULL)
	{
	  end_of_line=strchr(text,'\n');
	  if (end_of_line!=NULL)
	    {
	      end_character=(*end_of_line);
	      (*end_of_line)=0;
	    }

	  // Ajout d'une ligne

	  if (y==f_cursor_line)
	    {

	      if (!f_focused)
		m_expand_line(f_cursor_line);

	      // On met de ct la fin de la premire ligne avant insertion
	      // (aprs le point d'insertion)

	      string=m_line_to_string(f_cursor_line)->string;

	      start_of_1st_string_length=f_cursor_col-1;
	      insertion_place=string+start_of_1st_string_length;

	      end_of_1st_string_length=m_useful_length(insertion_place);

	      end_of_1st_string=new char [end_of_1st_string_length+1];
	      memcpy(end_of_1st_string,insertion_place,end_of_1st_string_length);
	      end_of_1st_string[end_of_1st_string_length]=0;

	      // On ajoute la premire ligne du texte  insrer

	      line_length=strlen(text);

	      tmp=start_of_1st_string_length+line_length;
	      new_string=new char[tmp+1];
	      ptr=new_string;
	      memcpy(ptr,string,start_of_1st_string_length);
	      ptr+=start_of_1st_string_length;
	      memcpy(ptr,text,line_length);
	      new_string[tmp]=0;

	      m_set_line(f_cursor_line,new_string);
	      delete []new_string;

	    }
	  else {
	    m_insert_line(y, end_of_line==NULL ? "" : text, NULL);
          }

	  // On passe  la ligne suivante */

	  if (end_of_line!=NULL)
	  {
	    (*end_of_line)=end_character;
	    text=end_of_line+1;
	    y++;
	  }
	}

      // On insre au dbut de la ligne
      // la fin de la premire ligne

      DEBUG_TEST(y!=f_cursor_line);

      line_length=strlen(text);

      tmp=line_length+end_of_1st_string_length;
      new_string=new char[tmp+1];
      ptr=new_string;
      memcpy(ptr,text,line_length);
      ptr+=line_length;
      memcpy(ptr,end_of_1st_string,end_of_1st_string_length);
      new_string[tmp]=0;

      m_set_line(y,new_string);
      delete []new_string;

      delete []end_of_1st_string;

      f_open=was_open;

      m_display_lines(f_cursor_line,f_nb_lines);
    }

  m_set_cursor_position(f_cursor_col,f_cursor_line,TRUE);
  m_text_changed_callback();
}


/****************************************************************************/
/* m_get_zone                                                               */
/*--------------------------------------------------------------------------*/
/* Retourne une zone du texte aprs avoir allou un buffer pour le stocker  */
/* - si first_col vaut f_nb_cols+1, seul le saut de ligne de first_line est */
/*   pris en compte                                                         */
/* - si last_col vaut 0, la zone va jusqu'au saut de ligne de la ligne      */
/*   (last_line-1)                                                          */
/****************************************************************************/

char *TTextZone::m_get_zone(int first_col,int first_line,int last_col,int last_line)
{
  char *buffer;
  char *ptr;

  PStringNode node;
  char *string;

  register int i;
  int line_length;
  int tmp;

  unsigned size;

  m_free_zone_buffer();

  size=m_get_zone_size(first_col,first_line,last_col,last_line);

  if (size!=1)
    {
//      set_new_handler(0);
      buffer = new(nothrow) char [size];
//      set_new_handler(TApplication::m_new_handler);

      if (buffer==NULL)
	size=1;
    }

  if (size==1)
    {
      buffer=new char [1];
      ptr=buffer;
    }
  else
    {
      ptr=buffer;
      node=m_line_to_string(first_line);
      string=node->string;

      if (first_line==last_line)
	{
	  line_length=strlen(string)-(first_col-1);
	  string+=(first_col-1);

	  tmp=(last_col-first_col+1);
	  if (line_length>tmp)
	    line_length=tmp;
	  if (line_length<0)
	    line_length=0;

	  memcpy(ptr,string,line_length);
	  ptr+=line_length;

	  if (tmp>line_length)
	    {
	      tmp-=line_length;
	      memset(ptr,' ',tmp);
	      ptr+=tmp;
	    }
	}
      else
	{
	  line_length=m_useful_length(string);

	  if (first_col<=line_length)
	    {
	      tmp=(line_length-first_col+1);
	      memcpy(ptr,string+first_col-1,tmp);
	      ptr+=tmp;
	    }

	  (*ptr)='\n';
	  ptr++;

	  for (i=first_line+1;i<last_line;i++)
	    {
	      node=node->next;
	      string=node->string;
	      tmp=m_useful_length(string);
	      memcpy(ptr,string,tmp);
	      ptr+=tmp;
	      (*ptr)='\n';
	      ptr++;
	    }

	  node=node->next;
	  string=node->string;
	  line_length=strlen(string);
	  tmp=last_col;

	  if (line_length>tmp)
	    line_length=tmp;
	  memcpy(ptr,string,line_length);
	  ptr+=line_length;
	  if (tmp>line_length)
	    {
	      tmp-=line_length;
	      memset(ptr,' ',tmp);
	      ptr+=tmp;
	    }
	}
    }

  // 0 terminal

  (*ptr)=0;

  DEBUG_TEST(size==(unsigned)(ptr-buffer+1));

  f_zone_buffer=buffer;
  return(f_zone_buffer);
}

/****************************************************************************/
/* m_free_zone_buffer                                                       */
/*--------------------------------------------------------------------------*/
/* Libre le buffer allou par la fonction m_get_zone.                      */
/* A appeler quand la zone retourne par m_get_zone n'est plus utile.       */
/****************************************************************************/

void TTextZone::m_free_zone_buffer()
{
  if (f_zone_buffer!=NULL)
    {
      delete []f_zone_buffer;
      f_zone_buffer=NULL;
    }
}

/****************************************************************************/
/* m_get_line                                                               */
/*--------------------------------------------------------------------------*/
/* Retourne une ligne du texte.                                             */
/* La ligne peut tre manipule sans consquence pour le texte.             */
/****************************************************************************/

char *TTextZone::m_get_line(int line_nb)
{
  char *string;
  int  length;

  if (f_return_string!=NULL)
    delete []f_return_string;

  if ((line_nb<1) || (line_nb>f_nb_lines))
    string="";
  else
    string=m_line_to_string(line_nb)->string;

  length=m_useful_length(string);

  f_return_string=new char[length+1];
  strncpy(f_return_string,string,length);
  f_return_string[length]=0;
  return(f_return_string);
}

/****************************************************************************/
/* m_save_text                                                              */
/*--------------------------------------------------------------------------*/
/* Sauvegarde du texte dans un fichier.                                     */
/* Retourne FALSE en cas d'chec.                                           */
/****************************************************************************/

boolean TTextZone::m_save_text(const char *filename)
{
  FILE *file;

  register int i;
  char *line;
  TMousePointer pointer;

  boolean error;

//int handle = sopen(filename, O_WRONLY|O_CREAT|O_TEXT,SH_DENYWR,S_IWRITE);
//  if (handle<0)
//    return(FALSE);
//  file=fdopen(handle,"wt");
  file=fopen(filename,"wt"); // Fix: overwrite as text data
  if (file==NULL)
    {
//      close(handle);
      return(FALSE);
    }

  pointer=GetMousePointer();
  SetMousePointer(MP_HOURGLASS);

  error=FALSE;
  for (i=1;i<=f_nb_lines;i++)
    {
      line=m_get_line(i);

      if (fputs(line,file)==EOF)
	error=TRUE;
      else
	// if (fputs("\n",file)==EOF)
	if (i != f_nb_lines && fputs("\n",file)==EOF) // Fix: last line added '\n'
	  error=TRUE;

      if (error)
	{
	  fclose(file);
	  remove(filename);
	  return(FALSE);
	}
    }
  fclose(file);
  SetMousePointer(pointer);
  return(TRUE);
}

/****************************************************************************/
/* m_load_text                                                              */
/*--------------------------------------------------------------------------*/
/* Charge le texte contenu dans un fichier.                                 */
/* Retourne FALSE en cas d'chec.                                           */
/****************************************************************************/

boolean TTextZone::m_load_text(const char *filename)
{
  int handle;
  FILE *file;
  char *line;
  register int i;
  int n,len;
  boolean success;
  boolean was_open;
  boolean line_complete;
  TMousePointer pointer;

  if (!f_modification_enabled)
    return(FALSE);

  m_clear_text();

  handle = sopen(filename, O_RDONLY | O_TEXT, SH_DENYWR, S_IREAD);
  if (handle<0)
    return(FALSE);

  file=fdopen(handle,"rt");
  if (file==NULL)
    {
      close(handle);
      return(FALSE);
    }

  pointer=GetMousePointer();
  SetMousePointer(MP_HOURGLASS);

  i=2;
  n=f_nb_cols+1;
  line=new char [n];

  was_open=f_open;
  f_open=FALSE; // Pour ne pas rafficher l'cran  chaque ligne

  line_complete=TRUE;
  while (fgets(line,n,file))
    {
      len=strlen(line);

      if ((len>=1) && (line[len-1]=='\n'))
	line[len-1]=0;
      else
	line_complete=FALSE;

      m_insert_line(i,line,NULL);
      i++;

      if (!line_complete)
	{
	  line_complete=TRUE;
	  while (fgets(line,n,file))
	    {
	      len=strlen(line);
	      if (len<1)
		break;
	      else
		{
		  if (line[len-1]=='\n')
		    break;
		}
	    }
	}
    }

  success=(boolean)(feof(file)!=0);

  delete[]line;
  fclose(file);

  SetMousePointer(pointer);

  // En cas d'erreur

  if (!success)
    {
      m_clear_text();
      return(FALSE);
    }

  // On efface la premire ligne (ligne vide ajoute par m_clear_text)

  m_delete_line(1);

  // On affiche le tout

  f_open=was_open;
  m_display_text();
  m_set_cursor_position(f_cursor_col,f_cursor_line,TRUE);

  return(TRUE);
}

/****************************************************************************/
/* m_select_zone                                                            */
/*--------------------------------------------------------------------------*/
/* Slectionne une zone du texte                                            */
/* - si first_col vaut f_nb_cols+1, seul le saut de ligne de first_line est */
/*   pris en compte                                                         */
/* - si last_col vaut 0, la zone va jusqu'au saut de ligne de la ligne      */
/*   (last_line-1)                                                          */
/****************************************************************************/

void TTextZone::m_select_zone(int pos1_col,int pos1_line,int pos2_col,int pos2_line)
{
  int tmp;

  boolean was_selected;
  int     old_min_selected_line=1,
	  old_max_selected_line=1;

  boolean first_is_1;
  boolean same_line;


  tmp=f_nb_cols+1;
  if (pos1_col<1)           pos1_col=1;
  if (pos1_col>tmp)         pos1_col=tmp;
  if (pos2_col<1)           pos2_col=1;
  if (pos2_col>tmp)         pos2_col=tmp;
  if (pos1_line<1)          pos1_line=1;
  if (pos1_line>f_nb_lines) pos1_line=f_nb_lines;
  if (pos2_line<1)          pos2_line=1;
  if (pos2_line>f_nb_lines) pos2_line=f_nb_lines;

  f_first_selected_col=pos1_col;
  f_first_selected_line=pos1_line;
  f_last_selected_col=pos2_col;
  f_last_selected_line=pos2_line;

  same_line=(boolean)(pos1_line==pos2_line);

  if ((pos1_col==pos2_col) && (same_line))
    {
      m_unselect_zone();
      m_set_cursor_position(pos1_col,pos1_line,TRUE);
      return;
    }

  if (same_line)
    first_is_1=(boolean)(pos1_col<=pos2_col);
  else
    first_is_1=(boolean)(pos1_line<pos2_line);


  was_selected=f_zone_selected;
  if (was_selected)
    {
      old_min_selected_line=f_min_selected_line;
      old_max_selected_line=f_max_selected_line;
    }
  else
    f_zone_selected=TRUE;

  if (first_is_1)
    {
      f_min_selected_line=pos1_line;
      f_min_selected_col=pos1_col;
      f_max_selected_line=pos2_line;
      f_max_selected_col=pos2_col-1;
    }
  else
    {
      f_min_selected_line=pos2_line;
      f_min_selected_col=pos2_col;
      f_max_selected_line=pos1_line;
      f_max_selected_col=pos1_col-1;
    }

  if (!was_selected)
    m_display_lines(f_min_selected_line,f_max_selected_line);
  else
    m_display_lines(MIN(f_min_selected_line,old_min_selected_line),
                    MAX(f_max_selected_line,old_max_selected_line));

  m_set_cursor_position(pos2_col,pos2_line,TRUE);

  if (!was_selected)
    m_selection_status_changed_callback();
}

/****************************************************************************/
/* m_unselect_zone                                                          */
/*--------------------------------------------------------------------------*/
/* Dslectionne la partie du texte slectionne.                           */
/****************************************************************************/

void TTextZone::m_unselect_zone()
{
  if (f_zone_selected)
    {
      f_zone_selected=FALSE;
      m_display_lines(f_min_selected_line,f_max_selected_line);
      m_selection_status_changed_callback();
    }
}

/****************************************************************************/
/* m_get_selected_zone                                                      */
/*--------------------------------------------------------------------------*/
/* Si m_zone_selected() est TRUE, retourne les limites de la zone           */
/* slectionne.                                                            */
/****************************************************************************/

void TTextZone::m_get_selected_zone(int &min_col,int &min_line,int &max_col,int &max_line)
{
  min_col =f_min_selected_col;
  min_line=f_min_selected_line;
  max_col =f_max_selected_col;
  max_line=f_max_selected_line;
}

/****************************************************************************/
/* m_cut_selection                                                          */
/*--------------------------------------------------------------------------*/
/* Dcoupe la zone slectionne et la met dans le presse-papiers.           */
/****************************************************************************/

void TTextZone::m_cut_selection()
{
  if (!f_modification_enabled)
    return;

  if (f_zone_selected)
    {
      m_copy_selection();
      if (((JPGetClipboard())[0])!=0)
	m_clear_selection();
    }
}

/****************************************************************************/
/* m_copy_selection                                                         */
/*--------------------------------------------------------------------------*/
/* Copie la zone slectionne dans le presse-papiers.                       */
/****************************************************************************/

void TTextZone::m_copy_selection()
{
  if (f_zone_selected)
    {
      JPSetClipboard(m_get_zone(f_min_selected_col,f_min_selected_line,
                                f_max_selected_col,f_max_selected_line));
      m_free_zone_buffer();
    }
}

/****************************************************************************/
/* m_paste                                                                  */
/*--------------------------------------------------------------------------*/
/* Colle le contenu du presse-papiers  l'emplacement du curseur.           */
/****************************************************************************/

void TTextZone::m_paste()
{
  if (!f_modification_enabled)
    return;

  m_insert_text(JPGetClipboard());
}

/****************************************************************************/
/* m_clear_selection                                                        */
/*--------------------------------------------------------------------------*/
/* Efface la zone slectionne.                                             */
/****************************************************************************/

void TTextZone::m_clear_selection()
{
  if (!f_modification_enabled)
    return;

  if (f_zone_selected)
    {
      m_clear_zone(f_min_selected_col,f_min_selected_line,
		    f_max_selected_col,f_max_selected_line);
      m_set_cursor_position(f_cursor_col,f_cursor_line,TRUE);
    }
}

/****************************************************************************/
/* m_line_to_string                                                         */
/*--------------------------------------------------------------------------*/
/* Retourne un pointeur sur une ligne du texte                              */
/* (la ligne doit exister).                                   		    */
/****************************************************************************/

PStringNode TTextZone::m_line_to_string(int  line)
{
  PStringNode node=NULL;
  register int  i;
  int  last_i;
  boolean forward=TRUE;


  int  gap[4];
  int  choice;

  DEBUG_TEST((line>=1) && (line<=f_nb_lines));

  if (line==f_current_string_line)
    return(f_current_string);

  // On cherche le point de dpart du parcours de la liste
  // le plus intressant entre :

  gap[0]=abs(line-f_current_string_line);      // l'lment courant
  gap[1]=abs(line-f_first_visible_line);       // le premier visible
  gap[2]=line-1;                               // le premier lment
  gap[3]=f_nb_lines-line;                      // le dernier lment


  if (gap[0]<gap[1])
    choice=0;
  else
    choice=1;

  if (gap[2]<gap[choice])
    choice=2;

  if (gap[3]<gap[choice])
    choice=3;

  // Sens de parcours de la liste et point de depart

  switch(choice)
    {
      case 0 : // Elment courant
	       forward=(boolean)(line>f_current_string_line);
	       node=f_current_string;
	       break;
      case 1 : // Premier visible
	       forward=(boolean)(line>f_first_visible_line);
	       node=f_first_visible_string;
	       break;
      case 2 : // Premier lement
	       forward=TRUE;
	       node=f_text;
	       break;
      case 3 : // Dernier lment
	       forward=FALSE;
	       node=f_last_string;
	       break;
    }


   // Parcours de la liste

  last_i=gap[choice];

  switch (forward)
    {
      case TRUE : for (i=0;i<last_i;i++)
		    node=node->next;
		  break;
      case FALSE: for (i=0;i<last_i;i++)
		    node=node->last;
		  break;
    }

  f_current_string=node;
  f_current_string_line=line;

  return(node);
}
/****************************************************************************/
/* m_set_nb_lines_changed_callback                                          */
/*--------------------------------------------------------------------------*/
/* Dfinition du callback associ  la modification du nombre de lignes du  */
/* texte                                                                    */
/****************************************************************************/

void TTextZone::m_set_nb_lines_changed_callback(void (*nb_lines_changed_action)(PObject, const char *),
                                                const char *nb_lines_changed_argument)
{
  SetCallback(f_nb_lines_changed_action,f_nb_lines_changed_argument,
	      nb_lines_changed_action,nb_lines_changed_argument);
}

/****************************************************************************/
/* m_set_text_changed_callback                                              */
/*--------------------------------------------------------------------------*/
/* Dfinition du callback associ  la modification du contenu de la zone   */
/* de texte                                                                 */
/****************************************************************************/

void TTextZone::m_set_text_changed_callback(void (*text_changed_action)(PObject, const char *),
					    const char *text_changed_argument)

{
  SetCallback(f_text_changed_action,f_text_changed_argument,
	      text_changed_action,text_changed_argument);
}

/****************************************************************************/
/* m_set_cursor_col/line_changed_callback                                   */
/*--------------------------------------------------------------------------*/
/* Dfinition du callback associ au dplacement du curseur                 */
/****************************************************************************/

void TTextZone::m_set_cursor_line_changed_callback(void (*cursor_line_changed_action)(PObject, const char *),
					           const char *cursor_line_changed_argument)

{
  SetCallback(f_cursor_line_changed_action,f_cursor_line_changed_argument,
	      cursor_line_changed_action,cursor_line_changed_argument);
}

void TTextZone::m_set_cursor_col_changed_callback(void (*cursor_col_changed_action)(PObject, const char *),
					          const char *cursor_col_changed_argument)

{
  SetCallback(f_cursor_col_changed_action,f_cursor_col_changed_argument,
	      cursor_col_changed_action,cursor_col_changed_argument);
}

void TTextZone::m_set_cursor_pos_changed_callback(void (*cursor_pos_changed_action)(PObject, const char *),
					          const char *cursor_pos_changed_argument)

{
  SetCallback(f_cursor_pos_changed_action,f_cursor_pos_changed_argument,
	      cursor_pos_changed_action,cursor_pos_changed_argument);
}

/****************************************************************************/
/* m_set_visible_lines_changed_callback                                     */
/*--------------------------------------------------------------------------*/
/* Dfinition du callback associ au changement de la partie visible du     */
/* texte (en hauteur).                                                      */
/****************************************************************************/

void TTextZone::m_set_visible_lines_changed_callback(void (*visible_lines_changed_action)(PObject, const char *), const char *visible_lines_changed_argument)
{
  SetCallback(f_visible_lines_changed_action,f_visible_lines_changed_argument,
	      visible_lines_changed_action,visible_lines_changed_argument);
}

void TTextZone::m_set_visible_cols_changed_callback(void (*visible_cols_changed_action)(PObject,const char *), const char *visible_cols_changed_argument)
{
  SetCallback(f_visible_cols_changed_action,f_visible_cols_changed_argument,
	      visible_cols_changed_action,visible_cols_changed_argument);
}

/****************************************************************************/
/* m_set_selection_status_changed_callback                                  */
/*--------------------------------------------------------------------------*/
/* Dfinition du callback associ au changement d'tat de la slection d'une*/
/* partie du texte (une zone est slectionne ou il n'y a pas de zone       */
/* slectionne).                                                           */
/****************************************************************************/

void TTextZone::m_set_selection_status_changed_callback(void (*selection_status_changed_action)(PObject, const char *), const char *selection_status_changed_argument)
{
  SetCallback(f_selection_status_changed_action,f_selection_status_changed_argument,
              selection_status_changed_action,selection_status_changed_argument);
}

/*ͻ*/
/*                           METHODES PROTEGEES                           */
/*ͼ*/

/****************************************************************************/
/* m_set_line                                                               */
/*--------------------------------------------------------------------------*/
/* Modifie une des lignes du texte                                          */
/****************************************************************************/

void TTextZone::m_set_line(int line_nb, const char *string, PLineProperties properties)
{
  PStringNode node;
  int length,size;
  int tab_size;
  int added_size;
  int col;

  char *ptr;

  if (!f_modification_enabled)
    return;

  // Validit des arguments

  if ((line_nb<1) || (line_nb>f_nb_lines))
    return;

  // Dselection

  m_unselect_zone();


  if (strchr(string,'\n')!=NULL)
    string="";

  // Longueur de la chaine (en comptant les tabulations)

  length=strlen(string);
  ptr=strchr(string,'\t');
  added_size=0;
  while (ptr!=NULL)
    {
      col=(int)(ptr-string)+added_size;
      tab_size=f_tab_size-(col%f_tab_size);
      added_size+=(tab_size-1);
      length+=(tab_size-1);
      ptr=strchr(ptr+1,'\t');
    }

  if (length>f_nb_cols)
    length=f_nb_cols;


  // Creation de la chaine

  node=m_line_to_string(line_nb);
  if (node->properties!=NULL)
    delete node->properties;
  node->properties=properties;
  delete []node->string;

  node->string=new char[length+1];
  (node->string)[length]=0;

  // Remplissage de la chaine

  const char* ptr1 = string;
  char* ptr2 = node->string;
  ptr=strchr(ptr1,'\t');
  added_size=0;
  while ((length>0) && (ptr!=NULL))
    {
      col=(int)(ptr-string)+added_size;
      tab_size=f_tab_size-(col%f_tab_size);
      added_size+=(tab_size-1);

      size=(int)(ptr-ptr1);
      if (size>length)
	size=length;
      memcpy(ptr2,ptr1,size);
      length-=size;
      ptr1+=size;
      ptr2+=size;

      size=tab_size;
      if (size>length)
	size=length;
      memset(ptr2,' ',size);
      length-=size;
      ptr1=ptr+1;
      ptr2+=size;

      ptr=strchr(ptr1,'\t');
    }

  memcpy(ptr2,ptr1,length);

  if (f_cursor_line==line_nb)
    {
      f_cursor_line=0;  // Pour ventuellement agrandir la chane
      m_set_cursor_position(f_cursor_col,line_nb,FALSE);
    }

  m_display_lines(line_nb,line_nb);
  m_text_changed_callback();
}


/****************************************************************************/
/* m_insert_line                                                            */
/*--------------------------------------------------------------------------*/
/* Insertion d'une ligne dans le texte                                      */
/****************************************************************************/

void TTextZone::m_insert_line(int line_nb, const char *string, PLineProperties properties)
{
  PStringNode node_before;
  PStringNode node;

  int line_before;

  PStringNode new_node;

  if (!f_modification_enabled)
    return;

  // Validit des arguments

  if ((line_nb<1) || (line_nb>(f_nb_lines+1)))
    return;

  // Dselection

  m_unselect_zone();


  if (f_nb_lines==MAX_NB_LINES)
    return;

  new_node=new TStringNode;
  new_node->string=new char[1];
  new_node->string[0]=0;
  new_node->properties=NULL;

  // Position d'insertion

  if ((f_nb_lines+1)==line_nb)  // Ajout  la fin
    {
      line_before=f_nb_lines;
    }
  else
    {
      line_before=line_nb-1;
    }

  // Insertion en dbut de texte

  if (line_before==0)
    {
      node=f_text;
      if (node!=NULL)
	node->last=new_node;
      new_node->last=NULL;
      new_node->next=f_text; // Peut tre NULL
      f_text=new_node;
    }

  // Insertion en milieu/fin de liste

  else
    {
      node_before=m_line_to_string(line_before);

      node=node_before->next;
      if (node!=NULL)
	node->last=new_node;
      new_node->next=node;
      new_node->last=node_before;
      node_before->next=new_node;
    }

  // Mise  jour des champs

  if (line_before==f_nb_lines)
    f_last_string=new_node;

  f_nb_lines++;

  if (f_nb_lines==1)
    {
      m_set_first_visible_line(1);
      m_set_first_visible_col(1);
      m_set_cursor_position(1,1,FALSE);
    }
  else
    {
      if (f_current_string_line>line_before)
	f_current_string_line++;

      if (f_first_visible_line>line_before)
	f_first_visible_string=f_first_visible_string->last;

      if (f_cursor_line>line_before)
	{
	  f_cursor_line++;
	  m_set_cursor_position(f_cursor_col,f_cursor_line,FALSE);
	}
    }

  m_set_line(line_nb,string,properties);
  m_display_lines(line_nb+1,f_nb_lines);
  m_set_cursor_position(f_cursor_col,f_cursor_line,TRUE);

  m_nb_lines_changed_callback();
  m_text_changed_callback();
}


/****************************************************************************/
/* m_display                                                                */
/*--------------------------------------------------------------------------*/
/* Affichage de l'objet                                                     */
/****************************************************************************/

void TTextZone::m_display()
{
  int x1,y1,x2,y2;
  char char_left,char_right;
  register int i;

  if (!f_open)
    return;

  TObject::m_display();

  x1=m_get_x_in_window();
  y1=m_get_y_in_window();
  y2=y1+f_height-1;
  x2=x1+f_width-1;

  // Barre du haut si pas de titre

  f_window->m_textattr(((f_parent->m_get_background())<<4)+(unsigned)BLACK);
  if (f_caption[0]==0)
    {

      f_window->m_gotoxy(x1,y1-1);
      f_window->m_putch(SPECIAL_CHAR(SCH_TEXTZONE_UP_LEFT));
      f_window->m_putnch(f_width-2,SPECIAL_CHAR(SCH_TEXTZONE_UP));
      f_window->m_putch(SPECIAL_CHAR(SCH_TEXTZONE_UP_RIGHT));
    }

  // Barre du bas si pas d'ascenseur horizontal

  if (f_horizontal_scrollbar==NULL)
    {
      f_window->m_gotoxy(x1,y2+1);
       f_window->m_putch(SPECIAL_CHAR(SCH_TEXTZONE_BOTTOM_LEFT));
      f_window->m_putnch(f_width-2,SPECIAL_CHAR(SCH_TEXTZONE_BOTTOM));
      f_window->m_putch(SPECIAL_CHAR(SCH_TEXTZONE_BOTTOM_RIGHT));
    }

  // Barre de gauche
  // Barre de droite si pas d'ascenseur vertical

  if (f_horizontal_scrollbar!=NULL)
    y2--;

  char_left=SPECIAL_CHAR(SCH_TEXTZONE_LEFT);
  char_right=SPECIAL_CHAR(SCH_TEXTZONE_RIGHT);

  f_window->m_set_normal_attr(f_background);

  for (i=(f_caption[0]==0)?y1:(y1+1);i<=y2;i++)
    {
      f_window->m_gotoxy(x1,i);
      f_window->m_putch(char_left);
      if (f_vertical_scrollbar==NULL)
	{
	  f_window->m_gotoxy(x2,i);
	  f_window->m_putch(char_right);
	}
    };

  // Coin bas droite si 2 ascenseurs

  if ((f_horizontal_scrollbar!=NULL) && (f_vertical_scrollbar!=NULL))
    {
      f_window->m_textattr((LIGHTGRAY<<4)+(unsigned)BLACK);
      f_window->m_gotoxy(x2-1,y2+1);
      f_window->m_puts(SPECIAL_STRING(SST_TEXTZONE_SB_CORNER));
    }

  m_display_text();
}


/****************************************************************************/
/* m_display_focus_depending_part                                           */
/*--------------------------------------------------------------------------*/
/* Affichage de la partie de l'objet dont l'aspect dpend du focus          */
/****************************************************************************/

void TTextZone::m_display_focus_depending_part()
{
  m_display_caption();

  if (f_zone_selected)
    m_display_lines(f_min_selected_line,f_max_selected_line);
}


/****************************************************************************/
/* m_display_lines                                                          */
/*--------------------------------------------------------------------------*/
/* Affichage de certaines lignes du texte                                   */
/* Ne complte pas avec des lignes vides si ces lignes ne vont pas jusqu'en */
/* bas de la zone d'affichage du texte.                                     */
/****************************************************************************/

void TTextZone::m_display_lines(int first_line,int last_line)
{
  int       x,y1,y2,tmp;
  register  int   i;

  PStringNode string_node;

  if (!f_open)
    return;

  if (   (first_line>last_line)
      || (last_line<1)
      || (first_line>f_nb_lines))
    return;

  x=m_get_x_in_window()+f_text_rel_x;
  y1=m_get_y_in_window()+f_text_rel_y;
  y2=y1+f_text_height-1;


  // Affichage de la partie du texte comportant des lignes

  if (first_line<f_first_visible_line)
    first_line=f_first_visible_line;

  i=y1+first_line-f_first_visible_line;
  if (i>y2)
    return;

  tmp=i+(last_line-first_line);
  if (tmp<y2)
    y2=tmp;

  string_node=m_line_to_string(first_line);

  while ((i<=y2) && (string_node!=NULL))
    {
      f_window->m_gotoxy(x,i);
      m_display_string_node(string_node,first_line);
      string_node=string_node->next;
      first_line++;
      i++;
    }
}

/****************************************************************************/
/* m_lose_focus                                                             */
/*--------------------------------------------------------------------------*/
/* Appele quand l'objet perd le focus                                      */
/****************************************************************************/

void TTextZone::m_lose_focus()
{
  if (!f_focused)
    return;
  TObject::m_lose_focus();
  if (f_focused)
    return;

  m_reduce_line(f_cursor_line);

  HideTextCursor();
}

/****************************************************************************/
/* m_take_focus                                                             */
/*--------------------------------------------------------------------------*/
/* Appele quand l'objet prend le focus                                     */
/****************************************************************************/

void TTextZone::m_take_focus()
{
  if (f_focused)
    return;
  TObject::m_take_focus();
  if (!f_focused)
    return;

  m_expand_line(f_cursor_line);

  // Pour afficher ou non le curseur
  m_set_cursor_position(f_cursor_col,f_cursor_line,FALSE);
}


/***************************************************/
/* m_set_active : L'objet devient/n'est plus actif */
/* ------------   (ne modifie pas l'affichage)     */
/***************************************************/

void TTextZone::m_set_active(boolean active)
{
  TObject::m_set_active(active);

  // Pour afficher ou non le curseur
  m_set_cursor_position(f_cursor_col,f_cursor_line,FALSE);
}


/*********************************************/
/* m_set_open : Ouvre/Ferme l'objet          */
/* ----------   (ne modifie pas l'affichage) */
/*********************************************/

void TTextZone::m_set_open(boolean open)
{
  if (open!=f_open)
    {
      TObject::m_set_open(open);

      if (f_focused)
	{
	  // Pour afficher ou non le curseur
	  m_set_cursor_position(f_cursor_col,f_cursor_line,FALSE);
	}
    }
}

/****************************************************************************/
/* m_display_string_node                                                    */
/*--------------------------------------------------------------------------*/
/* Affiche une ligne du texte.                                              */
/* La position d'affichage doit dj tre tablie                           */
/****************************************************************************/

void TTextZone::m_display_string_node(PStringNode node,int line)
{
  char *string;
  int  string_length;

  unsigned normal_attr,selected_attr;

  int  min_selected_col,max_selected_col;


  string=node->string;
  string_length=strlen(string);

  // Couleurs d'affichage

  selected_attr=((f_focused) && (f_window->m_is_active()))?
		 f_window->m_get_inverse_attr(f_background)
		:((DARKGRAY<<4)+(unsigned)WHITE);
  normal_attr  =f_window->m_get_normal_attr(f_background);

  // Partie selectionnee de la ligne

  m_get_line_selected_part(line,min_selected_col,max_selected_col);

  // Affichage de la ligne

  if (max_selected_col!=0)
    {
      m_display_string_part(string,string_length,1,min_selected_col-1,normal_attr);
      m_display_string_part(string,string_length,min_selected_col,max_selected_col,selected_attr);
    }
  m_display_string_part(string,string_length,max_selected_col+1,f_first_visible_col+f_text_width-1,normal_attr);
}

/****************************************************************************/
/* m_display_string_part                                                    */
/*--------------------------------------------------------------------------*/
/* Affiche une partie d'une ligne, slectionne ou non                      */
/* string, string_length : Texte de la ligne en cours + longueur de la      */
/*			   chaine					    */
/* min_col,max_col       : Partie de la ligne  afficher                    */
/* attribute             : Couleur d'affichage                              */
/****************************************************************************/

void TTextZone::m_display_string_part(const char *string, int string_length,
  				      int min_col, int max_col,
                                      unsigned attribute)
{
  int zone_length;
  int last_visible_col=f_first_visible_col+f_text_width-1;

  if (   (min_col>max_col)
      || (max_col<f_first_visible_col)
      || (min_col>last_visible_col))
    return;

  if (max_col>last_visible_col)
    max_col=last_visible_col;

  if (min_col<f_first_visible_col)
    min_col=f_first_visible_col;

  string+=min_col-1;
  string_length-=(min_col-1);

  zone_length=max_col-min_col+1;

  if (string_length<0)
    string_length=0;

  f_window->m_textattr(attribute);

  if (string_length<=zone_length)
    {
      if (string_length!=0)
	f_window->m_puts(string);
      f_window->m_putnch(zone_length-string_length,' ');
    }
  else
    {
      char* ptr=(char*)string+zone_length;
      char old_char=(*ptr);
      (*ptr)=0;

      f_window->m_puts(string);
      (*ptr)=old_char;
    }

}

/****************************************************************************/
/* m_get_line_selected_part                                                 */
/*--------------------------------------------------------------------------*/
/* Indique la partie de la ligne qui est slectionne.                      */
/* Si aucune partie de la ligne n'est slectionne, les valeurs retournes  */
/* sont 0,0								    */
/****************************************************************************/

void TTextZone::m_get_line_selected_part(int line,int &min_selected_col,int &max_selected_col)
{
  // Affichage de la ligne

  if (   (!f_zone_selected)
      || ((f_zone_selected) && ((line<f_min_selected_line) || (line>f_max_selected_line))))
    {
      min_selected_col=0;
      max_selected_col=0;
    }

  else
    {
      if (f_min_selected_line==f_max_selected_line) // == line
	{
	  min_selected_col=f_min_selected_col;
	  max_selected_col=f_max_selected_col;
	}
      else
	{
	  if (line==f_min_selected_line)
	    {
	      min_selected_col=f_min_selected_col;
	      max_selected_col=f_nb_cols;
	    }
	  else
	    {
	      if (line==f_max_selected_line)
		{
		  min_selected_col=1;
		  max_selected_col=f_max_selected_col;
		}
	      else
		{
		  min_selected_col=1;
		  max_selected_col=f_nb_cols;
		}
	    }
	}
    }
}


/****************************************************************************/
/* m_scrollbar_value_changed_call                                           */
/*--------------------------------------------------------------------------*/
/* Appel si la valeur de l'ascenseur change.                               */
/* Appelle le callback associ s'il y en a un de dfini.                    */
/****************************************************************************/

//  Static

void TTextZone::m_vert_scrollbar_value_changed_call(PObject scrollbar, const char */*argument*/)
{
  static boolean in_callback=FALSE;

  PTextZone text_zone;
  unsigned scrollbar_value;

  if (!in_callback)
    {
      in_callback=TRUE;

      text_zone=(PTextZone)(((PScrollBar)(scrollbar))->m_get_parent());
      scrollbar_value=(unsigned)(((PScrollBar)scrollbar)->m_get_value());
      text_zone->m_set_first_visible_line(scrollbar_value);

      in_callback=FALSE;
    }
}

void TTextZone::m_horiz_scrollbar_value_changed_call(PObject scrollbar, const char */*argument*/)
{
  static boolean in_callback=FALSE;

  PTextZone text_zone;
  unsigned scrollbar_value;

  if (!in_callback)
    {
      in_callback=TRUE;

      text_zone=(PTextZone)(((PScrollBar)(scrollbar))->m_get_parent());
      scrollbar_value=(unsigned)(((PScrollBar)scrollbar)->m_get_value());
      text_zone->m_set_first_visible_col(scrollbar_value);

      in_callback=FALSE;
    }
}

/****************************************************************************/
/* m_nb_lines_changed_callback                                              */
/*--------------------------------------------------------------------------*/
/* Appel si le nombre de lignes du texte change.                           */
/* Appelle le callback associ s'il y en a un de dfini.                    */
/****************************************************************************/

void TTextZone::m_nb_lines_changed_callback()
{
  static boolean in_callback=FALSE;

  if (!in_callback)
    {
      in_callback=TRUE;

      if (f_vertical_scrollbar!=NULL)
	{
	  if (f_nb_lines<=f_text_height)
	    {
	      f_vertical_scrollbar->m_disable();
	      f_vertical_scrollbar->m_set_can_be_enabled(FALSE);
	    }
	  else
	    {
	      f_vertical_scrollbar->m_set_min_max_values(1L,f_nb_lines-f_text_height+1);
	      f_vertical_scrollbar->m_set_can_be_enabled(TRUE);
	      if (f_enabled)
		f_vertical_scrollbar->m_enable();
	    }
	}

      CallCallback(this,f_nb_lines_changed_action,f_nb_lines_changed_argument);

      in_callback=FALSE;
    }
}

/****************************************************************************/
/* m_text_changed_callback                                                  */
/*--------------------------------------------------------------------------*/
/* Appel en cas de modification du texte.                                  */
/* Appelle le callback associ s'il y en a un de dfini.                    */
/****************************************************************************/

void TTextZone::m_text_changed_callback()
{
  CallCallback(this,f_text_changed_action,f_text_changed_argument);
}

/****************************************************************************/
/* m_cursor_col/line_changed_callback                                       */
/*--------------------------------------------------------------------------*/
/* Appel en cas de changement de position du curseur. 			    */
/****************************************************************************/

void TTextZone::m_cursor_line_changed_callback()
{
  CallCallback(this,f_cursor_line_changed_action,f_cursor_line_changed_argument);
}

void TTextZone::m_cursor_col_changed_callback()
{
  CallCallback(this,f_cursor_col_changed_action,f_cursor_col_changed_argument);
}

void TTextZone::m_cursor_pos_changed_callback()
{
  CallCallback(this,f_cursor_pos_changed_action,f_cursor_pos_changed_argument);
}

/****************************************************************************/
/* m_visible_lines_changed_callback                                         */
/*--------------------------------------------------------------------------*/
/* Fonction appele quand la partie visible du texte (en hauteur) est       */
/* modifie.                                                                */
/* Appelle le callback associ s'il y en a un de dfini.                    */
/****************************************************************************/

void TTextZone::m_visible_lines_changed_callback()
{
  static boolean in_callback=FALSE;

  if (!in_callback)
    {
      in_callback=TRUE;

      if (f_vertical_scrollbar!=NULL)
	f_vertical_scrollbar->m_set_value((long)f_first_visible_line);
      CallCallback(this,f_visible_lines_changed_action,f_visible_lines_changed_argument);
      in_callback=FALSE;
    }
}

void TTextZone::m_visible_cols_changed_callback()
{
  static boolean in_callback=FALSE;

  if (!in_callback)
    {
      in_callback=TRUE;

      if (f_horizontal_scrollbar!=NULL)
	f_horizontal_scrollbar->m_set_value((long)f_first_visible_col);
      CallCallback(this,f_visible_cols_changed_action,f_visible_cols_changed_argument);
      in_callback=FALSE;
    }
}

/****************************************************************************/
/* m_selection_status_changed_callback                                      */
/*--------------------------------------------------------------------------*/
/* Fonction appele quand l'tat de la partie slectionne du texte change  */
/* (une partie est slectionne, ou aucune ne l'est).                       */
/* Appelle le callback associ s'il y en a un de dfini.                    */
/****************************************************************************/

void TTextZone::m_selection_status_changed_callback()
{
  CallCallback(this,f_selection_status_changed_action,f_selection_status_changed_argument);
}

/****************************************************************************/
/* m_left_button_pressed_event                                              */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a cliqu dans l'objet avec le bouton gauche (l'objet tant */
/* activable).                                                              */
/* Retourne TRUE si l'objet est intress par cet vnement.                */
/****************************************************************************/

boolean TTextZone::m_left_button_pressed_event(int x,int y)
{
  int x1,y1,x2,y2;

  if (!f_focused)
    {
      if (!m_set_focus())
        return(FALSE);
    }

  if (TObject::m_left_button_pressed_event(x,y))
    return(TRUE);

  x1=m_get_x()+f_text_rel_x;
  y1=m_get_y()+f_text_rel_y;
  x2=x1+f_text_width-1;
  y2=y1+f_text_height-1;

  if ((x<x1) || (x>x2) || (y<y1) || (y>y2))
    return(TRUE);

  m_unselect_zone();

  m_set_cursor_position(x-x1+f_first_visible_col,
			y-y1+f_first_visible_line,
			TRUE);

  JPRefresh();

  m_mouse_move_button_pressed(x,y,FALSE);
  return(TRUE);
}

/****************************************************************************/
/* m_left_button_double_click_event                                         */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a double-cliqu dans l'objet avec le bouton gauche         */
/* (l'objet tant activable et ayant dj subi l'vnement                  */
/* m_left_button_pressed_event).                                            */
/* Retourne TRUE si l'objet est intress par cet vnement.                */
/****************************************************************************/

boolean TTextZone::m_left_button_double_click_event(int x,int y)
{
  int button_state;
  int x1,x2,y1,y2;
  int length;

  if (TObject::m_left_button_double_click_event(x,y))
    return(TRUE);

  x1=m_get_x()+f_text_rel_x;
  y1=m_get_y()+f_text_rel_y;
  x2=x1+f_text_width-1;
  y2=y1+f_text_height-1;

  if ((x<x1) || (x>x2) || (y<y1) || (y>y2))
    return(FALSE);

  m_set_cursor_position(1,
			y-y1+f_first_visible_line,
                        TRUE);

  length=m_line_length(f_cursor_line);

  if (length!=0)
    m_select_zone(1,f_cursor_line,
		  length+1,f_cursor_line);
  JPRefresh();

  // Si on bouge sans relcher

  button_state=LEFT_BUTTON_PRESSED;
  x2=x;
  y2=y;

  while (   (button_state==LEFT_BUTTON_PRESSED)
         && (x2==x) && (y2==y))
    GetMouseState(x2,y2,button_state);

  if (button_state==LEFT_BUTTON_PRESSED)
    m_mouse_move_button_pressed(x,y,TRUE);
  return(TRUE);
}

/****************************************************************************/
/* m_key_pressed_event                                                      */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur une touche qui est propose  l'objet         */
/* (qui est activable).                                                     */
/* Retourne TRUE si l'objet est intress par cette touche.                 */
/****************************************************************************/

boolean TTextZone::m_key_pressed_event(TKey key)
{
  if (TObject::m_key_pressed_event(key))
    return(TRUE);

  if (!f_focused)
    return(FALSE);


  if (IsPrintableKey(key))
    {
      m_character_hit(key.character);
      return(TRUE);
    }

  switch (key.character)
    {
      case RETURN          : m_return_hit();
			     return(TRUE);
      case CTRL_Y          : m_ctrl_y_hit();
			     return(TRUE);
      case BACKSPACE       : m_backspace_hit();
			     return(TRUE);
      case TAB             : m_tab_hit();
			     return(TRUE);

      case SHIFT_TAB       : return(TRUE);

      case LEFT            : m_left_hit(FALSE);
			     return(TRUE);
      case SHIFT_LEFT      : m_left_hit(TRUE);
			     return(TRUE);

      case RIGHT           : m_right_hit(FALSE);
			     return(TRUE);
      case SHIFT_RIGHT     : m_right_hit(TRUE);
			     return(TRUE);

      case UP              : m_up_hit(FALSE);
			     return(TRUE);
      case SHIFT_UP        : m_up_hit(TRUE);
			     return(TRUE);

      case DOWN            : m_down_hit(FALSE);
			     return(TRUE);
      case SHIFT_DOWN      : m_down_hit(TRUE);
			     return(TRUE);

      case HOME            : m_home_hit(FALSE);
			     return(TRUE);
      case SHIFT_HOME      : m_home_hit(TRUE);
			     return(TRUE);

      case END             : m_end_hit(FALSE);
			     return(TRUE);
      case SHIFT_END       : m_end_hit(TRUE);
			     return(TRUE);

      case CTRL_HOME       : m_ctrl_home_hit(FALSE);
			     return(TRUE);
      case CTRL_SHIFT_HOME : m_ctrl_home_hit(TRUE);
			     return(TRUE);

      case CTRL_PGUP       : m_ctrl_page_up_hit(FALSE);
			     return(TRUE);
      case CTRL_SHIFT_PGUP : m_ctrl_page_up_hit(TRUE);
			     return(TRUE);

      case CTRL_END        : m_ctrl_end_hit(FALSE);
			     return(TRUE);
      case CTRL_SHIFT_END  : m_ctrl_end_hit(TRUE);
			     return(TRUE);

      case CTRL_PGDN       : m_ctrl_page_down_hit(FALSE);
			     return(TRUE);
      case CTRL_SHIFT_PGDN : m_ctrl_page_down_hit(TRUE);
			     return(TRUE);

      case PGUP            : m_page_up_hit(FALSE);
			     return(TRUE);
      case SHIFT_PGUP      : m_page_up_hit(TRUE);
			     return(TRUE);

      case PGDN            : m_page_down_hit(FALSE);
			     return(TRUE);
      case SHIFT_PGDN      : m_page_down_hit(TRUE);
			     return(TRUE);

      case CTRL_LEFT       : m_ctrl_left_hit(FALSE);
			     return(TRUE);
      case CTRL_SHIFT_LEFT : m_ctrl_left_hit(TRUE);
			     return(TRUE);

      case CTRL_RIGHT      : m_ctrl_right_hit(FALSE);
			     return(TRUE);
      case CTRL_SHIFT_RIGHT: m_ctrl_right_hit(TRUE);
			     return(TRUE);

      case CTRL_INSERT     : m_ctrl_insert_hit();
			     return(TRUE);

      case INSERT          : m_insert_hit(FALSE);
			     return(TRUE);
      case SHIFT_INSERT    : m_insert_hit(TRUE);
			     return(TRUE);

      case DELETE          : m_delete_hit(FALSE);
			     return(TRUE);
      case SHIFT_DELETE    : m_delete_hit(TRUE);
			     return(TRUE);

    }

  return(FALSE);
}


/****************************************************************************/
/* m_mouse_move_button_pressed                                              */
/*--------------------------------------------------------------------------*/
/* Dplacement de la souris avec le bouton enfonc                          */
/* (la zone text doit dj avoir le focus).                                 */
/****************************************************************************/

void TTextZone::m_mouse_move_button_pressed(int x,int y,boolean selection)
{
  int x1,y1;
  int col,line;

  int button_state;
  boolean scroll;

  boolean first_scroll=TRUE;

  int new_cursor_col,new_cursor_line;
  int last_visible_col,last_visible_line;

  x1=m_get_x()+f_text_rel_x;
  y1=m_get_y()+f_text_rel_y;

  button_state=LEFT_BUTTON_PRESSED;
  scroll=FALSE;

  while (button_state==LEFT_BUTTON_PRESSED)
    {

      col=x-x1+f_first_visible_col;
      line=y-y1+f_first_visible_line;
      last_visible_col=f_first_visible_col+f_text_width-1;
      last_visible_line=f_first_visible_line+f_text_height-1;

      /*-----------------------------------*/
      /* Gauche/Droite de la zone de texte */
      /*-----------------------------------*/

      if (col<f_first_visible_col)
	{
	  if (f_first_visible_col>1)
	    {
	      new_cursor_col=f_first_visible_col-1;
	      scroll=TRUE;
	    }
	  else
	    new_cursor_col=f_first_visible_col;
	}
      else
	{
	  if (col>last_visible_col)
	    {
	      if (last_visible_col<=f_nb_cols)
		{
		  new_cursor_col=last_visible_col+1;
		  scroll=TRUE;
		}
	      else
		new_cursor_col=f_nb_cols+1;
	    }
	  else
	    new_cursor_col=MIN(col,(f_nb_cols+1));
	}

      /*------------------------------*/
      /* Haut/Bas de la zone de texte */
      /*------------------------------*/

      if (line<f_first_visible_line)
	{
	  if (f_first_visible_line>1)
	    {
	      new_cursor_line=f_first_visible_line-1;
	      scroll=TRUE;
	    }
	  else
	    new_cursor_line=f_first_visible_line;
	}
      else
	{
	  if (line>last_visible_line)
	    {
	      if (last_visible_line<f_nb_lines)
		{
		  new_cursor_line=last_visible_line+1;
		  scroll=TRUE;
		}
	      else
		new_cursor_line=f_nb_lines;
	    }
	  else
	    new_cursor_line=MIN(line,f_nb_lines);
	}

      /*-----------------------*/
      /* Dans la zone de texte */
      /*-----------------------*/

      if (!selection)
	{
	  if (   (col>=f_first_visible_col)
	      && (col<=last_visible_col)
	      && (line>=f_first_visible_line)
	      && (line<=last_visible_line))
	    {
	      selection=TRUE;
	      f_first_selected_col=new_cursor_col;
	      f_first_selected_line=new_cursor_line;
	    }
	}

      /*----------------------------------------------*/
      /* Affichage du texte avec la slection visible */
      /*----------------------------------------------*/

      if (   (new_cursor_col!=f_cursor_col)
	  || (new_cursor_line!=f_cursor_line))
	{
	  if (selection)
	    m_select_zone(f_first_selected_col,
			  f_first_selected_line,
			  new_cursor_col,
			  new_cursor_line);
	  else
	    m_set_cursor_position(new_cursor_col,new_cursor_line,TRUE);
	  JPRefresh();

	  if (scroll)
	    {
	      scroll=FALSE;
	      if (first_scroll)
		{
		  delay(TEXTZONE_FIRST_SCROLL_SPEED);
		  first_scroll=FALSE;
		}
	      else
		delay(TEXTZONE_SCROLL_SPEED);
	    }
	}

      /*-------------------------------------*/
      /* Lecture du nouvel tat de la souris */
      /*-------------------------------------*/

      GetMouseState(x,y,button_state);
    }
}


/*ͻ*/
/*                            METHODES PRIVEES                            */
/*ͼ*/


/****************************************************************************/
/* m_display_caption                                                        */
/*--------------------------------------------------------------------------*/
/* Affichage de la lgende de l'objet                                       */
/****************************************************************************/

void TTextZone::m_display_caption()
{
  if (!f_open)
    return;

  // S'il y a un titre, barre de menus

 if (f_caption[0]!=0)
    {
      f_window->m_gotoxy(m_get_x_in_window(),m_get_y_in_window());
      if ((f_focused) && (f_window->m_is_active()))
	f_window->m_textattr((BLUE<<4)+(unsigned)WHITE);
      else
	f_window->m_textattr((LIGHTGRAY<<4)+(unsigned)BLACK);

      f_window->m_put_caption(f_caption,f_enabled,f_width,CENTERED_LEFT);
    }

}

/****************************************************************************/
/* m_display_text                                                           */
/*--------------------------------------------------------------------------*/
/* Affichage du texte de la zone de texte                                   */
/****************************************************************************/

void TTextZone::m_display_text()
{
  int x1,y1,y2;
  PStringNode string_node;
  int line;

  register int i;

  if (!f_open)
    return;

  x1=m_get_x_in_window()+f_text_rel_x;
  y1=m_get_y_in_window()+f_text_rel_y;
  y2=y1+f_text_height-1;

  // Affichage de la partie du texte non vide

  string_node=f_first_visible_string;
  i=y1;
  line=f_first_visible_line;

  while ((i<=y2) && (string_node!=NULL))
    {
      f_window->m_gotoxy(x1,i);
      m_display_string_node(string_node,line);
      string_node=string_node->next;
      line++;
      i++;
    }

  // On complte avec des lignes vides

  f_window->m_set_normal_attr(f_background);
  while (i<=y2)
    {
      f_window->m_gotoxy(x1,i);
      f_window->m_putnch(f_text_width,' ');
      i++;
    }

}


/****************************************************************************/
/* m_character_hit                                                          */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur une touche                                    */
/* correspondant  un caractre affichable                                  */
/****************************************************************************/

void TTextZone::m_character_hit(int character)
{
  char *string;

  if (!f_modification_enabled)
    return;

  string=m_line_to_string(f_cursor_line)->string;

  m_clear_selection();

  if (f_cursor_col==(f_nb_cols+1))
    {
      PlaySound(BIP1);
      return;
    }

  switch (f_cursor_style)
    {
      case INSERT_CURSOR : m_insert_nch(1,character);
                           break;

      case RECOVER_CURSOR: string[f_cursor_col-1]=character;
                           m_display_lines(f_cursor_line,f_cursor_line);
			   m_set_cursor_position(f_cursor_col+1,f_cursor_line,TRUE);
			   m_text_changed_callback();

			   break;
    }

}

/****************************************************************************/
/* m_move_cursor_with_key                                                   */
/*--------------------------------------------------------------------------*/
/* Dplacement du curseur suite  l'appui sur une touche                    */
/****************************************************************************/

void TTextZone::m_move_cursor_with_key_to(boolean shift_pressed,int new_cursor_col,int new_cursor_line)
{
  // Appui sur shift

  if (shift_pressed)
    {
      if (!f_zone_selected)
        {
          f_first_selected_col=f_cursor_col;
          f_first_selected_line=f_cursor_line;
        }

      m_select_zone(f_first_selected_col,
		    f_first_selected_line,
                    new_cursor_col,
		    new_cursor_line);
    }

  // Pas d'appui sur shift

  else
    {
       m_unselect_zone();
       m_set_cursor_position(new_cursor_col,new_cursor_line,TRUE);
    }
}

/****************************************************************************/
/* m_ctrl_y_hit                                                             */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <CTRL-Y>                          */
/****************************************************************************/

void TTextZone::m_ctrl_y_hit()
{
  boolean lines_after_are_empty=TRUE;
  register int i;

  if (!f_modification_enabled)
    return;

  // m_unselect_zone effectu par m_delete_line

  m_delete_line(f_cursor_line);

  i=f_cursor_line;
  while (i<=f_nb_lines)
    {
      if (m_line_length(i)!=0)
	{
	  lines_after_are_empty=FALSE;
	  i=f_nb_lines;
	}
      i++;
    }

  if (lines_after_are_empty)
    {
      m_set_cursor_position(1,f_cursor_line,TRUE);
    }
}


/****************************************************************************/
/* m_left_hit                                                               */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <Flche gauche>                   */
/****************************************************************************/

void TTextZone::m_left_hit(boolean shift_pressed)
{
  m_move_cursor_with_key_to(shift_pressed,
                            f_cursor_col-1,
                            f_cursor_line);
}

/****************************************************************************/
/* m_right_hit                                                              */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <Flche droite>                   */
/****************************************************************************/

void TTextZone::m_right_hit(boolean shift_pressed)
{
  m_move_cursor_with_key_to(shift_pressed,
                            f_cursor_col+1,
                            f_cursor_line);
}

/****************************************************************************/
/* m_up_hit                                                                 */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <Flche haut>                     */
/****************************************************************************/

void TTextZone::m_up_hit(boolean shift_pressed)
{
  m_move_cursor_with_key_to(shift_pressed,
                            f_cursor_col,
                            f_cursor_line-1);
}

/****************************************************************************/
/* m_down_hit                                                               */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <Flche basse>                    */
/****************************************************************************/

void TTextZone::m_down_hit(boolean shift_pressed)
{
  m_move_cursor_with_key_to(shift_pressed,
                            f_cursor_col,
                            f_cursor_line+1);
}

/****************************************************************************/
/* m_ctrl_left_hit                                                          */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <CTRL>+<Flche gauche>            */
/****************************************************************************/

void TTextZone::m_ctrl_left_hit(boolean shift_pressed)
{
  int line;
  char *ptr;

  PStringNode node=m_line_to_string(f_cursor_line);
  char *string=node->string;

  if (f_cursor_col>1)
    {
      line=f_cursor_line;
      ptr=string+f_cursor_col-2;
      while ((ptr>=string) && ((*ptr)==' '))
        ptr--;
      while ((ptr>=string) && ((*ptr)!=' '))
        ptr--;
    }
  else
    {
      if (f_cursor_line==1)
        return;

      line=f_cursor_line-1;

      node=node->last;
      string=node->string;
      ptr=strchr(string,0);

      while ((ptr>=string) && ((*ptr)==' '))
        ptr--;
      while ((ptr>=string) && ((*ptr)!=' '))
        ptr--;
    }

  m_move_cursor_with_key_to(shift_pressed,
                            (int)(ptr-string+2),
                            line);


}

/****************************************************************************/
/* m_ctrl_right_hit                                                         */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <CTRL>+<Flche droite>            */
/****************************************************************************/

void TTextZone::m_ctrl_right_hit(boolean shift_pressed)
{
  int line;
  char *ptr;
  char old_char;

  PStringNode node=m_line_to_string(f_cursor_line);
  char *string=node->string;
  int  length=m_useful_length(string);

  if (f_cursor_col<=length)
    {
      line=f_cursor_line;
      old_char=string[length];
      string[length]=0;
      ptr=string+f_cursor_col;
      while (((*ptr)!=0) && ((*ptr)!=' '))
	ptr++;
      while (((*ptr)!=0) && ((*ptr)==' '))
        ptr++;
      string[length]=old_char;
    }
  else
    {
      if (f_cursor_line==f_nb_lines)
        return;

      line=f_cursor_line+1;

      node=node->next;
      string=node->string;
      ptr=string;
      while (((*ptr)!=0) && ((*ptr)==' '))
        ptr++;
    }

  m_move_cursor_with_key_to(shift_pressed,
                            (int)(ptr-string+1),
                            line);

}

/****************************************************************************/
/* m_home_hit                                                               */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <Home>                            */
/****************************************************************************/

void TTextZone::m_home_hit(boolean shift_pressed)
{
  m_move_cursor_with_key_to(shift_pressed,
                            1,f_cursor_line);
}

/****************************************************************************/
/* m_end_hit                                                                */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <End>                             */
/****************************************************************************/

void TTextZone::m_end_hit(boolean shift_pressed)
{
  m_move_cursor_with_key_to(shift_pressed,
                            m_line_length(f_cursor_line)+1,
                            f_cursor_line);
}

/****************************************************************************/
/* m_ctrl_home_hit                                                          */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <CTRL>+<Home>                     */
/****************************************************************************/

void TTextZone::m_ctrl_home_hit(boolean shift_pressed)
{
  m_move_cursor_with_key_to(shift_pressed,
			    1,1);
}

/****************************************************************************/
/* m_ctrl_end_hit                                                           */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <CTRL>+<End>                      */
/****************************************************************************/

void TTextZone::m_ctrl_end_hit(boolean shift_pressed)
{
  m_move_cursor_with_key_to(shift_pressed,
                            m_line_length(f_nb_lines)+1,
                            f_nb_lines);
}

/****************************************************************************/
/* m_page_up_hit                                                            */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <PageUp>                          */
/****************************************************************************/

void TTextZone::m_page_up_hit(boolean shift_pressed)
{
  m_set_first_visible_line(f_first_visible_line-f_text_height);
  m_move_cursor_with_key_to(shift_pressed,
                            f_cursor_col,
                            f_cursor_line-f_text_height);
}

/****************************************************************************/
/* m_page_down_hit                                                          */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <PageDown>                        */
/****************************************************************************/

void TTextZone::m_page_down_hit(boolean shift_pressed)
{
  m_set_first_visible_line(f_first_visible_line+f_text_height);
  m_move_cursor_with_key_to(shift_pressed,
                            f_cursor_col,
                            f_cursor_line+f_text_height);
}

/****************************************************************************/
/* m_ctrl_page_up_hit                                                       */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <CTRL>+<PageUp>                   */
/****************************************************************************/

void TTextZone::m_ctrl_page_up_hit(boolean shift_pressed)
{
  m_ctrl_home_hit(shift_pressed);
}

/****************************************************************************/
/* m_ctrl_page_down_hit                                                     */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <CTRL>+<PageDown>                 */
/****************************************************************************/

void TTextZone::m_ctrl_page_down_hit(boolean shift_pressed)
{
  m_ctrl_end_hit(shift_pressed);
}


/****************************************************************************/
/* m_return_hit                                                             */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <RETURN>                          */
/****************************************************************************/

void TTextZone::m_return_hit()
{
  int col;
  boolean only_spaces_before;
  char *string;
  char *ptr,*ptr2;
  int useful_length;

  if (!f_modification_enabled)
    return;

  string=m_line_to_string(f_cursor_line)->string;
  useful_length=m_useful_length(string);
  if (useful_length!=0)
    {
      ptr2=string+f_cursor_col-1;
      ptr=string;
      only_spaces_before=TRUE;

      while ((ptr<ptr2) && (only_spaces_before))
	{
	  if ((*ptr)==' ')
	    ptr++;
	  else
	    only_spaces_before=FALSE;
	}
    }
  else
    only_spaces_before=TRUE;

  if ((only_spaces_before) && (useful_length!=0))
    {
      col=f_cursor_col;
      m_set_cursor_position(1,f_cursor_line,FALSE);
      m_insert_text("\n");
      m_set_cursor_position(1,f_cursor_line+1,FALSE);
      string=m_line_to_string(f_cursor_line)->string;
      ptr=string;
      while ((*ptr)==' ')
	ptr++;

      DEBUG_TEST((*ptr)!=0);

      m_set_cursor_position((int)(ptr-string+1),f_cursor_line,TRUE);
    }
  else
    {
      m_insert_text("\n");
      m_set_cursor_position(1,f_cursor_line+1,FALSE);

      col=m_get_previous_tab(f_nb_cols);
      if (col!=1)
	m_insert_nch(col-1,' ');
      m_set_cursor_position(f_cursor_col,f_cursor_line,TRUE);
    }
}

/****************************************************************************/
/* m_backspace_hit                                                          */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <BackSpace>                       */
/****************************************************************************/

void TTextZone::m_backspace_hit()
{
  register int i;
  PStringNode node;
  char *string;
  boolean only_spaces_before;

  if (!f_modification_enabled)
    return;


  if (f_zone_selected)
    m_clear_selection();
  else
    {
      if (f_cursor_col>1)
	{
	  node=m_line_to_string(f_cursor_line);
	  string=node->string;
	  only_spaces_before=TRUE;
	  i=f_cursor_col-2;

	  while ((only_spaces_before) && (i>=0))
	    {
	      if (string[i]!=' ')
		only_spaces_before=FALSE;
	      i--;
	    }

	  if (only_spaces_before)
	    i=m_get_previous_tab(f_cursor_col-1);
	  else
	    i=f_cursor_col-1;
	  m_clear_zone(i,f_cursor_line,f_cursor_col-1,f_cursor_line);
	}
      else
	{
	  if (f_cursor_line>1)
	    m_clear_zone(m_line_length(f_cursor_line-1)+1,f_cursor_line-1,0,f_cursor_line);
	}
    }
}

/****************************************************************************/
/* m_tab_hit                                                                */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <Tab>                             */
/****************************************************************************/

void TTextZone::m_tab_hit()
{
  int length;
  register int i;
  char *string;

  register char *ptr;

  if (!f_modification_enabled)
    return;

  if (f_cursor_line>1)
    {
      i=f_cursor_line-1;
      while (m_line_length(i)==0)
	{
	  i--;
	  if (i==0)
	    break;
	}

      if (i!=0)
	{
	  string=m_line_to_string(i)->string;
	  length=m_useful_length(string);

	  if (f_cursor_col<=length)
            {
              ptr=string+f_cursor_col-1;
              while (((*ptr)!=' ') && ((*ptr)!=0) )
                ptr++;

              if ((*ptr)!=0)
                {
                  ptr++;
                  while (((*ptr)==' ') && ((*ptr)!=0))
		    ptr++;
		}

	      i=(int)(ptr-string)+1-f_cursor_col;

	      if (i>=1)
		{
		  m_insert_nch(i,' ');
		  return;
		}
	    }
	}
    }

  m_insert_nch(f_tab_size-((f_cursor_col-1)%f_tab_size),' ');
}

/****************************************************************************/
/* m_insert_hit                                                             */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <Insert>                          */
/****************************************************************************/

void TTextZone::m_insert_hit(boolean shift_pressed)
{

  if (shift_pressed)
    {
      if (!f_modification_enabled)
	return;
      m_paste();
    }
  else
    {
      if (f_cursor_style==INSERT_CURSOR)
	f_cursor_style=RECOVER_CURSOR;
      else
	f_cursor_style=INSERT_CURSOR;

      // Le curseur est visible quand on arrive ici
      SetTextCursorStyle(f_cursor_style);
    }
}

/****************************************************************************/
/* m_ctrl_insert_hit                                                        */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <CTRL>+<Insert>                   */
/****************************************************************************/

void TTextZone::m_ctrl_insert_hit()
{
  m_copy_selection();
}

/****************************************************************************/
/* m_delete_hit                                                             */
/*--------------------------------------------------------------------------*/
/* L'utilisateur a appuy sur les touches <Delete>                          */
/****************************************************************************/

void TTextZone::m_delete_hit(boolean shift_pressed)
{
  register int i;
  int tmp;
  int line_length;
  char *string;

  if (!f_modification_enabled)
    return;

  if (shift_pressed)
    m_cut_selection();
  else
    {
      if (f_zone_selected)
	m_clear_selection();
      else
        {
	  string=m_line_to_string(f_cursor_line)->string;
          line_length=m_useful_length(string);

          if (f_cursor_col<=line_length)
            {
              tmp=line_length-1;
              if (f_cursor_col-1<=tmp)
                {
                  for (i=f_cursor_col-1;i<tmp;i++)
                    string[i]=string[i+1];
                  string[tmp]=' ';
                }
              m_display_lines(f_cursor_line,f_cursor_line);
	      m_text_changed_callback();
	    }
          else
            {
              if (f_cursor_line<f_nb_lines)
                m_clear_zone(f_cursor_col,f_cursor_line,0,f_cursor_line+1);
            }
        }
    }
}

/****************************************************************************/
/* m_useful_length                                                          */
/*--------------------------------------------------------------------------*/
/* Retourne la longueur utile d'une chane de caractres (sans compter les  */
/* espaces de fin de chane).                                               */
/****************************************************************************/

int TTextZone::m_useful_length(char *string)
{
  char *ptr;
  char first_char;

  if (string[0]==0)
    return(0);

  first_char=string[0];
  string[0]=1;        // Sentinelle

  ptr=strchr(string,0);

  JPDEBUG_TEST(DEBUG_ERROR_31,((ptr-string)<=f_nb_cols));

  ptr--;
  while ((*ptr)==' ')
    ptr--;

  string[0]=first_char;

  if ((ptr==string) && (first_char==' '))
    return(0);

  return((int)(ptr-string+1));
}


/****************************************************************************/
/* m_reduce_line                                                            */
/*--------------------------------------------------------------------------*/
/* Rduit une des lignes du texte (lui enlve ses espaces de fin)           */
/****************************************************************************/

void TTextZone::m_reduce_line(int line_nb)
{
  PStringNode node=m_line_to_string(line_nb);
  char *string=node->string;
  char *new_string;
  int  new_length;


  new_length=m_useful_length(string);
  new_string=new char [new_length+1];
  strncpy(new_string,string,new_length);
  new_string[new_length]=0;

  delete []string;
  node->string=new_string;
}

/****************************************************************************/
/* m_expand_line                                                            */
/*--------------------------------------------------------------------------*/
/* Expanse une ligne du texte pour qu'elle ait f_nb_cols comme longueur.    */
/* Seule la ligne sur laquelle est situe le curseur est expanse (lorsque  */
/* la zone de texte a le focus).                                            */
/****************************************************************************/

void TTextZone::m_expand_line(int line_nb)
{
  PStringNode node=m_line_to_string(line_nb);
  char *string=node->string;

  char *new_string;
  char *ptr1,*ptr2;

  new_string=new char [f_nb_cols+1];
  strcpy(new_string,string);
  ptr1=new_string+strlen(new_string);
  ptr2=new_string+f_nb_cols;

  while (ptr1<ptr2)
    (*ptr1++)=' ';
  (*ptr2)=0;

  delete []string;
  node->string=new_string;
}

/****************************************************************************/
/* m_set_cursor_position                                                    */
/*--------------------------------------------------------------------------*/
/* Modifie la position du curseur.                                          */
/* Si make_cursor_visible est TRUE, modifie ventuellement la partie visible*/
/* du texte pour que le curseur soit visible.                               */
/* - Si la zone texte a le focus, f_cursor_line doit auparavant contenir le */
/*   numro de la ligne expanse (ou 0 s'il n'y en a pas).                  */
/****************************************************************************/

void TTextZone::m_set_cursor_position(int col,int line,boolean make_cursor_visible)
{
  int old_line;
  boolean cursor_in_visible_part;
  boolean col_changed,line_changed;

  int last_visible_col,last_visible_line;


  if (col<1)
    col=1;
  if (col>(f_nb_cols+1))
    col=f_nb_cols+1;

  if (line<1)
    line=1;
  if (line>f_nb_lines)
    line=f_nb_lines;

  line_changed=(boolean)(f_cursor_line!=line);
  col_changed =(boolean)(f_cursor_col!=col);

  f_cursor_col=col;

  if (f_cursor_line!=line)
    {
      old_line=f_cursor_line;
      f_cursor_line=line;
      if (f_focused)
	{
	  if (old_line!=0)
	    m_reduce_line(old_line);
	  m_expand_line(f_cursor_line);
	}
    }

  // Le curseur sort de la zone visible

  last_visible_col=f_first_visible_col+f_text_width-1;
  last_visible_line=f_first_visible_line+f_text_height-1;

  cursor_in_visible_part=(boolean)(   (f_cursor_col>=f_first_visible_col)
				   && (f_cursor_col<=last_visible_col)
				   && (f_cursor_line>=f_first_visible_line)
                                   && (f_cursor_line<=last_visible_line));

  if ((!cursor_in_visible_part) && (make_cursor_visible))
    {
      if (f_cursor_col<f_first_visible_col)
        m_set_first_visible_col(f_cursor_col);
      if (f_cursor_line<f_first_visible_line)
	m_set_first_visible_line(f_cursor_line);
      if (f_cursor_col>last_visible_col)
        m_set_first_visible_col(f_cursor_col-f_text_width+1);
      if (f_cursor_line>last_visible_line)
	m_set_first_visible_line(f_cursor_line-f_text_height+1);
      cursor_in_visible_part=TRUE;
    }

  // Si la zone a le focus, on raffiche le curseur

  if (f_focused)
    {
      if (   (f_window->m_is_active())
	  && (cursor_in_visible_part))
	{
	  SetTextCursorAt(m_get_x()+f_text_rel_x+f_cursor_col-f_first_visible_col,
			  m_get_y()+f_text_rel_y+f_cursor_line-f_first_visible_line,
			  f_cursor_style);
	}
      else
	HideTextCursor();
    }

  // Appel des callbacks de changement de position du curseur

  if (line_changed)
    m_cursor_line_changed_callback();

  if (col_changed)
    m_cursor_col_changed_callback();

  if ((line_changed) || (col_changed))
    m_cursor_pos_changed_callback();
}

/****************************************************************************/
/* m_get_zone_size                                                          */
/*--------------------------------------------------------------------------*/
/* Retourne la taille ncessaire pour stocker une zone du texte (0 terminal */
/* compris). Utilis par m_get_zone()                                       */
/****************************************************************************/

unsigned TTextZone::m_get_zone_size(int first_col,int first_line,int last_col,int last_line)
{
  PStringNode node;
  register int i;
  int line_length;
  unsigned long size;

  if (!m_correct_zone_limits(first_col,first_line,last_col,last_line))
    return(1);

  size=0;

  // Caractres

  if (first_line==last_line)
    size+=(last_col-first_col+1);
  else
    {
      node=m_line_to_string(first_line);
      line_length=m_useful_length(node->string);

      if (first_col<=line_length)
        size+=(line_length-first_col+1);

      for (i=first_line+1;i<last_line;i++)
	{
	  node=node->next;
	  size+=m_useful_length(node->string);
	}
      size+=last_col;
    }

  // Sauts de lignes

  size+=(last_line-first_line);

  // 0 terminal

  size++;

  if (size>UINT_MAX)
    size=1;

  return((unsigned)size);
}

/****************************************************************************/
/* m_correct_zone_limits                                                    */
/*--------------------------------------------------------------------------*/
/* Corrige les coordonnes fournies pour dsigner une zone.                 */
/* Retourne FALSE si la correction ne permet pas d'obtenir une zone correcte*/
/****************************************************************************/

boolean TTextZone::m_correct_zone_limits(int &first_col,int &first_line,int &last_col,int &last_line)
{
  int tmp;

  if (first_line<1)
    {
      first_line=1;
      first_col=1;
    }
  else
    {
      tmp=f_nb_cols+1;
      if (first_col<1)
        first_col=1;
      if (first_col>tmp)
	first_col=tmp;
    }


  if (last_line>f_nb_lines)
    {
      last_line=f_nb_lines;
      last_col=f_nb_cols;
    }
  else
    {
      if (last_col<0)
	last_col=0;
      if (last_col>f_nb_cols)
	last_col=f_nb_cols;
    }

  if (last_line<first_line)
    return(FALSE);

  if ((last_line==first_line) && (last_col<first_col))
    return(FALSE);

  return(TRUE);
}

/****************************************************************************/
/* m_get_previous_tab                                                       */
/*--------------------------------------------------------------------------*/
/* Recherche le prcdent alignement de texte dans les lignes prcdant le  */
/* curseur (dplacement suite  un backspace...)                            */
/****************************************************************************/

int TTextZone::m_get_previous_tab(int max_value)
{
  PStringNode node;
  int line;
  char *string;
  register int i;

  line=f_cursor_line;
  node=m_line_to_string(line);
  while (TRUE)
    {
      line--;
      if (line==0)
        return(1);

      node=node->last;
      string=node->string;

      // ce ne peut pas tre la ligne tendue

      if (string[0]!=0)
        {
	  i=0;
          while ((string[i]==' ') && (i<max_value))
            i++;
          if (i!=max_value)
	    return(i+1);
	}
    }
}

/****************************************************************************/
/* m_insert_nch                                                             */
/*--------------------------------------------------------------------------*/
/* Insertion de n caractres identiques dans le texte  la position du      */
/* curseur.                                                                 */
/****************************************************************************/

void TTextZone::m_insert_nch(int nb_chars,char character)
{
  char *string;
  char *insertion_place;
  int length_taken;
  int start_length,end_length;
  int tmp;

  DEBUG_TEST((f_focused) && (f_modification_enabled));

  if (nb_chars<=0)
    return;

  m_unselect_zone();

  string=m_line_to_string(f_cursor_line)->string;

  start_length=f_cursor_col-1;
  insertion_place=string+start_length;

  end_length=m_useful_length(insertion_place);

  if (end_length<0)
    end_length=0;

  length_taken=start_length+nb_chars;

  if ((length_taken<f_nb_cols) && (end_length!=0))
    {
      tmp=f_nb_cols-length_taken;
      if (end_length<tmp)
        tmp=end_length;
      memmove(insertion_place+nb_chars,insertion_place,tmp);
    }

  tmp=f_nb_cols-start_length;
  if (nb_chars<tmp)
    tmp=nb_chars;
  memset(insertion_place,character,tmp);

  m_display_lines(f_cursor_line,f_cursor_line);
  m_set_cursor_position(f_cursor_col+nb_chars,f_cursor_line,TRUE);

  m_text_changed_callback();
}

