/*							atanl.c
 *
 *	Inverse circular tangent, long double precision
 *	(arctangent)
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, atanl();
 *
 * y = atanl( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns radian angle between -pi/2 and +pi/2 whose tangent
 * is x.
 *
 * Range reduction is from four intervals into the interval
 * from zero to	 tan( pi/8 ).  The approximant uses a rational
 * function of degree 3/4 of the form x + x**3 P(x)/Q(x).
 *
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE	-10, 10	   150000	1.3e-19	    3.0e-20
 *
 */

/*							atanl.c */

/*
Cephes Math Library Release 2.7:  May, 1998
Copyright 1984, 1990, 1998 by Stephen L. Moshier
Modified for DJGPP/GCC by KB Williams,
kbwms@aol.com, December 2001 & December 2003
*/

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

/* *INDENT-OFF* */
static long double P[] =
{
    -8.6863818178092187535440E-1L,
    -1.4683508633175792446076E1L,
    -6.3976888655834347413154E1L,
    -9.9988763777265819915721E1L,
    -5.0894116899623603312185E1L,
};
static long double Q[] =
{
/*   1.00000000000000000000E0L,*/
     2.2981886733594175366172E1L,
     1.4399096122250781605352E2L,
     3.6144079386152023162701E2L,
     3.9157570175111990631099E2L,
     1.5268235069887081006606E2L,
};

/* tan( 3*pi/8 ) */
static const long double T3P8  =  2.41421356237309504880169L;

/* tan( pi/8 ) */
static const long double TP8   =  4.1421356237309504880169e-1L;

/* pi/4 */
/* #defined in fdlibml.h PIO4L = 7.8539816339744830961566E-1L */
/* *INDENT-ON* */


long double atanl(long double Arg)
{
    long double p, q, Retval, x, y, z;

    if (Arg == 0.0L)
    {
	Retval = Arg;
    }
    else if (isnanl(Arg))
    {
	Retval = Arg;
    }
    else if (isinfl(Arg))
    {
	Retval = copysignl(PIO2L, Arg);
    }
    else if (fpclassifyl(Arg) == FP_SUBNORMAL)
    {
	Retval = Arg;
    	__math_set_errno(ERANGE);
	__fp_raise_except(FE_UNDERFLOW);
    }
    else
    {
	/* make argument positive */
	x = fabsl(Arg);

	/* range reduction */
	if (x > T3P8)
	{
	    y = PIO2L;
	    x = -(1.0L / x);
	}

	else if (x > TP8)
	{
	    y = PIO4L;
	    x = (x - 1.0L) / (x + 1.0L);
	}
	else
	    y = 0.0L;

	/* rational form in x**2 */

	z = x * x;

	p = P[4] + z * (P[3] + z * (P[2] + z * (P[1] + z * P[0])));
	q = Q[4] + z * (Q[3] + z * (Q[2] + z * (Q[1] + z * (Q[0] + z))));
	y = y + (p / q) * z * x + x;

	Retval = copysignl(y, Arg);
    }
    return Retval;
}
