/*							exp2.c
 *
 *	Base 2 exponential function
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, exp2();
 *
 * y = exp2( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns 2 raised to the x power.
 *
 * Range reduction is accomplished by separating the argument
 * into an integer k and fraction f such that
 *     x    k  f
 *    2  = 2  2.
 *
 * A Pade' form
 *
 *   1 + 2x P(x**2) / (Q(x**2) - x P(x**2) )
 *
 * approximates 2**x in the basic range [-0.5, 0.5].
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE    -1022,+1024   30000       1.8e-16     5.4e-17
 *
 *
 * See exp.c for comments on error amplification.
 *
 *
 * ERROR MESSAGES:
 *
 *   message         condition      value returned
 * exp underflow    x < -MAXL2        0.0
 * exp overflow     x > MAXL2         MAXNUM
 *
 * For DEC arithmetic, MAXL2 = 127.
 * For IEEE arithmetic, MAXL2 = 1024.
 */


/*
Cephes Math Library Release 2.8:  June, 2000
Copyright 1984, 1995, 2000 by Stephen L. Moshier
*/
// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, December 2003


#include <errno.h>
#include <fenv.h>
#include <float.h>
#include <fdlibml.h>

/* *INDENT-OFF* */

// -----------
// (double)NaN
// -----------
static union
{
    unsigned long long U_Value;
    double  D_Value;
}
NAND = { 0x7ff8000000000000LL };


static double P[] = {
 2.30933477057345225087E-2,
 2.02020656693165307700E1,
 1.51390680115615096133E3,
};
static double Q[] = {
/* 1.00000000000000000000E0,*/
 2.33184211722314911771E2,
 4.36821166879210612817E3,
};


static double polevl(double, double *, int);
static double p1evl(double, double *, int);

static double polevl(x, coef, N)
double  x;
double  coef[];
int     N;
{
    double  ans;
    int     i;
    double *p;

    p = coef;
    ans = *p++;
    i = N;

    do
	ans = ans * x + *p++;
    while (--i);

    return (ans);
}

/*                                          N
 * Evaluate polynomial when coefficient of x  is 1.0.
 * Otherwise same as polevl.
 */

static double p1evl(x, coef, N)
double  x;
double  coef[];
int     N;
{
    double  ans;
    double *p;
    int     i;

    p = coef;
    ans = x + *p++;
    i = N - 1;

    do
	ans = ans * x + *p++;
    while (--i);

    return (ans);
}
/* *INDENT-ON* */

// ==================================
// exp2 - Base 2 exponential function
// ==================================

double exp2(x)
double  x;
{
    double  px, xx;
    int     n;

    if (isnand(x))
    {
	return (NAND.D_Value);		/* NaN */
    }

    if (x >= DBL_MAX_EXP)
    {
    	if ((isinfd)(x))
	{
	    return x;
	}
	else
	{
    	    __math_set_errno(ERANGE);
	    __fp_raise_except(FE_OVERFLOW);
	    return (HUGE_VAL);		/* +Inf */
	}
    }

    if (x < DBL_MIN_EXP-1)
    {
    	if ((isinfd)(x))
	{
	    return (0.0);
	}
	else
	{
	    __math_set_errno(ERANGE);
	    __fp_raise_except(FE_UNDERFLOW);
	}
    }

    xx = x;				/* save x */
/* separate into integer and fractional parts */
    px = floor(x + 0.5);
    n = px;
    x = x - px;				/* x in [-0.5, +0.5] */

/* rational approximation
 * exp2(x) = 1 +  2xP(xx)/(Q(xx) - P(xx))
 * where xx = x**2
 */
    xx = x * x;
    px = x * polevl(xx, P, 2);
    x = px / (p1evl(xx, Q, 2) - px);
    x = 1.0 + ldexp(x, 1);

/* scale by power of 2 */
    {
    	int errno_saved = errno;
	x = ldexp(x, n);
	__math_set_errno(errno_saved);
    }
    return (x);
}
