// ------
// fmaf.c
// ------
//
// Name
//
// Function fmaf - float precision floating-point multiply-add
//
// Synopsis:
//
// #include <math.h>
//
// float fmaf(float x, float y, float z);
//
// Purpose:
//
// Computes (x*y) + z rounded as one ternary operation.  The product-sum
// is computed as though to infinite precision and rounded once according
// to the current rounding mode.
//
// The procedure is:
//
// 1. Examine parameters in calling sequence for legitimacy:
//
//    o If either x or y is a NaN, a NaN is returned
//    o If either x or y is an infinity and the other is zero:
//
//	1. set errno to EDOM;
//	2. set invalid floating-point exception;
//	3. return a NaN.
//
//    o If neither x nor y is an infinity and the other zero,
//	and z is a NaN, return a NaN.
//
// 2. Calculate product-sum
//
//    If the final result is not finite (whether a NaN or Inf):
//
//	1. set errno to EDOM;
//	2. set invalid floating-point exception;
//	3. return a NaN.
//

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

float fmaf(float x, float y, float z)
{
    float Retval;

    if ((isnanf)(x) || (isnanf)(y))
    {
	Retval = NAN;
    }
    else if (((isinff)(x) && (y == 0.0))
    	 ||  ((isinff)(y) && (x == 0.0)))
    {
    	__math_set_errno(EDOM);
	__fp_raise_except(FE_INVALID);
	Retval = NAN;
    }
    else if ((isnanf)(z))
    {
	Retval = NAN;
    }
    else				// Calculate Product-Sum
    {
# if 0
    	float	HeadP, TailP, HeadZ, TailZ;

	feclearexcept(FE_OVERFLOW | FE_UNDERFLOW);

	SlMpyFlt(x, y, &HeadP, &TailP);
	DlAddFlt(HeadP, TailP, z, 0.0l, &HeadZ, &TailZ);

	Retval = HeadZ + TailZ;
# endif
	feclearexcept(FE_OVERFLOW | FE_UNDERFLOW);

	Retval = x*y+z;

	if (!isfinitef(Retval))
	{
	    __math_set_errno(EDOM);
	    __fp_raise_except(FE_INVALID);
	    Retval = NAN;
	}
	else if (fetestexcept(FE_OVERFLOW | FE_UNDERFLOW))
	{
	    __math_set_errno(ERANGE);
	    Retval = NAN;
	}
    }

    return Retval;
}
