// --------
// hypotl.c
// --------
//
// Returns the long double precision square root of the sum of
// squares of its two long double precision arguments.	This
// value is often referred to as the Euclidean distance.
//
// Written for DJGPP/GCC by KB Williams, kbwms@aol.com
// July 2003
//
// From C99:
// ---------
// Description
//
// The hypot function computes the square root of the sum of
// the squares of x and y, without undue overflow or underflow.
// A range error may occur.
//
// Returns
//
// The hypot functions return the value of the square root of
// the sum of the squares.
//
// The hypot functions:
//
//	o hypot(x, y), hypot(y, x), and hypot(x, -y)  are
//	  equivalent.
//
//	o hypot(x, y) returns +Inf if x is infinite, even if y is a
//	  NaN.
//
//	o hypot(x, +-0) is equivalent to fabs(x).
//
// =================================================================
//
// 1. Analyze arguments.
//
//	If x == +-Inf || y == +-Inf, Retval = +Inf
//	else if x == NaN || y == NaN, Retval = NaN
//
// 2. else get absolute values of arguments
//	x = fabsl(x);
//	y = fabsl(y);
//
// 3. Set Retval as follows:
//
//	if	x == 0, Retval = y
//	else if y == 0, Retval = x
//	else if x >= y, Retval = CalcHypot(x, y)
//	else		Retval = CalcHypot(y, x)
//
//      If Retval from CalcHypot is an infinity,
//	    set errno to ERANGE
//
// 4. return Retval
//
// 5. Function CalcHypot(x,y) proceeds as follows;
//
//    a. Guaranteed that x & y > 0, x >= y
//       Will calculate, with scaling before and after:
//
//    b. Determine whether x >> y; (if so, return x)
//	    Get exponents of x (= Expx) and y (= Expy);
//    c. if difference in exponents (D = Expx - Expy)
//	    exceeds 32 (D >= 0, since x >= y),
//	    set Retval = x;
//	 else proceed as follows:
//    d. Calculate scale factor:
//	    S = (Expx+Expy)/2
//    e. Scale x & y by 2^(-S) using ldexpl()
//    f. Calculate provisional final result:
//	    F = sqrtl(x*x + y*y)
//
//    g. Calculate final exponent
//	    E = exponent of F + S
//	 if E is too large (E > LDBL_MAX_EXP,
//	    set Retval to HUGE_VALL
//	 else if E is too small (E < LDBL_MIN_EXP),
//	    set Retval to F * 2^S
//	    set errno to ERANGE
//	    raise floating point UNDERFLOW exception
//	 else
//	    set Retval to F * 2^S
//
//    h. return Retval
//

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>
#include <stdio.h>
// --------------------------------------------
// Prototype for principal calculation function
// --------------------------------------------
static
long double CalcHypot(long double, long double);

long double hypotl(long double x, long double y)
{
    long double Retval;

    if (isinfl(x) || isinfl(y))
    {
	Retval = HUGE_VALL;
    }
    else if (isnanl(x) || isnanl(y))
    {
	Retval = NAN;
    }
    else
    {
	x = fabsl(x);
	y = fabsl(y);

	Retval = (x == 0) ? y :
		 (y == 0) ? x :
		 (x >= y) ? CalcHypot(x, y) : CalcHypot(y, x);
    }
    return Retval;
}
				// ---------------------------
#define BASE	0x3ffe		// Relative zero for exponents
				// ---------------------------

// The scaling procedure used below avoids underflow and overflow by
// scaling the magnitudes of x and y by the average of their exponents
// before multiplication and then rescaling the result.  This sequence
// is the the result of ideas provided by Steve Moshier.
//

static
long double CalcHypot(long double x, long double y)
{
    // Guaranteed that x & y > 0, x >= y

    int		Check, Diff, Expx, Expy, Scale;
    LDBL	A, B, Final, Retval, Unused;

    // Step b.

    Unused = frexpl(x, &Expx);
    Unused = frexpl(y, &Expy);

    // Step c.

    Diff = Expx - Expy;

    // If x is much larger than y, return x

    if (Diff > (LDBL_MANT_DIG>>1))
    {
	Retval = x;
    }
    else
    {
	// Steps d. & e.

    	// Calculate scaler
	Scale = (Expx + Expy) >> 1;

	// Scale x & y

	A = ldexpl(x, -Scale);
	B = ldexpl(y, -Scale);

	// Step f.
	Final = sqrtl(A*A + B*B);

	// Step g

	// Calculate final exponent

	Unused = frexpl(Final, &Check);
	Check += Scale;

	if (Check > LDBL_MAX_EXP)
	{				// Overflow
	    __math_set_errno(ERANGE);
	    Retval = HUGE_VALL;
	    __fp_raise_except(FE_OVERFLOW);
	}
	else if (Check < LDBL_MIN_EXP)
	{				// Underflow
	    Retval = ldexpl(Final, Scale);
	    __math_set_errno(ERANGE);		// (Might need more code)
	    __fp_raise_except(FE_UNDERFLOW);
	}
	else
	{   // Insert final exponent
	    Retval = ldexpl(Final, Scale);
	}
    }
    // Step h.
    return Retval;
}

#ifdef	TEST

#include <stdio.h>

int
main(void)
{
static int TestNum;

fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);

  __math_set_errno(0);
  fprintf(stderr, "hypotl(LDBL_MIN/2, LDBL_MIN) =\t\t %.21Lg\n",
  		hypotl(LDBL_MIN/2.0L, LDBL_MIN));
  fprintf(stderr, "__math_set_errno(%d\n", errno));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr, "hypotl(3, 4) =\t\t\t %25.17Le\n", hypotl(3., 4.));
# if 1
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(3*10^150, 4*10^150) =\t %30.21Lg\n",
	 hypotl(3.e+150L, 4.e+150L));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(3*10^306, 4*10^306) =\t %30.21Lg\n",
	 hypotl(3.e+306L, 4.e+306L));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(3*10^-320, 4*10^-320) =\t %30.21Lg\n",
	 hypotl(3.e-320L, 4.e-320L));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(0.7*DBL_MAX, 0.7*DBL_MAX) =%30.21Lg\n",
	 hypotl(0.7L*DBL_MAX, 0.7L*DBL_MAX));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(DBL_MAX, 1.0) =\t\t %30.21Lg\n", hypotl(DBL_MAX, 1.0L));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(1.0, DBL_MAX) =\t\t %30.21Lg\n", hypotl(1.0L, DBL_MAX));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(0.0, DBL_MAX) =\t\t %30.21Lg\n", hypotl(0.0L, DBL_MAX));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(3*10^2463, 4*10^2463) =\t %30.21Lg\n",
	 hypotl(3.e+2463L, 4.e+2463L));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(3*10^2467, 4*10^2467) =\t %30.21Lg\n",
	 hypotl(3.e+2467L, 4.e+2467L));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(3*10^4930, 4*10^4930) =\t %30.21Lg\n",
	 hypotl(3.e+4930L, 4.e+4930L));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(3*10^-4930, 4*10^-4930) =\t %30.21Lg\n",
	 hypotl(3.e-4930L, 4.e-4930L));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(0.7*LDBL_MAX, 0.7*LDBL_MAX) =%30.21Lg\n",
	 hypotl(0.7L*LDBL_MAX, 0.7L*LDBL_MAX));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(LDBL_MAX, 1.0) =\t\t %30.21Lg\n", hypotl(LDBL_MAX, 1.0L));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(1.0, LDBL_MAX) =\t\t %30.21Lg\n", hypotl(1.0L, LDBL_MAX));
fprintf(stderr, "------------TEST %3d------------\n", ++TestNum);
  fprintf(stderr,"hypotl(0.0, LDBL_MAX) =\t\t %30.21Lg\n", hypotl(0.0L, LDBL_MAX));
# endif
  return 0;
}

#endif

