// ------------
// llrintf.c
// ------------
//
// From ISO C99:
//
// Function llrintf
//
// Synopsis
//
//	#include <math.h>
//	long long int llrintf(float x);
//
// Description
//
// Using the current rounding direction, function llrintf
// rounds its argument to the nearest integer value and
// returns a numeric value of type long long int. If the
// rounded value is outside the range of type long long
// int, the value of the result is undefined and global
// variable errno is set to ERANGE.
//
// Returns
//
// The llrintf function returns the rounded integer value.
//
// Remarks (C99 paragraph 7.9.6.5)
//
// The llrintf function provides floating-to-integer conversion
// as prescribed by IEC 60559. It rounds according to the current
// rounding direction.	If the rounded value is outside the range
// of the return type, the numeric result is undefined and the
// invalid exception is raised.	 When no other exception is raised
// during the conversion and the result differs from the argument,
// the inexact exception is raised.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, September 2003
//

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long long llrintf(float Arg)
{
    int		CrntExcepts, Exponent;
    float	FltIntVal;

    CrntExcepts = fetestexcept(FE_ALL_EXCEPT);

    ROUND_TO_INT(Arg, FltIntVal);

    (void)frexpf(FltIntVal, &Exponent);

    if ((Exponent > 63) || ! isfinitef(Arg))
    {
	__math_set_errno(EDOM);
	FltIntVal = -LLONG_MAX;
	__fp_raise_except(FE_INVALID);
    }
    else if (fetestexcept(FE_ALL_EXCEPT) == CrntExcepts)
    {
	if (Arg != FltIntVal)
	{
	    __fp_raise_except(FE_INEXACT);
	}
    }

    return (long long) FltIntVal;
}
