// ----------
// llround.c
// ----------
//
// From ISO C99:
//
// The llround function
//
// Synopsis
//
//	#include <math.h>
//	long long int llround(double x);
//
//
// Description
//
// The llround function rounds its argument to the nearest
// integer value, rounding halfway cases away from zero,
// regardless of the current rounding direction. If the
// rounded value is outside the range of the return type,
// the numeric result is unspecified. A range error may occur
// if the magnitude of x is too large.
//
//
// Returns
//
// The llround function returns the rounded integer value.
//
//
// Remarks
//
// The llround function differs from the llrintl function
// (with its default rounding direction) just in that the
// llround function rounds halfway cases away from zero,
// and may (but need not) raise the inexact exception for
// non-integer arguments that round to within the range
// of the return type.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, March 2002, revised September 2003
//

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long long
llround(double Arg)
{
    int		CrntExcepts, Exponent;
    double	DblIntVal;
    
    CrntExcepts = fetestexcept(FE_ALL_EXCEPT);

    DblIntVal = roundl(Arg);

    (void)frexp(DblIntVal, &Exponent);

    if ((Exponent > 63) || ! isfinited(Arg))
    {
	__math_set_errno(EDOM);
	DblIntVal = -LLONG_MAX;
	__fp_raise_except(FE_INVALID);
    }
    else if (fetestexcept(FE_ALL_EXCEPT) == CrntExcepts)
    {
	if (Arg != DblIntVal)
	{
	    __fp_raise_except(FE_INEXACT);
	}
    }

    return (long long) DblIntVal;
}
