// ----------
// llroundf.c
// ----------
//
// From ISO C99:
//
// The llroundf function
//
// Synopsis
//
//	#include <math.h>
//	long long int llroundf(float x);
//
//
// Description
//
// The llroundf function rounds its argument to the nearest
// integer value, rounding halfway cases away from zero,
// regardless of the current rounding direction. If the
// rounded value is outside the range of the return type,
// the numeric result is unspecified. A range error may occur
// if the magnitude of x is too large.
//
//
// Returns
//
// The llroundf function returns the rounded integer value.
//
//
// Remarks
//
// The llroundf function differs from the llrintf function
// (with its default rounding direction) just in that the
// llroundf function rounds halfway cases away from zero,
// and may (but need not) raise the inexact exception for
// non-integer arguments that round to within the range
// of the return type.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, March 2002, revised September 2003
//

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long long
llroundf(float Arg)
{
    int		CrntExcepts, Exponent;
    float	FltIntVal;
    
    CrntExcepts = fetestexcept(FE_ALL_EXCEPT);

    FltIntVal = roundf(Arg);

    (void)frexpf(FltIntVal, &Exponent);

    if ((Exponent > 63) || ! isfinitef(Arg))
    {
	__math_set_errno(EDOM);
	FltIntVal = -LLONG_MAX;
	__fp_raise_except(FE_INVALID);
    }
    else if (fetestexcept(FE_ALL_EXCEPT) == CrntExcepts)
    {
	if (Arg != FltIntVal)
	{
	    __fp_raise_except(FE_INEXACT);
	}
    }

    return (long long) FltIntVal;
}
