// ----------
// llroundl.c
// ----------
//
// From ISO C99:
//
// The llroundl function
//
// Synopsis
//
//	#include <math.h>
//	long long int llroundl(long double x);
//
//
// Description
//
// The llroundl function rounds its argument to the nearest
// integer value, rounding halfway cases away from zero,
// regardless of the current rounding direction. If the
// rounded value is outside the range of the return type,
// the numeric result is unspecified. A range error may occur
// if the magnitude of x is too large.
//
//
// Returns
//
// The llroundl function returns the rounded integer value.
//
//
// Remarks
//
// The llroundl function differs from the llrintl function
// (with its default rounding direction) just in that the
// llroundl function rounds halfway cases away from zero,
// and may (but need not) raise the inexact exception for
// non-integer arguments that round to within the range
// of the return type.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, March 2002, revised September 2003
//

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long long
llroundl(long double Arg)
{
    int		CrntExcepts, Exponent;
    long double LdblIntVal;
    
    CrntExcepts = fetestexcept(FE_ALL_EXCEPT);

    LdblIntVal = roundl(Arg);

    (void)frexpl(LdblIntVal, &Exponent);

    if ((Exponent > 63) || ! isfinitel(Arg))
    {
	__math_set_errno(EDOM);
	LdblIntVal = -LLONG_MAX;
	__fp_raise_except(FE_INVALID);
    }
    else if (fetestexcept(FE_ALL_EXCEPT) == CrntExcepts)
    {
	if (Arg != LdblIntVal)
	{
	    __fp_raise_except(FE_INEXACT);
	}
    }

    return (long long) LdblIntVal;
}
