/*							log2.c
 *
 *	Base 2 logarithm
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, log2();
 *
 * y = log2( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the base 2 logarithm of x.
 *
 * The argument is separated into its exponent and fractional
 * parts.  If the exponent is between -1 and +1, the base e
 * logarithm of the fraction is approximated by
 *
 *     log(1+x) = x - 0.5 x**2 + x**3 P(x)/Q(x).
 *
 * Otherwise, setting  z = 2(x-1)/x+1),
 *
 *     log(x) = z + z**3 P(z)/Q(z).
 *
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      0.5, 2.0    30000       2.0e-16     5.5e-17
 *    IEEE      exp(+-700)  40000       1.3e-16     4.6e-17
 *
 * In the tests over the interval [exp(+-700)], the logarithms
 * of the random arguments were uniformly distributed.
 *
 * ERROR MESSAGES:
 *
 * log2 singularity:  x = 0; returns -HUGE_VAL
 * log2 domain:       x < 0; returns NAN
 */

/*
Cephes Math Library Release 2.8:  June, 2000
Copyright 1984, 1995, 2000 by Stephen L. Moshier
*/
//
// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, November 2003
//

#include <errno.h>
#include <fenv.h>
#include <fdlibml.h>

#define UNK

/* *INDENT-OFF* */

// -----------
// (double)NaN
// -----------
static union
{
    unsigned long long U_Value;
    double  D_Value;
}
NAND = { 0x7ff8000000000000LL };


/* Coefficients for log(1+x) = x - x**2/2 + x**3 P(x)/Q(x)
 * 1/sqrt(2) <= x < sqrt(2)
 */


#ifdef UNK
static double P[] =
{
 1.01875663804580931796E-4,
 4.97494994976747001425E-1,
 4.70579119878881725854E0,
 1.44989225341610930846E1,
 1.79368678507819816313E1,
 7.70838733755885391666E0,
};
static double Q[] =
{
/* 1.00000000000000000000E0, */
 1.12873587189167450590E1,
 4.52279145837532221105E1,
 8.29875266912776603211E1,
 7.11544750618563894466E1,
 2.31251620126765340583E1,
};
#define LOG2EA 0.44269504088896340735992
#endif

#ifdef IBMPC
static unsigned short P[] =
{
0x1bb0,0x93c3,0xb4c2,0x3f1a,
0x52f2,0x3f56,0xd6f5,0x3fdf,
0x6911,0xed92,0xd2ba,0x4012,
0xeb2e,0xc63e,0xff72,0x402c,
0xc84d,0x924b,0xefd6,0x4031,
0xdcf8,0x7d7e,0xd563,0x401e,
};
static unsigned short Q[] =
{
/*0x0000,0x0000,0x0000,0x3ff0,*/
0xef8e,0xae97,0x9320,0x4026,
0xc033,0x4e19,0x9d2c,0x4046,
0xbdbd,0xa326,0xbf33,0x4054,
0xae21,0xeb5e,0xc9e2,0x4051,
0x25b2,0x9e1f,0x200a,0x4037,
};
static unsigned short L[5] = {0x0bf8,0x94ae,0x551d,0x3fdc};
#define LOG2EA (*(double *)(&L[0]))
#endif

/* Coefficients for log(x) = z + z**3 R(z)/S(z),
 * where z = 2(x-1)/(x+1)
 * 1/sqrt(2) <= x < sqrt(2)
 */

#ifdef UNK
static double R[3] =
{
-7.89580278884799154124E-1,
 1.63866645699558079767E1,
-6.41409952958715622951E1,
};
static double S[3] =
{
/* 1.00000000000000000000E0,*/
-3.56722798256324312549E1,
 3.12093766372244180303E2,
-7.69691943550460008604E2,
};
/* log2(e) - 1 */
#define LOG2EA 0.44269504088896340735992
#endif

#ifdef IBMPC
static unsigned short R[12] =
{
0x0e84,0xdc6c,0x443d,0xbfe9,
0x7b6b,0x7302,0x62fc,0x4030,
0x2a20,0x1122,0x0906,0xc050,
};
static unsigned short S[12] =
{
/*0x0000,0x0000,0x0000,0x3ff0,*/
0x6d0a,0x43ec,0xd60d,0xc041,
0xe40e,0x112a,0x8180,0x4073,
0x3f3b,0x19b3,0x0d89,0xc088,
};
#endif

static double polevl(double, double *, int);
static double p1evl(double, double *, int);

/* *INDENT-ON* */

static double polevl(x, coef, N)
double  x;
double  coef[];
int     N;
{
    double  ans;
    int     i;
    double *p;

    p = coef;
    ans = *p++;
    i = N;

    do
	ans = ans * x + *p++;
    while (--i);

    return (ans);
}

/*                                          N
 * Evaluate polynomial when coefficient of x  is 1.0.
 * Otherwise same as polevl.
 */

static double p1evl(x, coef, N)
double  x;
double  coef[];
int     N;
{
    double  ans;
    double *p;
    int     i;

    p = coef;
    ans = x + *p++;
    i = N - 1;

    do
	ans = ans * x + *p++;
    while (--i);

    return (ans);
}

// ---------------------------------
// log2 - Calculate base 2 logarithm
// ---------------------------------

double log2(x)
double  x;
{
    int     e;
    double  y;
    volatile double z;

    if (isnand(x))
    {
	z = (NAND.D_Value);		/* NaN */
    }
    /* Test for domain */
    else if (x <= 0.0)
    {
	if (x == 0.0)
	{
	    __math_set_errno(ERANGE);
	    z = (-HUGE_VAL);
	    __fp_raise_except(FE_DIVBYZERO);
	}
	else
	{
	    __math_set_errno(EDOM);
	    z = (NAND.D_Value);
	    __fp_raise_except(FE_INVALID);
	}
    }
    else if ((isinfd)(x))			/* +Inf */
    {
	z = (x);
    }
    else
    {
	/* separate mantissa from exponent */

	/* Note, frexp is used so that denormal numbers
	 * will be handled properly.
	 */
	x = frexp(x, &e);

	/* logarithm using log(x) = z + z**3 P(z)/Q(z),
	 * where z = 2(x-1)/x+1)
	 */

	if ((e > 2) || (e < -2))
	{
	    if (x < SQRTH)
	    {				/* 2( 2x-1 )/( 2x+1 ) */
		e -= 1;
		z = x - 0.5;
		y = 0.5 * z + 0.5;
	    }
	    else
	    {				/*  2 (x-1)/(x+1)   */
		z = x - 0.5;
		z -= 0.5;
		y = 0.5 * x + 0.5;
	    }

	    x = z / y;
	    z = x * x;
	    y = x * (z * polevl(z, (double *)R, 2) / p1evl(z, (double *)S, 3));
	}
	else
	{

	    /* logarithm using log(1+x) = x - .5x**2 + x**3 P(x)/Q(x) */

	    if (x < SQRTH)
	    {
		e -= 1;
		x = ldexp(x, 1) - 1.0;	/*  2x - 1  */
	    }
	    else
	    {
		x = x - 1.0;
	    }

	    z = x * x;
	    y = x * (z * polevl(x, (double *)P, 5) / p1evl(x, (double *)Q, 5)) - ldexp(z, -1);
	}

/* Multiply log of fraction by log2(e)
 * and base 2 exponent by 1
 *
 * ***CAUTION***
 *
 * This sequence of operations is critical and it may
 * be horribly defeated by some compiler optimizers.
 */
	z = y * LOG2EA;
	z += x * LOG2EA;
	z += y;
	z += x;
	z += e;
    }
    return (z);
}
