// ------------
// lrintf.c
// ------------
//
// From ISO C99:
//
// Function lrintf
//
// Synopsis
//
//	#include <math.h>
//	long int lrintf(float x);
//
// Description
//
// The lrintf function rounds its argument to the nearest
// integer value, rounding according to the current rounding
// direction. If the rounded value is outside the range of
// the return type, the numeric result is unspecified.	A
// range error may occur if the magnitude of x is too large.
// In this case, global variable errno is set to ERANGE.
//
// Returns
//
// The lrintf function returns the rounded integer value.
//
// Remarks
//
// The lrintf function provides floating-to-integer conversion
// as prescribed by IEC 60559. It rounds according to the current
// rounding direction.	If the rounded value is outside the range
// of the return type, the numeric result is undefined and the
// invalid exception is raised.	 When no other exception is raised
// during the conversion and the result differs from the argument,
// the inexact exception is raised.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, October 2003
//
#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long int lrintf(float Arg)
{
    int		CrntExcepts, Exponent;
    float	FltIntVal;

    CrntExcepts = fetestexcept(FE_ALL_EXCEPT);

    ROUND_TO_INT(Arg, FltIntVal);

    (void)frexpf(FltIntVal, &Exponent);

    if ((Exponent > 31) || !isfinitef(Arg))
    {
	__math_set_errno(EDOM);
	FltIntVal = -LONG_MAX;
	__fp_raise_except(FE_INVALID);
    }
    else if (fetestexcept(FE_ALL_EXCEPT) == CrntExcepts)
    {
	if (Arg != FltIntVal)
	{
	    __fp_raise_except(FE_INEXACT);
	}
    }

    return (long int) FltIntVal;
}
