// ------------
// lroundf.c
// ------------
//
// From ISO C99:
//
// Function lroundf
//
// Synopsis
//
//	#include <math.h>
//	long int lroundf(float x);
//
// Description
//
// The lroundf function differs from the lrintf function
// (with its default rounding direction) just in that the
// lroundf function rounds halfway cases away from zero.
// 
// The inexact exception is raised for non-integer arguments 
// that round to within the range of the return type.
// 
// The invalid exception is raised if the magnitude of x 
// is too large.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, September 2003
//
//
// Returns
//
// The lroundf function returns the rounded integer value.
//
// Remarks
//
// The lroundf function provides floating-to-integer conversion
// as prescribed by IEC 60559. It rounds according to the current
// rounding direction.	If the rounded value is outside the range
// of the return type, the numeric result is undefined and the
// invalid exception is raised.	 When no other exception is raised
// during the conversion and the result differs from the argument,
// the inexact exception is raised.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, September 2003
//
#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long int lroundf(float Arg)
{
    int		CrntExcepts, Exponent;
    float	FltIntVal;

    CrntExcepts = fetestexcept(FE_ALL_EXCEPT);

    FltIntVal = roundf(Arg);

    (void)frexpf(FltIntVal, &Exponent);

    if ((Exponent > 31) || !isfinitef(Arg))
    {
	__math_set_errno(EDOM);
	FltIntVal = -LONG_MAX;
	__fp_raise_except(FE_INVALID);
    }
    else if (fetestexcept(FE_ALL_EXCEPT) == CrntExcepts)
    {
	if (Arg != FltIntVal)
	{
	    __fp_raise_except(FE_INEXACT);
	}
    }

    return (long int) FltIntVal;
}
