// ----------
// lroundl.c
// ----------
//
// From ISO C99:
//
// The lroundl function
//
// Synopsis
//
//	#include <math.h>
//	long int lroundl(long double x);
//
//
// Description
//
// The lroundl function rounds its argument to the nearest
// integer value, rounding halfway cases away from zero,
// regardless of the current rounding direction. If the
// rounded value is outside the range of the return type,
// the numeric result is unspecified. A range error occurs
// if the magnitude of x is too large.
//
//
// Returns
//
// The lroundl function returns the rounded integer value.
//
//
// Remarks
//
// The lroundl function differs from the lrintl function
// (with its default rounding direction) just in that the
// lroundl function rounds halfway cases away from zero.
// 
// The inexact exception is raised for non-integer arguments 
// that round to within the range of the return type.
// 
// The invalid exception is raised if the magnitude of x 
// is too large.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, March 2002, revised September 2003
//

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long int
lroundl(long double Arg)
{
    int		CrntExcepts, Exponent;
    long double LdblIntVal;

    CrntExcepts = fetestexcept(FE_ALL_EXCEPT);

    LdblIntVal = roundl(Arg);
    
    (void)frexpl(LdblIntVal, &Exponent);

    if ((Exponent > 31) || ! isfinitel(Arg))
    {
	__math_set_errno(EDOM);
	LdblIntVal = -LONG_MAX;
	__fp_raise_except(FE_INVALID);
    }
    else if (fetestexcept(FE_ALL_EXCEPT) == CrntExcepts)
    {
	if (Arg != LdblIntVal)
	{
	    __fp_raise_except(FE_INEXACT);
	}
    }

    return (long int) LdblIntVal;
}
