// ------------
// remainderl.c
// ------------
//
// Function remainderl()
//
// From ISO C99:
//
// Synopsis
//
//	#include <math.h>
//	long double remainderl(long double x, long double y);
//
// Description
//
// The remainderl function computes the remainder x REM y
// required by IEC 60559.
//
// Returns
//
// The remainderl function returns the value of x REM y.
// If the remainder is zero, it is returned with the sign of x.
//
// Special Values
//
//	o remainderl(+-0, y) returns +-0 if y is not zero.
//
//	o remainderl(x, y) returns a NaN and raises the invalid
//	  exception if x is infinite or y is zero.
//
//	o remainderl(x, +-Inf) returns x if x is not infinite.
//
// Written for DJGPP/GNU by KB Williams,
// kbwms@aol.com, October 2003
//
#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

long double
remainderl (long double x, long double y);

long double
remainderl (long double x, long double y)
{
    long double NewX, Retval;
    int		CrntRndDir, IntPart, YexpBits;

    if (isnanl(x) || isnanl(y))
    {
    	Retval = NAN;
    }
    else if (isinfl(x) || (y == 0))
    {
       	__math_set_errno(EDOM);
    	Retval = NAN;
	__fp_raise_except(FE_INVALID);
    }
    else if (isinfl(y))
    {
    	Retval = x;
    }
    else
    {
        GET_LDOUBLE_EXP(YexpBits, y);

        if ((YexpBits & 0x7fff) < 0x7ffc)	// Reduce x
	    NewX = fmodl(x, 8 * y);		// x < 8 * y
        else
            NewX = x;

        CrntRndDir = fegetround();
        fesetround(FE_TONEAREST);               // Rounds to nearest or even

        IntPart = rintl(NewX / y);		// x/y rounded to int

	fesetround(CrntRndDir);

        Retval = NewX - IntPart * y;

	if (Retval == 0)
	    Retval = copysignl(Retval, x);
    }
    return Retval;
}
