// ---------
// remquof.c
// ---------
//
// Function remquof()
//
// Synopsis
//
//	#include <math.h>
//	float remquof(float x, float y,	int *quo);
//
// Description
//
// Function remquof computes the same remainder as function
// remainder.  In the object pointed to by quo, remquof
// stores the signed lowest three bits (range -7 to +7) of the
// integer value closest to the quotient x/y . This partial
// quotient can be used in some argument reduction algorithms.
//
// Returns
//
// Function remquof returns x REM y (as required by IEC 60559).
// Also, in the object pointed to by quo, function remquof stores
// the signed lowest three bits of the integer value closest to
// the quotient x/y.
//
// Written for DJGPP/GCC by KB Williams,
// kbwms@aol.com, October 2003
//
#include <errno.h>
#include <fenv.h>
#include <float.h>
#include <fdlibml.h>

float
remquof(float x, float y, int *quo)
{
    float Retval;

    *quo   = 0;

    if ((isnanf)(x) || (isnanf)(y))
    {
    	Retval = NAN;
    }
    else if (isinff(x) || (y == 0))
    {
       	__math_set_errno(EDOM);
    	Retval = NAN;
	__fp_raise_except(FE_INVALID);
    }
    else if ((isinff)(y))
    {
    	Retval = x;
    }
    else
    {
	float	NewX;
	int	CrntRndDir, IntPart, Exponent;

        (void)frexpf(y, &Exponent);

        if (Exponent < (FLT_MAX - 2))		// Reduce x
	    NewX = fmodf(x, 8 * y);		// x < 8 * y
        else
            NewX = x;

        CrntRndDir = fegetround();
        fesetround(FE_TONEAREST);

        IntPart = rintf(NewX / y);		// x/y rounded to int

        fesetround(CrntRndDir);

        Retval = NewX - IntPart * y;

	if (Retval == 0)
	    Retval = copysignf(Retval, x);

	*quo = (IntPart >= 0) ? IntPart & 7	// get lowest 3 bits
			      : -((-IntPart) & 7);
    }

    return  Retval;
}
