/* sinhl.c -- long double version of sinh.c.
 * Conversion to long double by KB Williams,
 * kbwms@aol.com, December 2001, November 2003,
 * January 2004
 */

/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */

/* sinhl(x)
 * Method :
 * mathematically sinh(x) if defined to be (exp(x)-exp(-x))/2
 *	1. Replace x by |x| (sinhl(-x) = -sinhl(x)).
 *	2.
 *						     E + E/(E+1)
 *	    0	     <= x <= 25	    :  sinhl(x) := --------------,
 *							 2
 *			where E = expm1l(x)
 *
 *	    25	     <= x <= lnovft :  sinhl(x) := expl(x)/2
 *	    lnovft   <= x <= ln2ovft:  sinhl(x) := expl(x/2)/2 * expl(x/2)
 *	    ln2ovft  <	x	    :  sinhl(x) := HUGE_VALL (overflow)
 *
 */

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

/* MAXLOGL is 1.1356523406294143949492E4L or log(LDBL_MAX) */

#ifdef __STDC__
long double sinhl(long double Arg)
#else
long double sinhl(Arg)
long double Arg;
#endif
{
    long double AbsArg, Retval, T;

    AbsArg = fabsl(Arg);

    /* if |Arg| is NaN, Inf, or too small ... */
    if (!isfinitel(AbsArg))
    {
	Retval = AbsArg;
    }	 
    
    else if (fpclassifyl(AbsArg) == FP_SUBNORMAL)
    {
    	Retval = AbsArg;
	__math_set_errno(ERANGE);
	__fp_raise_except(FE_UNDERFLOW);
    }	    		       

    /* if |Arg| in (0,25], calculate (E+E/(E+1))) */
    else if (AbsArg < 25.0L)
    {
	T = expm1l(AbsArg);
	if (AbsArg < 1.0L)
	{		/* for small |Arg|, use more stable form */
	    Retval = (T+T) - T * T/(T + 1.0L);
	}
	else
	{
	    Retval = (T + T/(T + 1.0L));
	}
	Retval *= 0.5L;
    }

    /* if |Arg| in [25, log(maxdouble)] return 0.5*exp(|Arg|) */
    else if (AbsArg < MAXLOGL)
    {
	Retval = 0.5L * expl(AbsArg);
    }

    /* if |Arg| in [log(maxdouble), log(overflow threshold)] */
    else if (AbsArg < (MAXLOGL + LOGE2L))
    {
	T = expl(0.5L * AbsArg);	/* avoid overflow */
	Retval = T * 0.5L;
	Retval *= T;
    }

    /* if |Arg| > overflow threshold, sinhl(Arg) overflow */
    else
    {
	__math_set_errno(ERANGE);
	Retval = HUGE_VALL;
	__fp_raise_except(FE_OVERFLOW);
    }

    return (signbitl(Arg) ? -Retval : Retval);
}
