// -----------
// slmpydbl.c
// -----------
//
// Function SlMpyDbl
//
// Multiplies two single-length double precision arguments using
// extra-precision arithmetic.
//
// Returns a double-length double result in space provided in the
// calling sequence.
//
// If no errors are detected, zero is returned as a function value.       .
// If one of the arguments is either a NaN or Infinity, -1 is returned.
//
// int SlMpyDbl(double ArgA, double ArgB,
//              double *HeadC, double *TailC);
//
// Procedure is:
//
// 1. Separate each argument as follows:
//
//      o sign        - one bit in unsigned int
//      o exponent    - signed int
//      o significand - 53 bits left justified in unsigned long long
//
// 2. Perform extra-length multiplication of significands
// 3. Left justify result (one bit at most) and set bias as needed
// 4. Build double precision head and tail of result
// 5. Adjust sign of each as needed
//
//

#include <fdlibml.h>
#include <fenv.h>

void    MpyUll(ULLONG, ULLONG, ULLONG *, ULLONG *);

#define HI_BIT	0x8000000000000000ULL
static int
ParseDbl(double ArgA, int *SignA, int *ExpA, ULLONG * FracBitsA)
{
    int     Retval = 0;

    if (!isfinited(ArgA))
    {
	Retval = -1;
    }
    else
    {
    	double	 Temp;
	*SignA = (ArgA < 0);
	Temp = ldexp(frexp(fabs(ArgA), ExpA), 53);
	*FracBitsA = ((ULLONG)Temp) << 11;
    }
    return Retval;
}
int
SlMpyDbl(double ArgA, double ArgB, double *HeadC, double *TailC)
{
    int     ErrFlg;
    int     ExpA, ExpB, SignA, SignB;
    ULLONG  FracBitsA, FracBitsB;

    ErrFlg = ParseDbl(ArgA, &SignA, &ExpA, &FracBitsA);
    if (ErrFlg != 0)
    {
	ErrFlg = 1;
    }
    else
    {
	ErrFlg = ParseDbl(ArgB, &SignB, &ExpB, &FracBitsB);

	if (ErrFlg != 0)
	{
	    ErrFlg = 2;
	}
	else
	{
	    int		Ignored;
	    ULLONG	HeadBits,   TailBits;
	    ULLONG	MyHeadBits, MyTailBits;
	    double	MyHeadC,    MyTailC;

	    MpyUll(FracBitsA, FracBitsB, &HeadBits, &TailBits);

	    if ((HeadBits & HI_BIT) == 0)
	    {
		HeadBits = (HeadBits << 1) + (TailBits >> 63);
		TailBits <<= 1;
		--ExpA;
	    }
	    MyHeadBits = HeadBits & 0xfffffffffffff800ULL;
	    MyHeadC = ldexp(frexp((double)MyHeadBits, &Ignored), ExpA+ExpB);

	    MyTailBits = ((HeadBits ^ MyHeadBits) << 53) + (TailBits >> 11);
	    MyTailC = ldexp(frexp((double)MyTailBits, &Ignored), -53+ExpA+ExpB);

	    if (SignA == SignB)
	    {
	    	*HeadC = MyHeadC;
	    	*TailC = MyTailC;
	    }
	    else
	    {
	    	*HeadC = -MyHeadC;
	    	*TailC = -MyTailC;
	    }
	}
    }

    return ErrFlg;
}
