// -----------
// slmpyflt.c
// -----------
//
// Function SlMpyFlt
//
// Multiplies two single-length float precision arguments using
// extra-precision arithmetic.
//
// Returns a float-length float result in space provided in the
// calling sequence.
//
// If no errors are detected, zero is returned as a function value.       .
// If one of the arguments is either a NaN or Infinity, -1 is returned.
//
// int SlMpyFlt(float  ArgA,  float ArgB,
//		float *HeadC, float *TailC);
//
// Procedure is:
//
// 1. Split each argument into two pieces
//
//    ErrFlg = SplitFlt(ArgA, &ExpA, &HeadA, &TailA);
//
//    If ErrFlg is -1, ArgA is not finite, return ErrFlg
//
//    ErrFlg = SplitFlt(ArgB, &ExpB, &HeadB, &TailB);
//
//    If ErrFlg is -1, ArgB is not finite, return ErrFlg
//
// 2. Calculate product
//
//    a. Calculate Head of C
//	 *HeadC = P1 + P2:
//
//	 P1 = HeadA * HeadB
//	 P2 = HeadA*TailB + TailA*HeadB
//
//	 P1 consists of the highest-valued bits of the product
//	 P2 consists of the middle bits of the product
//	 HeadC consists of 24 bits of product-sum, namely all
//		of P1 plus the upper bits of P2
//
//    b. Calculate Tail of C
//	 *TailC = ((P1 - *HeadC) + P2) + TailA*TailB
//
//	 TailC consists of the lower bits of P2 and (possibly)
//	 some of TailA*TailB
//

#include <fdlibml.h>
#include <fenv.h>

int SlMpyFlt(float ArgA, float ArgB,
     	     float *HeadC, float *TailC)
{
    int     ErrFlg;
    int	    ExpA, ExpB;
    float   HeadA, HeadB, TailA, TailB;

    ErrFlg = SplitFlt(ArgA, &ExpA, &HeadA, &TailA);

    if (ErrFlg != 0)
    {
    	ErrFlg = 1;
    }
    else
    {
    	ErrFlg = SplitFlt(ArgB, &ExpB, &HeadB, &TailB);

	if (ErrFlg != 0)
	{
	    ErrFlg = 2;
	}
	else
	{
	    int		CrntRndMode;
	    volatile
	    	float	MyHeadC, MyTailC, P1, P2;

	    CrntRndMode = fegetround();
	    fesetround(FE_TOWARDZERO);

	    P1 = HeadA * HeadB;
	    P2 = HeadA*TailB + TailA*HeadB;

	    MyHeadC = P1 + P2;
	    MyTailC = (P1 - MyHeadC);
	    MyTailC += P2;
	    MyTailC += TailA*TailB;

	    *HeadC = ldexpf(MyHeadC, ExpA+ExpB);
	    *TailC = ldexpf(MyTailC, ExpA+ExpB);

	    fesetround(CrntRndMode);
	}
    }

    return ErrFlg;
}
