// ----------
// spltldbl.c
// ----------
//
// Function SplitLdbl
//
// Breaks up long double floating point argument into two 32-bit
// numbers of the same type.
//
// int SplitLdbl(Arg, Exp, Head, Tail)
//
//      long double Arg;                // Input
//      int	    *Exp;		// Exponent of Arg
//      long double *Head;              // Upper 32 bits of Arg
//      long double *Tail;              // Lower 32 bits of Arg
//
// Returns -1 if Arg is +-Inf or NaN
//
// Procedure is:
//
// 1. Scale Arg in [.5, 1), store exponent in Exp
// 2. Get fraction bits as a 64-bit integer value - FracBits
// 3. Break into two 32-bit unsigned long long int values
//    HeadBits - Upper 32 bits are left justified in 64-bit ULLONG
//    TailBits - Lower 32 bits are right justified in 64-bit ULLONG
// 4. Convert Head and Tail to long double, correctly signed and
//    scaled in [.5, 1) with exponent on Tail 64 less that Head's
//

#include <fdlibml.h>

int
SplitLdbl(long double Arg, int *Exp,
    long double *Head, long double *Tail)
{
    int     Retval = 0;

    if (!isfinitel(Arg))
    {
	Retval = -1;
    }
    else
    {
    	int	Unused;
	ULLONG  FracBits, HeadBits, TailBits;

    	LDBL	MyArg = frexpl(Arg, Exp);

	GET64_LDOUBLE(Unused, FracBits, MyArg);

	HeadBits = FracBits & 0xffffffff00000000ULL;
	TailBits = FracBits ^ HeadBits;

	*Head = copysignl(ldexpl((LDBL)HeadBits, -64), Arg);
	*Tail = copysignl(ldexpl((LDBL)TailBits, -64), Arg);
    }

    return Retval;
}
