/*							tanl.c
 *
 *	Circular tangent, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, tanl();
 *
 * y = tanl( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the circular tangent of the radian argument x.
 *
 * Range reduction is modulo pi/4.  A rational function
 *	 x + x**3 P(x**2)/Q(x**2)
 * is employed in the basic interval [0, pi/4].
 *
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE     +-1.07e9	      30000	1.9e-19	    4.8e-20
 *
 * ERROR MESSAGES:
 *
 *   message	     condition		value returned
 * tan total loss   x > 2^39		    0.0
 *
 */
/*							cotl.c
 *
 *	Circular cotangent, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * long double x, y, cotl();
 *
 * y = cotl( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the circular cotangent of the radian argument x.
 *
 * Range reduction is modulo pi/4.  A rational function
 *	 x + x**3 P(x**2)/Q(x**2)
 * is employed in the basic interval [0, pi/4].
 *
 *
 *
 * ACCURACY:
 *
 *			Relative error:
 * arithmetic	domain	   # trials	 peak	      rms
 *    IEEE     +-1.07e9	     30000	1.9e-19	    5.1e-20
 *
 *
 * ERROR MESSAGES:
 *
 *   message	     condition		value returned
 * cot total loss   x > 2^39		   inaccurate
 * cot singularity  x = 0		   HUGE_VALL
 *
 */

/*
Cephes Math Library Release 2.7:  May, 1998
Copyright 1984, 1990, 1998 by Stephen L. Moshier
*/
// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, December 2003

#include <errno.h>
#include <fdlibml.h>
#include <fenv.h>

/* *INDENT-OFF* */
static long double P[] =
{
-1.3093693918138377764608E4L,
 1.1535166483858741613983E6L,
-1.7956525197648487798769E7L,
};
static long double Q[] =
{
/* 1.0000000000000000000000E0L,*/
 1.3681296347069295467845E4L,
-1.3208923444021096744731E6L,
 2.5008380182335791583922E7L,
-5.3869575592945462988123E7L,
};
static long double DP1 = 7.853981554508209228515625E-1L;
static long double DP2 = 7.946627356147928367136046290398E-9L;
static long double DP3 = 3.061616997868382943065164830688E-17L;

#ifdef USE_LOSSTH
static long double LOSSTH = 5.49755813888e11L; /* 2^39 */
#endif
/* *INDENT-ON* */

static long double tancotl(long double, int);

long double tanl(x)
long double x;
{
    if (x == 0.0L)
	return (x);
    if (isnanl(x))
	return (x);

// Check for denormal Argument

    if (fpclassifyl(x) == FP_SUBNORMAL)
    {
    	__math_set_errno(ERANGE);
	__fp_raise_except(FE_UNDERFLOW);
	return (x);
    }

    if ((isinfl)(x))
    {
    	__math_set_errno(EDOM);
	__fp_raise_except(FE_INVALID);
	return NAN;
    }
    return (tancotl(x, 0));
}

long double cotl(x)
long double x;
{

    if (x == 0.0L)
    {
    	__math_set_errno(EDOM);
	__fp_raise_except(FE_DIVBYZERO);
	return HUGE_VALL;		// Singularity, Return INFINITY
    }
    return (tancotl(x, 1));
}

static long double tancotl(xx, cotflg)
long double xx;
int     cotflg;
{
    long double x, y, z, zz;
    int     j, sign;

/* make argument positive but save the sign */
    if (xx < 0.0L)
    {
	x = -xx;
	sign = -1;
    }
    else
    {
	x = xx;
	sign = 1;
    }
/* compute x mod PIO4 */
    y = floorl(x / PIO4L);

/* strip high bits of integer part */
    z = ldexpl(y, -4);
    z = floorl(z);			/* integer part of y/16 */
    z = y - ldexpl(z, 4);		/* y - 16 * (y/16) */

/* integer and fractional part modulo one octant */
    j = z;

/* map zeros and singularities to origin */
    if (j & 1)
    {
	j += 1;
	y += 1.0L;
    }

    z = ((x - y * DP1) - y * DP2) - y * DP3;

    zz = z * z;

    if (zz > 1.0e-20L)
    {
	long double p, q;
	p = zz * (P[2] + zz * (P[1] + zz * P[0]));
	q = Q[3] + zz * (Q[2] + zz * (Q[1] + zz * (Q[0] + zz)));
	y = z + z * (p / q);
    }
    else
	y = z;

    if (j & 2)
    {
	if (cotflg)
	    y = -y;
	else
	    y = -1.0L / y;
    }
    else
    {
	if (cotflg)
	    y = 1.0L / y;
    }

    if (sign < 0)
	y = -y;

    return (y);
}
