/*	catan.c
 *
 *	Complex circular arc tangent
 *
 *
 *
 * SYNOPSIS:
 *
 * double complex catan();
 * double complex z, w;
 *
 * w = catan (z);
 *
 *
 *
 * DESCRIPTION:
 *
 * If
 *     z = x + iy,
 *
 * then
 *          1       (    2x     )
 * Re w  =  - arctan(-----------)  +  k PI
 *          2       (     2    2)
 *                  (1 - x  - y )
 *
 *               ( 2         2)
 *          1    (x  +  (y+1) )
 * Im w  =  - log(------------)
 *          4    ( 2         2)
 *               (x  +  (y-1) )
 *
 * Where k is an arbitrary integer.
 *
 * catan(z) = -i catanh(iz).
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC       -10,+10      5900       1.3e-16     7.8e-18
 *    IEEE      -10,+10     30000       2.3e-15     8.5e-17
 * The check catan( ctan(z) )  =  z, with |x| and |y| < PI/2,
 * had peak relative error 1.5e-16, rms relative error
 * 2.9e-17.  See also clog().
 */
/*
Cephes Math Library Release 2.1:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004


#include "complex.h"
#include <float.h>
#include <math.h>

#include "redupi.c"

double  complex
catan(z)
double complex z;
{
    double complex w;
    double  a, t, x, x2, y;

    x = creal(z);
    y = cimag(z);

//    if ((x == 0.0) && (y > 1.0))
//	goto ovrf;

    x2 = x * x;
    a = 1.0 - x2 - (y * y);
//    if (a == 0.0)
//	goto ovrf;

    t = 0.5 * atan2(2.0 * x, a);
    w = redupi(t);

    t = y - 1.0;
    a = x2 + (t * t);
    if (a == 0.0)
	goto ovrf;

    t = y + 1.0;
    a = (x2 + (t * t)) / a;
    if (a == 0.0)
	goto ovrf;

    w = w + (0.25 * log(a)) * I;
    return (w);

  ovrf:
    //mtherr ("catan", OVERFLOW);
    w = DBL_MAX + DBL_MAX * I;
    return (w);
}
