/*							catanl()
 *
 *	Complex circular arc tangent
 *
 *
 *
 * SYNOPSIS:
 *
 * long double complex catanl();
 * long double complex z, w;
 *
 * w = catanl( z );
 *
 *
 *
 * DESCRIPTION:
 *
 * If
 *     z = x + iy,
 *
 * then
 *          1       (    2x     )
 * Re w  =  - arctan(-----------)  +  k PI
 *          2       (     2    2)
 *                  (1 - x  - y )
 *
 *               ( 2         2)
 *          1    (x  +  (y+1) )
 * Im w  =  - log(------------)
 *          4    ( 2         2)
 *               (x  +  (y-1) )
 *
 * Where k is an arbitrary integer.
 *
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC       -10,+10      5900       1.3e-16     7.8e-18
 *    IEEE      -10,+10     30000       2.3e-15     8.5e-17
 * The check catan( ctan(z) )  =  z, with |x| and |y| < PI/2,
 * had peak relative error 1.5e-16, rms relative error
 * 2.9e-17.  See also clog().
 */
/*
Cephes Math Library Release 2.1:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004


#include "complex.h"
#include <math.h>
#include <float.h>
#include "redupil.c"

long double complex
catanl(long double complex z)
{
    long double complex w;
    long double a, t, x, x2, y;

    x = creall(z);
    y = cimagl(z);

    x2 = x * x;
    a = 1.0L - x2 - (y * y);

    t = atan2l(2.0L * x, a) * 0.5L;
    w = redupil(t);

    t = y - 1.0L;
    a = x2 + (t * t);
    if (a == 0.0L)
    {
	w = LDBL_MAX + LDBL_MAX * I;
    }
    else
    {

	t = y + 1.0L;
	a = (x2 + (t * t)) / a;
	if (a == 0.0L)
	{
	    w = LDBL_MAX + LDBL_MAX * I;
	}
	else
	{
	    w = w + (0.25L * logl(a)) * I;
	}
    }
    return (w);
}
