/*	ccotl.c
 *
 *	Complex circular cotangent
 *
 *
 *
 * SYNOPSIS:
 *
 * long double complex ccotl();
 * long double complex z, w;
 *
 * w = ccotl( z );
 *
 *
 *
 * DESCRIPTION:
 *
 * If
 *     z = x + iy,
 *
 * then
 *
 *           sin 2x  -  i sinh 2y
 *     w  =  --------------------.
 *            cosh 2y  -  cos 2x
 *
 * On the real axis, the denominator has zeros at even
 * multiples of PI/2.  Near these points it is evaluated
 * by a Taylor series.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC       -10,+10      3000       6.5e-17     1.6e-17
 *    IEEE      -10,+10     30000       9.2e-16     1.2e-16
 * Also tested by ctan * ccot = 1 + i0.
 */
/*
Cephes Math Library Release 2.1:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004



#include "complex.h"
#include <math.h>
#include <float.h>
#include "redupil.c"
#include "ctansl.c"

long double complex
ccotl(long double complex z)
{
    long double complex w;
    long double d;

    d = coshl(2.0L * cimagl(z)) - cosl(2.0L * creall(z));

    if (d && fabsl(d) < 0.25L)
    {		
	d = ctansl(z);
    }
    if (d == 0.0L)
    {
	//mtherr( "ccot", OVERFLOW );
	w = LDBL_MAX + LDBL_MAX * I;
    }
    else
    {		
	w = sinl(2.0L * creall(z)) / d + (-sinhl(2.0L * cimagl(z)) / d) * I;
    }
    return (w);
}
