/*	clgamma.c
 *
 *	Natural logarithm of complex gamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * #include <complex.h>
 * double complex x, y, clgamma();
 *
 * y = clgamma( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the base e (2.718...) logarithm of the complex gamma
 * function of the argument.
 *
 * The logarithm of the gamma function is approximated by the
 * logarithmic version of Stirling's asymptotic formula.
 * Arguments of real part less than 14 are increased by recurrence.
 * The cosecant reflection formula is employed for arguments
 * having real part less than -14.
 *
 * Arguments greater than MAXLGM return DBL_MAX and an error
 * message.  MAXLGM = 2.556348e305 for IEEE arithmetic.
 *
 *
 *
 * ACCURACY:
 *
 *
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      -20,20      500000      1.4e-14     4.5e-16
 *    IEEE     -100,100     100000                  1.6e-16
 * The error criterion was relative when the function magnitude
 * was greater than one but absolute when it was less than one.
 */
/*
Cephes Math Library Release 2.7:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004


#include "complex.h"
#include <float.h>
#include <math.h>


/* Asymptotic expansion of log gamma  */
static double A[] = {
#if 0
    -1.3924322169059011164274322169059011164274E0,
    1.7964437236883057316493849001588939669435E-1,
    -2.9550653594771241830065359477124183006536E-2,
    6.4102564102564102564102564102564102564103E-3,
#endif
    -1.9175269175269175269175269175269175269175E-3,	// 691./360360.
    8.4175084175084175084175084175084175084175E-4,	// 1./1188.
    -5.9523809523809523809523809523809523809524E-4,	// 1./1680.
    7.9365079365079365079365079365079365079365E-4,      // 1./12600.
    -2.7777777777777777777777777777777777777778E-3,     // 1./360.
    8.3333333333333333333333333333333333333333E-2       // 1./12.
};
/* log( sqrt( 2*pi ) ) */
static double LS2PI = 0.91893853320467274178;
static double LOGPI = 1.14472988584940017414;

#define MAXLGM 2.556348e305

/* Logarithm of gamma function */

double complex
clgamma(x)
double complex x;
{
    double  p, q, a;
    double complex c, w, u, v;
    int     i, cj;

    cj = 0;
    if (cimag(x) < 0)
    {
	cj = 1;
	x = conj(x);
    }

/* Reflection formula -z gamma(-z) gamma(z) = pi / sin(pi z) */
    if (creal(x) < -14.0)
    {
	q = creal(x);
	p = floor(q);
	if ((p == q) && cimag(x) == 0.0)
	    goto loverf;
	if (fabs(cimag(x)) > 36.7)
	{
	    /* sin z grows exponentially with Im(z).  Find ln sin(pi z)
	       from |sin z| = sqrt( sin^2 x + sinh^2 y),
	       arg sin z = arctan(tanh y / tan x).  */
	    c = PI * cimag(x) - M_LN2 + I * PI * (0.5 - q);
	    c = LOGPI - c - clgamma(1.0 - x);
	}
	else
	{
	    /* Reduce sine arg mod pi.  */
	    u = csin(PI * (x - p));
	    if (u == 0.0)
		goto loverf;
	    w = clgamma(1.0 - x);
	    c = LOGPI - clog(u) - w;
	    /* Adjust for reduced sine arg.  */
	    //cimag(c) += PI * p;
	    c += I * (PI * p);
	}
	goto ldone;
    }
    w = 0.0;
    if (creal(x) < 14.0)
    {
	/* To satisfy Im {clgamma(z)} = arg cgamma(z), accumulate
	   arg u during the recurrence.  */
	a = 0.0;
	w = 1.0;
	p = 0.0;
	u = x;
	while (creal(u) < 14.0)
	{
	    if (u == 0.0)
		goto loverf;
	    w *= u;
	    a += carg(u);
	    p += 1.0;
	    u = x + p;
	}
	x = u;
	w = -log(cabs(w)) - I * a;
    }

    if (creal(x) > MAXLGM)
    {
      loverf:
	// mtherr( "clgamma", OVERFLOW );
	c = DBL_MAX + DBL_MAX * I;
	goto ldone;
    }

    c = (x - 0.5) * clog(x) - x + LS2PI + w;

    if (cabs(x) > 1.0e8)
	goto ldone;

    v = 1.0 / (x * x);
    u = A[0];
    for (i = 1; i < 6; i++)
    {
	u = u * v + A[i];
    }
    c = c + u / x;

  ldone:
    if (cj)
	c = conj(c);
    return (c);
}
