/*	csqrtf.c
 *
 *	Complex square root
 *
 *
 *
 * SYNOPSIS:
 *
 * float complex csqrtf();
 * float complex z, w;
 *
 * w = csqrtf( z );
 *
 *
 *
 * DESCRIPTION:
 *
 *
 * If z = x + iy,  r = |z|, then
 *
 *                       1/2
 * Re w  =  [ (r + x)/2 ]   ,
 *
 *                       1/2
 * Im w  =  [ (r - x)/2 ]   .
 *
 * Cancellation error in r-x or r+x is avoided by using the
 * identity  2 Re w Im w  =  y.
 *
 * Note that -w is also a square root of z.  The root chosen
 * is always in the right half plane and Im w has the same sign as y.
 *
 *
 *
 * ACCURACY:
 *
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      -10,+10    1,000,000    1.8e-7       3.5e-8
 *
 */
/*
Cephes Math Library Release 2.1:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004


#include "complex.h"
#include <float.h>
#include <math.h>

float   complex
(csqrtf)(float complex z)
{
    float complex w;
    float   x, y, r, t;

    x = crealf(z);
    y = cimagf(z);

    if (y == 0.0f)
    {
	if (x < 0.0f)
	{
	    w = 0.0f + sqrtf(-x) * I;
	    //return (w);
	}
	else if (x == 0.0f)
	{
	    w = 0.0f + y * I;
	    //return (w);
	}
	else
	{
	    w = sqrtf(x) + y * I;
	    //return (w);
	}
    }

    if (x == 0.0f)
    {
	r = fabsf(y);
	r = sqrtf(0.5f * r);
	if (y > 0)
	    w = r + r * I;
	else
	    w = r - r * I;
	//return (w);
    }
    else
    {
    	int	ea, es;
	int	ex, ey;

	// Extract exponents of real (x) and imaginary (y) parts

	frexpf(x, &ex);
	frexpf(y, &ey);

	// Calculate scale & rescale factors

	es = ((ex + ey) >> 2);		// rescale

	ea = es << 1;			// scale		

	// Scale x and y

	x = ldexpf(x, -ea);
	y = ldexpf(y, -ea);

	w = x + y * I;
	r = cabsf(w);

        if (x > 0)
        {
	    t = sqrtf(0.5f * r + 0.5f * x);
	    r = ldexpf(fabsf((0.5f * y) / t), es);
	    t = ldexpf(t, es);
        }
        else
        {
	    r = sqrtf(0.5f * r - 0.5f * x);
	    t = ldexpf(fabsf((0.5f * y) / r), es);
	    r = ldexpf(r, es);
        }
    
	if (y < 0)
	    w = t - r * I;
	else
	    w = t + r * I;
    }

    return (w);
}
