/*  csqrtl.c
 *  --------
 *  Complex square root
 *
 *
 * SYNOPSIS:
 *
 * long double complex csqrtl();
 * long double complex z, w;
 *
 * w = csqrtl( z );
 *
 *
 * DESCRIPTION:
 *
 *
 * If z = x + iy,  r = |z|, then
 *
 *                       1/2
 * Re w  =  [ (r + x)/2 ]   ,
 *
 *                       1/2
 * Im w  =  [ (r - x)/2 ]   .
 *
 * Cancellation error in r-x or r+x is avoided by using the
 * identity  2 Re w Im w  =  y.
 *
 * Note that -w is also a square root of z.  The root chosen
 * is always in the right half plane and Im w has the same sign as y.
 *
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      -10,+10     500000      1.1e-19     3.0e-20
 *
 */
/*
Cephes Math Library Release 2.1:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004


#include "complex.h"
#include <math.h>

long double complex
csqrtl(long double complex z)
{
    long double complex w;
    long double x, y, r, t;

    x = creall(z);
    y = cimagl(z);

    if (y == 0.0L)
    {
	if (x < 0.0L)
	{
	    w = 0.0L + sqrtl(-x) * I;
	    //return (w);
	}
	else
	{
	    w = sqrtl(x) + 0.0L * I;
	    //return (w);
	}
    }


    else if (x == 0.0L)
    {
	r = fabsl(y);
	r = sqrtl(0.5L * r);
	if (y > 0.0L)
	    w = r + r * I;
	else
	    w = r - r * I;
	//return (w);
    }
    else
    {
    	int	ea, es;
	int	ex, ey;

	// Extract exponents of real (x) and imaginary (y) parts

	frexpl(x, &ex);
	frexpl(y, &ey);

	// Calculate scale & rescale factors

	es = ((ex + ey) >> 2);		// rescale

	ea = es << 1;			// scale		

	// Scale x and y

	x = ldexpl(x, -ea);
	y = ldexpl(y, -ea);

	w = x + y * I;
	r = cabsl(w);

        if (x > 0)
        {
	    t = sqrtl(0.5L * r + 0.5L * x);
	    r = ldexpl(fabsl((0.5L * y) / t), es);
	    t = ldexpl(t, es);
        }
        else
        {
	    r = sqrtl(0.5L * r - 0.5L * x);
	    t = ldexpl(fabsl((0.5L * y) / r), es);
	    r = ldexpl(r, es);
        }
    
	if (y < 0)
	    w = t - r * I;
	else
	    w = t + r * I;
    }
        		       
    return (w);
}
