/*	ctan.c
 *
 *	Complex circular tangent
 *
 *
 *
 * SYNOPSIS:
 *
 * double complex ctan();
 * double complex z, w;
 *
 * w = ctan (z);
 *
 *
 *
 * DESCRIPTION:
 *
 * If
 *     z = x + iy,
 *
 * then
 *
 *           sin 2x  +  i sinh 2y
 *     w  =  --------------------.
 *            cos 2x  +  cosh 2y
 *
 * On the real axis the denominator is zero at odd multiples
 * of PI/2.  The denominator is evaluated by its Taylor
 * series near these points.
 *
 * ctan(z) = -i ctanh(iz).
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC       -10,+10      5200       7.1e-17     1.6e-17
 *    IEEE      -10,+10     30000       7.2e-16     1.2e-16
 * Also tested by ctan * ccot = 1 and catan(ctan(z))  =  z.
 */

/*
Cephes Math Library Release 2.1:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004


#include "complex.h"
#include <float.h>
#include <math.h>

#include "redupi.c"
#include "ctans.c"

double  complex
ctan(z)
double complex z;
{
    double complex w;
    double  d;

    d = cos(2.0 * creal(z)) + cosh(2.0 * cimag(z));

    if (d && (fabs(d) < 0.25))
	d = ctans(z);

    if (d == 0.0)
    {
	w = DBL_MAX + DBL_MAX * I;
    }
    else
    {		
	w = sin(2.0 * creal(z)) / d + (sinh(2.0 * cimag(z)) / d) * I;
    }	
    return (w);
}
