/*	ctgammaf
 *
 *	Complex gamma function, single precision
 *
 *
 *
 * SYNOPSIS:
 *
 * #include <complex.h>
 * float complex x, y, ctgammaf();
 *
 * y = ctgammaf( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns complex-valued gamma function of the complex argument.
 *
 * Arguments |Re(x)| <= 14 are increased by recurrence.
 * Large arguments are handled by Stirling's formula. Large negative
 * arguments are made positive using the reflection formula.  
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      -10,10      100000      5.2e-6      5.8e-7
 *    IEEE      -22,22      100000      1.1e-5      1.2e-6
 *    IEEE      -27,27      100000      4.6e-5      1.5e-6
 *
 * Error for arguments outside the test range will be larger
 * owing to error amplification by the exponential function.
 *
 */
/*
Cephes Math Library Release 2.7:  March, 1998
Copyright 1984, 1998 Stephen L. Moshier
*/
// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004

#include "complex.h"
#include <float.h>
#include <math.h>

/* Stirling's formula for the gamma function */
#define NSTIR 7
static float STIR[NSTIR] = {
    -5.92166437353693882865E-4,
    6.97281375836585777429E-5,
    7.84039221720066627474E-4,
    -2.29472093621399176955E-4,
    -2.68132716049382716049E-3,
    3.47222222222222222222E-3,
    8.33333333333333333333E-2
};
#define MAXSTIR 26.0
/* square root of 2*pi	*/
static float SQTPI = 2.50662827463100050242f;

/* Gamma function computed by Stirling's formula.  */

static float complex cstirf(float complex);

static float complex
cstirf(x)
float complex x;
{
    float complex y, v, w;
    int     i;

    w = 1.0f / x;

    y = STIR[0];
    for (i = 1; i < NSTIR; i++)
    {
	y = y * w + STIR[i];
    }

    w = 1.0f + w * y;
    if (cabsf(x) > MAXSTIR)
    {					/* Avoid overflow in pow() */
	v = cpowf(x, 0.5f * x - 0.25f);
	y = v * (v * cexpf(-x));
    }
    else
    {
	y = cpowf(x, x - 0.5f) * cexpf(-x);
    }
/* y = cpowf( x, x - 0.5f ) * cexpf(-x); */

    y = SQTPI * y * w;
    return (y);
}



float complex
ctgammaf(x)
float complex x;
{
    float   p, q;
    float complex c, u;

    if (fabsf(crealf(x)) > 13.0f)
    {
	if (crealf(x) < 0.0f)
	{
	    q = crealf(x);
	    p = floorf(q);
	    if ((p == q) && (cimagf(x) == 0.0f))
	    {
	      cgoverf:
		// mtherr( "ctgammaf", OVERFLOW );
		return (FLT_MAX + I * FLT_MAX);
	    }
	    /* Compute complex sinf(pi x)  */
	    c = csinf(M_PI * x);
	    /* Reflection formula.  */
	    c = M_PI / (c * ctgammaf(1.0f - x));
	}
	else
	{
	    c = cstirf(x);
	}
	return (c);
    }

    c = 1.0f;
    p = 0.0f;
    u = x;
    while (crealf(u) < 13.0f)
    {
	if ((crealf(u) == 0.0f) && (cimagf(u) == 0.0f))
	    goto cgoverf;
	c *= u;
	p += 1.0f;
	u = x + p;
    }
    u = cstirf(u);
    return (u / c);
}
