/*	ctgammal.c
 *
 *	Complex gamma function, long double precision
 *
 *
 *
 * SYNOPSIS:
 *
 * #include <complex.h>
 * long double complex x, y, cgammal();
 *
 * y = ctgammal( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns complex-valued gamma function of the complex argument.
 *
 * Arguments |Re(x)| <= 20 are increased by recurrence.
 * Large arguments are handled by Stirling's formula. Large negative
 * arguments are made positive using the reflection formula.  
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 * 80-bit long double:
 *    IEEE      -10,10       40000      4.1e-18     7.0e-19
 *    IEEE      -20,20       40000      9.1e-18     1.0e-18
 *    IEEE     -100,100      40000      5.9e-17     7.4e-18
 * 128-bit long double:
 *    IEEE      -10,10       30000      4.9e-32     8.7e-33
 *    IEEE     -100,100      45000      1.2e-31     1.7e-32
 *
 * Error for arguments outside the test range will be larger
 * owing to error amplification by the exponential function.
 *
 */
/*
Cephes Math Library Release 2.1:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
*/

// Modified for DJGPP/GCC by KB Williams,
// kbwms@aol.com, April 2004


#include "complex.h"
#include <float.h>
#include <math.h>

#ifdef LD128BITS
#define NGITER 50.0L
//#define NLGITER 50.0L
//#define LGMXINT 44.4L
#define GSMALL 1.e-17L
#else
#define NGITER 20.0L
//#define NLGITER 16.0L
//#define LGMXINT 78.3L
#define GSMALL 1.e-9L
#endif

/* square root of 2*pi	*/
#define SQTPIL	2.50662827463100050241576528481104525L

/* Gamma function computed by Stirling's formula.  */

static long double complex cstirfl(long double complex);
long double complex
cstirfl(x)
long double complex x;
{
    long double complex y, w;
    int     i;
/* Stirling's formula for the gamma function */
#define NSTIR 18
static const long double STIR[NSTIR] = {
    1.50561130400264244123842218771311273E-2L,
    1.79540117061234856107699407722226331E-1L,
    -2.48174360026499773091565836874346432E-3L,
    -2.95278809456991205054406510546938244E-2L,
    5.40164767892604515180467508570241736E-4L,
    6.40336283380806979482363809026579583E-3L,
    -1.62516262783915816898635123980270998E-4L,
    -1.91443849856547752650089885832852254E-3L,
    7.20489541602001055908571930225015052E-5L,
    8.39498720672087279993357516764983445E-4L,
    -5.17179090826059219337057843002058823E-5L,
    -5.92166437353693882864836225604401187E-4L,
    6.97281375836585777429398828575783308E-5L,
    7.84039221720066627474034881442288850E-4L,
    -2.29472093621399176954732510288065844E-4L,
    -2.68132716049382716049382716049382716E-3L,
    3.47222222222222222222222222222222222E-3L,
    8.33333333333333333333333333333333333E-2L,
};


    w = 1.0L / x;

    y = STIR[0];
    for (i = 1; i < NSTIR; i++)
    {
	y = y * w + STIR[i];
    }

    w = 1.0L + w * y;
#if 1
    y = cpowl(x, x - 0.5L) * cexpl(-x);
#else
    y = (x - 0.5L) * clogl(x) - x;
    y = cexpl(y);
#endif
    y = SQTPIL * y * w;
    return (y);
}



long double complex
ctgammal(x)
long double complex x;
{
    long double complex c, u;
    long double p, q;
    int     cj;

    cj = 0;
    if (cimagl(x) < 0.0L)
    {
	cj = 1;
	x = conj(x);
    }

    q = creall(x);
    if (fabsl(q) > NGITER)
    {
	if (q < 0.0L)
	{
	    p = floorl(q);
	    if ((p == q) && (cimagl(x) == 0.0L))
	    {
		//mtherr("cgammal", OVERFLOW);
		c = LDBL_MAX + I * LDBL_MAX;
		goto gamdone;
	    }
	    /* Compute complex sinl(pi x)  */
	    c = csinl(M_PIl * x);
	    /* Reflection formula.  */
	    c = M_PIl / (c * ctgammal(1.0L - x));
	}
	else
	{
	    c = cstirfl(x);
	}
	goto gamdone;
    }

    c = 1.0L;
    p = 0.0L;
    u = x;
    while (creall(u) < NGITER)
    {
	if ((fabsl(creall(u)) < GSMALL) && (fabsl(cimagl(u)) < GSMALL))
	    goto small;
	c *= u;
	p += 1.0L;
	u = x + p;
    }
    u = cstirfl(u);
    c = u / c;
    goto gamdone;


  small:
    if ((x) == 0.0L)
    {
	//mtherr("cgammal", SING);
	c = LDBL_MAX + LDBL_MAX * I;
	goto gamdone;
    }
    else
	c =
	    1.0L / (((1.0L +
		    0.57721566490153286060651209008240243L * u) * u) * c);

  gamdone:

    if (cj)
	c = conj(c);
    return (c);
}
