/*
 * Board.cs - Handle the playing board.
 *
 * Copyright (C) 2003  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Mahjongg
{

using System;

// This class implements the Mahjongg board logic, including where
// each piece is placed and detection of whether pieces can be selected.

public sealed class Board
{
    // Internal state.
    private Tile[,,] board;
    private int numTilesLeft;

    // Definition of the board layout.  'X' = normal piece,
    // '^' = piece half line up, '.' = piece half position across and up.
    private const String layer1 =
	    " XXXXXXXXXXXX  " +
	    "   XXXXXXXX    " +
	    "  XXXXXXXXXX   " +
	    " XXXXXXXXXXXX  " +
	    "^XXXXXXXXXXXX^^" +
	    "  XXXXXXXXXX   " +
	    "   XXXXXXXX    " +
	    " XXXXXXXXXXXX  ";
    private const String layer2 =
	    "               " +
	    "    XXXXXX     " +
	    "    XXXXXX     " +
	    "    XXXXXX     " +
	    "    XXXXXX     " +
	    "    XXXXXX     " +
	    "    XXXXXX     " +
	    "               ";
    private const String layer3 =
	    "               " +
	    "               " +
	    "     XXXX      " +
	    "     XXXX      " +
	    "     XXXX      " +
	    "     XXXX      " +
	    "               " +
	    "               ";
    private const String layer4 =
	    "               " +
	    "               " +
	    "               " +
	    "      XX       " +
	    "      XX       " +
	    "               " +
	    "               " +
	    "               ";
    private const String layer5 =
	    "               " +
	    "               " +
	    "               " +
	    "               " +
	    "      .        " +
	    "               " +
	    "               " +
	    "               ";

    // Constructor.
    public Board()
	    	{
				Init();
	    	}

    // Re-initialize the board for a new game.
    public void Init()
	    	{
				// Create the board array: 5 layers, 16 rows, 30 columns.
				board = new Tile [5, 16, 30];
				ClearBoard();

				// Shuffle the tiles into their placement order.
				Tile[] tiles = ShuffleTiles();

				// Position the tiles.
				int tile = 0;
				PlaceLayer(tiles, ref tile, layer1, 0);
				PlaceLayer(tiles, ref tile, layer2, 1);
				PlaceLayer(tiles, ref tile, layer3, 2);
				PlaceLayer(tiles, ref tile, layer4, 3);
				PlaceLayer(tiles, ref tile, layer5, 4);

				// We start with 144 tiles.
				numTilesLeft = 144;
	    	}

    // Build the full list of tiles and shuffle them.
    private Tile[] ShuffleTiles()
	    	{
				Tile[] tiles = new Tile [144];
				int posn = 0;
				Tile tile;

				// Populate the tile list.
				for(tile = Tile.Char_1; tile <= Tile.Bamboo_9; ++tile)
				{
		    		tiles[posn++] = tile;
		    		tiles[posn++] = tile;
		    		tiles[posn++] = tile;
		    		tiles[posn++] = tile;
				}
				for(tile = Tile.Season_1; tile <= Tile.Season_4; ++tile)
				{
		    		tiles[posn++] = tile;
				}
				for(tile = Tile.Wind_North; tile <= Tile.Dragon_White; ++tile)
				{
		    		tiles[posn++] = tile;
		    		tiles[posn++] = tile;
		    		tiles[posn++] = tile;
		    		tiles[posn++] = tile;
				}
				for(tile = Tile.Flower_1; tile <= Tile.Flower_4; ++tile)
				{
		    		tiles[posn++] = tile;
				}

				// Shuffle the deck.
				Random rand = new Random();
				int swapWith;
				for(posn = 0; posn < 144; ++posn)
				{
		    		swapWith = rand.Next(0, 143);
		    		tile = tiles[posn];
		    		tiles[posn] = tiles[swapWith];
		    		tiles[swapWith] = tile;
				}
				return tiles;
	    	}

    // Clear all board locations to "no tile.
    private void ClearBoard()
	    	{
				int x, y, layer;
				for(layer = 0; layer < 5; ++layer)
				{
		    		for(y = 0; y < 16; ++y)
		    		{
						for(x = 0; x < 30; ++x)
						{
			    			board[layer, y, x] = Tile.None;
						}
		    		}
				}
	    	}

    // Place a layer of tiles on the board.
    private void PlaceLayer(Tile[] tiles, ref int tile, String layer, int num)
	    	{
				int posn = 0;
				int x, y;
				for(y = 0; y < 8; ++y)
				{
		    		for(x = 0; x < 15; ++x)
		    		{
						switch(layer[posn++])
						{
			    			case 'X':
			    			{
								board[num, y * 2, x * 2] = tiles[tile];
								board[num, y * 2, x * 2 + 1]
				    				= Tile.TilePartLeft;
								board[num, y * 2 + 1, x * 2]
				    				= Tile.TilePartUp;
								board[num, y * 2 + 1, x * 2 + 1]
				    				= Tile.TilePartLeftUp;
								++tile;
			    			}
			    			break;

			    			case '^':
			    			{
								board[num, y * 2 - 1, x * 2] = tiles[tile];
								board[num, y * 2 - 1, x * 2 + 1]
				    				= Tile.TilePartLeft;
								board[num, y * 2, x * 2]
				    				= Tile.TilePartUp;
								board[num, y * 2, x * 2 + 1]
				    				= Tile.TilePartLeftUp;
								++tile;
			    			}
			    			break;

			    			case '.':
			    			{
								board[num, y * 2 - 1, x * 2 + 1] = tiles[tile];
								board[num, y * 2 - 1, x * 2 + 2]
				    				= Tile.TilePartLeft;
								board[num, y * 2, x * 2 + 1]
				    				= Tile.TilePartUp;
								board[num, y * 2, x * 2 + 2]
				    				= Tile.TilePartLeftUp;
								++tile;
			    			}
			    			break;
						}
		    		}
				}
	    	}

    // Find the layer that contains a tile at a position.  -1 if no tile,
    // or the tile cannot be selected.
    private int FindLayer(ref int x, ref int y)
	    	{
				int layer, nlayer;

				// Work down the layers until we find a tile.
				for(layer = 4; layer >= 0; --layer)
				{
		    		if(board[layer, y, x] != Tile.None)
		    		{
						break;
		    		}
				}
				if(layer < 0)
				{
		    		return -1;
				}

				// Find the upper-left corner of the tile.
				switch(board[layer, y, x])
				{
		    		case Tile.TilePartLeft:	--x; break;
		    		case Tile.TilePartUp:	--y; break;
		    		case Tile.TilePartLeftUp:   --x; --y; break;
				}

				// Check that the tile is not covered.
				for(nlayer = layer + 1; nlayer < 5; ++nlayer)
				{
		    		if(board[nlayer, y, x] != Tile.None)
		    		{
						return -1;
		    		}
		    		if(board[nlayer, y, x + 1] != Tile.None)
		    		{
						return -1;
		    		}
		    		if(board[nlayer, y + 1, x] != Tile.None)
		    		{
						return -1;
		    		}
		    		if(board[nlayer, y + 1, x + 1] != Tile.None)
		    		{
						return -1;
		    		}
				}

				// Check that the tile has no more than 1 sibling.
				int siblings = 0;
				if(x > 0)
				{
		    		if(board[layer, y, x - 1] != Tile.None)
		    		{
						++siblings;
		    		}
		    		else if(board[layer, y + 1, x - 1] != Tile.None)
		    		{
						++siblings;
		    		}
				}
				if((x + 2) < 30)
				{
		    		if(board[layer, y, x + 2] != Tile.None)
		    		{
						++siblings;
		    		}
		    		else if(board[layer, y + 1, x + 2] != Tile.None)
		    		{
						++siblings;
		    		}
				}
				if(siblings >= 2)
				{
					return -1;
				}

				// Return the layer that contains the tile that we found.
				return layer;
	    	}

    // Determine if a particular location on the board is selectable,
    // and return the tile at that location.  Returns "Tile.None" if
    // it isn't possible to select the location.
    public Tile CanSelect(int layer, int x, int y)
	    	{
				int flayer = FindLayer(ref x, ref y);
				if(flayer == layer)
				{
		    		return board[layer, y, x];
				}
				else
				{
		    		return Tile.None;
				}
	    	}

    // Get the layer of a selectable piece.  -1 if no selectable piece.
    public int GetLayer(int x, int y)
	    	{
				return FindLayer(ref x, ref y);
	    	}

    // Adjust the co-ordinates of a location to get the upper-left
    // corner of the tile at that location.
    public void Adjust(int layer, ref int x, ref int y)
	    	{
				switch(board[layer, y, x])
				{
		    		case Tile.TilePartLeft:	--x; break;
		    		case Tile.TilePartUp:	--y; break;
		    		case Tile.TilePartLeftUp:   --x; --y; break;
				}
	    	}

    // Get the tile at a particular location, or Tile.None if none.
    public Tile GetTile(int layer, int x, int y)
	    	{
				Adjust(layer, ref x, ref y);
				return board[layer, y, x];
	    	}
    public Tile GetTileDirect(int layer, int x, int y)
	    	{
				return board[layer, y, x];
	    	}

    // Remove a tile from a particular location.
    public void Remove(int layer, int x, int y)
	    	{
				Adjust(layer, ref x, ref y);
				board[layer, y, x] = Tile.None;
				board[layer, y, x + 1] = Tile.None;
				board[layer, y + 1, x] = Tile.None;
				board[layer, y + 1, x + 1] = Tile.None;
				--numTilesLeft;
	    	}

    // Find the location of two tiles that match, for hinting.
    // Returns false if nothing was found.
    public bool Hint(out int layer1, out int x1, out int y1,
		             out int layer2, out int x2, out int y2)
	    	{
				// TODO
				return false;
	    	}

    // Determine if the game is over due to the user winning.
    public bool UserWon
	    	{
				get
				{
		    		return (numTilesLeft == 0);
				}
	    	}

    // Determine if the game is over due to there being no more
    // tiles that can be matched.
    public bool UserLost
	    	{
				get
				{
		    		int layer1 = 0, x1 = 0, y1 = 0;
		    		int layer2 = 0, x2 = 0, y2 = 0;

		    		// If there are no tiles left, then the user didn't lose.
		    		if(numTilesLeft == 0)
		    		{
						return false;
		    		}

		    		// The user lost if we can't find any hints.
		    		return !Hint(out layer1, out x1, out y1,
				 				 out layer2, out x2, out y2);
				}
	    	}

}; // class Board

}; // namespace Mahjongg
