;; $Header: /cvsroot/erc/erc/erc.el,v 1.45 2001/09/10 15:15:18 mlang Exp $

;; erc.el - an Emacs IRC client

;; Author: Alexander L. Belikoff (abel@bfr.co.il)
;; Contributors: Sergey Berezin (sergey.berezin@cs.cmu.edu),
;;               Mario Lang (mlang@delysid.org),
;;               Alex Schroeder (alex@gnu.org)
;; Maintainers: Mario Lang (mlang@delysid.org), Alex Schroeder (alex@gnu.org)
;; Version: 2.1 ($Revision: 1.45 $)
;; Keywords: IRC, client, Internet

;; Copyright (C) 1997 Alexander L. Belikoff

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;; Commentary:

;; ERC is an IRC client for Emacs.

;; For more information, see the following URLs:
;; * http://www.sourceforge.net/projects/erc/
;; * http://www.emacswiki.org/cgi-bin/wiki.pl?EmacsIRCClient

;; Jul-26-2001. erc.el is now in CVS on SourceForge. I invite everyone
;; who wants to hack it to contact me <mlang@delysid.org> in order to
;; get write access on the CVS.

;; Planned: (please add your wildest ideas here)
;; * revise erc-keymap: There are still C-c c like keybindings used
;;   and some interactive functions are not bound.
;; * Find out a clean way to impelement functions like erc-join-channel
;;   without that overhead we have currently.
;; * Add content-filter to be able to filter on message content based on
;;   channel, nick or all.
;; * use the Emacs 21 header-line for displaying channel topic
;; * Make erc-mode-line customizable through a variable erc-mode-line-format(s)?
;; * Write DCC support using netcat for server ports.

;; Installation:

;; Put erc.el in your load-path, and put (require 'erc) in your .emacs.

;; Configuration:

;; Use M-x customize-group RET erc RET to get an overview
;; of all the variables you can tweak.

;; Usage:

;; To connect to an IRC server, do
;;
;; M-x erc-select RET
;;
;; After you are connected to a server, you can use C-h m or have a look at
;; the not (yet) completely finished IRC menu.

;; Changelog:

;; Since the original 2.0 version of Alexander and Sergey:
;;
;; * Make user-defineable variables customizable (defvar->defcustom).
;; * Removable of the dependency for edo-tools.el
;; * Added IRC menu.
;; * Slightly changed key-bindings in erc-mode.
;; * Hidden channel buffer tracking (thanks to Lathi).
;;   see erc-toggle-track-modified-channels
;; * Keyword highlighting
;; * Nickname completion using TAB
;; * Added support for filling messages (see variable erc-filling)
;; * quit-message washing (erc-wash-quit-reason)
;; * And several other tweaks...

;; Code:

(require 'comint)
(require 'font-lock)
(require 'ring)

(defvar erc-version-string "2.0"
  "ERC version. This is used by `erc-version' function")

(defvar erc-official-location "http://cvs.sourceforge.net/cgi-bin/viewcvs.cgi/~checkout~/erc/erc/erc.el?rev=HEAD&content-type=text/plain Comments: mailto://mlang@delysid.org"
  "Location of the ERC client on the Internet")

;;; tunable connection and authentication parameters

(defgroup erc nil
  "Emacs Internet Relay Chat client."
  :link '(url-link "http://www.emacswiki.org/cgi-bin/wiki.pl?EmacsIRCClient")
  :group 'processes)

(defcustom erc-server nil
  "IRC server to use.

See function `erc-compute-server' for more details on connection
parameters and authentication."
  :group 'erc
  :type 'string)

(defcustom erc-port nil
  "IRC port to use."
  :group 'erc
  :type '(choice (const nil)
		 string))

(defcustom erc-nick nil
  "Nickname to use.

Can be either a string, or a list of strings.
In the latter case, if the first nick in the list is already in use,
other nicks are tried in the list order.

See function `erc-compute-nick' for more details on connection
parameters and authentication."
  :group 'erc
  :type '(choice (const nil)
		 (string :tag "Nickname")
		 (repeat string)))

(defcustom erc-user-full-name nil
  "User full name.

See function `erc-compute-full-name' for more details on connection
parameters and authentication."
  :group 'erc
  :type '(choice (const nil) string function)
  :set (lambda (sym val)
	 (if (functionp val)
	     (set sym (funcall val))
	   (set sym val))))


(defvar erc-password nil
  "ERC password to use in authentication (not necessary)")

(defcustom erc-prompt-for-password t
  "Asks before using the default password, or whether to enter a new one."
  :group 'erc
  :type 'boolean)

;; tunable GUI stuff

(defcustom erc-prompt "ERC>"
  "Prompt used by ERC. Trailing whitespace is not required"
  :group 'erc
  :type 'string)

;; Hmm, is this useful at all. If so, we would need to do it better, because
;; one looses nickname completion when turning this variable on.
(defcustom erc-prompt-interactive-input nil
  "*If you set this variable to t, the erc prompt gets a local-map text
property which allows you to enter text in the minbuffer, instead of the 
erc buffer itself."
  :group 'erc
  :type 'boolean)

(defcustom erc-filling nil
  "*If non-nil, ERC tries to do propper word-wrapping."
  :group 'erc
  :type 'boolean)

(defcustom erc-fill-prefix nil
  "fill-prefix for `erc-fill-region'. nil means fill with space,
a string means fill with this string."
  :group 'erc
  :type '(choice (const nil) string))

(defcustom erc-default-fill-column 78
  "Right margin for word-wrap in ERC buffers.
This value is a default, you can always change it through `set-fill-column'
interactively."
  :group 'erc
  :type 'integer)

(defcustom erc-action-separator "> "
  "Separator put between nick and it's action"
  :group 'erc
  :type 'string)

(defcustom erc-next-line-add-newlines nil
  "*If non-nil, \\[next-line] (`next-line') inserts newline at end of buffer."
  :group 'erc
  :type 'boolean)

(defcustom erc-timestamp-format nil
  "If non-nil, then it is a time format string, to be used by
`format-time-string' function and displayed with every message in the
ERC buffers. Value `nil' means do not timestamp messages.

Good examples are \"%T \" and \"%H:%M \"."
  :group 'erc
  :type '(choice (const nil)
		 (string)))

(defcustom erc-notice-prefix "*** "
  "Prefix for all notices."
  :group 'erc
  :type 'string)

(defcustom erc-notice-highlight-type 'all
  "Determines how to highlight notices. The following values are
allowed:

    'prefix - highlight notice prefix only
    'all    - highlight the entire notice

Any other value disables notice's highlighting altogether."
  :group 'erc
  :type '(choice (const :tag "highlight notice prefix only" prefix)
		 (const :tag "highlight the entire notice" all)
		 (const :tag "don't highlight notices at all" nil)))

(defcustom erc-pal-highlight-type 'nick
  "Determines how to highlight pal's messages. The following values are
allowed:

    'nick - highlight pal's nickname only
    'all  - highlight the entire message from pal

Any other value disables pal highlighting altogether."
  :group 'erc
  :type '(choice (const :tag "highlight pal's nickname only" nick)
		 (const :tag "highlight the entire message" all)
               (const :tag "don't highlight pal's messages" nil)))

(defvar erc-uncontrol-input-line t
  "Determines whether to interpret the control characters on the input line
before rewriting it.

The \"uncontrolled\" line usually looks much nicer (with bold and
other fonts).  There is no point in disabling it any more, since all
the color and font information is now preserved.  Consider it
depricated.")

(defvar erc-interpret-controls-p t 
  "Whether to interpret the colors and other highlighting info or not.
The interpreting can require a lot of resources and in chatty
channels, or in an emergency (message flood) it can be turned off to
save processing time.")

(defvar erc-multiline-input nil
  "Send multiple lines from the last `erc-prompt' to the end of buffer
  as a single message.  Depricated, do not use it.")

;; other tunable parameters

(defcustom erc-auto-discard-away t
  "If non-NIL, then sending anything, while being AWAY, automatically
discards the AWAY state"
  :group 'erc
  :type 'boolean)

(defcustom erc-play-sound t
  "*Play sound on SOUND ctcp requests (used in ICQ chat)"
  :group 'erc
  :type 'boolean)

(defcustom erc-sound-path nil
  "List of directories that contain sound samples to play on SOUND events."
  :group 'erc
  :type '(repeat directory))

(defcustom erc-default-sound nil
  "Play this sound if the requested file was not found"
  :group 'erc
  :type '(choice (const nil)
		 file))

(defcustom erc-play-command "play"
  "Command for playing sound samples"
  :group 'erc
  :type 'string)

(defcustom erc-page-function nil
  "A function name to process a \"page\" request.  The function must
take two arguments: SENDER and MSG, both strings.  Value `nil' for
this variable will cause the page message to appear in the minibuffer
window."
  :group 'erc
  :type '(choice (const nil)
		 (function-item)))

(defcustom erc-paranoid nil
  "Paranoid mode. If non-nil, then all incoming CTCP requests (will be
shown, except visible ones (like ACTION and SOUND)."
  :group 'erc
  :type 'boolean)

(defcustom erc-disable-ctcp-replies nil
  "If non-nil, then all incoming CTCP requests that normally require
an automatic reply (like VERSION or PING) will be ignored.  Good to
set if some hacker is trying to flood you away."
  :group 'erc
  :type 'boolean)

(defcustom erc-anonymous-login t
  "Be paranoid, don't give away your machine name"
  :group 'erc
  :type 'boolean)

(defcustom erc-email-userid "user"
  "Use this as your email user ID"
  :group 'erc
  :type 'string)

(defcustom erc-pals nil
  "List of pals on IRC."
  :group 'erc
  :type '(repeat string))

(defcustom erc-highlight-strings nil
  "List of regular expressions to highlight in all incoming messages."
  :group 'erc
  :type '(repeat string))

(defcustom erc-host-danger-highlight nil
  "List of regexps for hosts to highlight as 'dangerous',
like offenders, etc."
  :group 'erc
  :type '(repeat regexp))

(defcustom erc-ignore-list nil 
  "The list of regular expressions to match against identities of the
form `nick!login@host'.  If an id matches, the message from the person
will not be processed."
  :group 'erc
  :type '(repeat regexp))

(defvar erc-flood-protect 'normal
  "Enables flood protection when non-nil.  Flooding is sending too
much information to the server in too short time, which may result in
loosing connection.  

If the value is 'strict, use a more strict limits provided in
`erc-flood-limit2', otherwise use \"normal\" limits in `erc-flood-limit'.")

(defvar erc-flood-limit '(1000 25 5) 
  "Is a 3-element list (BYTES LINES SEC), defining the flood threshold:
at most BYTES bytes or LINES lines in messages within SEC seconds from
each other.  When either byte or line limit is exceeded, ERC stops
sending anything to the server, except for pings and one-line manual
user's commands.")

(defvar erc-flood-limit2 '(300 10 5)
  "Similar to `erc-flood-protect', but normally much more strict.  It
will be used instead of `erc-flood-protect' in critical situations
(detected flood, explicit user request, etc.).  Currently, it is
turned on when the flood limit is exceeded for the first time.")

(defcustom erc-startup-file-list
  '("~/.ercrc.el" "~/.ercrc" "~/.ircrc" ".ercrc.el" ".ercrc")
  "List of files to try for a startup script. The first existant and
readable one will get executed.

If the filename ends with `.el' it is presumed to be an emacs-lisp
script and it gets (load)ed. Otherwise is is treated as a bunch of
regular IRC commands"
  :group 'erc
  :type '(repeat file))

(defcustom erc-script-path nil 
  "List of directories to look for a script in /load command.  The
script is first searched in the current directory, then in each
directory in the list."
  :group 'erc
  :type '(repeat directory))

(defcustom erc-script-echo t
  "*If not-NIL, echo the IRC script commands locally."
  :group 'erc
  :type 'boolean)

(defcustom erc-max-buffer-size 30000
  "*Maximum size of each ERC buffer. Used only when auto-truncation is enabled.
(see `erc-truncate-buffer' and `erc-insert-hook')."
  :group 'erc
  :type 'integer)

(defcustom erc-log-channels-directory nil
  "The directory to place log files for channels.  If not nil, all the
channel buffers are logged in separate files in that directory.  The
directory must exist, it will not be created."
  :group 'erc
  :type '(choice (const nil)
		 (directory :must-match)))

(defcustom erc-log-channels nil
  "Enables/disables logging of channel buffers.  Nil means do not log."
  :group 'erc
  :type 'boolean)

(defvar erc-grab-buffer-name "*erc-grab*"
  "The name of the buffer created by `erc-grab-region'.")

;; variables available for IRC scripts

(defvar erc-user-information "ERC User"
  "USER_INFORMATION IRC variable")

(defvar erc-autogreet-script "hello.el"
  "Script name to use for autogreeting.")

;;; Hooks

(defcustom erc-insert-hook nil
  "Hook to run on any text insertion into an ERC buffer.  Put
`erc-truncate-buffer' here to prevent unbounded buffer growth."
  :group 'erc
  :type 'hook
  :options '(erc-truncate-buffer))

(defcustom erc-message-hook nil
  "Run this hook on every message that arrives, from a channel or a
person, with arguments (MESSAGE TARGET NICK BUFFER-NAME IP LOGIN
SPEC).  The TARGET is either a channel name or your current nick (see
`erc-current-nick').  In the last case, it is a private message."
  :group 'erc
  :type 'hook)

(defcustom erc-join-hook nil
  "Run this hook with arguments (channel nick buffer IP login sspec)
on every join event, yours or someone else's.  Distinguish this by
comparing your nick with the argument."
  :group 'erc
  :type 'hook)

(defcustom erc-part-hook nil
  "Run this hook with arguments (channel nick buffer IP reason) on every part
event, yours or someone else's.  Distinguish this by comparing your nick
with the argument."
  :group 'erc
  :type 'hook)

(defcustom erc-quit-hook nil
  "Run this hook with arguments (nick IP reason) on every quit
event, yours or someone else's.  Distinguish this by comparing your nick
with the argument."
  :group 'erc
  :type 'hook)

(defcustom erc-disconnected-hook nil
  "Run this hook with arguments (nick IP reason) when disconnected,
before automatic reconnection.  Note, that `erc-quit-hook' might not
be run when we disconnect, simply because we do not necessarily
receive the QUIT event."
  :group 'erc
  :type 'hook)

(defcustom erc-kick-hook nil
  "Run this hook with arguments (target channel nick reason host login sspec)
on every kick event.  The target is the nick being kicked, all the
other info is for the kicker."
  :group 'erc
  :type 'hook)

(defcustom erc-mode-change-hook nil
  "Run this hook with arguments (nick target mode-string host login
sspec) on every mode change event.  The target is the nick or channel
whose mode is being changed, all the other info is for the person who
changed it."
  :group 'erc
  :type 'hook)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; This variable went buffer local for each connection
;(defvar erc-buffer-list nil
;  "The list of ERC buffers.  Dead (killed) buffers are gradually 
;  pruned away, but do not rely on all of them being alive.")

;; mode-specific tables

(defvar erc-mode-syntax-table nil
  "Syntax table used while in ERC mode.")

(cond ((not erc-mode-syntax-table)
       (setq erc-mode-syntax-table (make-syntax-table))
       (modify-syntax-entry ?\" ".   " erc-mode-syntax-table)
       (modify-syntax-entry ?\\ ".   " erc-mode-syntax-table)
       (modify-syntax-entry ?' "w   " erc-mode-syntax-table)
       ;; Make dabbrev-expand useful for nick names
       (modify-syntax-entry ?< "." erc-mode-syntax-table)
       (modify-syntax-entry ?> "." erc-mode-syntax-table)))

(defvar erc-mode-abbrev-table nil
  "Abbrev table used while in ERC mode.")

(define-abbrev-table 'erc-mode-abbrev-table ())

(defvar erc-mode-map nil)
(defvar erc-info-mode-map nil)

(cond ((not erc-mode-map)
       (setq erc-mode-map (make-sparse-keymap))
       (define-key erc-mode-map [(control return)] 'erc-send-paragraph)
       (if (not erc-multiline-input)
	   (define-key erc-mode-map "\C-m" 'erc-send-current-line))
       (define-key erc-mode-map "\M-p" 'erc-previous-command)
       (define-key erc-mode-map "\M-n" 'erc-next-command)
       (define-key erc-mode-map "\t" 'erc-complete)
       (define-key erc-mode-map "\C-cg" 'erc-grab-region)
       (define-key erc-mode-map "\C-ca" 'erc-input-action)
       (define-key erc-mode-map "\C-cs" 'erc-toggle-sound)
       (define-key erc-mode-map "\C-cc" 'erc-toggle-ctcp-autoresponse)
       (define-key erc-mode-map "\C-c\C-a" 'erc-bol)
       (define-key erc-mode-map "\C-c\C-c" 'erc-toggle-interpret-controls)
       (define-key erc-mode-map "\C-c\C-f" 'erc-toggle-flood-control)
       (define-key erc-mode-map "\C-c\C-i" 'erc-invite-only-mode)
       (define-key erc-mode-map "\C-c\C-j" 'erc-join-channel)
       (define-key erc-mode-map "\C-c\C-l" 'erc-save-buffer-in-logs)
       (define-key erc-mode-map "\C-c\C-m" 'erc-insert-mode-command)
       (define-key erc-mode-map "\C-c\C-n" 'erc-channel-names)
       (define-key erc-mode-map "\C-c\C-p" 'erc-part-from-channel)
       (define-key erc-mode-map "\C-c\C-r" 'erc-remove-text-properties-region)
       (define-key erc-mode-map "\C-c\C-t" 'erc-set-topic)
       ;; Menu-bar
       (define-key erc-mode-map [menu-bar irc]
	 (cons "IRC" (make-sparse-keymap "IRC")))
       (define-key erc-mode-map [menu-bar irc action]
	 '("Input Action" . erc-input-action))
       (define-key erc-mode-map [menu-bar irc join]
	 '("Join Channel" . erc-join-channel))
       (define-key erc-mode-map [menu-bar irc topic]
	 '("Set channel topic" . erc-set-topic))
       (define-key erc-mode-map [menu-bar irc leave]
	 '("Leave current channel" . erc-part-from-channel))
       (define-key erc-mode-map [menu-bar irc add-pal]
	 '("Add pal" . erc-add-pal))
       (define-key erc-mode-map [menu-bar irc delete-pal]
	 '("Delete pal" . erc-delete-pal))
       (define-key erc-mode-map [menu-bar irc mode]
	 '("Change mode" . erc-insert-mode-command))
       (define-key erc-mode-map [menu-bar irc invite-only]
	 '("Invite only" . erc-invite-only-mode))
       (define-key erc-mode-map [menu-bar irc select]
	 '("Select server" . erc-select))
       (define-key erc-mode-map [menu-bar irc toggle]
	 (cons "Toggle..." (make-sparse-keymap "IRC")))
       (define-key erc-mode-map [menu-bar irc toggle tracking]
	 '("Track hidden channel buffers" . erc-toggle-track-modified-channels))))

;; keymap for channel info buffers.  Empty at the moment.
(cond ((not erc-info-mode-map)
       (setq erc-info-mode-map (make-sparse-keymap))))


;; faces

;; Honestly, I have a horrible sense of color and the "defaults" below
;; are supposed to be really bad. But colors ARE required in IRC to
;; convey different parts of conversation. If you think you know better
;; defaults - send them to me.

;; Now colors are a bit nicer, at least to my eyes. 
;; You may still want to change them to better fit your background.-- S.B.

(defgroup erc-faces nil
  "Faces for ERC."
  :group 'erc)

(defface erc-default-face '((t)) "Default face." :group 'erc-faces)
  "ERC default face."
  :group
(defface erc-direct-msg-face '((t (:foreground "IndianRed")))
  "ERC face used for messages you receive in the main erc buffer."
  :group 'erc-faces)
(defface erc-input-face '((t (:foreground "brown")))
  "ERC face used for your input."
  :group 'erc-faces)
(defface erc-bold-face '((t (:bold t)))
  "ERC bold face."
  :group 'erc-faces)
(defface erc-inverse-face
  '((t (:foreground "White" :background "Black")))
  "ERC inverse face."
  :group 'erc-faces)
(defface erc-underline-face '((t (:underline t))) 
  "ERC underline face."
  :group 'erc-faces)
(defface erc-prompt-face
  '((t (:bold t :foreground "Black" :background"lightBlue2")))
  "ERC face for the prompt."
  :group 'erc-faces)
(defface erc-notice-face '((t (:bold t :foreground "SlateBlue")))
  "ERC face for notices."
  :group 'erc-faces)
(defface erc-action-face '((t (:bold t)))
  "ERC face for actions generated by /ME."
  :group 'erc-faces)
(defface erc-error-face '((t (:foreground "White" :background "Red")))
  "ERC face for errors."
  :group 'erc-faces)
(defface erc-host-danger-face '((t (:foreground "red")))
  "ERC face for people on dangerous hosts.
See `erc-host-danger-highlight'."
  :group 'erc-faces)
(defface erc-pal-face '((t (:bold t :foreground "Magenta")))
  "ERC face for your pals.
See `erc-pals'."
  :group 'erc-faces)
(defface erc-highlight-face '((t (:bold t :foreground "pale green")))
  "ERC face for you favorite regular expressions.
See `erc-highlight-strings'."
  :group 'erc-faces)

(defface fg:erc-color-face0 '((t (:foreground "White")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face1 '((t (:foreground "black")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face2 '((t (:foreground "blue4")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face3 '((t (:foreground "green4")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face4 '((t (:foreground "red")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face5 '((t (:foreground "brown")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face6 '((t (:foreground "purple")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face7 '((t (:foreground "orange")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face8 '((t (:foreground "yellow")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face9 '((t (:foreground "green")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face10 '((t (:foreground "lightblue1")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face11 '((t (:foreground "cyan")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face12 '((t (:foreground "blue")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face13 '((t (:foreground "deeppink")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face14 '((t (:foreground "gray50")))
  "ERC face."
  :group 'erc-faces)
(defface fg:erc-color-face15 '((t (:foreground "gray90")))
  "ERC face."
  :group 'erc-faces)

(defface bg:erc-color-face0 '((t (:background "White")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face1 '((t (:background "black")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face2 '((t (:background "blue4")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face3 '((t (:background "green4")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face4 '((t (:background "red")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face5 '((t (:background "brown")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face6 '((t (:background "purple")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face7 '((t (:background "orange")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face8 '((t (:background "yellow")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face9 '((t (:background "green")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face10 '((t (:background "lightblue1")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face11 '((t (:background "cyan")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face12 '((t (:background "blue")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face13 '((t (:background "deeppink")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face14 '((t (:background "gray50")))
  "ERC face."
  :group 'erc-faces)
(defface bg:erc-color-face15 '((t (:background "gray90")))
  "ERC face."
  :group 'erc-faces)

(defun erc-get-bg-color-face (n)
  "Fetches the right face for the color number 0-15."
  (if (stringp n) (setq n (string-to-number n)))
  (if (not (numberp n))
      (progn 
	(message (format "erc-get-bg-color-face: n is NaN: %S" n))
	(beep)
	'default)
    (if (> n 16) 
	(progn
	  (erc-log (format "   Wrong color: %s" n))
	  (setq n (mod n 16))))
    (cond 
     ((= n 0) 'bg:erc-color-face0)
     ((= n 1) 'bg:erc-color-face1)
     ((= n 2) 'bg:erc-color-face2)
     ((= n 3) 'bg:erc-color-face3)
     ((= n 4) 'bg:erc-color-face4)
     ((= n 5) 'bg:erc-color-face5)
     ((= n 6) 'bg:erc-color-face6)
     ((= n 7) 'bg:erc-color-face7)
     ((= n 8) 'bg:erc-color-face8)
     ((= n 9) 'bg:erc-color-face9)
     ((= n 10) 'bg:erc-color-face10)
     ((= n 11) 'bg:erc-color-face11)
     ((= n 12) 'bg:erc-color-face12)
     ((= n 13) 'bg:erc-color-face13)
     ((= n 14) 'bg:erc-color-face14)
     ((= n 15) 'bg:erc-color-face15)
     (t (erc-log (format "   Wrong color: %s" n)) 'default))))

(defun erc-get-fg-color-face (n)
  "Fetches the right face for the color number 0-15."
  (if (stringp n) (setq n (string-to-number n)))
  (if (not (numberp n))
      (progn 
	(message (format "erc-get-fg-color-face: n is NaN: %S" n))
	(beep)
	'default)
    (if (> n 16) 
	(progn
	  (erc-log (format "   Wrong color: %s" n))
	  (setq n (mod n 16))))
    (cond 
     ((= n 0) 'fg:erc-color-face0)
     ((= n 1) 'fg:erc-color-face1)
     ((= n 2) 'fg:erc-color-face2)
     ((= n 3) 'fg:erc-color-face3)
     ((= n 4) 'fg:erc-color-face4)
     ((= n 5) 'fg:erc-color-face5)
     ((= n 6) 'fg:erc-color-face6)
     ((= n 7) 'fg:erc-color-face7)
     ((= n 8) 'fg:erc-color-face8)
     ((= n 9) 'fg:erc-color-face9)
     ((= n 10) 'fg:erc-color-face10)
     ((= n 11) 'fg:erc-color-face11)
     ((= n 12) 'fg:erc-color-face12)
     ((= n 13) 'fg:erc-color-face13)
     ((= n 14) 'fg:erc-color-face14)
     ((= n 15) 'fg:erc-color-face15)
     (t (erc-log (format "   Wrong color: %s" n)) 'default))))

;; debugging support

(defvar erc-log-p nil
  "Do debug logging if non-NIL")

(defvar erc-debug-log-file
  (concat (expand-file-name ".") "/ERC.debug")

  "Debug log file name")


(defmacro erc-log (string)
  "Logs STRING if logging is on (see `erc-log-p')"

  `(if erc-log-p
       (erc-log-aux ,string)))


(defun erc-log-aux (string)
  "Do the debug logging of STRING"

  (let ((cb (current-buffer))
	(point 1)
	(was-eob nil)
	(session-buffer (and (boundp 'erc-process)
			     (processp erc-process)
			     (process-buffer erc-process))))
    (if session-buffer
	(progn
	  (set-buffer session-buffer)
	  (if (not (and erc-dbuf (bufferp erc-dbuf) (buffer-live-p erc-dbuf)))
	      (progn
		(make-variable-buffer-local 'erc-dbuf)
		(setq erc-dbuf (get-buffer-create 
				(concat "*ERC-DEBUG: " 
					erc-session-server "*")))))
	  (set-buffer erc-dbuf)
	  (setq point (point))
	  (setq was-eob (eobp))
	  (goto-char (point-max))
	  (insert (concat "** " string "\n"))
	  (if was-eob (goto-char (point-max))
	    (goto-char point))
	  (set-buffer cb))
      (message (concat "ERC: ** " string)))))

;; Last active buffer, to print server messages in the right place

(defvar erc-active-buffer nil
  "The current active buffer, the one where the user typed the last
command." )

;; now the mode activation routines

;;;###autoload
(defun erc-mode ()
  "Major mode for Emacs IRC
Special commands:

\\{erc-mode-map}

Turning on text-mode runs the hook `erc-mode-hook'."
  (interactive)
  (kill-all-local-variables)
  (use-local-map erc-mode-map)
  (setq mode-name "ERC")
  (setq major-mode 'erc-mode)
  (setq local-abbrev-table erc-mode-abbrev-table)
  (set-syntax-table erc-mode-syntax-table)
  (make-variable-buffer-local 'next-line-add-newlines)
  (setq next-line-add-newlines erc-next-line-add-newlines)
  (setq fill-column erc-default-fill-column)
  (make-variable-buffer-local 'paragraph-separate)
  (make-variable-buffer-local 'paragraph-start)
  (setq paragraph-separate (concat "\C-l\\|\\(^" erc-prompt "\\)"))
  (setq paragraph-start (concat "\\(" erc-prompt "\\)"))
  ;; Saving log file on exit
  (if erc-log-channels-directory
      (progn
	(auto-save-mode -1)
	(setq buffer-offer-save t)
	(setq buffer-file-name (concat erc-log-channels-directory "/" 
				       (buffer-name buffer) ".txt"))
	      (if (boundp 'local-write-file-hooks)
		  (setq local-write-file-hooks 
			'(erc-save-buffer-in-logs)) ;Emacs >=19
		(make-local-variable 'write-file-hooks)
		(setq write-file-hooks              ;Emacs 18
		      '(erc-save-buffer-in-logs)))))
  (erc-input-ring-setup)
  ;; Run the mode hooks
  (run-hooks 'erc-mode-hook))

;;;###autoload
(defun erc-info-mode ()
  "Major mode for Emacs IRC Channel Info buffers.
Special commands:

\\{erc-mode-map}."
  
  (interactive)
  (kill-all-local-variables)

  (use-local-map erc-info-mode-map)
  (setq mode-name "ERC Info")
  (setq major-mode 'erc-info-mode)
  (toggle-read-only 1))


;; activation

(defconst erc-default-server "irc.funet.fi"
  "IRC server to use if it cannot be detected otherwise")

(defconst erc-default-port "irc"
  "IRC port to use if it cannot be detected otherwise")


(defcustom erc-join-buffer 'buffer
  "Determines how to display the newly created IRC buffer.
   'window - in another window, 
   'frame - in another frame,
   any other value - in place of the current buffer"
  :group 'erc
  :type '(choice (const frame)
		 (const window)
		 (const buffer)))

(defcustom erc-join-info-buffer 'disable
  "Determines how to display the INFO buffer for a channel on join.
Values can be 'frame - in another frame, 'window - another window in
the same frame, 'split - split the current window in two and display
it in the lower subwindow.  Any other value will leave the info buffer
invisible."
  :group 'erc
  :type '(choice (const frame)
		 (const window)
		 (const split)
		 (const disable)))

(defcustom erc-frame-alist nil
  "*Alist of frame parameters for creating erc frames.
A value of `nil means to use `default-frame-alist'."
  :group 'erc
  :type '(repeat (cons :format "%v"
                       (symbol :tag "Parameter")
                       (sexp :tag "Value"))))

(defcustom erc-frame-dedicated-p nil
  "*Non-nil means the erc frames are dedicated to that buffer.
This only has effect when `erc-join-buffer' and `erc-join-info-buffer'
are set to `frame'."
  :group 'erc
  :type 'boolean)

(defun erc-compute-buffer-name (server &optional tgt nick)
  "Computes the buffer name for the current server SERVER, target TGT,
 and nick NICK.  If TGT and/or NICK are non-nil, then the buffer name
 includes the target and nick.  Thus, every nick has it's own buffer
 for every target (channel)."

  (concat (if tgt (format "%s!" (downcase tgt)) "")
	  (if nick (format "%s@" (downcase nick)) "")
	  (format (if (not (or tgt nick)) "*ERC: %s*" "%s")
		  (downcase server))))

(defun erc-get-buffer-create (server &optional tgt nick)
  "Combines `get-buffer-create' and `erc-compute-buffer-name'.
   Takes SERVER, TGT, and NICK, returns BUFFER object.
   Puts the buffer into the list of ERC buffers, if not there."
  (let ((cb (current-buffer))
	(buf (get-buffer-create (erc-compute-buffer-name  server tgt nick))))
    (if (and (boundp 'erc-process) (processp erc-process))
	(set-buffer (process-buffer erc-process)))
    (if (and (boundp 'erc-buffer-list) (not (member buf erc-buffer-list)))
	(setq erc-buffer-list (cons buf erc-buffer-list)))
    (set-buffer cb)
    buf))

(defun erc-get-buffer (server &optional tgt nick)
  "Combines `get-buffer' and `erc-compute-buffer-name'.
   Takes SERVER, TGT, and NICK, returns BUFFER object."
  (get-buffer (erc-compute-buffer-name server tgt nick)))

(defun erc-prune-dead-buffers (buffers)
  "Leaves only live buffers from BUFFERS and returns the rest in the list."
  (let ((res nil))
    (while buffers
      (if (buffer-live-p (car buffers))
	  (setq res (cons (car buffers) res)))
      (setq buffers (cdr buffers)))
    res))

(defun erc-find-buffer (server &optional tgt buffer-list strict)
  "Finds the right buffer for the SERVER and (optionally) target TGT.
If BUFFER-LIST is supplied, then the buffer is searched in this list.
Otherwise the `erc-buffer-list' is used.  If no buffer can match the
specification, the current buffer is returned.  However, if STRICT is
non-nil, then nil is returned in case there is no buffer that matches
the requirement precisely."
  (let ((cb (current-buffer))
	(session-buf (and (boundp 'erc-process) (processp erc-process)
			  (process-buffer erc-process)))
	(res nil))
    (if session-buf (set-buffer session-buf))
    (if (null buffer-list)
	(if (boundp 'erc-buffer-list) (setq buffer-list erc-buffer-list)
	  (progn 
	    (message "Warning: erc-buffer-list is undefined")
	    (erc-log "Warning: erc-buffer-list is undefined"))))
    (setq buffer-list (erc-prune-dead-buffers buffer-list))
    (setq server (downcase server))
    (if tgt (setq tgt (downcase tgt)))
    ;;(message "buffer-list = %S, server = %s, tgt = %s" buffer-list server tgt)
    (while (and buffer-list (not res))
      (set-buffer (car buffer-list))
;	  (message "buf=%s, server=%s, tgt=%s" (current-buffer) erc-session-server (erc-default-target))
      (if (and (string= (downcase erc-session-server) server)
	       (string= (erc-default-target) tgt))
	  (progn
	    (setq res (car buffer-list))))
      (setq buffer-list (cdr buffer-list)))
    (if (and (not res) (not strict))
	(setq res cb))
    (set-buffer cb)
    res))

(defun erc-rename-buffer (buffer)
  "Bring the name of the BUFFER in sync with the current nick.  Don't
 touch the target."
  ())

(defun erc (&optional server port nick full-name connect passwd tgt-list channel process)
  "Run ERC.  If CONNECT is non-nil, connect to the server.  
Otherwise assume already connected and just create a separate buffer
for the new target CHANNEL."
  (let ((buffer (erc-get-buffer-create server channel nick)))
    (if (not (eq buffer (current-buffer)))
	(progn
	  (if erc-log-p
	      ;; we can't log to debug buffer, it may not exist yet
	      (message (format "erc: current buffer %s, switching to %s"
			       (current-buffer) buffer)))
	  (cond ((eq erc-join-buffer 'window)
		 (switch-to-buffer-other-window buffer))
		((eq erc-join-buffer 'frame)
                 (funcall '(lambda (frame)
                             (raise-frame frame)
                             (select-frame frame))
                          (make-frame (or erc-frame-alist
                                          default-frame-alist)))
                 (switch-to-buffer buffer)
                 (when erc-frame-dedicated-p
                   (set-window-dedicated-p (selected-window) t)))
		(t (switch-to-buffer buffer)))))
    (set-buffer buffer)
    (setq erc-active-buffer buffer)
    (erc-mode)
    ;; go to the end of the buffer and open a new line
    ;; (the buffer may have existed)
    (goto-char (point-max))
    (open-line 1)
    (goto-char (point-max))

    ;; make local variables

    ;; connection parameters
    (make-variable-buffer-local 'erc-session-server)
    (make-variable-buffer-local 'erc-session-port)
    (make-variable-buffer-local 'erc-session-user-full-name)
    (make-variable-buffer-local 'erc-process)
    (setq erc-process process)
    (make-variable-buffer-local 'erc-insert-marker)
    (setq erc-insert-marker (make-marker))
    (set-marker erc-insert-marker (point))
    ;; stack of default recepients
    (make-variable-buffer-local 'def-rcpts)
    (setq def-rcpts tgt-list)
    ;; stack for user's nicknames
    (make-variable-buffer-local 'nick-stk)
    (setq nick-stk ())
    ;; assoc list of pairs (TIME-OF-PING-REQUEST-SENT . DESTINATION)
    (make-variable-buffer-local 'pings)
    (setq pings ())
    ;; last incomplete line read
    (make-variable-buffer-local 'prev-rd)
    (setq prev-rd "")
    ;; Channel members: only used in channel buffers.  It's a list of
    ;; user info entries '(nick op host email full-name ...).  Only
    ;; nick and op predicate must be present, other fields are not
    ;; required.
    (make-variable-buffer-local 'channel-members)
    (setq channel-members nil)
    ;; A topic string for the channel
    (make-variable-buffer-local 'channel-topic)
    (setq channel-topic "")
    ;; list of strings representing modes.  E.g. '("i" "m" "s" "b Quake!*@*")
    ;; (not sure the ban list will be here, but why not)
    (make-variable-buffer-local 'channel-modes)
    (setq channel-modes nil)
    ;; limit on the number of users on the channel (mode +l)
    (make-variable-buffer-local 'channel-user-limit)
    (setq channel-user-limit nil)
    ;; last peers (sender and receiver)
    (make-variable-buffer-local 'last-peers)
    (setq last-peers '(nil . nil))
    ;; last invitation channel
    (make-variable-buffer-local 'invitation)
    (setq invitation nil)
    ;; away flag
    (make-variable-buffer-local 'away)
    (setq away nil)
    ;; Server channel list
    (make-variable-buffer-local 'channel-list)
    (setq channel-list ())
;;;; Some flood protection stuff
    ;; time of last command sent
    (make-variable-buffer-local 'last-sent-time)
    (make-variable-buffer-local 'last-ping-time)
    ;; time of last CTCP response/request sent
    (make-variable-buffer-local 'last-ctcp-time)
    (setq last-sent-time (erc-current-time))
    (setq last-ping-time (erc-current-time))
    (setq last-ctcp-time (erc-current-time))
    (make-variable-buffer-local 'erc-lines-sent)
    (setq erc-lines-sent 0)
    (make-variable-buffer-local 'erc-bytes-sent)
    (setq erc-bytes-sent 0)
    ;; user requested quit
    (make-variable-buffer-local 'quitting)
    (setq quitting nil)
    ;; login-time 'nick in use' error
    (make-variable-buffer-local 'bad-nick)
    (setq bad-nick nil)
    ;; whether we have logged in
    (make-variable-buffer-local 'erc-logged-in)
    (setq erc-logged-in nil)
    ;; The local copy of `erc-nick' - the list of nicks to choose 
    (make-variable-buffer-local 'erc-default-nicks)
    (setq erc-default-nicks (if (consp erc-nick) erc-nick (list erc-nick)))
    ;; password stuff
    (make-variable-buffer-local 'password)
    (setq password passwd)
    ;; debug output buffer
    (make-variable-buffer-local 'erc-dbuf)
    (setq erc-dbuf 
	  (if erc-log-p
	      (get-buffer-create (concat "*ERC-DEBUG: " server "*"))
	    nil))
    (make-variable-buffer-local 'erc-buffer-list)
    (setq erc-buffer-list (list buffer))
    (erc-determine-parameters server port nick full-name)
    (if connect (erc-connect))
    (erc-update-mode-line)
    (goto-char (point-max))
    (open-line 1)
    (goto-char (point-max))
    (set-marker (process-mark erc-process) (point))
    (set-marker erc-insert-marker (point))
    (erc-display-prompt)
    (goto-char (point-max))))

;; interactive startup

(defvar erc-server-history-list nil
  "IRC server interactive selection history list")

(defvar erc-nick-history-list nil
  "Nickname interactive selection history list")

(defun erc-already-logged-in (server nick)
  "Returns nil if the user is logged in to the server with this nick.
This is determined by looking for the appropriate buffer and checking
whether the connection is still alive.

Do not rely on this function, it's only a guess, and may be unreliable."
  (let ((buf (erc-get-buffer server nil nick)))
    (if buf 
	(progn
	  (set-buffer buf)
	  (and (boundp 'erc-process) 
	       (processp erc-process)
	       (equal (process-status erc-process) 'open)))
      nil)))


(if (not (fboundp 'read-passwd))
    (defun read-passwd (prompt)
      "Substitute for read-passwd in early emacsen"
      (read-from-minibuffer prompt)))




;;;###autoload
(defun erc-select (&optional server port nick)
  "Interactively select connection parameters and run ERC."
  (interactive)
  (if (null server) (setq server erc-server))
  (if (null port) (setq port erc-port))
  (setq nick (erc-compute-nick nick))
  (let* ((server (read-from-minibuffer
		  "IRC server: " server nil nil 'erc-server-history-list))
	 (port
	  (erc-string-to-port
	   (read-from-minibuffer "IRC port: "
				 (erc-port-to-string (or port "irc")))))
	 (nick 
	  (if (erc-already-logged-in server nick)
	      (read-from-minibuffer
	       (format "%s is in use. Choose new nickname: " nick)
	       nick
	       nil nil 'erc-nick-history-list)
	    (read-from-minibuffer
	     "Nickname: " nick
	     nil nil 'erc-nick-history-list)))
	 (passwd (if erc-prompt-for-password
		     (if (and erc-password
			      (y-or-n-p "Use the default password? "))
			 erc-password
		       (read-passwd "Password: "))
		   erc-password)))
    (if (and passwd (string= "" passwd))
	(setq passwd nil))
    (while (erc-already-logged-in server  nick)
      ;; hmm, this is a problem when using multiple connections to a bnc
      ;; with the same nick. Currently this code prevents using more than one
      ;; bnc with the same nick. actually it would be nice to have
      ;; bncs transparent, so that erc-compute-buffer-name displays
      ;; the server one is connected to. 
      (setq nick (read-from-minibuffer
		  (format "%s is in use. Choose new nickname: " nick)
		  nick
		  nil nil 'erc-nick-history-list)))
    (erc server port nick erc-user-full-name;; full name
	 t passwd)))

;;; process management

(defun erc-connect ()
  "Perform the connection and login."
  (message "Connecting to %s:%s..." erc-session-server erc-session-port)
  (setq erc-process (open-network-stream
	      (format "erc-%s-%s" erc-session-server erc-session-port)
	      (current-buffer)
	      erc-session-server
	      erc-session-port))
  (message "Connecting to %s:%s... done" erc-session-server erc-session-port)
  (set-process-sentinel erc-process 'erc-process-sentinel)
  (set-process-filter erc-process 'erc-process-filter)
  (set-marker (process-mark erc-process) (point))
  (set-marker erc-insert-marker (point))
  (erc-log "\n\n\n********************************************\n")
  (message "Logging in as \'%s\'..." (erc-current-nick))
  (erc-login)
  ;; wait with script loading until we receive a confirmation (first
  ;; MOTD line)
  )


(defun erc-process-filter (erc-process string)
  "Filter messages from server"
  (erc-log (concat "erc-process-filter: " string))
  (let ((ob (current-buffer))
	(nb (process-buffer erc-process)))
    (set-buffer nb)
    (mapcar (lambda (s)
	      (cond ((and (consp s) (car s)) ;; (string . buffer) pair
		     (erc-display-line (car s) (cdr s))
;		     (erc-display-line (concat (car s) "\n") (cdr s))
		     )
		    ((stringp s)
		     (erc-display-line s)
;		     (erc-display-line (concat s "\n"))
		     )))
	    (mapcar 'erc-parse-line-from-server
		    (erc-split-multiline string)))
    (set-buffer ob)))


(defun erc-process-sentinel (cproc event)
  "Sentinel function for ERC process."
  (erc-log (format
	    "SENTINEL: proc: %S  status: %S  event: %S (quitting: %S)"
	    erc-process (process-status erc-process) event quitting))

  (save-excursion
    (set-buffer (process-buffer cproc))
;    (goto-char (point-max))

    (run-hook-with-args 'erc-disconnected-hook 
			(erc-current-nick) (system-name) "")
    (if (string= event "exited abnormally with code 256\n")

	;; Sometimes (eg on a /LIST command) ERC happens to die with
	;; an exit code 256. The icrii client also shows this behavior
	;; and it restarts itself. So do I.

	(cond
	 ((not quitting)
	  (open-line 1)
	  (goto-char (point-max))
	  (insert
	   (erc-highlight-error
	    "Connection failed! Re-establishing connection...\n"))
	  (erc-update-mode-line)
	  (setq last-sent-time 0)
	  (setq last-lines-sent 0)
	  (erc-connect))

	 (t
	  (let* ((wd (window-width))
		(msg "*** ERC finished ***")
		(off (/ (- wd (length msg)) 2))
		(s ""))
	    (if (> off 0)
		(setq s (make-string off ?\ )))
	    (insert (concat "\n\n" s msg "\n")))))
      (insert (concat "\n\n*** ERC terminated: " event "\n"))))
  (goto-char (point-max))
  (erc-update-mode-line))


;;; I/O interface
  
;; send interface

;;; Mule stuff: send international characters the right way in Mule
;;; capable emacsen, and don't screw those who can't do ISO code pages.

;; encode ISO to clear text, if possible
(or (fboundp 'encode-coding-string)
    (defun encode-coding-string (x y &optional z) x))
;; decode string from possibly ISO encoding
(or (fboundp 'decode-coding-string)
    (defun decode-coding-string (x y &optional z) x))

(defun erc-encode-coding-string (s)
  "Supposed to encode ISO-characters in \"clear text\" format.
 Doesn't seem to work with colors, so does nothing at the moment"
  (encode-coding-string s 'ctext)
;   (let ((res "")
; 	(i 0)
; 	(nexti 0)
; 	(l (length s)))
;     (while nexti
;       (setq nexti (string-match "[\C-b\C-_\C-c\C-v]" (substring s i)))
;       (if nexti (setq nexti (+ nexti i)))
;       (message "nexti = %s" nexti)
;       (setq res (concat res (encode-coding-string (substring s i nexti)
; 						  'ctext)))
;       (if nexti
; 	  (progn
; 	    (setq res (concat res (substring s nexti (1+ nexti))))
; 	    (setq i (1+ nexti)))))
;     res))
;;  s
)

(defun erc-decode-coding-string (s)
  "Supposed to decode ISO-characters from \"clear text\" format.
 Doesn't seem to work with colors, so does nothing at the moment"
  ;; (decode-coding-string s 'ctext)
  s)

(defun erc-flood-exceeded-p (line)
  "Determines whether the flood limits are exceeded or not by the LINE.
 It also maintains all the flood control variables."
  ;; First, clean up if no messages for long enough time
  (let ((flood (cond ((eq erc-flood-protect 'strict) erc-flood-limit2)
		     (erc-flood-protect erc-flood-limit)
		     (t nil))))
    (if (or (not flood)
	    (< (+ last-sent-time (nth 2 flood))
	       (erc-current-time)))
	(progn
	  (setq erc-lines-sent 0)
	  (setq erc-bytes-sent 0)))
    ;; Update the variables for the new line
    (setq erc-lines-sent (1+ erc-lines-sent))
    (setq erc-bytes-sent (+ (length line) erc-bytes-sent))
    (setq last-sent-time (erc-current-time))
  ;; Now do what they ask
    (and flood
	 (let ((bytes (nth 0 flood))
	       (lines (nth 1 flood)))
	   (or (and lines (> erc-lines-sent lines))
	       (and bytes (> erc-bytes-sent bytes)))))))

(defun erc-send-command (l &optional force)
  "Send command line L to IRC server.  If the optional FORCE is non-nil,
send the command even if the flood guard is in effect and the
limit is exceeded.

Return non-nil if the line is actually sent, nil otherwise.
The command line must contain neither prefix nor trailing `\\n'"
  (erc-log (concat "erc-send-command: " l))
  (if (and away erc-auto-discard-away)
      (erc-process-away nil))
  (let ((ob (current-buffer))
	(buf (if (and (boundp 'erc-process) (processp erc-process))
		 (process-buffer erc-process)
	       nil))
	(res nil))
    (if buf
	(progn
	  (set-buffer buf)
	  (let ((exceeded (erc-flood-exceeded-p l)))
	    (if (or force (not exceeded))
		(progn
		  (if exceeded 
		      (progn
			(message "Warning: flood exceeded, but send forced")
			(erc-log-aux 
			 (format "ERC FLOOD PROTECT: flood exceeded, but send forced on %S\n"
				 l))))
		  (process-send-string erc-process 
		       (concat (erc-encode-coding-string l) "\n"))
		  (setq res t))
	    (progn 
	      (if (not erc-disable-ctcp-replies)
		  (progn
		    (setq erc-disable-ctcp-replies t)
		    (erc-display-line 
		     (erc-highlight-error
		      "FLOOD PROTECTION: Automatic CTCP responses turned off.\n")
		     ob)))
	      (if (not (eq erc-flood-protect 'strict))
		  (progn
		    (setq erc-flood-protect 'strict)
		    (erc-display-line 
		     (erc-highlight-error
		      "FLOOD PROTECTION: Switched to Strict Flood Control mode.\n")
		     ob)))
	      (message "ERC FLOOD PROTECT: line not sent: %S" l)
	      (erc-log-aux (format "ERC FLOOD PROTECT: line not sent: %S" l)))))
	  (set-buffer ob))
      (progn
	(message "ERC: No process running")
	(beep)))
    res))

(defun erc-send-ctcp-message (tgt l &optional force)
  "Send CTCP message L for TGT. If TGT is nil the message is not sent.
The command must contain neither prefix nor trailing `\\n'"
  (cond
   (tgt
    (erc-log (format "erc-send-CTCP-message: [%s] %s" tgt l))
    (erc-send-command (format "PRIVMSG %s :\C-a%s\C-a" tgt l) force))))

(defun erc-send-ctcp-notice (tgt l &optional force)
  "Send CTCP notice L for TGT. If TGT is nil the message is not sent.
The command must contain neither prefix nor trailing `\\n'"
  (cond
   (tgt
    (erc-log (format "erc-send-CTCP-notice: [%s] %s" tgt l))
    (erc-send-command (format "NOTICE %s :\C-a%s\C-a" tgt l) force))))

(defun erc-send-action (tgt str &optional force)
  "Send CTCP ACTION information described by STR to TGT"
  (erc-send-ctcp-message tgt (format "ACTION %s" str) force)
  (put-text-property 0 (length str) 
		     'face 'erc-action-face str)
  (erc-display-line (concat (erc-current-nick) 
			    erc-action-separator str)
		    (current-buffer)))

;; command tables

(defconst erc-command-table
  ;; MSG and NOTICE are put first to ease the search.
  '(("MSG" erc-cmd-msg "")
    ("NOTICE" erc-cmd-msg "")
    ("AWAY" erc-cmd-away "")
    ("BYE" erc-cmd-quit "")
    ("CHANNEL" erc-cmd-join "")
    ("COMMENT" nil "")
    ("CTCP" erc-cmd-ctcp "")
    ("DATE" erc-cmd-time "")
    ("DESCRIBE" erc-cmd-describe "")
    ("EXIT" erc-cmd-quit "")
    ("JOIN" erc-cmd-join "")
    ("KICK" erc-cmd-kick "")
    ("LEAVE" erc-cmd-part "")
    ("LOAD" erc-cmd-load "")
    ("ME" erc-cmd-me "")
    ("NICK" erc-cmd-nick "")
    ("PART" erc-cmd-part "")
    ("PING" erc-cmd-ping "")
    ("QUERY" erc-cmd-query "")
    ("QUIT" erc-cmd-quit "")
    ("SERVER" erc-cmd-server "")
    ("SET" erc-cmd-not-implemented "")
    ("SIGNOFF" erc-cmd-quit "")
    ("SOUND" erc-cmd-sound "")
    ("TIME" erc-cmd-time "")
    ("NAMES" erc-cmd-names "")
;    ("NAMES" erc-cmd-default-channel "")
    ("TOPIC" erc-cmd-topic ""))

  "Table of input commands supported by ERC. Each entry has a form
\(NAME FUNCTION HELP\)

NAME is a command name in uppercase. FUNCTION is a command handler or NIL to
ignore the command. This function must take two strings as arguments:
a command name and the rest of the command line. HELP is a character string
containing one-line help for the command.

There may be a default handler for all other commands. It must be represented
by an entry with NAME as an empty string")


;; display interface

(defun erc-display-line-buffer (string buffer) 
"Display STRING in the ERC.  Auxiliary function used in
`erc-display-line'.  After the insertion, runs `erc-insert-hook'.

The line gets filtered to interpret the control characters. If STRING is NIL,
the function does nothing"
  (when string
    (save-excursion;; to restore original buffer and point
      (set-buffer (or buffer (process-buffer erc-process)))
      (save-excursion;; to restore point in the new buffer
	(let ((insert-position (or (marker-position erc-insert-marker)
				   (point-max)))
	      (string (erc-interpret-controls string)))
	  (unless (string-match "\n$" string)
	    (setq string (concat string "\n")))
	  (erc-log (concat "erc-display-line: " string 
			   (format "(%S)" string) " in buffer " 
			   (format "%s" buffer)))
	  (goto-char insert-position)
	  ;; decode the ISO code page before inserting (for mule'ed emacsen)
	  (insert (erc-decode-coding-string string))
	  (when (and (boundp 'erc-insert-marker)
		     (markerp erc-insert-marker))
	    (set-marker erc-insert-marker (point)))
	  ;; Filling
	  (erc-fill-region insert-position (point))
	  ;; run hook
	  (run-hooks 'erc-insert-hook))))))

(defun erc-fill-region (start end)
  "Fill region from START to END.
START and END should be markers."
  ;; Manage filling.
;  (message "filling from %S to %S" start end)
  (when erc-filling
    (let ((current-pos (point-marker))
	  (fill-prefix erc-fill-prefix))
      (unless fill-prefix
	(goto-char start)
	(when (looking-at "\\(\\S-+ \\)")
	  (let ((len (length (match-string 1))))
	    (setq fill-prefix (make-string len ? )))))
      (fill-region start end))))

(defun erc-display-line (string &optional buffer)
  "Display STRING in the ERC BUFFER.
All screen output must be done through this function.  If BUFFER is nil
or omitted, the default ERC buffer for the `erc-session-server' is used.
The BUFFER can be an actual buffer, a list of buffers, 'all or 'active.
If BUFFER = 'all, the string is displayed in all the ERC buffers for the
current session. 'active means the current active buffer
\(`erc-active-buffer').  If the buffer can't be resolved, the current
buffer is used.  After the insertion, runs `erc-insert-hook' in each
buffer.

The line gets filtered to interpret the control characters. If STRING is
NIL, the function does nothing"
  (let ((bufs (cond ((bufferp buffer) (list buffer))
		    ((listp buffer) buffer)
		    ((eq 'all buffer) 
		     (and (boundp 'erc-buffer-list) erc-buffer-list))
		    ((eq 'active buffer) (and erc-active-buffer
					      (list erc-active-buffer)))
		    ((and (boundp 'erc-process)(processp erc-process))
		     (list (process-buffer erc-process)))
		    (t (list (current-buffer)))))
	(new-bufs nil))
    (while bufs
      (if (buffer-live-p (car bufs))
	  (progn
	    (erc-display-line-buffer string (car bufs))
	    (setq new-bufs (cons (car bufs) new-bufs))))
      (setq bufs (cdr bufs)))
    (if (null new-bufs)
	(if (and (boundp 'erc-process) (processp erc-process))
	    (erc-display-line-buffer string (process-buffer erc-process))
	  (erc-display-line-buffer string (current-buffer))))
    (if (and (eq 'all buffer) (boundp 'erc-buffer-list))
	(setq erc-buffer-list new-bufs))))

(defun erc-send-current-line ()
  "Parse current line and send it to IRC."
  ;; Plan of attack:
  ;; save the string (without prompt) to s
  ;; return if s is empty
  ;; if not last line - go to the last one and one more
  ;; clear current line
  ;; print either <nick> message or nick> message or prompt in prompt face
  ;; uncontrol & print the line
  ;; send line to the parser
  ;; set the process marker
  ;; open new line and print the prompt there
  (interactive)
  (setq erc-active-buffer (current-buffer))
  (let* ((l (erc-parse-current-line))
	 (s (concat (car l) (cdr l))))
    (cond
     ((string-match "^\\s-*\n*$" s)
      (message "Blank line - ignoring...")
      (ding))
     (t
      (cond ((> (count-lines (progn (end-of-line) (point)) (point-max)) 0)
	     (goto-char (point-max))
	     (open-line 1)
	     (goto-char (point-max))))
      (delete-region
       (progn (beginning-of-line) (point))
       (progn (end-of-line) (point)))
      (if (eq (elt s 0) ?/)
	  ; This trick is wrong, because the ring gets messed,
	  ; up and changing s doesnt do any good in gerneal.
;	  (if (string= (substring s 0 3) "/me")
;	      (progn 
;		(insert (concat (erc-current-nick) erc-action-separator))
;		(setq s (substring s 4)))
	  (erc-display-prompt)
	(insert (concat "<" (erc-current-nick) "> ")))
      ;; First, merge the color info with the user supplied controls
;      (erc-log (concat "erc-send-current-line: about to merge controls in "
;		       s))
      ;; remove the last "\n" from the string - for nicer display
      (let ((eos (string-match "[\n]+$" s)))
	(if eos (setq s (substring s 0 eos))))
      (setq s (erc-merge-controls s))
;      (erc-log (format "Done with s: %S" s))
      ;; now start decorating the string anew
      (put-text-property 0 (length s) 'face 'erc-input-face s)
      (insert (if erc-uncontrol-input-line
		  (erc-interpret-controls s)
		s))
      (erc-fill-region (progn (beginning-of-line) (point))
		       (progn (end-of-line) (point)))
      (erc-add-to-input-ring s)
      (goto-char (point-max))
      (open-line 1)
      (goto-char (point-max))
      (set-marker (process-mark erc-process) (point))
      (set-marker erc-insert-marker (point))
      (erc-display-prompt)
      (erc-process-input-line (concat s "\n") t)))))

(defun erc-send-paragraph (&optional force)
  "Send multiple lines starting from the first `erc-prompt' before the
point till the end of buffer.  If a non-nil argument is provided,
force sending even under excess flood condition."
  (interactive "P")
  (setq erc-active-buffer (current-buffer))
  (let* ((pnt (point))
	 (start (let ((re (concat "^" erc-prompt))
		      (doit t))
		  (while (and doit (/= (point) (point-min)))
		    (backward-char)
		    (if (looking-at re) (setq doit nil)))
		  (+ (point) (length erc-prompt) 1)))
	 (end (point-max))
	 (str (buffer-substring start end)))
    (delete-region start end)
    (goto-char (point-max))
    (setq str (erc-merge-controls str))
    (erc-load-irc-script-lines (erc-split-multiline-safe str) force t)
    (goto-char (point-max));; do it again, in case loading moves the point
    (open-line 1)
    (goto-char (point-max))
    (set-marker (process-mark erc-process) (point))
    (set-marker erc-insert-marker (point))
    (erc-display-prompt)))

(defun erc-process-input-line (line &optional force)
  "Translate LINE to an RFC1459 command and send it based on a table
in `erc-command-table'.  Returns non-nil if the command is actually
sent to the server, and nil otherwise.

If the command in the LINE is not in the table, it is passed to
`erc-cmd-default'. If LINE is not a command (ie. doesn't start with /<WORD>)
then it is sent as a message.

See documentation for `erc-command-table' for more details.

An optional FORCE argument forces sending the line when flood protection
is in effect."
  (cond					; invoke method in the table
   ((string-match "^\\s-*/\\([A-Za-z]+\\)\\(\\s-+.*\\|\\s-*\\)$" line)
    (let* ((cmd (format "%s" (upcase (match-string 1 line))))
	   (args (format "%s" (match-string 2 line)))
	   (e (assoc cmd erc-command-table))
	   (fn (if e
		   (nth 1 e)
		 #'erc-cmd-default)))
      (erc-log (format "proc-input: lookup: [%S] [%S] -> %S" cmd args e))
      (if (not (funcall fn cmd args force))
	  (progn
	    (erc-display-line (erc-highlight-error "Bad syntax\n")
			      (current-buffer))
	    nil)
	t)))
   (t					; send as text
    (let ((r (erc-default-target)))
      (if r
	  (erc-send-command (format "PRIVMSG %s :%s" r line) force)
	(progn 
	  (erc-display-line (erc-highlight-error "No target\n")
			    (current-buffer))
	  nil))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                    Input commands handlers
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun erc-cmd-not-implemented (cmd line &optional force)
  (erc-log (format "cmd: NOT IMPLEMENTED: %s %s" cmd line))
  (erc-display-line (erc-highlight-error "Not implemented\n"))
  t)

(defun erc-cmd-default (cmd line &optional force)
  (erc-log (format "cmd: DEFAULT: %s %s" cmd line))
  (erc-send-command (concat cmd line) force)
  t)

(defun erc-cmd-default-channel (cmd line &optional force)
  (let ((tgt (erc-default-target))
	(msg line))
    (if (string-match "^\\s-*\\(#\\S-+\\)\\(.*\\)$" line)
	(setq tgt (match-string 1 line)
	      msg (match-string 2 line)))
    (if tgt
	(progn
	  (erc-log (format "cmd: DEFAULT: %s %s %s" cmd tgt line))
	  (erc-send-command (concat cmd " " tgt line) force))
      (erc-display-line (erc-highlight-error "No target\n")
			(current-buffer))))
  t)

(defun erc-cmd-away (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(.*\\)$" line)
    (let ((reason (match-string 1 line)))
      (erc-log (format "cmd: AWAY: %s" reason))
      (erc-send-command
       (if (string= reason "")
	   "AWAY"
	 (concat "AWAY :" reason)) force))
    t)
   (t
    nil)))

(defun erc-cmd-ctcp (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(\\S-+\\)\\s-+\\(.*\\)$" line)
    (let ((nick (match-string 1 line))
	  (cmd (match-string 2 line)))
      (erc-log (format "cmd: CTCP [%s]: [%s]" nick cmd))
      (erc-send-ctcp-message nick cmd force))
    t)
   (t
    nil)))
     
(defun erc-cmd-describe (cmd line &optional force)
  (cond
   ((string-match
     "^\\s-*\\(\\S-+\\)\\s-\\(.*\\)$" line)
    (let ((dst (match-string 1 line))
	  (s (match-string 2 line)))
      (erc-log (format "cmd: DESCRIBE: [%s] %s" dst s))
      (erc-send-action dst s force))
    t)
   (t
    nil)))

(defun erc-cmd-join (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(\\S-*\\)\\s-*$" line)
    (let ((s (match-string 1 line))
	  (chnl nil))
      (erc-log (format "cmd: JOIN: %s" s))
      (cond
       ((string= (upcase s) "-INVITE")
	(if invitation
	    (setq chnl invitation
		  invitation nil)
	  (message "You've got no invitation")))
       (t
	(setq chnl s)))
      (cond
       (chnl
	(erc-send-command (format "JOIN %s" chnl) force)
	(erc-send-command (format "MODE %s" chnl) force))))
    t)
   (t
    nil)))

(defvar erc-last-channel-names nil 
"If non-nil, it is a list of names in a channel whose names are about
to be listed.  On channels with too many people, the server sends
names in several lines, and this variable is used to save the extra
information (host IP, login name, etc.) that has been collected so
far, and that /names #channel doesn't provide.")

(defun erc-cmd-names (cmd line &optional force)
  "Clears the channel name list first, then sends the command"
  (let ((tgt (erc-default-target))
	(msg line))
    (if (string-match "^\\s-*\\(#\\S-+\\)\\(.*\\)$" line)
	(setq tgt (match-string 1 line)
	      msg (match-string 2 line)))
    (if (and tgt (string-match "^#" tgt))
	(progn
	  (erc-log (format "cmd: DEFAULT: %s %s %s" cmd tgt line))
	  (setq erc-last-channel-names (erc-get-channel-members tgt))
	  ;;(message "erc-cmd-names: %s: names=%S" tgt erc-last-channel-names)
	  (erc-refresh-channel-members tgt "")
 	  (erc-send-command (concat cmd " " tgt line) force))
      (erc-display-line (erc-highlight-error "No default channel\n")
			(current-buffer))))
  t)
    
(defun erc-cmd-kick (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(\\#S-+\\)\\s-+\\(\\S-+\\)\\(\\s-.*\\)?$" line)
    (let ((ch (match-string 1 line))
	  (nick (match-string 2 line))
	  (reason (match-string 3 line)))
      (if (not reason) (setq reason ""))
      (erc-log (format "cmd: KICK: %s/%s: %s" nick ch reason))
      (erc-send-command (format "KICK %s %s :%s" ch nick reason) force))
    t)
   ((string-match "^\\s-*\\(\\S-+\\)\\(\\s-.*\\)?$" line)
    (let ((ch (erc-default-target))
	  (nick (match-string 1 line))
	  (reason (match-string 2 line)))
      (if ch
	  (progn
	    (if (not reason) (setq reason ""))
	    (erc-log (format "cmd: KICK: %s/%s: %s" nick ch reason))
	    (erc-send-command (format "KICK %s %s :%s" ch nick reason) force))
	(erc-display-line (erc-highlight-error "No default channel\n")
			  (current-buffer))))
    t)
   (t
    nil)))

(defun erc-cmd-load (cmd line &optional force)
  "Loads the script provided in the LINE.  If LINE continues beyond
the file name, the rest of it is put in a (local) variable
`erc-script-args', which can be used in elisp scripts.

The optional FORCE argument is ignored here - you can't force loading
a script after exceeding the flood threshold."
  (cond
   ((string-match "^\\s-*\\(\\S-+\\)\\(.*\\)$" line)
    (let* ((file0 (match-string 1 line))
	   (erc-script-args (match-string 2 line))
	   (file (erc-find-file file0 erc-script-path)))
      (erc-log (format "cmd: LOAD: %s" file))
      (cond
       ((not file)
	(erc-display-line
	 (erc-highlight-error (concat "Cannot find file " file0))))
       ((not (file-readable-p file))
	(erc-display-line
	 (erc-highlight-error (concat "Cannot read file " file))))
       (t
	(message "Loading \'%s\'..." file)
	(erc-load-script file)
	(message "Loading \'%s\'...done" file))))
    t)
   (t
    nil)))

(defun erc-cmd-me (cmd line &optional force)
  "Pose some action."
  (cond
   ((string-match "^\\s-\\(.*\\)$" line)
    (let ((s (match-string 1 line)))
      (erc-log (format "cmd: ME: %s" s))
      (erc-send-action (erc-default-target) s force))
    t)
   (t
    nil)))
   
(defun erc-cmd-msg (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(\\S-+\\)\\(\\s-*$\\|\\s-+.*\\)" line)
    (let ((tgt (match-string 1 line))
	  (s (match-string 2 line)))
      (erc-log (format "cmd: MSG(%s): [%s] %s" cmd tgt s))
      (if (string-match "^ " s) (setq s (substring s 1)))
      (cond
       ((string= tgt ",")
	(if (car last-peers)
	    (setq tgt (car last-peers))
	  (setq tgt nil)))
       ((string= tgt ".")
	(if (cdr last-peers)
	    (setq tgt (cdr last-peers))
	  (setq tgt nil))))
      (cond
       (tgt
	(setcdr last-peers tgt)
	(erc-send-command
	 (if (string= cmd "MSG")
	     (format "PRIVMSG %s :%s" tgt s)
	   (format "NOTICE %s :%s" tgt s)) force))
       (t
	(erc-display-line (erc-highlight-error "No target\n")))))
    t)
   (t
    nil)))

(defun erc-cmd-nick (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(\\S-+\\)\\s-*$" line)
    (let ((nick (match-string 1 line)))
      (erc-log (format "cmd: NICK: %s (bad-nick: %S)" nick bad-nick))
      (erc-send-command (format "NICK %s" nick) force)
      (cond (bad-nick
	     (erc-push-nick nick)
	     (erc-update-mode-line)
;	     (erc-rename-all-buffers);; no need so far
	     (setq bad-nick nil))))
    t)
   (t
    nil)))

(defun erc-cmd-part (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(#\\S-+\\)\\(\\s-+.*\\)?$" line)
    (let ((ch (match-string 1 line))
	  (msg (match-string 2 line)))
      (if msg 
	  (setq msg (substring msg 1))
	(setq msg ""))
      (erc-log (format "cmd: PART: %s: %s" ch msg))
      (erc-send-command
       (if (string= msg "")
	   (format "PART %s" ch)
	 (format "PART %s :%s" ch msg)) force))
    t)
   ((string-match "^\\(.*\\)$" line)
    (let ((ch (erc-default-target))
	  (msg (match-string 1 line)))
      (if (not msg) (setq msg ""))
      (if (and ch (string-match "^#" ch))
	  (progn
	    (erc-log (format "cmd: PART: %s: %s" ch msg))
	    (erc-send-command
	     (if (string= msg "")
		 (format "PART %s" ch)
	       (format "PART %s :%s" ch msg)) force))
	(erc-display-line (erc-highlight-error "No target\n")
			  (current-buffer))))
    t)
   (t
    nil)))

(defun erc-cmd-ping (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(\\S-+\\)\\s-*$" line)
    (let* ((s (match-string 1 line))
	   (ct (erc-current-time))
	   (str (format "%f" ct))
	   (nn (if (string-match "^\\([0-9]+\\)" str)
	       (match-string 1 str) "0"))
	   (ob (current-buffer)))
      (erc-log (format "cmd: PING: %s" s))
      (erc-send-ctcp-message s (format "PING %s" nn) force)
      (if (and (boundp 'erc-process) (processp erc-process))
	  (set-buffer (process-buffer erc-process)))
      (setq pings (cons (cons ct (erc-trim-string s)) pings))
      (set-buffer ob))
    t)
   (t
    nil)))

(defun erc-cmd-query (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(\\S-*\\)\\s-*$" line)
    (let ((ob (current-buffer))
	  (s (match-string 1 line))
	  (session-buffer (and (boundp 'erc-process)
			       (processp erc-process)
			       (process-buffer erc-process)))
	  (buffer nil))
      (erc-log (format "cmd: QUERY [%s]" s))
      (if (string= s "")
	  (erc-delete-query)
	(progn
	  (if (and session-buffer
		   (buffer-live-p session-buffer))
	      (set-buffer session-buffer))
	  (erc erc-session-server erc-session-port (erc-current-nick) 
	       erc-session-user-full-name nil nil def-rcpts 
	       s erc-process)
	  (setq buffer (erc-find-buffer erc-session-server s))
	  (if buffer (set-buffer buffer))
	  (erc-add-query s)))
      (erc-update-mode-line))
    t)
   (t
    nil)))

(defun erc-cmd-quit (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(.*\\)$" line)
    (let* ((s (match-string 1 line))
	   (ob (current-buffer))
	   (buffer (if (and (boundp 'erc-process)(processp erc-process))
		       (process-buffer erc-process) nil)))
      (if (and buffer (bufferp buffer)) (set-buffer buffer))
      (erc-log (format "cmd: QUIT: %s" s))
      (setq quitting t)
      (erc-send-command
       (if (or (not s) (string= s ""))
	   (format
	    "QUIT :\C-bERC\C-b v%s (IRC client for Emacs)"; - \C-b%s\C-b"
	    erc-version-string) ; erc-official-location)
	 (format "QUIT :%s" s)) force)
      (set-buffer ob))
    t)
   (t
    nil)))

(defun erc-cmd-server (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(\\S-+\\)\\s-*$" line)
    (let ((nserver (match-string 1 line)))
      (erc-log (format "cmd: SERVER: %s" nserver))
      (erc-select nserver nil (erc-current-nick)))
    t)
   (t
    nil)))

;(defun erc-cmd-server (cmd line)
;  (cond
;   ((string-match "^\\s-*\\(\\S-+\\)\\s-*$" line)
;    (let ((nserver (match-string 1 line)))
;      (erc-log (format "cmd: SERVER: %s" nserver))
;      (let ((quitting t))
;	(delete-process erc-process)
;	(setq erc-session-server nserver)
;	(erc-connect))
;      (erc-update-mode-line))
;    t)
;   (t
;    nil)))

(defun erc-cmd-time (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(.*\\)$" line)
    (let ((args (match-string 1 line)))
      (erc-log (format "cmd: TIME: %s" args))
      (erc-send-command (concat "TIME " args) force))
    t)
   (t
    nil)))

(defun erc-cmd-topic (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(#\\S-+\\)\\s-\\(.*\\)$" line)
    (let ((ch (match-string 1 line))
	  (topic (match-string 2 line)))
      (erc-log (format "cmd: TOPIC [%s]: %s" ch topic))
      (erc-send-command (format "TOPIC %s :%s" ch topic) force))
    t)
   ((string-match "^\\s-*\\(.*\\)$" line)
    (let ((ch (erc-default-target))
	  (topic (match-string 1 line)))
      (if (and ch (string-match "^#" ch))
	  (progn
	    (erc-log (format "cmd: TOPIC [%s]: %s" ch topic))
	    (erc-send-command (format "TOPIC %s :%s" ch topic) force))
	(erc-display-line (erc-highlight-error "No target\n")
			  (current-buffer))))
    t)
   (t
    nil)))

(defun erc-cmd-sound (cmd line &optional force)
  (cond
   ((string-match "^\\s-*\\(\\S-+\\)\\(\\s-.*\\)?$" line)
    (let ((file (match-string 1 line))
	  (msg (match-string 2 line))
	  (tgt (erc-default-target)))
      (if (null msg) 
	  (setq msg "")
	;; remove the first white space
	(setq msg (substring msg 1)))
      (if tgt
	  (progn
	    (erc-send-ctcp-message tgt (format "SOUND %s %s" file msg) force)
	    (if erc-play-sound (erc-play-sound file)))
	  (erc-display-line (erc-highlight-error "No target\n")
			    (current-buffer)))
      t))
   (t nil)))

;;;; End of IRC commands

(defun erc-reformat-command (line)
  "Re-format command in the line

Removes leading [and trailing?] spaces and converts a command to uppercase.
Returns a string untouched, if it doesn't contain a command"

  (if (string-match "^\\s-*/\\([A-Za-z]+\\)\\(\\s-+.*\\|\\s-*\\)$" line)
      (concat "/" (upcase (match-string 1 line)) " " (match-string 2 line))
    line))

(defun erc-make-property-list (fg bg bold ul inv)
  "Compiles the list of properties from the FG, BG colors and BOLD,
   UL (underline) and INV (inverse) flags."
  (let ((res nil))
    (if ul (setq res (cons 'erc-underline-face res)))
    (if bold (setq res (cons 'erc-bold-face res)))
    (if fg (setq res (cons (if inv
			       (erc-get-bg-color-face fg)
			     (erc-get-fg-color-face fg))
			   res)))
    (if bg (setq res (cons (if inv
			       (erc-get-fg-color-face bg)
			     (erc-get-bg-color-face bg))
			   res)))
    (if (and inv (not (or fg bg))) (setq res (cons 'erc-inverse-face res)))
    res))

(defun erc-prepend-properties (obj start end fg bg bold ul inv)
  "Prepends text properties from START to END to the OBJ (string or
   buffer) that are determined by BG, FG colors (0-15) and BOLD, UL
   (underline) and INV (inverse) flags"
  (let ((props (erc-make-property-list fg bg bold ul inv)))
    (if props
	(progn
	  ;;(erc-log (format "prepending property list: %s" props))
	  (if fg (put-text-property start end 'erc-fg fg obj))
	  (if bg (put-text-property start end 'erc-bg bg obj))
	  (if bold (put-text-property start end 'erc-bold bold obj))
	  (if ul (put-text-property start end 'erc-ul ul obj))
	  (if inv (put-text-property start end 'erc-inv inv obj))
	  (font-lock-prepend-text-property start end 'face props obj)))))

(defun erc-decode-controls (line)
  "Converts all 'erc-* properties of LINE into explicit control
characters that can be sent back to the server.  Useful for resending
a colored line just by pasting it at the prompt, or for grabbing color
pop-ups that other people send."

  (let ((pos 0)
	(next-pos 0)
	(fg   nil)
	(bg   nil)
	(bold nil)
	(ul   nil)
	(inv  nil)
	;; previous values
	(col nil)
	(bg0 nil)
	(bold0 nil)
	(ul0   nil)
	(inv0  nil)
	(res ""))
    (while pos
      (setq fg   (get-text-property pos 'erc-fg line)
	    bg   (get-text-property pos 'erc-bg line)
	    bold (get-text-property pos 'erc-bold line)
	    ul   (get-text-property pos 'erc-ul line)
	    inv  (get-text-property pos 'erc-inv line))
      (setq next-pos (next-property-change pos line))
      ;; put "end of color" only if not at the beginning of line,
      ;; and the color was on.  Also put it if the bg color dissapeared
      (if (or (and col (not fg) (not bg) (/= pos 0))
	      (and (not bg) bg0))
	  (setq res (concat res "\C-c")))
      (if fg 
	  (progn
	    (setq res (concat res (char-to-string ?\^c) (format "%02d" fg)))
	    (if bg (setq res (concat res (format ",%02d" bg))))))
      (if (or (and bold (not bold0))
	      (and (not bold) bold0))
	  (setq res (concat res (char-to-string ?\^b))))
      (if (or (and ul (not ul0))
	      (and (not ul) ul0))
	  (setq res (concat res (char-to-string ?\^_))))
      (if (or (and inv (not inv0))
	      (and (not inv) inv0))
	  (setq res (concat res (char-to-string ?\^v))))
      (setq col (or fg bg)
	    bg0 bg
	    bold0 bold
	    ul0 ul
	    inv0 inv)
      (setq res (concat res (substring line pos next-pos)))
      (setq pos next-pos))
    res))

(defun erc-interpret-controls (line)
  "Translate control characters in the line to faces and beeps."
  ;;(erc-log (format "start: %S" (current-time)))
  (cond
   ((and (not (null line)) erc-interpret-controls-p)
    (let ((res "")
	  (i 0)
	  (j 0)
	  (len (length line))
	  (parts nil);; for partitioning the string into segments
	  (bg nil) ;; number 0-15 (color)
	  (fg nil) ;; number 0-15 (color)
	  (bold nil)    ;; boolean
	  (ul nil);; boolean (underline)
	  (inv nil);; boolean (inverse) - not sure how it'll work...
	  (hl-start 0);; position in the string where it starts
	  (hl-face nil);; the list of faces ready for 
	               ;; "font-lock-prepend-text-property"
	  (prop-list nil));; list of arguments for `erc-prepend-properties'

      ;; First, decompose the string into parts without controls, and
      ;; the controls by themselves, and make a list of corresponding
      ;; substrings.  Then we'll combine them one chunk at a time, not
      ;; by one character, as was before.
      (setq j (string-match "\\(\C-b\\|\C-c\\|\C-_\\|\C-v\\|\C-g\\|\C-o\\)"
			    (substring line i)))
      (while j
	(let* ((rest (substring line j))
	       (s1 (substring line i j)) ;; "clean" part of the string
	       (ctrl (cond ((string-match "\\(\\(\C-c\\([0-9]\\([0-9]\\)?\\(,[0-9]\\([0-9]\\)?\\)?\\)?\\)\\|\C-b\\|\C-_\\|\C-v\\|\C-g\\|\C-o\\)" rest)
			    (setq i (+ j (match-end 1)))
			    (match-string 1 rest))
			   (t (erc-log-aux (concat "ERROR: erc-interpret-controls: no match in " rest))
			      (setq j nil)
			      ""))))
	  (if (not (string= s1 "")) (setq parts (cons s1 parts)))
	  (setq parts (cons ctrl parts))
	  (setq j (string-match "\\(\C-b\\|\C-c\\|\C-_\\|\C-v\\|\C-g\\|\C-o\\)"
			    (substring line i)))
	  (if j (setq j (+ i j)))))

      (if (< i len) (setq parts (cons (substring line i) parts)))
      (setq parts (reverse parts))
      (setq i 0 j 0)
      ;;(erc-log (format "computed list of parts: %S" (current-time)))

      ;;(erc-log (format "erc-interpret-controls: parts = %S" parts))
	  
      ;; The parts of the string are ready:
      (while parts
	(let ((c (string-to-char (car parts))))
;; 	(erc-log (format "intp-c: i %d c [%c] st %S fc %S"
;; 			 i c hl-start hl-face))
	  (cond ((char-equal c ?\^g)	; beep
		 (ding t))
		
		;; bold
		((char-equal c ?\^b)
;;	       (erc-log	(format "intp-c: BOLD i %d j %d st %S" i j hl-start))
		 (setq prop-list (cons (list hl-start j fg bg bold ul inv)
				       prop-list))
		 ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
		 (setq bold (not bold) 
		       hl-start j))
		
		;; underline
		((char-equal c ?\^_)
		 (setq prop-list (cons (list hl-start j fg bg bold ul inv)
				       prop-list))
		 ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
		 (setq ul (not ul) 
		       hl-start j))

		;; inverse
		((char-equal c ?\^v)
		 (setq prop-list (cons (list hl-start j fg bg bold ul inv)
				       prop-list))
		 ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
		 (setq inv (not inv) 
		       hl-start j))
		;; mIRC color -- S.B.
		((char-equal c ?\^c)
		 (setq prop-list (cons (list hl-start j fg bg bold ul inv)
				       prop-list))
		 ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
		 ;; clear the fg color specs by default (keep the bg)
		 (setq hl-start j
		       fg nil
;;		       bg nil
		       )
		 (let ((ccode (substring (car parts) 1)))
		   (if (string-match 
			"^\\([0-9]\\([0-9]\\)?\\(,[0-9]\\([0-9]\\)?\\)?\\)"
			ccode)
		       (let ((cspec (match-string 1 ccode)))
			 (cond ((string-match "^\\([0-9]+\\),\\([0-9]+\\)" cspec)
				(setq fg (string-to-number 
					  (match-string 1 cspec))
				      bg (string-to-number
					  (match-string 2 cspec))))
			       (t (setq fg (string-to-number cspec)))))
		     ;; otherwise, reset the colors
		     (setq bg nil))))
		;; End of text property -- S.B.
		((char-equal c ?\^o);; - end of all properties (??)
		 (setq prop-list (cons (list hl-start j fg bg bold ul inv)
				       prop-list))
		 ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
		 (setq bg nil
		       fg nil
		       bold nil
		       ul nil
		       inv nil
		       hl-start j))
;		((char-equal c ?\^q)
;		 ;; I don't know how to interpret this.  Just skip it for now.
;		 nil)
;		((char-equal c ?\^r)
;		 ;; I don't know how to interpret this.  Just skip it for now.
;		 nil)
		(t
		 (setq res (concat res (car parts)))
		 (setq j (+ j (length (car parts)))))))
	(setq parts (cdr parts)))
      (setq prop-list (cons (list hl-start j fg bg bold ul inv)
			    prop-list))
      ;;(erc-prepend-properties hl-start j res fg bg bold ul inv)
      ;; now prepend all the properties
      (while prop-list
	(apply 'erc-prepend-properties (cons res (car prop-list)))
	(setq prop-list (cdr prop-list)))
      ;;(erc-log (format "done: %S" (current-time)))
      ;;(erc-log (concat "erc-interpret-controls: res = " res))
      res))
   ((not (null line)) line)
   (t nil)))

;(defun erc-interpret-controls (line)
;  "Translate control characters in the line to faces and beeps"

;  (cond
;   ((not (null line))
;    (let ((res "")
;	  (i 0)
;	  (j 0)
;	  (len (length line))
;	  (bg nil) ;; number 0-15 (color)
;	  (fg nil) ;; number 0-15 (color)
;	  (bold nil)    ;; boolean
;	  (ul nil);; boolean (underline)
;	  (inv nil);; boolean (inverse) - not sure how it'll work...
;	  (hl-start 0);; position in the string where it starts
;	  (hl-face nil));; the list of faces ready for 
;                        ;; "font-lock-prepend-text-property"
;      (while (< i len)
;	(let ((c (string-to-char (substring line i))))
;;; 	(erc-log (format "intp-c: i %d c [%c] st %S fc %S"
;;; 			 i c hl-start hl-face))
;	  (cond ((char-equal c ?\^g)	; beep
;		 (ding t))
		
;		;; bold
;		((char-equal c ?\^b)
;;;	       (erc-log	(format "intp-c: BOLD i %d j %d st %S" i j hl-start))
;		 (erc-prepend-properties hl-start j res fg bg bold ul inv)
;		 (setq bold (not bold) 
;		       hl-start j))
		
;		;; underline
;		((char-equal c ?\^_)
;		 (erc-prepend-properties hl-start j res fg bg bold ul inv)
;		 (setq ul (not ul) 
;		       hl-start j))

;		;; inverse
;		((char-equal c ?\^v)
;		 (erc-prepend-properties hl-start j res fg bg bold ul inv)
;		 (setq inv (not inv) 
;		       hl-start j))
;		;; ICQ color -- S.B.
;		((char-equal c ?\^c)
;		 (erc-prepend-properties hl-start j res fg bg bold ul inv)
;		 ;; clear the fg color specs by default (keep the bg)
;		 (setq hl-start j
;		       fg nil
;;;		       bg nil
;		       )
;		 (let* ((rest (substring line (1+ i)))
;			(add-i 0)) ;; how much to advance 'i'
;		   (if (string-match 
;			"^\\([0-9]\\([0-9]\\)?\\(,[0-9]\\([0-9]\\)?\\)?\\)"
;			rest)
;		       (let ((cspec (match-string 1 rest)))
;			 (setq add-i (length cspec))
;			 (cond ((string-match "^\\([0-9]+\\),\\([0-9]+\\)" cspec)
;				(setq fg (string-to-number 
;					  (match-string 1 cspec))
;				      bg (string-to-number
;					  (match-string 2 cspec))))
;			       (t (setq fg (string-to-number cspec)))))
;		     ;; otherwise, reset the colors
;		     (setq bg nil))
;		   (setq i (+ i add-i))))
;		;; End of text property -- S.B.
;		((char-equal c ?\^o);; - end of all properties (??)
;		 (erc-prepend-properties hl-start j res fg bg bold ul inv)
;		 (setq bg nil
;		       fg nil
;		       bold nil
;		       ul nil
;		       inv nil
;		       hl-start j))
;;		((char-equal c ?\^q)
;;		 ;; I don't know how to interpret this.  Just skip it for now.
;;		 nil)
;;		((char-equal c ?\^r)
;;		 ;; I don't know how to interpret this.  Just skip it for now.
;;		 nil)
;		(t
;		 (setq res (concat res (char-to-string c)))
;		 (if (not (char-equal (string-to-char "\n") c))
;		     (put-text-property j (1+ j) 'face
;					(get-char-property i 'face line) res))
;		 (setq j (1+ j)))))
;	(setq i (1+ i)))
;      (erc-prepend-properties hl-start j res fg bg bold ul inv)
;      (erc-log (concat "erc-interpret-controls: res = " res))
;      res))

;   (t nil)))


(defun erc-merge-controls (line)
  "Translate control characters in the line to faces and beeps, then
 converts all properties into control symbols back (to mix both user
 entered and existing colors), and removes all the text properties from
 the string.

 This allows copying a colored line and sending it preserving the
 colors, and also to edit it without decoding the colors first."
  (erc-decode-controls line)
;  (erc-decode-controls
;   (erc-interpret-controls line))
  )

(defun erc-grab-region (start end)
  "Converts all the IRC text properties in each line of the region
into controls and writes them to a separate buffer.  The resulting
text can be used directly as a script to generate this text again."
  (interactive "r")
  (setq erc-active-buffer (current-buffer))
  (save-excursion
    (let* ((cb (current-buffer))
	   (buf (generate-new-buffer erc-grab-buffer-name))
	   (region (buffer-substring start end))
	   (lines (erc-split-multiline-safe region)))
      (set-buffer buf)
      (while lines
	(insert (concat (erc-decode-controls (car lines)) "\n"))
	(setq lines (cdr lines)))
      (set-buffer cb)
      (switch-to-buffer-other-window buf))))

(defun erc-display-prompt (&optional buffer pos)
  "Display ERC prompt (defined by `erc-prompt' variable) in BUFFER (if
 NIL or not provided - current buffer) using `erc-prompt-face' at the
 POS (if not NIL) or current position."

  (let ((ob (current-buffer))
	(pnt nil))
    (if (and buffer (bufferp buffer)) (set-buffer buffer))
    (setq pnt (point))
    ;; convert pos from mark-or-point to point
    (if pos (progn (goto-char pos)(setq pos (point))(goto-char pnt)))
    (if (> (length erc-prompt) 0)
	(let ((s (concat erc-prompt " ")))
	  (put-text-property 0 (length erc-prompt) 'face 'erc-prompt-face s)
	  (put-text-property (length erc-prompt) (length s)
			     'face 'erc-input-face s)
	  (if erc-prompt-interactive-input
	      (progn
		(put-text-property (length erc-prompt) (length s)
				   'local-map (erc-interactive-input-map) s)
		(put-text-property (length erc-prompt) (length s)
				   'front-sticky '(local-map) s)))
	  (if pos 
	      (progn (goto-char pos)
		     (insert s)
		     (if (>= pnt pos) (setq pnt (+ pnt (length s))))
		     (goto-char pnt))
	    (insert s))))
    (set-buffer ob)))


;; interactive operations

(defun erc-interactive-input-map ()
  (let ((lm (make-sparse-keymap)))
    (loop for c from 32 to 127 
	  do (define-key lm (vector c) 'erc-input-message))
;    (define-key lm [?\C-i] 'erc-complete)
    lm))

(defun erc-input-message ()
  (interactive)
  (let ((minibuffer-allow-text-properties t)
	;; This doesnt help, seems as hippie-expand insists on the point
	;; of the buffer ???
;	(channel-members channel-members)
	(read-map minibuffer-local-map))
    (define-key read-map [?\C-i] 'erc-complete)
    (insert (read-from-minibuffer "Message: " (string last-command-char) read-map))
    (erc-send-current-line)))

(defvar erc-action-history-list ()
  "History list for interactive action input")

(defun erc-input-action ()
  "Interactively input a user action and send it to IRC."
  (interactive "")
  (setq erc-active-buffer (current-buffer))
  (let ((action (read-from-minibuffer 
		 "Action: " nil nil nil erc-action-history-list)))
    (if (not (string-match "^\\s-*$" action))
	(progn
	  (erc-send-action (erc-default-target) action)))))

(defun erc-join-channel (channel)
  "Prompt for a channel name to join.
If POINT is at beginning of a channel name, use that as default."
  (interactive
   (list
    (let ((chnl (if (looking-at "\\(#[^ ]+\\)")
		    (match-string 1)
		  ""))
	  (table (if (and (boundp 'erc-process) (processp erc-process))
		     (progn
		       (set-buffer (process-buffer erc-process))
		       channel-list)
		   nil)))
      (completing-read "Join channel: " table nil nil (cons chnl 0)))))
  (erc-cmd-join "/JOIN" channel))

(defun erc-part-from-channel (reason)
  "Part from the current channel and prompt for a reason."
  (interactive
   (list
    (if (and (boundp 'reason) (stringp reason) (not (string= reason "")))
	reason
      (read-from-minibuffer (concat "Leave " (erc-default-target) ", Reason? ") (cons "No reason" 0)))))
  (erc-cmd-part "PART" (concat (erc-default-target)" " reason)))

(defun erc-set-topic (topic)
  "Prompt to set topic for the current channel."
  (interactive 
   (list
    (read-from-minibuffer (concat "Set topic of "
				  (erc-default-target)
				  ": ")
			  (when (boundp 'channel-topic)
			    (cons channel-topic 0)))))
  (erc-cmd-topic "TOPIC" topic))

			  
;; nick completion

(defun erc-complete ()
  "Complete nick at point.
See `erc-try-complete-nick' for more technical info."
  (interactive)
  (let ((hippie-expand-try-functions-list '(erc-try-complete-nick)))
    (hippie-expand nil)))

(defcustom erc-nick-completion 'all
  "Determine how the list of nicks is determined during nick completion.
See `erc-complete-nick' for information on how to activate this.

pals:   Use `erc-pals'.
all:    All channel members.

You may also provide your own function that returns a list of completions,
or you may use an arbitrary list expression."
  :type '(choice (const :tag "List of pals" pals)
		 (const :tag "All channel members" all)
		 (repeat :tag "List" (string :tag "Nick"))
		 function
		 sexp)
  :group 'erc)

(defun erc-try-complete-nick (old)
  "Complete nick at point.
This is a function to put on `hippie-expand-try-functions-list'.
Then use \\[hippie-expand] to expand nicks.
The type of completion depends on `erc-nick-completion'."
  (cond ((eq erc-nick-completion 'pals)
	 (try-complete-erc-nick old erc-pals))
	((eq erc-nick-completion 'all)
	 (try-complete-erc-nick old (mapcar (function car) channel-members)))
	((functionp 'erc-nick-completion)
	 (try-complete-erc-nick old (erc-nick-completion)))
	(t
	 (try-complete-erc-nick old erc-nick-completion))))

(defun try-complete-erc-nick (old completions)
  "Try to complete current word depending on `erc-try-complete-nick'.
The argument OLD has to be nil the first call of this function, and t
for subsequent calls (for further possible completions of the same
string).  It returns t if a new completion is found, nil otherwise.  The
second argument COMPLETIONS is a list of completions to use.  Actually,
it is only used when OLD is nil.  It will be copied to `he-expand-list'
on the first call.  After that, it is no longer used."
  (let (expansion)
    (unless old
      (let ((alist (if (consp (car completions))
		       completions
		     (mapcar (lambda (s)
			       (list (concat s ": ")))
			     completions)))); make alist if required
	(he-init-string (he-dabbrev-beg) (point))
	(unless (string= he-search-string "")
	  (setq expansion 
		(try-completion he-search-string alist)))
	(when (eq expansion t)
	  (setq expansion nil))
	(when (string= expansion he-search-string)
	  (with-output-to-temp-buffer "*Completions*"
	    (display-completion-list
	     (all-completions he-search-string alist)))
	  (setq expansion nil))))
    (if expansion
	(progn
	  (he-substitute-string expansion)
	  t)  
      (progn
	(if old (he-reset-string))
	nil))))

;;; Track hidden channel buffers for new messages

(defadvice switch-to-buffer (after erc-remove-channel-from-modded-list activate)
  "Check if buffer is in the `erc-modified-channel-string'."
  (if erc-tracking-modified-channels
      (erc-remove-channel-from-modded-list)))

(defun erc-remove-channel-from-modded-list () 
  "If the current buffer is an ERC channel, it will be removed  
from the `erc-modified-channel-string' and therefore removed from the  
mode-line." 
  (interactive) 
  (if (eq major-mode 'erc-mode) 
      (let ((this-channel (erc-default-target))) 
        (if (and this-channel 
		 (string-match (concat "\\(,?" this-channel "\\)")
			       erc-modified-channel-string))
            ;; only do this work if we are in an erc buffer and 
            ;; the buffer is in the list of channels modified 
            (progn  
              ;;(message "purging %s" this-channel) 
              (setq erc-modified-channel-string
                    (concat (substring erc-modified-channel-string
				       0 (match-beginning 1))
			    (substring erc-modified-channel-string
				       (match-end 1))))
              (if (equal erc-modified-channel-string " []")
		  (setq erc-modified-channel-string "")) 
              ))))) 

(defun erc-track-modified-channels () 
  "Hook function for `erc-insert-hook' to check if the current  
buffer should be added to the modeline as a hidden, modified  
channel.  Assumes it will only be called when current-buffer 
is in erc-mode." 
  (let ((this-channel (erc-default-target))) 
    (if (and (not (get-buffer-window (buffer-name))) 
	     this-channel
             (not (string-match this-channel erc-modified-channel-string))) 
        ;; only do this work if the buffer isn't visible and 
        ;; isn't already in our list 
        (progn  
          (setq erc-modified-channel-string
                (if (equal erc-modified-channel-string "")
		    (concat " [" this-channel "]") 
                  (concat (substring erc-modified-channel-string 0 -1)
			  "," this-channel "]"))) 
          )))) 
 
(defvar erc-tracking-modified-channels nil 
  "Internal variable used to indicate if modified channel 
tracking is currently in use. A value of t indicates that 
tracking is on and nil that tracking is off.
Use the function `erc-toggle-track-modified-channels' to toggle
channel tracking either interactively or in your .emacs
after loading erc.el.")
 
(defvar erc-modified-channel-string nil
  "Internal string used for displaying modified channels in the 
mode line.") 
 
(defun erc-toggle-track-modified-channels () 
  "The defun either initializes the tracking of modified channels  
or uninitializes the tracking of modified channels based on the  
value of `erc-tracking-modified-channels'" 
  (interactive) 
  (if erc-tracking-modified-channels 
      (progn                             
        (setq global-mode-string
	      (delq 'erc-modified-channel-string global-mode-string)) 
        (remove-hook 'erc-insert-hook 'erc-track-modified-channels)
	(setq erc-tracking-modified-channels nil)
	(message "Deactivated channel tracking..."))
    
    (progn           
      (or global-mode-string 
          (setq global-mode-string '(""))) 
      (or (memq 'erc-modified-channel-string global-mode-string) 
          (setq global-mode-string
		(append global-mode-string '(erc-modified-channel-string))))
      (setq erc-tracking-modified-channels t)
      (setq erc-modified-channel-string "")
      (add-hook 'erc-insert-hook 'erc-track-modified-channels)
      (erc-update-mode-line)
      (message "Channel tracking activaed...")))) 

;;; movement of point

; heavily borrowed from comint.el's `comint-bol' and `comint-skip-prompt'
(defun erc-bol ()
  "Place point at the beginning of the line just after the `erc-prompt'."
  (interactive)
  (beginning-of-line)
  (let ((eol (save-excursion (end-of-line) (point))))
    (when (and (looking-at (concat "^" (regexp-quote erc-prompt)))
	       (<= (match-end 0) eol))
      (goto-char (match-end 0))
      (if (looking-at " ")
	  (forward-char 1)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;;                        IRC SERVER INPUT HANDLING
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun erc-parse-line-from-server (string)
  "Parse the message received from server.
Returns the valuable part of the message."
  (erc-log (concat "parse: input: " string))
  (cond
   ;; numeric replies
   ((string-match
     "^:\\S-+\\s-+\\([0-9][0-9][0-9]\\)\\s-+\\S-+\\s-+\\(.*\\)$" string)
    (erc-process-num-reply
     (string-to-number (match-string 1 string)) (match-string 2 string)))
   ;; closing message
   ((string-match "^ERROR\\s-+:Closing\\s-+Link" string)
    (erc-update-mode-line)
    nil)
   ;; ping messages
   ((string-match "^\\s-*PING\\s-+:\\(\\S-+\\)" string)
    (let ((pinger (match-string 1 string)))
      (erc-log (format "PING: %s" pinger))
      (progn
	;; ping response to the server MUST be forced, or you can lose big
	(erc-send-command (format "PONG :%s" pinger) t)
	(if erc-paranoid 
	    (message "ERC: server ping (last: %s sec. ago)"
		     (erc-time-diff last-ping-time (erc-current-time))))
	(setq last-ping-time (erc-current-time)))
      nil))
   ;; pong messages
   ((string-match
     (concat "^:"
	     erc-session-server
	     "\\s-+PONG\\s-+\\(\\S-+\\)\\s-+:\\s-*\\(\\S-+\\)")
     string)
    nil)
   ;; server NOTICE messages
   ((or (string-match (concat "^\\s-*:" erc-session-server "\\s-+NOTICE\\s-+"
			      (erc-current-nick) "\\s-*:\\s-*") string)
	(string-match "^\\s-*NOTICE\\s-+\\S-+\\s-*:\\s-*" string))
    
    (erc-highlight-notice
     (format "%s%s" erc-notice-prefix (substring string (match-end 0)))))
   ;; PRIVMSG's and CTCP messages
   ((string-match
     "^:\\(\\S-+\\)\\s-+\\(PRIVMSG\\|NOTICE\\)\\s-+\\(\\S-+\\)\\s-+:[ \t]*\\(.*\\)$" string)
    (let ((sspec (match-string 1 string))
	  (cmd (match-string 2 string))
	  (tgt (match-string 3 string))
	  (msg (match-string 4 string)))
      (if (erc-ignored-user-p sspec)
	  (progn
	    (message "ignored %s from %s to %s" cmd sspec tgt)
	    nil)
	(let* ((sndr (erc-parse-user sspec))
	       (nick (nth 0 sndr))
	       (login (nth 1 sndr))
	       (host (nth 2 sndr))
	       (msgp (string= cmd "PRIVMSG"))
	       ;; S.B. downcase *both* tgt and current nick
	       (privp (string= (downcase tgt) (downcase (erc-current-nick))))
	       (mark-s (concat (if erc-timestamp-format
				   (format-time-string erc-timestamp-format
						       (current-time))
				 "")
			       (if msgp (if privp "*" "<") "-")))
	       (mark-e (if msgp (if privp "*" ">") "-"))
	       (s nil)
	       (buffer nil))
	  (erc-log (format "MSG: %s TO %s FROM %S <%s>" cmd tgt sndr msg))
	  (cond
	   ((string-match "^\C-a\\([^\C-a]*\\)\C-a?$" msg)
	    (setq s
		  (if (string= cmd "PRIVMSG")
		      (erc-process-ctcp-request nick (match-string 1 msg)
						host login)
		    (progn
		      (erc-process-ctcp-response nick (match-string 1 msg))
		      ;(setq buffer erc-active-buffer)
		      ))))
	   (t
	    (setcar last-peers nick)
	    (cond
	     ;; private message/notice
	     (privp
	      (setq s
		    (if away
			(format "%s%s%s <%s>  %s"
				mark-s nick mark-e
				(format-time-string "%a %b %d %H:%M"
						    (current-time))
				msg)
		      (format "%s%s%s %s" mark-s nick mark-e msg)))
	      (put-text-property 0 (length s) 'face 'erc-direct-msg-face s)
	      (if (not (get-buffer-window (process-buffer erc-process)))
		  (message "IRC message from %s: %s" nick msg)))
	     ;; channel message/notice
	     (t
	      (setq s (format "%s%s%s %s" mark-s nick mark-e msg))
	      (put-text-property 0 (length s) 'face 'erc-default-face s)))
	    (erc-log (format "DEBUG: apres cond: s: %s" s))
	    ;; mark the pal
	    (if (erc-pal-p nick)
		(cond
		 ((equal erc-pal-highlight-type 'nick)
		  (put-text-property (length mark-s)
				     (+ (length mark-s) (length nick))
				     'face 'erc-pal-face s))
		 ((equal erc-pal-highlight-type 'all)
		  (put-text-property (length mark-s) 
				     (+ (length s) (length nick))
				     'face 'erc-pal-face s))
		 (t nil)))
	    ;; mark the dangerous host
	    (if (erc-host-danger-p host)
		(cond
		 ((equal erc-pal-highlight-type 'nick)
		  (put-text-property (length mark-s)
				     (+ (length mark-s) (length nick))
				     'face 'erc-host-danger-face s))
		 ((equal erc-pal-highlight-type 'all)
		  (put-text-property (length mark-s) 
				     (+ (length s) (length nick))
				     'face 'erc-host-danger-face s))
		 (t nil)))
	    ;; highlight misc. strings
	    (if erc-highlight-strings
		(erc-highlight-strings s))))
	  (if (not buffer)
	      (setq buffer (erc-find-buffer erc-session-server 
					    (if privp nick tgt))))
	  (let ((ob (current-buffer)))
	    (if buffer (set-buffer buffer))
	    ;; update the chat partner info.  Add to the list if private
	    ;; message.  We will accumulate private identities indefinitely
	    ;; at this point.
	    (if (erc-update-channel-member (if privp nick tgt) nick nick 
					   privp nil nil host login)
		(erc-update-channel-info-buffer (if privp nick tgt)))
	    (run-hook-with-args 'erc-message-hook
				msg tgt nick (buffer-name buffer) host
				login sspec)
	    (set-buffer ob))
	  (cons s buffer)))))
;    (cons s (erc-get-buffer erc-session-server tgt (erc-current-nick)))))

   ;; MODE messages
   ((string-match
     "^:\\(\\S-+\\)\\s-+MODE\\s-+\\(\\S-+\\)\\s-+\\(.*\\)\\s-*$"
     string)
    (let* ((sspec (match-string 1 string))
	   (tgt (match-string 2 string))
	   (mode (match-string 3 string))
	   (sndr (erc-parse-user sspec))
	   (nick (nth 0 sndr))
	   (login (nth 1 sndr))
	   (host (nth 2 sndr)))
      (erc-log (format "MODE: %s -> %s: %s" nick tgt mode))
      ;; dirty hack
      (let ((ob (current-buffer))
	    (buf (cond ((string-match "^#" tgt)
			(erc-find-buffer erc-session-server tgt))
		       (erc-active-buffer erc-active-buffer)
		       (t (erc-find-buffer erc-session-server tgt))))
	    (res (if (or (string= login "") (string= host ""))
		     (erc-highlight-notice
		      (format "%s%s has changed mode for %s to %s"
			      erc-notice-prefix nick tgt mode))
		   (erc-highlight-notice
		    (format "%s%s (%s@%s) has changed mode for %s to %s"
			    erc-notice-prefix nick login host tgt mode)))
		 ))
	(if buf (set-buffer buf))
	(erc-update-modes tgt mode nick host login)
	(run-hook-with-args 'erc-mode-change-hook 
			    nick tgt mode host login sspec)
	(set-buffer ob)
	(cons res buf))))
;       (erc-get-buffer erc-session-server tgt (erc-current-nick)))))

   ;; NICK messages
   ((string-match
     "^:\\(\\S-+\\)\\s-+NICK\\s-+:\\s-*\\(.*\\)\\s-*$" string)
    (let* ((sspec (match-string 1 string))
	   (nn (match-string 2 string))
	   (sndr (erc-parse-user sspec))
	   (nick (nth 0 sndr))
	   (login (nth 1 sndr))
	   (host (nth 2 sndr))
	   (bufs (erc-buffer-list-with-nick nick)))
      (erc-log (format "NICK: %s -> %s" nick nn))
      (erc-update-member-all-channels nick nn nil nil nil host login)
      (erc-update-channel-info-buffers)
      (cond
       ((string= nick (erc-current-nick))
	(erc-push-nick nn)
	(erc-update-mode-line)
	(cons
	 (erc-highlight-notice
	  (format "%sYour new nickname is %s" erc-notice-prefix nn))
	 'active))
       (t
	(erc-handle-user-status-change 'nick
				       (list nick login host)
				       (list nn))
	(cons (erc-highlight-notice
	       (format "%s%s (%s@%s) has renamed himself to %s"
		       erc-notice-prefix nick login host nn))
	      bufs)))))

   ;; Initial NICK messages (server-side nick resolution)
   ((string-match
     "^:\\(\\S-+\\)\\s-+NICK\\s-+\\(.*\\)\\s-*$" string)
    (let* ((nn (match-string 2 string))
	   (nick (match-string 1 string))
	   (bufs (erc-buffer-list-with-nick nick)))
      (erc-log (format "NICK: %s -> %s" nick nn))
      (erc-update-member-all-channels nick nn)
      (erc-update-channel-info-buffers)
      (cond
       ((string= nick (erc-current-nick))
	(erc-push-nick nn)
	(erc-update-mode-line)
	(cons
	 (erc-highlight-notice
	  (format "%sYour new nickname is %s" erc-notice-prefix nn))
	 'active))
       (t
	(erc-handle-user-status-change 'nick
				       (list nick nil nil)
				       (list nn))
	(cons (erc-highlight-notice
	       (format "%s%s (%s@%s) has renamed himself to %s"
		       erc-notice-prefix nick "???" "???" nn))
	      bufs)))))

   ;; TOPIC messages
   ((string-match
     "^:\\(\\S-+\\)\\s-+TOPIC\\s-+\\(\\S-+\\)\\s-+:\\s-*\\(.*\\)\\s-*$" string)
    (let* ((sspec (match-string 1 string))
	   (ch (match-string 2 string))
	   (topic (match-string 3 string))
	   (sndr (erc-parse-user sspec))
	   (nick (nth 0 sndr))
	   (login (nth 1 sndr))
	   (host (nth 2 sndr))
	   (time (format-time-string "%T %m/%d/%y" (current-time))))
      (erc-log (format "TOPIC: %s for %s -> %s" nick ch topic))
      (erc-update-channel-member ch nick nick nil nil nil host login)
      (erc-update-channel-topic ch (format "%s\C-c (%s, %s)" topic nick time))
      (erc-update-channel-info-buffer ch)
      (cons
       (erc-highlight-notice
	(format "%s%s (%s@%s) has set the topic for %s: %s"
		erc-notice-prefix nick login host ch topic))
       (erc-find-buffer erc-session-server ch))))
;       (erc-get-buffer erc-session-server ch (erc-current-nick)))))

   ;; KICK messages
   ((string-match
     "^:\\(\\S-+\\)\\s-+KICK\\s-+\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+:\\s-*\\(.*\\)\\s-*$" string)
    (let* ((sspec (match-string 1 string))
	   (ch (match-string 2 string))
	   (tgt (match-string 3 string))
	   (reason (match-string 4 string))
	   (sndr (erc-parse-user sspec))
	   (nick (nth 0 sndr))
	   (login (nth 1 sndr))
	   (host (nth 2 sndr))
	   (buffer (erc-find-buffer erc-session-server ch)))
;	   (buffer (erc-get-buffer erc-session-server ch (erc-current-nick))))
      (erc-log (format "KICK: %s by %s: %s" tgt nick reason))
      (erc-remove-channel-member ch tgt)
      (erc-update-channel-info-buffer ch)
      (run-hook-with-args 'erc-kick-hook tgt ch nick reason host login sspec)
      (cons
       (erc-highlight-notice
	(cond
	 ((string= tgt (erc-current-nick))
	  (erc-delete-default-channel ch buffer)
	  (erc-update-mode-line buffer)
	  (format "%sYou have been kicked by %s off channel %s: %s"
		  erc-notice-prefix nick ch reason))
	 ((string= nick (erc-current-nick))
	  (format "%sYou have kicked %s off channel %s: %s"
		  erc-notice-prefix tgt ch reason))
	 (t
	  (format "%s%s (%s@%s) has kicked %s off channel %s: %s"
		  erc-notice-prefix nick login host tgt ch reason))))
       buffer)))

   ;; INVITE messages
   ((string-match
     (concat "^:\\(\\S-+\\)\\s-+INVITE\\s-+"
	     (erc-current-nick)
	     "\\s-+:\\s-*\\(.*\\)\\s-*$")
     string)
    (let* ((sspec (match-string 1 string))
	   (chnl (match-string 2 string))
	   (sndr (erc-parse-user sspec))
	   (nick (nth 0 sndr))
	   (login (nth 1 sndr))
	   (host (nth 2 sndr)))
      (erc-log (format "INVITE: by %s to %s" sspec chnl))
      (setq invitation chnl)
      (cons
       (erc-highlight-notice
	(format "%s%s (%s@%s) invites you to channel %s"
		erc-notice-prefix nick login host chnl))
       'active)))

   ;; JOIN messages
   ((string-match "^:\\(\\S-+\\)\\s-+JOIN\\s-+:\\s-*\\(\\S-+\\)"
		  string)
    (let* ((sspec (match-string 1 string))
	   (chnl (match-string 2 string))
	   (sndr (erc-parse-user sspec))
	   (nick (nth 0 sndr))
	   (login (nth 1 sndr))
	   (host (nth 2 sndr))
	   (buffer nil))
      (erc-log (format "JOIN: %S" sndr))
      ;; strip the stupid combined JOIN facility (IRC 2.9)
      (if (string-match "^\\(.*\\)?\^g.*$" chnl)
	  (setq chnl (match-string 1 chnl)))
      (let* ((ob (current-buffer))
	     (info-buf nil)
	     (str (cond
		   ;;;; If I have joined a channel
		   ;; downcasing nicks - don't want to rely on the server's
		   ;; correct casing
		   ((string= (downcase nick) (downcase (erc-current-nick)))
		    (erc erc-session-server erc-session-port nick 
			 erc-session-user-full-name nil nil def-rcpts 
			 chnl erc-process)
		    (setq buffer (erc-find-buffer erc-session-server chnl))
		    (if buffer
			(progn
			  (set-buffer buffer)
			  (erc-add-default-channel chnl)
			  ;; display the channel info buffer
			  ;;(message "JOIN: info-buf=%S" info-buf)
			  (setq info-buf (erc-find-channel-info-buffer chnl))
			  (cond ((eq erc-join-info-buffer 'frame)
				 (switch-to-buffer-other-frame info-buf))
				((eq erc-join-info-buffer 'window)
				 (switch-to-buffer-other-window info-buf))
				((eq erc-join-info-buffer 'split)
				 (split-window-vertically)
				 (switch-to-buffer-other-window info-buf)))
			  ;; and return to the channel buffer...
			  ;; boy, how to put the focus back into the channel
			  ;; window now???
			  (set-buffer buffer)))
		    (setq erc-last-channel-names nil)
		    (erc-refresh-channel-members chnl "")
		    (erc-update-mode-line)
		    (erc-highlight-notice
		     (format "%s%sYou have joined channel %s"
			     (if erc-timestamp-format
				 (format-time-string erc-timestamp-format
						     (current-time)) "")
			     erc-notice-prefix chnl)))
		   (t
		    (setq buffer (erc-find-buffer erc-session-server chnl))
		    (erc-highlight-notice
		     (format "%s%s%s (%s@%s) has joined channel %s"
			     (if erc-timestamp-format
				 (format-time-string erc-timestamp-format
						     (current-time)) "")
			     erc-notice-prefix nick login host chnl)))))
	     (res (cons str buffer)))
	(if buffer (set-buffer buffer))
;	(message "Running join-hook with buffer=%S" buffer-name)
	(erc-update-channel-member chnl nick nick t nil nil host login)
	(erc-update-channel-info-buffer chnl)
	(run-hook-with-args 'erc-join-hook chnl nick (buffer-name buffer)
			    host login sspec)
	;; on join, we want to stay in the new channel buffer
	;;(set-buffer ob)
	res)))

   ;; PART messages
   ((string-match
     "^:\\(\\S-+\\)\\s-+PART\\s-+\\(\\S-+\\)\\s-*\\(:\\s-*\\(.*\\)\\)?$"
     string)
    (let* ((sspec (match-string 1 string))
	   (chnl (match-string 2 string))
	   (reason (match-string 4 string))
	   (sndr (erc-parse-user sspec))
	   (nick (nth 0 sndr))
	   (login (nth 1 sndr))
	   (host (nth 2 sndr))
	   (buffer (erc-find-buffer erc-session-server chnl)))
;	   (buffer (erc-get-buffer erc-session-server chnl (erc-current-nick))))
      (erc-log (format "PART: %s by %s: %s" chnl sspec reason))
      (erc-remove-channel-member chnl nick)
      (erc-update-channel-info-buffer chnl)
      (run-hook-with-args 'erc-part-hook
			  chnl nick (buffer-name buffer) host reason)
      (cons
       (cond
	((string= nick (erc-current-nick))
	 (erc-delete-default-channel chnl buffer)
	 (erc-update-mode-line buffer)
	 (erc-highlight-notice
	  (format "%s%sYou have left channel %s"
		  (if erc-timestamp-format
		      (format-time-string erc-timestamp-format
					  (current-time)) "")
		  erc-notice-prefix chnl)))
	(t
	 (erc-highlight-notice
	  (format "%s%s%s (%s@%s) has left channel %s%s"
		  (if erc-timestamp-format
		      (format-time-string erc-timestamp-format
					  (current-time)) "")
		  erc-notice-prefix nick login host chnl
		  (if (and reason (not (string= reason "")))
		      (concat ": " reason)
		    "")))))
       	 buffer)))

   ;; QUIT messages
   ((string-match "^:\\(\\S-+\\)\\s-+QUIT\\s-+:\\s-*\\(.*\\)$"
		  string)
    (let* ((sspec (match-string 1 string))
	   (reason (match-string 2 string))
	   (sndr (erc-parse-user sspec))
	   (nick (nth 0 sndr))
	   (login (nth 1 sndr))
	   (host (nth 2 sndr))
	   (bufs (erc-buffer-list-with-nick nick)))
      (erc-remove-member-all-channels nick)
      (erc-update-channel-info-buffers)
      (run-hook-with-args 'erc-quit-hook nick host reason)
      (setq reason (erc-wash-quit-reason reason nick login host))
      (cons
       (erc-highlight-notice
	(format "%s%s%s (%s@%s) has quit: %s" 
		(if erc-timestamp-format
		    (format-time-string erc-timestamp-format
					(current-time)) "")
		erc-notice-prefix nick login host reason))
       bufs)
      ))

   ;; other
   (t string)))


(defun erc-wash-quit-reason (reason nick login host)
  "Remove duplicate text from quit reason in relation to nick (user@host) 
information.
Returns reason unmodified if nothing can be removed."
  (or (when (string-match (concat "^\\(Read error\\) to "
				  nick "\\[" host "\\]: "
				  "\\(.+\\)$") reason)
	(concat (match-string 1 reason) ": " (match-string 2 reason)))
      (when (string-match (concat "^\\(Ping timeout\\) for "
				  nick "\\[" host "\\]$") reason)
	(match-string 1 reason))
      reason))
  
(defun erc-process-num-reply (n s)
  "Process numeric reply from server. N is a numeric code,
S is a trailing message"
  
  (erc-log (format "process-num-reply: [%d] %s" n s))
  (cond

   ;; endings to ignore
   ((or (= n 315)
	(= n 318)
	(= n 365)
	(= n 366)
	(= n 369)
	(= n 374)
	(= n 376))
    nil)

   ;; error replies
   
   ;; login-time 'nickname in use' message
   ((and (= n 433)
	 (string-match "^\\(\\S-+\\)\\s-*:" s))
    (let* ((nick (match-string 1 s))
	   (newnick (nth 1 erc-default-nicks)))
      (setq bad-nick t)
      ;; try to use a different nick
      (if erc-default-nicks (setq erc-default-nicks (cdr erc-default-nicks)))
      (if (not newnick)
	  (setq newnick (concat nick "^")))
      (erc-cmd-nick "NICK" newnick t)
      (erc-highlight-error 
       (format "%sNickname \'%s\' is already in use, trying %s"
	       erc-notice-prefix nick newnick))))

   ;; other error replies
   ((and (>= n 400)
	(< n 500))
    (cons (erc-highlight-error (concat erc-notice-prefix s))
	  'active))

   ;; "good" numeric replies

   ;; MOTD, INFO and other weird messages
   ((and
     (or (= n 371) (= n 372) (= n 375) (= n 376)
	 (and (>= n 1) (<= n 4))
	 (and (>= n 251) (<= n 255)))
     (string-match "^:\\s-*\\(.*\\)$" s))
    (let ((msg (match-string 1 s)))
      (erc-log (format "MOTD/%d: %s" n msg))
      (if (not erc-logged-in)
	  (progn
	    (setq erc-logged-in t)
	    (message "Logging in as \'%s\'... done" (erc-current-nick))
	    ;; execute a startup script
	    (let ((f (erc-select-startup-file)))
	      (if f
		  (progn
		    ;;(message "Loading \'%s\'..." f)
		    (erc-load-script f)
		    ;;(message "Loading \'%s\'... done" f)
		    )))))
      (cons (erc-highlight-notice (format "%s%s" erc-notice-prefix msg))
	    'active)))

   ;; LUSEROP, LUSERUNKNOWN, LUSERCHANNELS
   ((and (>= n 252) (<= n 254)
	 (string-match "^\\s-*\\([0-9]+\\)\\s-*:\\s-?\\(.*\\)$" s))
    (let ((nops (match-string 1 s))
	  (msg (match-string 2 s)))
      (cons (erc-highlight-notice
	     (format "%s%s %s" erc-notice-prefix nops msg))
	    'active)))
   
   ;; ADMIN messages
   ((and (>= n 256) (<= n 259)
	 (string-match "^:\\s-*\\(.*\\)\\s-*$" s))
    (let ((msg (match-string 1 s)))
      (cons (erc-highlight-notice (concat erc-notice-prefix msg))
	    'active)))
   
   ;; AWAY notice
   ((and (= n 301)
	 (string-match "^\\(\\S-+\\)\\s-+:\\s-*\\(.*\\)$" s))
    (let ((nick (match-string 1 s))
	  (msg (match-string 2 s)))
      (cons (erc-highlight-notice
	     (format "%s%s is AWAY: %s" erc-notice-prefix nick msg))
	    'active)))

   ;; AWAY messages
   ((and
     (or (= n 305) (= n 306))
     (string-match "^:\\s-*\\(.*\\)\\s-*$" s))
    (erc-process-away (= n 306))
    (let ((msg (match-string 1 s)))
      (cons (erc-highlight-notice (concat erc-notice-prefix msg))
	    'active)))
   
   ;; WHOIS/WAS notices
   ((and (or (= n 311) (= n 314))
	 (string-match
	  "^\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+\\*\\s-+:\\s-*\\(.*\\)$"
	  s))
    (let ((nick (match-string 1 s))
	  (user (match-string 2 s))
	  (host (match-string 3 s))
	  (fname (match-string 4 s)))
      (erc-update-member-all-channels nick nick nil nil nil host user fname)
      (erc-update-channel-info-buffers)
      (cons (erc-highlight-notice
	     (format "%s%s %s %s (%s@%s)"
		     erc-notice-prefix nick
		     (if (= n 311)
			 "is"
		       "was")
		     fname user host))
	    'active)))

   ;; WHOISOPERATOR
   ((and (= n 313)
	 (string-match "^\\s-*\\(\\S-+\\)\\s-*:\\s-?\\(.*\\)$" s))
    (let ((nick (match-string 1 s))
	  (msg (match-string 2 s)))
      (cons (erc-highlight-notice
	     (format "%s%s %s" erc-notice-prefix nick msg))
	    'active)))
   
   ;; IDLE notice
   ((and (= n 317)
	 (string-match
	  "^\\(\\S-+\\)\\s-+\\([0-9]+\\)\\s-*\\([0-9]+\\)?\\s-*:\\s-*seconds idle.*$"
	  s))
    (let* ((nick (match-string 1 s))
	   (nsec (string-to-number (match-string 2 s)))
	   (time-str (match-string 3 s))
	   (time (if time-str
		     (format-time-string "%T %m/%d/%y" 
					 (erc-string-to-emacs-time time-str))
		   nil)))
      (erc-update-member-all-channels nick nick nil nil nil nil nil nil
				      (if time 
					  (format "on since %s" time) nil))
      (erc-update-channel-info-buffers)
      (cons (erc-highlight-notice 
	     (format "%s%s is idle for %s%s"
		     erc-notice-prefix
		     nick
		     (erc-sec-to-time nsec)
		     (if time (format ", on since %s" time) "")))
	    'active)))

   ;; server notice
   ((and (= n 312)
	 (string-match
	  "^\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+:\\s-*\\(.*\\)\\s-*$" s))
    (let* ((nick (match-string 1 s))
	   (saddr (match-string 2 s))
	   (srvr (match-string 3 s)))
      (cons (erc-highlight-notice
	     (format "%s%s is/was on server %s (%s)"
		     erc-notice-prefix nick saddr srvr))
	    'active)))

   ;; channels notice
   ((and (= n 319)
	 (string-match "^\\(\\S-+\\)\\s-+:\\s-*\\(.*\\)\\s-*$" s))
    (let* ((nick (match-string 1 s))
	   (chnls (match-string 2 s)))
      (cons (erc-highlight-notice
	     (format "%s%s is on channel(s): %s"
		     erc-notice-prefix nick chnls))
	    'active)))

   ;; LIST header
   ((= n 321)
    (progn
      (setq channel-list ())
      (cons (erc-highlight-notice (format "%s%s" erc-notice-prefix s))
	    'active)))
   
   ;; LIST notice
   ((and (= n 322)
	 (string-match
	  "^\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+:\\s-*\\(.*\\)\\s-*$" s))
    (let* ((chnl (match-string 1 s))
	   (nv (match-string 2 s))
	   (topic (match-string 3 s))
	   (res (erc-highlight-notice
		 (if (string= topic "")
		     (format "%s%s [%s]" erc-notice-prefix chnl nv)
		   (format "%s%s [%s]: %s" erc-notice-prefix chnl nv topic)))))
      (add-to-list 'channel-list (list chnl))
      (erc-update-channel-topic chnl topic)
      (erc-update-channel-info-buffer chnl)
      (cons res 'active)))

   ;; LIST footer
   ((and (= n 323)
	 (string-match
	  "^\\s-*:\\(.*\\)$" s))
    (let* ((msg (match-string 1 s)))
      (cons (erc-highlight-notice (format "%s%s" erc-notice-prefix msg))
	    'active)))

   ;; WHO notice
   ((and (= n 352)
	 (string-match
	  "^\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+:\\([0-9]+\\)\\s-+\\(.*\\)\\s-*$" s))
    (let* ((chnl (match-string 1 s))
	   (usr (match-string 2 s))
	   (hst (match-string 3 s))
	   (srv (match-string 4 s))
	   (nick (match-string 5 s))
	   (fl (match-string 6 s))
	   (hc (match-string 7 s))
	   (fn (match-string 8 s)))
      (cons (erc-highlight-notice
	     (format "%-11s %-10s %-4s %s@%s (%s)" chnl nick fl usr hst fn))
	    'active)))

   ;; users on the channel
   ((and (= n 353)
	 (string-match "^[*=]\\s-+\\(\\S-+\\)\\s-+:\\s-*\\(.*\\)\\s-*$" s))
    (let ((chnl (match-string 1 s))
	  (users (match-string 2 s)))
      (erc-refresh-channel-members chnl users t)
      (erc-update-channel-info-buffer chnl)
      (cons
       (erc-highlight-notice
	(format "%sUsers on %s: %s" erc-notice-prefix chnl users))
       'active)))
;       (erc-get-buffer erc-session-server chnl (erc-current-nick)))))

   ;; INVITE response
   ((and (= n 341)
	 (string-match "^\\s-*\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-*$" s))
    (let ((nick (match-string 1 s))
	  (chnl (match-string 2 s)))
      (cons
       (erc-highlight-notice
       (format "%s%s was invited to channel %s" erc-notice-prefix nick chnl))
       (erc-find-buffer erc-session-server chnl))))

   ;; TOPIC notice
   ((and (= n 332)
	 (string-match "^\\(\\S-+\\)\\s-+:\\(.*\\)$" s))
    (let ((chnl (match-string 1 s))
	  (topic (match-string 2 s)))
      (erc-log (format "TOPIC [332]: chnl=%S, topic=%S" chnl topic))
      (erc-update-channel-topic chnl topic)
      (erc-update-channel-info-buffer chnl)
      (cons (erc-highlight-notice
		 (format "%s%s topic: %s" erc-notice-prefix chnl topic))
	    'active)))

   ;; who set the topic and when
   ((and (= n 333)
	 (string-match "^\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+\\([0-9]+\\)$" s))
    (let* ((chnl (match-string 1 s))
	   (nick (match-string 2 s))
	   (time-str (match-string 3 s))
	   (time (format-time-string "%T %m/%d/%y"
				     (erc-string-to-emacs-time time-str)))
	   (res (erc-highlight-notice
		 (format "%s%s: topic set by %s, %s"
			 erc-notice-prefix chnl nick time))))
      (erc-update-channel-topic chnl (format "\C-c (%s, %s)" nick time)
				'append)
      (erc-update-channel-info-buffer chnl)
      (cons res 'active)))

   ;; channel or nick modes
   ((and (= n 324)
	 (string-match "^\\(\\S-+\\)\\s-+\\(.*\\)$" s))
    (let* ((chnl (match-string 1 s))
	   (modes (match-string 2 s))
	   (s (format "%s%s modes: %s" erc-notice-prefix chnl modes)))
      (erc-set-modes chnl modes)
      (cons (erc-highlight-notice s)
	    (erc-find-buffer erc-session-server chnl))))

   ;; Channel living time
   ((and (= n 329)
	 (string-match "^\\(\\S-+\\)\\s-+\\(.*\\)$" s))
    (let* ((chnl (match-string 1 s))
	   (time (erc-string-to-emacs-time (match-string 2 s)))
	   (s (format "%s%s was created on %s"
		      erc-notice-prefix 
		      chnl 
		      (format-time-string "%A %Y-%m-%d %X" time))))
      (cons (erc-highlight-notice s) 'active)))
   ;; channel ban list (response for /mode #channel +b)
   ((and (= n 367)
	 (string-match 
	  "^\\s-*\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+\\([0-9]+\\)"
	  s))
    (let* ((chnl (match-string 1 s))
	   (mask (match-string 2 s))
	   (nick (match-string 3 s))
	   (time (erc-string-to-emacs-time (match-string 4 s)))
	   (s (format "%s is banned in %s by %s at %s"
		      mask chnl nick
		      (format-time-string "%T %m/%d/%y" time))))
      (cons (erc-highlight-notice s) 'active)))

   ;; links
   ((and (= n 364)
	 (string-match
	  "^\\(\\S-+\\)\\s-+\\(\\S-+\\)\\s-+:\\s-*\\([0-9]+\\)\\s-+\\(.*\\)\\s-*$" s))
    (let ((mask (match-string 1 s))
	   (svr (match-string 2 s))
	   (hc (match-string 3 s))
	   (sinfo (match-string 4 s)))
      (cons (erc-highlight-notice
	     (format "%s%-20s %-20s %2s %s"
		     erc-notice-prefix mask svr hc sinfo))
	    'active)))

   ;; time
   ((and (= n 391)
	 (string-match "^\\(\\S-+\\)\\s-*:\\s-*\\(.*\\)$" s))
    (let ((svr (match-string 1 s))
	  (tm (match-string 2 s)))
      (cons (erc-highlight-notice (format "%s%s: %s"
					  erc-notice-prefix svr tm))
	    'active)))

   ((and (= n 331)
	 (string-match "^\\s-*\\(\\S-+\\)\\s-*:\\s-*\\(.*\\)\\s-*$" s))
    (let ((chnl (match-string 1 s))
	  (topic (match-string 2 s)))
;;    (erc-log (format "process-num-reply: got the 331 one with chnl=%s, topic=%s" chnl topic))
      (cons
       (erc-highlight-notice
	(format "Topic on %s: %s" chnl topic))
       (erc-find-buffer erc-session-server chnl))))
;       (erc-get-buffer erc-session-server chnl (erc-current-nick)))))

   ;; other
   (t
    (cons (erc-highlight-notice (format "%s[%d] %s" erc-notice-prefix n s))
	  'active))))


(defun erc-process-ctcp-request (sndr msg &optional host login)
  "Process incoming CTCP request.
SNDR is sender's nickname. MSG is the message contents. The function
returns a string to be displayed or NIL"
  (erc-log (format "process-ctcp-request: [%s] %s" sndr msg))
;;; Don't display everything, only those that usually go silent
;  (if erc-paranoid
;      (erc-display-line
;       (erc-highlight-error
;	(format "==> CTCP request from %s: %s\n" sndr msg))))
  (cond
   ;; ACTION
   ((string-match "^ACTION\\s-\\(.*\\)\\s-*$" msg)
    (let ((s (match-string 1 msg)))
      (put-text-property 0 (length s) 'face 'erc-action-face s)
      (concat (if erc-timestamp-format
		  (format-time-string erc-timestamp-format
				      (current-time)) "")
	      sndr erc-action-separator s)))
   ;; CLIENTINFO
   ((string-match "^CLIENTINFO\\(\\s-*\\|\\s-+.*\\)$" msg)
    (if erc-paranoid
	(erc-display-line
	 (erc-highlight-error
	  (format "==> CTCP request from %s (%s@%s): %s\n" 
		  sndr login host msg))))
    (if (not erc-disable-ctcp-replies)
	(let ((s (erc-client-info (erc-trim-string (match-string 1 msg)))))
	  (erc-send-ctcp-notice sndr (format "CLIENTINFO %s" s))))
    nil)
   ;; ECHO
   ((string-match "^ECHO\\s-+\\(.*\\)\\s-*$" msg)
    (if erc-paranoid
	(erc-display-line
	 (erc-highlight-error
	  (format "==> CTCP request from %s (%s@%s): %s\n" sndr login host msg))))
    (if (not erc-disable-ctcp-replies)
	(let ((s (match-string 1 msg)))
	  (erc-send-ctcp-notice sndr (format "ECHO %s" s))
	  (put-text-property 0 (length s) 'face 'erc-action-face s)
	  (concat sndr " [ECHO]> " s))))
   ;; FINGER
   ((string= "FINGER" msg)
    (if erc-paranoid
	(erc-display-line
	 (erc-highlight-error
	  (format "==> CTCP request from %s (%s@%s): %s\n" sndr login host msg))))
    (if (not erc-disable-ctcp-replies)
        (let ((s (if erc-anonymous-login
		 (format "FINGER I'm %s." (erc-current-nick))
	       (format "FINGER %s (%s@%s)."
		       (user-full-name)
		       (user-login-name)
		       (system-name))))
	  (ns (erc-time-diff last-sent-time (erc-current-time))))
      (if (> ns 0)
	  (setq s (concat s " Idle for " (erc-sec-to-time ns))))
      (erc-send-ctcp-notice sndr s)))
    nil)
   ;; PING
   ((string-match "^PING\\s-+\\([0-9]+\\)" msg)
    (if erc-paranoid
	(erc-display-line
	 (erc-highlight-error
	  (format "==> CTCP request from %s (%s@%s): %s\n" sndr login host msg))))
    (if (not erc-disable-ctcp-replies)
	(erc-send-ctcp-notice sndr (format "PING %s" (match-string 1 msg))))
    nil)
   ;; TIME
   ((string= msg "TIME")
    (if erc-paranoid
	(erc-display-line
	 (erc-highlight-error
	  (format "==> CTCP request from %s (%s@%s): %s\n" sndr login host msg))))
    (if (not erc-disable-ctcp-replies)
	(erc-send-ctcp-notice sndr (format "TIME %s" (current-time-string))))
    nil)
   ;; USERINFO
   ((string= msg "USERINFO")
    (if erc-paranoid
	(erc-display-line
	 (erc-highlight-error
	  (format "==> CTCP request from %s (%s@%s): %s\n" sndr login host msg))))
    (if (not erc-disable-ctcp-replies)
	(erc-send-ctcp-notice sndr (format "USERINFO %s"
					   erc-user-information)))
    nil)
   ;; VERSION request
   ((string= "VERSION" msg)
    (if erc-paranoid
	(erc-display-line
	 (erc-highlight-error
	  (format "==> CTCP request from %s (%s@%s): %s\n" sndr login host msg))))
    (if (not erc-disable-ctcp-replies)
	(erc-send-ctcp-notice
	 sndr
	 (format "VERSION \C-bERC\C-b v%s - an IRC client for emacs (\C-b%s\C-b)"
		 erc-version-string
		 erc-official-location)))
    nil)
   ;; SOUND request
   ((string-match "SOUND\\s-+\\(\\S-+\\)\\(\\(\\s-+.*\\)\\|\\(\\s-*\\)\\)$" msg)
    (let ((sound (match-string 1 msg))
	  (comment (match-string 2 msg)))
      (if erc-play-sound (erc-play-sound sound))
      (erc-highlight-notice
       (format "%s%s plays %s:%s" erc-notice-prefix sndr sound comment))))
   ;; PAGE request
   ((string-match "PAGE\\(\\s-+.*\\)?$" msg)
    (let* ((m (match-string 1 msg))
	   (page-msg (if m (erc-interpret-controls (substring m 1))
		       "[no message]")))
      (if m (setq m (substring m 1)))
      (erc-highlight-notice
       (if erc-page-function
	   (progn
	     (apply erc-page-function (list sndr page-msg))
	     (concat erc-notice-prefix sndr " pages you: " m))
	 (progn
	   (message (concat "PAGE from " sndr "(" login "@" 
			    host "): " page-msg))
	   (beep)
	   (concat erc-notice-prefix sndr " pages you: " m))))))
   ;; all other requests
   (t
    (erc-highlight-notice
     (format "%sUnknown CTCP request from %s (%s@%s): %s"
	     erc-notice-prefix sndr login host msg)))))

(defun erc-process-ctcp-response (sndr msg)
  "Process incoming CTCP response.
SNDR is sender's nickname.
MSG is the message contents."
  (erc-log (format "process-ctcp-response: [%s] %s" sndr msg))
  (cond
   ;; ECHO
   ((string-match "^ECHO\\s-+\\(.*\\)\\s-*$" msg)
    (let ((s (format "%s [ECHO]> %s" nick (match-string 1 msg))))
      (put-text-property 0 (length s) 'face 'erc-action-face s)
      s))
   ;; CLIENTINFO
   ((string-match "^CLIENTINFO\\s-+\\(.*\\)\\s-*$" msg)
    (let ((s (format "%s [CLIENTINFO]> %s" nick (match-string 1 msg))))
      (put-text-property 0 (length s) 'face 'erc-action-face s)
      s))
   ;; FINGER
   ((string-match "^FINGER\\s-+\\(.*\\)\\s-*$" msg)
    (let ((s (format "%s [FINGER]> %s" nick (match-string 1 msg))))
      (put-text-property 0 (length s) 'face 'erc-action-face s)
      s))
   ;; PING
   ((string-match "^PING\\s-+\\([0-9]+\\)" msg)
    (let* ((pt (string-to-number (concat (match-string 1 msg) ".0")))
	   (ct (erc-current-time))
	   (ns (erc-time-diff pt ct))
	   (pair (assoc pt pings)))
      (cond
       (pair
	;; do not delete our pings - we may have several replies if a
	;; ping was sent to a channel
	;(setq pings (delete pair pings))
	(erc-highlight-notice
	 (format "%sPing time to %s%s is %s"
		 erc-notice-prefix
		 sndr 
		 (if (string= sndr (cdr pair)) ""
		   (concat " (" (cdr pair) ")"))
		 (erc-sec-to-time ns))))
       (t
	(erc-highlight-error
	 (format "Unexpected PING response from %s (time %s)" nick pt))))))
   ;; TIME
   ((string-match "^TIME\\s-+\\(.*\\)\\s-*$" msg)
    (let ((s (format "%s [TIME]> %s" nick (match-string 1 msg))))
      (put-text-property 0 (length s) 'face 'erc-action-face s)
      s))
   ;; VERSION response
   ((string-match "^VERSION\\s-+\\(.*\\)\\s-*$" msg)
    (let ((s (format "%s [VERSION]> %s" nick (match-string 1 msg))))
      (put-text-property 0 (length s) 'face 'erc-action-face s)
      s))
   (t
    (erc-highlight-notice
     (format "%sUnknown CTCP message from %s: %s"
	     erc-notice-prefix nick msg)))))

(defun erc-process-away (away-p)
  "Do additional processing of user going away (if AWAY-P is non-nil),
or coming back"
  (cond
   (away-p
    (setq away (current-time)))
   (t
    (let ((away-time away))
      ;; away must be set to NIL BEFORE sending anything to prevent
      ;; an infinite recursion
      (setq away nil)
      (erc-send-action
       (erc-default-target)
       (if away-time
	   (format "is back (gone for %s)"
		   (erc-sec-to-time (erc-time-diff
				     (erc-emacs-time-to-erc-time away-time)
				     (erc-current-time))))
	 "is back")))))
  (erc-update-mode-line))

;;;; List of channel members handling

(defun erc-refresh-channel-members (chnl names-string &optional add)

  "If there is a buffer for CHNL, updates the `channel-members'
variable for that buffer according to NAMES-STRING - the string
listing all the names on the channel.  If optional ADD is non-nil, do
not remove existing names from the list."

  (let ((names nil)
	(name nil)
	(op nil)
	(voice nil)
	(res nil)
	(buf (and (string-match "^#" chnl)
		  (boundp 'erc-session-server)
		  erc-session-server
		  (erc-find-buffer erc-session-server chnl nil t)))
	(ob (current-buffer)))
    (if buf
	(progn
	  (set-buffer buf)
	  (while (string-match "^\\s-*\\(\\S-+\\)\\(\\s-+.*$\\|$\\)" 
			       names-string)
	    (setq names (cons (match-string 1 names-string) names))
	    (setq names-string (match-string 2 names-string)))
	  (if (not add) (setq channel-members nil))
	  (while names
	    (cond ((string-match "^@\\(.*\\)$" (car names))
		   (setq name (match-string 1 (car names))
			 op 'on
			 voice 'off))
		  ((string-match "^+\\(.*\\)$" (car names))
		   (setq name (match-string 1 (car names))
			 op 'off
			 voice 'on))
		  (t (setq name (car names)
			   op 'off
			   voice 'off)))
	    (erc-update-channel-member chnl name name add op voice)
	    (setq names (cdr names)))
	  (set-buffer ob)))))

(defun erc-update-channel-member (chnl nick new-nick
				       &optional add op voice host email
				       full-name info)

  "Updates the user info in the channel CHNL.  The user's NICK will be
changed to NEW-NICK (provide the same nick if not changed).  If ADD is
non-nil, add the user if it is not in the list.  The other optional
arguments OP, HOST, EMAIL and FULL-NAME change the appropriate fields.
INFO is the additional info (sign-on time, comments, etc.), VOICE is
the voice mode in the channel.  Note, that OP = NIL means the status
does not change, so use 'ON or 'OFF to set the OP status instead of T
and NIL.  The same goes for VOICE.

Returns non-nil if the info is actually updated."

  (let ((buf (and ;;(string-match "^#" chnl) update for private chat too
		  (boundp 'erc-session-server)
		  erc-session-server
		  (erc-find-buffer erc-session-server chnl)))
	(ob (current-buffer))
	(res nil)
	(names nil)
	(entry nil)
	(ently-local nil)
	(new-entry nil)
	(changed nil))
    (if buf
	(progn
	  (set-buffer buf)
	  (setq names (and (boundp 'channel-members)
			   channel-members))
	  (setq entry-local (assoc nick names))
	  (setq entry (or entry-local
			  (assoc nick erc-last-channel-names)))
	  (if entry
	      (progn
		(erc-log (format "update-member: entry = %S" entry))
		;;(message "update-member: entry = %S, add=%s" entry add)
		(let* ((nick0 (nth 0 entry))
		       (op0 (nth 1 entry))
		       (voice0 (nth 2 entry))
		       (host0 (nth 3 entry))
		       (email0 (nth 4 entry))
		       (full0 (nth 5 entry))
		       (info0 (nth 6 entry)))
		  (setq new-entry (list new-nick
					(cond ((eq op 'on) t)
					      ((eq op 'off) nil)
					      (entry-local op0)
					      (t nil))
					(cond ((eq voice 'on) t)
					      ((eq voice 'off) nil)
					      (entry-local voice0)
					      (t nil))
					(if host host host0)
					(if email email email0)
					(if full-name full-name full0)
					(if info info info0))))
		(cond ((and entry-local (not (equal entry-local new-entry)))
		       (erc-log (format "update-member: new-entry = %S"
					new-entry))
		       (while names
			 (let* ((entry0 (car names))
				(nick0 (nth 0 entry0)))
			   (setq res (cons (if (string= (downcase nick)
							(downcase nick0))
					       new-entry entry0)
					   res)))
			 (setq names (cdr names)))
		       (setq channel-members (reverse res))
		       (setq changed t))
		      ;; no nick in the list yet, add it if needed
		      ((and (not entry-local) add)
		       (setq channel-members
			     (cons new-entry channel-members))
		       (setq changed t))))
	    (if add
		(progn
		  (setq channel-members (cons (list new-nick
						    (cond ((eq op 'on) t)
							  ((eq op 'off) nil)
							  (t op))
						    (cond ((eq voice 'on) t)
							  ((eq voice 'off) nil)
							  (t voice))
						    host email full-name info)
					      names))
		  (setq changed t))))
	  (set-buffer ob)))
    changed))

(defun erc-get-channel-members (chnl) 
  "Returns the value of `channel-members' in the channel buffer assigned
to CHNL."
  (let ((buf (and (string-match "^#" chnl)
		  (boundp 'erc-session-server)
		  erc-session-server
		  (erc-find-buffer erc-session-server chnl)))
	(ob (current-buffer))
	(names nil))
    (if buf
	(progn
	  (set-buffer buf)
	  (setq names (and (boundp 'channel-members)
			   channel-members))
	  (set-buffer ob)))
    names))
	  

(defun erc-remove-channel-member (chnl nick) 

  "Takes CHNL and NICK, removes the NICK form the channel's CHNL
membership list."
  (setq nick (downcase nick))
  (let* ((buf (and (boundp 'erc-session-server)
		   erc-session-server
		   (erc-find-buffer erc-session-server chnl)))
	 (names nil)
	 (ob (current-buffer))
	 (res nil))
    (if buf
	(progn
	  (set-buffer buf)
	  (setq names (and (boundp 'channel-members) channel-members))
	  (while names
	    (if (not (string= (downcase (car (car names))) nick))
		(setq res (cons (car names) res)))
	    (setq names (cdr names)))
	  (setq channel-members (reverse res))
	  (set-buffer ob)))))

(defun erc-update-channel-topic (chnl topic &optional modify)
  "Finds a buffer for the channel CHNL and sets the TOPIC for it.  If
optional MODIFY is 'append or 'prepend, then append or prepend the
TOPIC string to the current topic."

  (let* ((buf (and (string-match "^#" chnl)
		   (boundp 'erc-session-server)
		   erc-session-server
		   (erc-find-buffer erc-session-server chnl)))
	 (ob (current-buffer)))
    (if buf
	(progn
	  (set-buffer buf)
	  (if (boundp 'channel-topic)
	      (cond ((eq modify 'append)
		     (setq channel-topic (concat channel-topic topic)))
		    ((eq modify 'prepend)
		     (setq channel-topic (concat topic channel-topic)))
		    (t (setq channel-topic topic))))
	  (set-buffer ob)))))

(defun erc-set-modes (tgt mode-string)
  "Set the modes for the TGT provided as MODE-STRING."
  (let* ((modes (erc-parse-modes mode-string))
	 (add-modes (nth 0 modes))
	 (remove-modes (nth 1 modes))
	 ;; list of triples: (mode-char 'on/'off argument)
	 (arg-modes (nth 2 modes)))
    (cond ((string-match "^#" tgt); channel modes
	   (let ((buf (and (boundp 'erc-session-server)
			   erc-session-server
			   (erc-find-buffer erc-session-server tgt)))
		 (ob (current-buffer)))
	     (if buf
		 (progn
		   (set-buffer buf)
		   (setq channel-modes add-modes)
		   (setq channel-user-limit nil)
		   (while arg-modes
		     (let ((mode (nth 0 (car arg-modes)))
			   (onoff (nth 1 (car arg-modes)))
			   (arg (nth 2 (car arg-modes))))
		       (cond ((string-match "^[Ll]" mode)
			      (erc-update-channel-limit tgt onoff arg))
			     (t nil)))
		     (setq arg-modes (cdr arg-modes)))
		   (erc-update-channel-info-buffer tgt)
		   (set-buffer ob)))))
	  (t nil)) ; we do not keep our nick's modes yet
    ))

(defun erc-sort-strings (strings) 

  "Sorts the list of STRINGS in the lexicographic order.  Does not
have a side effect."

  (let ((lst nil))
    (while strings
      (setq lst (cons (car strings) lst))
      (setq strings (cdr strings)))
    (sort lst 'string<)))

(defun erc-parse-modes (mode-string)

  "Return a list of 3 elements: '(add-modes remove-modes arg-modes).
The add-modes and remove-modes are lists of single-character strings
for modes without parameters to add and remove respectively.  The
arg-modes is a list of triples of the form '(mode-char on/off
argument)."

  (if (string-match "^\\s-*\\(\\S-+\\)\\(\\s-.*$\\|$\\)" mode-string)
      (let ((chars (mapcar 'char-to-string (match-string 1 mode-string)))
	    (args-str (match-string 2 mode-string));; arguments in channel modes
	    (args nil)
	    (add-modes nil)
	    (remove-modes nil)
	    (arg-modes nil); list of triples: (mode-char 'on/'off argument)
	    (add-p t))
	;; make the argument list
	(while (string-match "^\\s-*\\(\\S-+\\)\\(\\s-+.*$\\|$\\)" args-str)
	  (setq args (cons (match-string 1 args-str) args))
	  (setq args-str (match-string 2 args-str)))
	(setq args (reverse args))
	;; collect what modes changed, and match them with arguments
	(while chars
	  (cond ((string= (car chars) "+") (setq add-p t))
		((string= (car chars) "-") (setq add-p nil))
		((string-match "^[ovbOVB]" (car chars))
		 (setq arg-modes (cons (list (car chars)
					     (if add-p 'on 'off)
					     (if args (car args) nil))
				       arg-modes))
		 (if args (setq args (cdr args))))
		((string-match "^[Ll]" (car chars))
		 (setq arg-modes (cons (list (car chars)
					     (if add-p 'on 'off)
					     (if (and add-p args)
						 (car args) nil))
				       arg-modes))
		 (if (and add-p args) (setq args (cdr args))))
		(add-p (setq add-modes (cons (car chars) add-modes)))
		(t (setq remove-modes (cons (car chars) remove-modes))))
	  (setq chars (cdr chars)))
	(setq add-modes (reverse add-modes))
	(setq remove-modes (reverse remove-modes))
	(setq arg-modes (reverse arg-modes))
	(list add-modes remove-modes arg-modes))
    nil))

(defun erc-update-modes (tgt mode-string &optional nick host login)
  "Updates the mode information for TGT, provided as MODE-STRING.
Optional arguments: NICK, HOST and LOGIN - the attributes of the
person who changed the modes."
  (let* ((modes (erc-parse-modes mode-string))
	 (add-modes (nth 0 modes))
	 (remove-modes (nth 1 modes))
	 ;; list of triples: (mode-char 'on/'off argument)
	 (arg-modes (nth 2 modes)))
    ;; now parse the modes changes and do the updates
    (cond ((string-match "^#" tgt); channel modes
	   (let ((buf (and (boundp 'erc-session-server)
			   erc-session-server
			   (erc-find-buffer erc-session-server tgt)))
		 (ob (current-buffer)))
	     (if buf
		 (progn
		   (set-buffer buf)
		   (if (not (boundp 'channel-modes))
		       (progn
			 (make-variable-buffer-local 'channel-modes)
			 (setq channel-modes nil)))
		   (while remove-modes
		     (setq channel-modes
			   (erc-delete-string (car remove-modes)
					      channel-modes))
		     (setq remove-modes (cdr remove-modes)))
		   (while add-modes
		     (setq channel-modes
			   (cons (car add-modes) channel-modes))
		     (setq add-modes (cdr add-modes)))
		   (setq channel-modes (erc-sort-strings channel-modes))
		   (while arg-modes
		     (let ((mode (nth 0 (car arg-modes)))
			   (onoff (nth 1 (car arg-modes)))
			   (arg (nth 2 (car arg-modes))))
		       (cond ((string-match "^[oO]" mode)
			      (erc-update-channel-member tgt arg arg nil 
							 onoff))
			     ((string-match "^[Vv]" mode)
			      (erc-update-channel-member tgt arg arg nil 
							 nil onoff))
			     ((string-match "^[Ll]" mode)
			      (erc-update-channel-limit tgt onoff arg))
			     (t nil)); only ops are tracked now
		       (setq arg-modes (cdr arg-modes))))
		   (erc-update-channel-info-buffer tgt)))))
	  ;; nick modes - ignored at this point
	  (t nil))))

(defun erc-update-channel-limit (chnl onoff n)
  "Updates the limit on the number of users."
  (message "chnl=%S, onoff=%S, n=%S" chnl onoff n)
  (let ((chnl-buf (and ;;(string-match "^#" chnl); allow private chat too
		   (boundp 'erc-session-server)
		   erc-session-server
		   (erc-find-buffer erc-session-server chnl)))
	(ob (current-buffer)))
    (if (and chnl-buf
	     (or (not (eq onoff 'on))
		 (and (stringp n) (string-match "^[0-9]+$" n))))
	(progn
	  (set-buffer chnl-buf)
	  (cond ((eq onoff 'on) (setq channel-user-limit (string-to-number n)))
		(t (setq channel-user-limit nil)))))))

(defun erc-find-channel-info-buffer (chnl)

  "Returns the channel info buffer.  If it doesn't exist, creates it.
The channel must be a string starting with `#' character.  If it is
not, nil is returned and no buffer is created."
  (let* ((chnl-buf (and ;;(string-match "^#" chnl); allow private chat too
			(boundp 'erc-session-server)
			erc-session-server
			(erc-find-buffer erc-session-server chnl)))
	 (ob (current-buffer))
	 (name nil)
	 (res nil))
    (if chnl-buf
	(progn
	  (set-buffer chnl-buf)
	  ;; if chnl is a real channel, the target must match
	  (if (and chnl
		   (or (not (string-match "^#" chnl))
		       (and (erc-default-target)
			    (string= (downcase (erc-default-target))
				     (downcase chnl)))))
	      (progn
		;;(message "test is true, chnl = %S" chnl)
		(setq name (concat (buffer-name chnl-buf) ":INFO"))
		(setq res (get-buffer name))
		;;(message "res (after 1st setq) = %S" res)
		(if (not res)
		    (progn
		      (setq res (get-buffer-create name))
		      ;;(message "res (after 2nd setq) = %S" res)
		      (set-buffer res)
		      (erc-info-mode)))))
	  (set-buffer ob)))
    ;;(message "res = %S" res)
    res))

(defun erc-update-channel-info-buffer (chnl)
  "This function updates the channel info buffer with current info,
like topic, modes, users, etc."

  (let* ((chnl-buf (and ;;(string-match "^#" chnl)
			(boundp 'erc-session-server)
			erc-session-server
			(erc-find-buffer erc-session-server chnl)))
	 (ob (current-buffer))
	 (info-buf (erc-find-channel-info-buffer chnl))
	 (names nil)
	 (topic nil)
	 (modes nil)
	 (info-point nil))
    ;;(message "chnl-buf=%s" chnl-buf)
    ;;(message "info-buf=%s" info-buf)
    ;; if no target, do not update anything
    (if (and chnl chnl-buf info-buf)
	(progn
	  (set-buffer chnl-buf)
	  (setq names (and (boundp 'channel-members) channel-members))
	  (setq topic (and (boundp 'channel-topic) channel-topic))
	  (setq modes (and (boundp 'channel-modes) channel-modes))
	  (setq limit (and (boundp 'channel-user-limit) channel-user-limit))
	  ;;(message "info-buf=%s" info-buf)
	  (set-buffer info-buf)
	  (setq info-point (point))
	  ;;(message "info-point = %s" info-point)
	  (toggle-read-only -1)
	  (erase-buffer)
	  (insert (erc-interpret-controls 
		   (cond ((string-match "^#" chnl)
			  (format "\C-c3Channel \C-b%s\C-b[%s]:\C-c %s" 
				  chnl (length names)
				  (if (and topic (not (string= "" topic)))
				      topic
				    "<no topic>")))
			 (t (format "\C-c5Private from \C-b%s\C-b[%s]\C-c"
				    chnl (length names))))))
	  (if modes 
	      (insert (concat "\nmodes: +"
			      (mapconcat 'identity modes ""))))
	  (if limit (insert (format "   user limit = %s" limit)))
	  (insert "\n\n")
	  (if names
	      (while names
		(let* ((entry (car names))
			(nick (nth 0 entry))
			(op (nth 1 entry))
			(voice (nth 2 entry))
			(host (nth 3 entry))
			(email (nth 4 entry))
			(full (nth 5 entry))
			(info (nth 6 entry)))
		(insert
		 (erc-interpret-controls 
		  (concat (if op "\C-c2@" " ")
			  (if voice "+" " ") nick " "
			  (if email email "")
			  (if host (concat "@" host) "")
			  (if full
			      (concat " " 
				      (if (> (length full) 25) 
					  (concat (substring full 0 22) "...")
					full))
			    "")
			  (if info (concat " " info) "") "\n"))))
		(setq names (cdr names))))
	  (goto-char info-point)
	  (toggle-read-only 1)
	  (set-buffer ob)))))

(defun erc-remove-member-all-channels (nick &optional buffer-list)
  "Does what it says in all the buffers for the current session."
  (let ((ob (current-buffer))
	(tgt nil))
    (if (and (boundp 'erc-process) (processp erc-process))
	(set-buffer (process-buffer erc-process)))
    (if (not buffer-list)
	(setq buffer-list (and (boundp 'erc-buffer-list)
			       erc-buffer-list)))
    (while buffer-list
      (if (buffer-live-p (car buffer-list))
	  (progn
	    (set-buffer (car buffer-list))
	    (setq tgt (erc-default-target))
	    (if (and tgt (string-match "^#" tgt))
		(erc-remove-channel-member tgt nick))))
      (setq buffer-list (cdr buffer-list)))
    (set-buffer ob)))

(defun erc-update-member-all-channels (nick new-nick
				       &optional add op voice host email 
				       full-name info
				       buffer-list)
  "Does what it says in all the buffers for the current session."
  (let ((ob (current-buffer))
	(tgt nil))
    (if (and (boundp 'erc-process) (processp erc-process))
	(set-buffer (process-buffer erc-process)))
    (if (not buffer-list)
	(setq buffer-list (and (boundp 'erc-buffer-list)
			       erc-buffer-list)))
    (while buffer-list
      (if (buffer-live-p (car buffer-list))
	  (progn
	    (set-buffer (car buffer-list))
	    (setq tgt (erc-default-target))
	    (if (and tgt (string-match "^#" tgt))
		(erc-update-channel-member tgt nick new-nick add op voice 
					   host email full-name info))))
      (setq buffer-list (cdr buffer-list)))
    (set-buffer ob)))

(defun erc-update-channel-info-buffers (&optional buffer-list)
  "Does what it says in all the buffers for the current session."
  (let ((ob (current-buffer))
	(tgt nil))
    (if (and (boundp 'erc-process) (processp erc-process))
	(set-buffer (process-buffer erc-process)))
    (if (not buffer-list)
	(setq buffer-list (and (boundp 'erc-buffer-list)
			       erc-buffer-list)))
    (while buffer-list
      (if (buffer-live-p (car buffer-list))
	  (progn
	    (set-buffer (car buffer-list))
	    (setq tgt (erc-default-target))
	    (if (and tgt (string-match "^#" tgt))
		(erc-update-channel-info-buffer tgt))))
      (setq buffer-list (cdr buffer-list)))
    (set-buffer ob)))

(defun erc-buffer-list-with-nick (nick &optional buffer-list)

  "Returns the list of buffers corresponding to channels with the NICK
member.  If optional BUFFER-LIST is non-nil, search in that list."

  (let ((session-buffer (and (boundp 'erc-process) (processp erc-process)
			     (process-buffer erc-process)))
	(names nil)
	(ob (current-buffer))
	(res nil))
    (setq nick (downcase nick))
    (if (and session-buffer (buffer-live-p session-buffer))
	(set-buffer session-buffer))
    (if (and (null buffer-list)
	     (boundp 'erc-buffer-list))
	(setq buffer-list erc-buffer-list))
    (while buffer-list
      (if (buffer-live-p (car buffer-list))
	  (progn
	    (set-buffer (car buffer-list))
	    (if (and (erc-default-target)
		     (string-match "^#" (erc-default-target))
		     (boundp 'channel-members))
		(progn
		  (setq names channel-members)
		  (while names
		    (if (string= nick (downcase (car (car names))))
			(progn 
			  (setq res (cons (car buffer-list) res))
			  (setq names nil))
		      (setq names (cdr names))))))))
      (setq buffer-list (cdr buffer-list)))
    res))
		
(defun erc-handle-user-status-change (typ nlh &optional l)
  "Handle changes in any user's status. So far, only nick change is handled.

Generally, the TYP argument is a symbol describing the change type, NLH is
a list containing the original nickname, login name and hostname for the user,
and L is a list containing additional TYP-specific arguments.

So far the following TYP/L pairs are supported:

       event                    TYP                    L

    nickname change            'nick                (NEW-NICK)

"
  (erc-log (format "user-change: type: %S  nlh: %S  l: %S" typ nlh l))
  (cond
   ;; nickname change
   ((equal typ 'nick)
    t)

   (t
    nil)))

(defun erc-highlight-notice (s)
  "Highlight notice message S and return it.
See also variable `erc-notice-highlight-type'"
  (cond
   ((equal erc-notice-highlight-type 'prefix)
    (put-text-property 0 (length erc-notice-prefix) 'face 'erc-notice-face s)
    s)
   ((equal erc-notice-highlight-type 'all)
    (put-text-property 0 (length s) 'face 'erc-notice-face s)
    s)
   (t s)))

(defun erc-highlight-strings (s)
  "Highlight strings in S and return it.
See also variable `erc-highlight-strings."
  (let ((strs erc-highlight-strings)
	str pos)
    (while strs
      (setq str (car strs)
	    strs (cdr strs)
	    pos (string-match str s))
      (while pos
	(put-text-property (match-beginning 0)
			   (match-end 0)
			   'face 'erc-highlight-face s)
	(setq pos (string-match str s (match-end 0)))))
    s))

(defun erc-highlight-error (s)
  "Highlight error message S and return it"
  (put-text-property 0 (length s) 'face 'erc-error-face s)
  s)

(defun erc-parse-user (string)
  "Parse IRC-type user specification (nick!login@host) to three separate
tokens and return them as a list"
  (erc-log (concat "parse-user: input: " string))
  (if (string-match "\\([^!]*\\)!\\([^@]*\\)@\\(.*\\)" string)
     (list (match-string 1 string)
	   (match-string 2 string)
	   (match-string 3 string))
    (list string "" "")))

(defun erc-parse-current-line ()
  "Parse current input line.
Returns a pair (PART1 . PART2), where PART1 is the input part before the point
and PART2 is the part after the point."
  (save-excursion
    (let* ((erc-prompt-regexp (concat "^" erc-prompt "[ \t]*"))
	   (p1 (point))
	   (p0 (progn (if erc-multiline-input
			  (search-backward-regexp erc-prompt-regexp)
			(beginning-of-line)) (point)))
	   (p2 (progn (if erc-multiline-input
			  (goto-char (point-max))
			(end-of-line)) (point)))
	   (l0 (buffer-substring p0 p2))
	   (l1 (buffer-substring p0 p1))
	   (l2 (buffer-substring p1 p2)))
;	   (l0 (buffer-substring-no-properties p0 p2))
;	   (l1 (buffer-substring-no-properties p0 p1))
;	   (l2 (buffer-substring-no-properties p1 p2)))
;;      (erc-log (format "parse-line: l0: %S  l1: %S  l2: %S\n" l0 l1 l2))
      (cond ((string-match erc-prompt-regexp l0)
	     (let ((i1 (match-end 0)))
	       (if (>= i1 (length l1))
		   (cons nil (substring l0 i1))
		   (cons (substring l1 i1) l2))))
	    (t (cons l1 l2))))))

(defun erc-split-multiline (string)
  "Split STRING, containing multiple lines and return them in a list"
  (interactive "")
  (let ((l ())
	(i0 0)
	(doit t))
    (while doit
      (let ((i (string-match "\r?\n" string i0))
	    (s (substring string i0)))
	(cond (i
	       (setq l
		     (cons
		      (concat prev-rd (substring string i0 i))
		      l))
	       (setq prev-rd "")
	       (setq i0 (match-end 0)))
	      ((> (length s) 0)
	       (setq prev-rd (concat prev-rd s))
	       (setq doit nil))
	      (t (setq doit nil)))))
    (reverse l)))

(defun erc-split-multiline-safe (string)
  "Split STRING, containing multiple lines and return them in a list.
  Do it only for STRING as the complete input, do not carry unfinished
  strings over to the next call."
  (interactive)
  (let ((l ())
	(i0 0)
	(doit t))
    (while doit
      (let ((i (string-match "\r?\n" string i0))
	    (s (substring string i0)))
	(cond (i (setq l (cons (substring string i0 i) l))
		 (setq i0 (match-end 0)))
	      ((> (length s) 0) 
	         (setq l (cons s l))(setq doit nil))
	      (t (setq doit nil)))))
    (reverse l)))

;; command history -- implemented using ring.el.

;; Should these be buffer local?  Currently the history is shared
;; accross all erc buffers.
(defvar erc-input-ring nil "Input ring for erc.")
(defvar erc-input-ring-index nil "Position in the input ring for erc.")

(defun erc-input-ring-setup ()
  "Do the setup required so that we can use comint style input rings.
Call this function when setting up the mode."
  (setq erc-input-ring (make-ring comint-input-ring-size)
	erc-input-ring-index nil))

(defun erc-add-to-input-ring (s)
  "Add string S to the input ring and reset history position."
  (ring-insert erc-input-ring s)
  (setq erc-input-ring-index 0))

(defun erc-previous-command ()
  "Replace current command with the previous one from the history."
  (interactive)
  (let* ((len (ring-length erc-input-ring))
	 (s (ring-ref erc-input-ring erc-input-ring-index)))
    (erc-replace-current-command s)
    (setq erc-input-ring-index (ring-plus1 erc-input-ring-index len))))

(defun erc-next-command ()
  "Replace current command with the next one from the history."
  (interactive)
  (let ((len (ring-length erc-input-ring)))
    (setq erc-input-ring-index (ring-minus1 erc-input-ring-index len))
    (let ((s (ring-ref erc-input-ring erc-input-ring-index)))
      (erc-replace-current-command s))))

(defun erc-replace-current-command (s)
  "Replace current command with string S."
  ;; delete line
  (delete-region
   (progn (beginning-of-line) (point))
   (progn (end-of-line) (point)))
  (erc-display-prompt)
  (insert s))

;; input ring handling

;; nick handling

(defun erc-push-nick (nick)
  "Push new nickname to a nickname stack"
  (let ((ob (current-buffer)))
    (if (and (boundp 'erc-process)(processp erc-process))
	(set-buffer (process-buffer erc-process)))
    (setq nick-stk (cons nick nick-stk))
    (set-buffer ob)))

(defun erc-pop-nick ()
  "Remove topmost nickname from a stack"
  (let ((ob (current-buffer)))
    (if (and (boundp 'erc-process)(processp erc-process))
	(set-buffer (process-buffer erc-process)))
    (if (null nick-stk)
	(error "Nickname stack empty")
      (setq nick-stk (cdr nick-stk)))
    (set-buffer ob)))

(defun erc-current-nick ()
  "Return current nickname (top of the nickname stack)"
  (let ((ob (current-buffer))
	(res nil))
    (if (and (boundp 'erc-process)(processp erc-process))
	(set-buffer (process-buffer erc-process)))
    (if nick-stk (setq res (car nick-stk)))
    (set-buffer ob)
    res))

(defun erc-original-nick ()
  "Return the original nickname the buffer was created with 
 (bottom of the nickname stack)"
  (let ((ob (current-buffer))
	(res nil))
    (if (and (boundp 'erc-process)(processp erc-process))
	(set-buffer (process-buffer erc-process)))
    (if nick-stk
	(setq res (nth (- (length nick-stk) 1) nick-stk)))
    (set-buffer ob)
    res))

;; default target handling

(defun erc-default-target ()
  "Returns current default target (as a character string) or NIL if none."
  (let ((tgt (car def-rcpts)))
    (cond
     ((not tgt) nil)
     ((listp tgt) (cdr tgt))
     (t tgt))))

(defun erc-add-default-channel (ch)
  "Add channel to the default channel list. "

;;; This is no longer true.  The channel is added to another window
;;; and we don't want to mess the target there.
;"If the current default
;recepient is of QUERY type, then push the new default channel *after*
;the head"

  (let ((d1 (car def-rcpts))
	(d2 (cdr def-rcpts))
	(chl (downcase ch)))
;    (if (and (listp d1)
;	     (equal (car d1) 'QUERY))
;	(setq def-rcpts
;	      (cons d1 (cons chl d2)))
      (setq def-rcpts
	    (cons chl def-rcpts))
;)
))

(defun erc-delete-default-channel (ch &optional buffer)
  "Delete channel from the default channel list."
  (let ((ob (current-buffer)))
    (if (and buffer (bufferp buffer))
	(set-buffer buffer))
    (setq def-rcpts (delete (downcase ch) def-rcpts))
    (set-buffer ob)))

(defun erc-add-query (nick)
  "Add QUERY'd nickname to the default channel list. The previous
default target of QUERY type gets removed"
  (let ((d1 (car def-rcpts))
	(d2 (cdr def-rcpts))
	(qt (cons 'QUERY (downcase nick))))
    (if (and (listp d1)
	     (equal (car d1) 'QUERY))
	(setq def-rcpts (cons qt d2))
      (setq def-rcpts (cons qt def-rcpts)))))

(defun erc-delete-query ()
  "Delete the topmost target if it is a QUERY"

  (let ((d1 (car def-rcpts))
	(d2 (cdr def-rcpts)))
    (if (and (listp d1)
	     (equal (car d1) 'QUERY))
	(setq def-rcpts d2)
      (error "Current target is not a QUERY"))))

;; pal stuff

(defun erc-delete-string (str lst)
  "Same as `delete', only non-destructive, and for lists of strings"
  (let ((acc nil))
    (while (and lst (not (equal str (car lst))))
      (setq acc (cons (car lst) acc))
      (setq lst (cdr lst)))
    (if lst (append acc (cdr lst)) acc)))

(defun erc-pal-p (nick)
  "Check whether NICK is in the pals list."
  (member nick erc-pals))


(defun erc-host-danger-p (host)
  "Check whether NICK is in the `erc-host-danger-highlight' list"
  (let ((l erc-host-danger-highlight)
	(res nil))
    (while l
      (if (string-match (car l) host)
	  (progn
	    (setq res t)
	    (setq l nil))
	(setq l (cdr l))))
    res))

(defun erc-ignored-user-p (spec)
  "Takes a full SPEC of a user in the form \"nick!login@host\", and
matches against all the regexp's in `erc-ignore-list'.  If anyone
match, returns that regexp, and nil otherwise."
  (let ((lst erc-ignore-list))
    (while (and lst (not (string-match (car lst) spec)))
      (setq lst (cdr lst)))
    (and lst (car lst))))

(defun erc-add-pal ()
  "Add pal interactively."
  (interactive)
  (let ((pal (downcase (read-from-minibuffer "Pal\'s nickname: "))))
    (if (erc-pal-p pal)
	(error (format "pal \"%s\" already on the list" pal))
      (setq erc-pals (cons pal erc-pals)))))

(defun erc-delete-pal ()
  "Delete pal interactively"
  (interactive)
  (let ((pal (downcase (read-from-minibuffer "Pal\'s nickname: "
					     nil nil nil 'erc-pals))))
    (if (not (erc-pal-p pal))
	(error (format "pal \"%s\" is not on the list" pal))
      (setq erc-pals (erc-delete-string pal erc-pals)))))

;; Sound stuff - S.B.

(defun erc-play-sound (file)
  "Plays a sound file located in one of the directories in `erc-sound-path'
   with a command `erc-play-command'."
  (let ((filepath (erc-find-file file erc-sound-path)))
    (if (and (not filepath) erc-default-sound)
	(setq filepath erc-default-sound))
    (cond ((and filepath (file-exists-p filepath))
;	   (start-process "erc-sound" nil erc-play-command filepath)
 	   (start-process "erc-sound" nil "/bin/tcsh"  "-c"
 			  (concat erc-play-command " " filepath))
	   )
	  (t (beep)))
    (erc-log (format "Playing sound file %S" filepath))))

;(defun erc-play-sound (file)
;  "Plays a sound file located in one of the directories in `erc-sound-path'
;   with a command `erc-play-command'."
;  (let ((filepath nil)
;	(paths erc-sound-path))
;    (while (and paths 
;		(progn (setq filepath (concat (car paths) "/" file))
;		       (not (file-exists-p filepath))))
;      (setq paths (cdr paths)))
;    (if (and (not (and filepath (file-exists-p filepath)))
;	     erc-default-sound)
;	(setq filepath erc-default-sound))
;    (cond ((and filepath (file-exists-p filepath))
;;	   (start-process "erc-sound" nil erc-play-command filepath)
; 	   (start-process "erc-sound" nil "/bin/tcsh"  "-c"
; 			  (concat erc-play-command " " filepath))
;	   )
;	  (t (beep)))
;    (erc-log (format "Playing sound file %S" filepath))))

(defun erc-toggle-sound (&optional arg)
  "Toggles playing sounds on and off.  With positive argument,
  turns them on.  With any other argument turns sounds off."
  (interactive "P")
  (cond ((and (numberp arg) (> arg 0))
	 (setq erc-play-sound t))
	(arg (setq erc-play-sound nil))
	(t (setq erc-play-sound (not erc-play-sound))))
  (message "ERC sound is %s" (if erc-play-sound "ON" "OFF")))

;; other "toggles"

(defun erc-toggle-ctcp-autoresponse (&optional arg)
  "Toggles automatic CTCP replies (like VERSION and PING) on and off.
 With positive argument, turns them on.  With any other argument turns
 them off."
  (interactive "P")
  (cond ((and (numberp arg) (> arg 0))
	 (setq erc-disable-ctcp-replies t))
	(arg (setq erc-disable-ctcp-replies nil))
	(t (setq erc-disable-ctcp-replies (not erc-disable-ctcp-replies))))
  (message "ERC CTCP replies are %s" (if erc-disable-ctcp-replies "OFF" "ON")))

(defun erc-toggle-flood-control (&optional arg)
  "Toggles between strict and normal and no flood control."
  (interactive "P")
  (cond ((and (numberp arg) (> arg 0))
	 (setq erc-flood-protect 'strict))
	(arg (setq erc-flood-protect 'normal))
	((eq erc-flood-protect 'strict)
	 (setq erc-flood-protect nil))
	(erc-flood-protect (setq erc-flood-protect 'strict))
	(t (setq erc-flood-protect 'normal)))
  (message "ERC flood control is %s" 
	   (cond ((eq erc-flood-protect 'strict) "STRICT")
		 (erc-flood-protect "NORMAL")
		 (t "OFF"))))

(defun erc-toggle-interpret-controls (&optional arg)
  "Toggles interpretation of colors and other control sequences in
messages on and off.  With positive argument, turns it on.  With any
other argument turns it off."
  (interactive "P")
  (cond ((and (numberp arg) (> arg 0))
	 (setq erc-interpret-controls-p t))
	(arg (setq erc-interpret-controls-p nil))
	(t (setq erc-interpret-controls-p (not erc-interpret-controls-p))))
  (message "ERC color interpretation %s"
	   (if erc-interpret-controls-p "ON" "OFF")))

;; Some useful channel and nick commands for fast key bindings

(defun erc-invite-only-mode (&optional arg)
  "Turns on the invite only mode (+i) for the current channel.  If the
non-nil argument is provided, turn this mode off (-i).  

This command is sent even if excess flood is detected."
  (interactive "P")
  (setq erc-active-buffer (current-buffer))
  (let ((tgt (erc-default-target))
	(erc-force-send t))
    (cond ((or (not tgt) (not (string-match "^#" tgt)))
	   (erc-display-line (erc-highlight-error "No target\n")
			     (current-buffer)))
	  (arg (erc-load-irc-script-lines (list (concat "/mode " tgt " -i")) 
					  t))
	  (t (erc-load-irc-script-lines (list (concat "/mode " tgt " +i"))
					t)))))

(defun erc-insert-mode-command ()
  "Inserts the line \"/mode <current target> \" at the cursor."
  (interactive)
  (let ((tgt (erc-default-target)))
    (if tgt (insert (concat "/mode " tgt " "))
      (erc-display-line (erc-highlight-error "No target\n")
			(current-buffer)))))

(defun erc-channel-names ()
  "Runs \"/names #channel\" in the current channel"
  (interactive)
  (setq erc-active-buffer (current-buffer))
  (let ((tgt (erc-default-target)))
    (if tgt (erc-load-irc-script-lines (list (concat "/names " tgt)))
      (erc-display-line (erc-highlight-error "No target\n")
			(current-buffer)))))

(defun erc-remove-text-properties-region (start end)
  "Clears the region from all the colors, etc."
  (interactive "r")
  (save-excursion
    (let ((s (buffer-substring start end)))
      (delete-region start end)
      (insert (format "%s" s)))))

;; script execution and startup

(defun erc-find-file (file &optional path)
  "Searches for a FILE as it is (in a current directory) and then
 directories in the PATH, if provided, and returns the first full name
 found, or NIL if none."
  (let ((filepath file))
    (if (file-readable-p filepath) filepath
      (progn
	(while (and path
		    (progn (setq filepath (concat (car path) "/" file))
			   (not (file-readable-p filepath))))
	  (setq path (cdr path)))
	(if path filepath nil)))))

(defun erc-select-startup-file ()
  "Select startup file with a script to execute. See also
the variable `erc-startup-file-list'"
  (let ((l erc-startup-file-list)
	(f nil))
    (while (and (not f) l)
      (if (file-readable-p (car l))
	  (setq f (car l)))
      (setq l (cdr l)))
    f))

(defun erc-find-script-file (file)
  "Searches for the FILE in the current directory and those provided
in `erc-script-path'."
  (erc-find-file file erc-script-path))

(defun erc-load-script (file)
  "Load a script from FILE.  FILE must be the full name, it is not
 searched in the `erc-script-path'.  If the filename ends with `.el',
 then load it as a emacs-lisp program. Otherwise, trieat it as a
 regular IRC script"
  (erc-log (concat "erc-load-script: " file))
  (cond
   ((string-match "\\.el$" file)
    (load file))
   (t
    (erc-load-irc-script file))))

(defun erc-process-script-line (line &optional args) 
"Does script-specific substitutions (script arguments, current nick,
server, etc.)  in the line and returns it.

Substitutions are: %C and %c = current target (channel or nick), 
%S %s = current server, %N %n = my current nick, and %x is x verbatim,
where x is any other character;
$* = the entire argument string, $1 = the first argument, $2 = the second,
end so on."
  (if (not args) (setq args ""))
  (let* ((arg-esc-regexp "\\(\\$\\(\\*\\|[1-9][0-9]*\\)\\)\\([^0-9]\\|$\\)")
	 (percent-regexp "\\(%.\\)")
	 (esc-regexp (concat arg-esc-regexp "\\|" percent-regexp))
	 (tgt (erc-default-target))
	 (server (and (boundp 'erc-session-server) erc-session-server))
	 (nick (erc-current-nick))
	 (res "")
	 (tmp nil)
	 (arg-list nil)
	 (arg-num 0))
    (if (not tgt) (setq tgt ""))
    (if (not server) (setq server ""))
    (if (not nick) (setq nick ""))
    ;; First, compute the argument list
    (setq tmp args)
    (while (string-match "^\\s-*\\(\\S-+\\)\\(\\s-+.*$\\|$\\)" tmp)
      (setq arg-list (cons (match-string 1 tmp) arg-list))
      (setq tmp (match-string 2 tmp)))
    (setq arg-list (reverse arg-list))
    (setq arg-num (length arg-list))
    ;; now do the substitution
    (setq tmp (string-match esc-regexp line))
    (while tmp
      ;;(message "beginning of while: tmp=%S" tmp)
      (let* ((hd (substring line 0 tmp))
	     (esc "")
	     (subst "")
	     (tail (substring line tmp)))
	(cond ((string-match (concat "^" arg-esc-regexp) tail)
	       (setq esc (match-string 1 tail))
	       (setq tail (substring tail (match-end 1))))
	      ((string-match (concat "^" percent-regexp) tail)
	       (setq esc (match-string 1 tail))
	       (setq tail (substring tail (match-end 1)))))
	;;(message "hd=%S, esc=%S, tail=%S, arg-num=%S" hd esc tail arg-num)
	(setq res (concat res hd))
	(setq subst 
	      (cond ((string= esc "") "")
		    ((string-match "^\\$\\*$" esc) args)
		    ((string-match "^\\$\\([0-9]+\\)$" esc)
		     (let ((n (string-to-number (match-string 1 esc))))
		       (message "n = %S, integerp(n)=%S" n (integerp n))
		       (if (<= n arg-num) (nth (1- n) arg-list) "")))
		    ((string-match "^%[Cc]$" esc) tgt)
		    ((string-match "^%[Ss]$" esc) server)
		    ((string-match "^%[Nn]$" esc) nick)
		    ((string-match "^%\\(.\\)$" esc) (match-string 1 esc))
		    (t (erc-log (format "BUG in erc-process-script-line: bad escape sequence: %S\n" esc))
		       (message "BUG IN ERC: esc=%S" esc)
		       "")))
	(setq line tail)
	(setq tmp (string-match esc-regexp line))
	(setq res (concat res subst))
	;;(message "end of while: line=%S, res=%S, tmp=%S" line res tmp)
	))
    (setq res (concat res line))
    res))

(defun erc-load-irc-script (file &optional force)
  "Load IRC script from FILE"
  (erc-log (concat "erc-load-script: " file))
  (let* ((cb (current-buffer))
	 ;; don't mangle the buffer if it's being edited
	 (str nil)
	 (b (generate-new-buffer " *erc-script*")))
    (set-buffer b)
    (insert-file file)
    (setq str (buffer-substring (point-min) (point-max)))
    (set-buffer cb)
    (kill-buffer b)
    (erc-load-irc-script-lines (erc-split-multiline-safe str) force)))

;(defun erc-load-irc-script (file)
;  "Load IRC script from FILE"

;  (erc-log (concat "erc-load-script: " file))
;  (let* ((cb (current-buffer))
;	 (pnt (point))
;	 ;; don't mangle the buffer if it's being edited
;	 (b (generate-new-buffer " *erc-script*"))
;	 (s "")
;	 (sp (concat erc-prompt " ")))
;    (set-buffer b)
;    (insert-file file)
;    (goto-char (point-min))
;    (set-buffer cb)
;    ;; prepare the prompt string for echo
;    (put-text-property 0 (length erc-prompt) 'face 'erc-prompt-face sp)
;    (put-text-property (length erc-prompt) (length sp)
;		       'face 'erc-input-face sp)
;    (while s
;      (erc-log (concat "erc-load-script: CMD: " s))
;      (if (not (string-match "^\\s-*$" s))
;	  (progn
;	    (if erc-script-echo
;		(progn
;;		  (erc-display-prompt cb erc-insert-marker)
;		  (put-text-property 0 (length s) 'face 'erc-input-face s)
;		  (erc-display-line (concat sp s) cb)))
;	    (erc-process-script-line s)))
;      (setq s (erc-get-command-from-buffer b)))
;    (kill-buffer b)))

(defun erc-load-irc-script-lines (lines &optional force noexpand)
  "Load IRC script LINES (a list of strings).  If optional NOEXPAND
 argument is non-nil, do not expand script-specific sequenced, process
 the lines verbatim.  Use this for multiline user input."
  (let* ((cb (current-buffer))
	 (pnt (point))
	 (s "")
	 (sp (concat erc-prompt " "))
	 (args (and (boundp 'erc-script-args) erc-script-args)))
    (if (and args (string-match "^ " args))
	(setq args (substring args 1)))
    ;; prepare the prompt string for echo
    (put-text-property 0 (length erc-prompt) 'face 'erc-prompt-face sp)
    (put-text-property (length erc-prompt) (length sp)
		       'face 'erc-input-face sp)
    (while lines
      (setq s (car lines))
      (erc-log (concat "erc-load-script: CMD: " s))
      (if (not (string-match "^\\s-*$" s))
	  (let ((line (if noexpand s (erc-process-script-line s args))))
	    (if (and (erc-process-input-line line force)
		     erc-script-echo)
		(progn
;		  (erc-display-prompt cb erc-insert-marker)
		  (put-text-property 0 (length line) 
				     'face 'erc-input-face line)
		  (erc-display-line (concat sp line) cb)))))
      (setq lines (cdr lines)))))

;;; Deprecated
(defun erc-get-command-from-buffer (b)
  "Read the current line from buffer B, move the point down
and return the line"
  ;; There may be a possible problem with a non-saved excursion
  ;; but if we do save it, we won't be able to remember the new
  ;; position in the command buffer
  (let ((cb (current-buffer))
	(s nil))
    (set-buffer b)
    (cond
     ((not (equal (point) (point-max)))
      (setq s (buffer-substring-no-properties
	       (progn (beginning-of-line) (point))
	       (progn (end-of-line) (point))))
      (beginning-of-line 2)))
    (set-buffer cb)
    s))

;; authentication

(defun erc-login ()
  "Perform user authentication at the IRC server"
  (erc-log (format "login: nick: %s, user: %s %s %s :%s"
		   (erc-current-nick)
		   (user-login-name)
		   (system-name)
		   erc-session-server
		   erc-session-user-full-name))
  (if password (erc-send-command (format "PASS %s" password))
    (message "Logging without password"))
  (erc-send-command (format "NICK %s" (erc-current-nick)))
  (erc-send-command
   (format "USER %s %s %s :%s"
	   ;; hacked - S.B.
	   (if erc-anonymous-login erc-email-userid (user-login-name))
	   (if erc-anonymous-login "128.129.130.131" (system-name))
	   erc-session-server
	   erc-session-user-full-name)))

;; connection properties' heuristics

(defun erc-determine-parameters (&optional server port nick name)
  "Determine the connection and authentication parameters and
sets the buffer local variables:

- erc-session-server
- erc-session-port
- nick-stk
- erc-session-full-name"
  (setq erc-session-server (erc-compute-server server)
	erc-session-port (or port erc-default-port)
	erc-session-user-full-name (erc-compute-full-name name))
  (erc-push-nick (erc-compute-nick nick)))
  
(defun erc-compute-server (server)
  "return the IRC server to use using the following order until a non-NIL
one is found:

- argument
- erc-server value
- value of IRCSERVER environment variable
- erc-default-server value"
  (or server
      erc-server
      (getenv "IRCSERVER")
      erc-default-server))

(defun erc-compute-nick (nick)
  "Return the user's nick using the following order until a non-NIL
one is found:

- argument
- erc-nick value
- value of IRCNICK environment variable
- user's login name"
  (or nick
      (if (consp erc-nick) (car erc-nick) erc-nick)
      (getenv "IRCNICK")
      (user-login-name)))


(defun erc-compute-full-name (name)
  "return the user's full name using the following order until a non-NIL
one is found:

- argument
- erc-user-full-name value
- value of IRCNAME environment variable
- user's full name from the system databases"
  (or name
      erc-user-full-name
      (getenv "IRCNAME")
      (if erc-anonymous-login "unknown" nil)
      (user-full-name)))

;; time routines

(defun erc-string-to-emacs-time (string)
  "Convert long number represented by the STRING into the list of
'(high low), compatible with emacs time format."
  (let* ((n (string-to-number (concat string ".0"))))
    (list (truncate (/ n 65536))
	  (truncate (mod n 65536)))))

(defun erc-emacs-time-to-erc-time (tm)
  "Convert Emacs time to a number of seconds since the epoch"
  (+ (* (nth 0 tm) 65536.0) (nth 1 tm)))
;  (round (+ (* (nth 0 tm) 65536.0) (nth 1 tm))))

(defun erc-current-time ()
  "Return current time as a number of seconds since the epoch"
  (erc-emacs-time-to-erc-time (current-time)))

(defun erc-time-diff (t1 t2)
  "Return time difference in seconds between T1 and T2 (T2 >= T1)"
  (- t2 t1))

(defun erc-time-gt (t1 t2)
  "Check whether T1 > T2"
  (> t1 t2))

(defun erc-sec-to-time (ns)
  "Convert seconds to a time string HH:MM.SS"
  (setq ns (truncate ns))
  (format "%02d:%02d.%02d"
	  (/ ns 3600)
	  (/ (% ns 3600) 60)
	  (% ns 60)))

;; info

(defconst erc-clientinfo-alist
  '(("ACTION" . "is used to inform about one's current activity")
    ("CLIENTINFO" . "gives help on CTCP commands supported by client")
    ("ECHO" . "echoes its arguments back")
    ("FINGER" . "shows user's name, location, and idle time")
    ("PING" . "measures delay between peers")
    ("TIME" . "shows client-side time")
    ("USERINFO" . "shows information provided by a user")
    ("VERSION" . "shows client type and version"))
  
  "Alist of CTCP CLIENTINFO for ERC commands")

(defun erc-client-info (s)
  "Return CTCP CLIENTINFO on command S. Is S is NIL or an empty string
then return general CLIENTINFO"

  (if (or (not s) (string= s ""))
      (concat
       (apply #'concat
	      (mapcar (lambda (e)
			(concat (car e) " "))
		      erc-clientinfo-alist))
       ": use CLIENTINFO <COMMAND> to get more specific information")
    (let ((h (assoc s erc-clientinfo-alist)))
      (if h
	  (concat s " " (cdr h))
	(concat s ": unknown command")))))

;;;; Hook functions

(defun erc-directory-writable-p (dir)
  "Determines whether the DIR is a writable directory.
  At this point, only determines whether it exists and is a directory.
  The rest is to be implemented."
  (let ((attr (file-attributes dir)))
    (and attr ;; it exists
	 (nth 0 attr) ;; it is a dir. or a sym. link
	 )))

(defun erc-truncate-buffer-to-size (size &optional buffer) 
  "Truncates the buffer to the size SIZE, leaving the bottom portion, if
  it is bigger than SIZE+512 characters.  If BUFFER is not provided,
  the current buffer is assumed.

  If `erc-log-channels' is non-nil and `erc-log-channels-directory' is
  a valid directory with write access, then append the cut portion of
  the buffer in the appropriate log file."
  (let ((ob (current-buffer)))
    (if (and buffer (get-buffer buffer))
	(set-buffer (get-buffer buffer)))
    (if (> (point-max) (+ size 512))
	(progn
	  (buffer-disable-undo)
	  (if (and erc-log-channels
		   erc-log-channels-directory
		   (erc-directory-writable-p erc-log-channels-directory))
		(append-to-file 1 (- (point-max) size) 
				(concat erc-log-channels-directory
					"/" (buffer-name) ".txt")))
	  (delete-region 1 (- (point-max) size))
	  (buffer-enable-undo)))
    (set-buffer ob)))

(defun erc-save-buffer-in-logs (&optional buffer)
  "When the logs are enabled, that is `erc-log-channels' is non-nil
and `erc-log-channels-directory' is a valid directory, appends the
entire BUFFER contents to the log file.  If BUFFER is not provided,
current buffer is used.

This is normally done on exit, to save the unsaved portion of the
buffer, since only the text that runs off the buffer limit is logged
automatically."
  (interactive)
  (if (not buffer) (setq buffer (current-buffer)))
  (let ((file buffer-file-name))
    (setq buffer-file-name nil)
    (if (and erc-log-channels
	     erc-log-channels-directory
	     (erc-directory-writable-p erc-log-channels-directory))
	(progn
	  (append-to-file (point-min) (point-max)
			  (concat erc-log-channels-directory
				  "/" (buffer-name buffer) ".txt"))
	  (erase-buffer)
	  (erc-display-prompt)
	  (goto-char (point-max))))
    (set-buffer-modified-p nil)
    (setq buffer-file-name file)))

(defun erc-truncate-buffer ()
  "Truncates the current buffer to `erc-max-buffer-size'.
Meant to be used in hooks, like `erc-insert-hook'."
  (interactive)
  (erc-truncate-buffer-to-size erc-max-buffer-size))

;; Join hook function - use it to run an autogreeting script
(defun erc-join-autogreet (chnl nick buffer &optional host login spec)
  "Runs an autogreet script provided by `erc-autogreet-script' with
arguments CHNL, NICK, and BUFFER name (all strings).  It is meant to
be used in `erc-join-hook'.

Keep in mind, that autogreets sometimes are considered poor style on
IRC.  Use it when you really need it."
  (save-excursion
    (erc-cmd-load "LOAD" (concat erc-autogreet-script " " chnl
				 " " nick " " buffer))))

;;;; miscellaneous

(defun erc-update-mode-line-buffer (buffer)
  "Update the mode line in a single ERC buffer BUFFER"
;  (erc-log (format "MODE LINE: erc-process: %S" (process-status erc-process)))
  (let ((ob (current-buffer)))
    (set-buffer buffer)
    (let ((name (buffer-name buffer))
	  (nick (erc-current-nick))
	  (tgt (or (erc-default-target) "0"))
	  (buffers erc-buffer-list))
      (setq mode-line-format
	    (list 
; 	   "-----ERC: "
; 	   erc-session-server
	     (if (boundp 'mode-line-mule-info) mode-line-mule-info "")
	     mode-line-modified
	     "%b"
	     "/" (erc-port-to-string erc-session-port)
	     (cond
	      ((not (equal (process-status erc-process) 'open))
	       " (CLOSED) ")
	      (away
	       (concat " (AWAY since "
		       (format-time-string "%a %b %d %H:%M" away)
		       ") "))
	      ;; Removing this makes erc-rename-all-buffers necessary
	      ;; to reflect current nick after renaming with /NICK correctly.
	      (t
;	       (concat " (" nick " -> " tgt ") ")
	       ""))
	     global-mode-string
	     "%[("
	     mode-name 
	     mode-line-process 
	     minor-mode-alist 
	     "%n" ")%]--"
	     (line-number-mode "L%l--") 
	     '(-3 . "%p") 
	     "-%-")))
    (set-buffer ob)))

(defun erc-update-mode-line (&optional buffer)
  "Update the mode line in one buffer BUFFER, if it is supplied, or
all ERC buffers otherwise."
  (if (and buffer (bufferp buffer))
      (erc-update-mode-line-buffer buffer)
    (let ((ob (current-buffer))
	  (buf (if (and (boundp 'erc-process) (processp erc-process))
		   (process-buffer erc-process) nil))
	  (bufs nil))
      (if buf (set-buffer buf))
      (setq bufs erc-buffer-list)
      (while bufs
	(if (buffer-live-p (car bufs))
	    (erc-update-mode-line-buffer (car bufs)))
	(setq bufs (cdr bufs)))
      (set-buffer ob))))

(defun erc-port-to-string (p)
  "Convert port P to string. P may be an integer or a service name"
  (if (integerp p)
      (int-to-string p)
    p))

(defun erc-string-to-port (s)
  "Convert string S to either integer port number or a service name"
  (let ((n (string-to-number s)))
    (if (= n 0)
	s
      n)))

(defun erc-version ()
  "Display ERC version information."
  (interactive)
  (message "ERC version %s" erc-version-string))

(defun erc-trim-string (s)
  "Trim leading and trailing spaces off the string"
  (cond
   ((not (stringp s)) nil)
   ((string-match "^\\s-*$" s)
    "")
   ((string-match "^\\s-*\\(.*\\S-\\)\\s-*$" s)
    (match-string 1 s))
   (t
    s)))

(provide 'erc)

;; end of $Source: /cvsroot/erc/erc/erc.el,v $
;;
;; Local Variables:
;; mode: outline-minor
;; outline-regexp: ";;+"
;; End:
