;;; erc-stamp.el --- Timestamping for Emacs IRC CLient

;; Copyright (C) 2002  Free Software Foundation, Inc.

;; Author: Mario Lang <mlang@delysid.org>
;; Keywords: comm, processes, timestamp
;; URL: http://www.emacswiki.org/cgi-bin/wiki.pl?ErcStamp

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; The code contained in this module is responsible for inserting
;; timestamps into ERC buffers.  In order to actually activate this,
;; you must call `erc-stamp-mode'.

;; You can choose between two different ways of inserting timestamps.
;; Customize `erc-insert-timestamp-function' and
;; `erc-insert-away-timestamp-function'.

;;; Code:

(require 'erc)
(require 'erc-compat)

(defconst erc-stamp-version "$Revision: 1.15 $"
  "ERC stamp mode revision.")

(defgroup erc-timestamp nil
  "For long conversation on IRC it is sometimes quite
useful to have individual messages timestamp.  This
group provides settings related to the format and display
of timestamp information in `erc-mode' buffer.

For timestamping to be activated, you just need to load `erc-stamp'
in your .emacs file or ineractively using `load-library'."
  :group 'erc)

(defcustom erc-timestamp-format "[%H:%M]"
  "*If set to a string, messages will be timestamped.
This string is processed using `format-time-string'.
Good examples are \"%T\" and \"%H:%M\".

If nil, timestamping is turned off."
  :group 'erc-timestamp
  :type '(choice (const nil)
		 (string)))

(defcustom erc-insert-timestamp-function 'erc-insert-timestamp-right
  "*Function to use to insert timestamps.

It takes a single argument STRING which is the final string
which all text-properties already appended.  This function only cares about
inserting this string at the right position.  Narrowing is in effect
while it is called, so (point-min) and (point-max) determine the region to
operate on."
  :group 'erc-timestamp
  :type '(choose (const :tag "Right" erc-insert-timestamp-right)
		 (const :tag "Left" erc-insert-timestamp-left)
		 function))

(defcustom erc-away-timestamp-format "<%H:%M>"
  "*Timestamp format used when marked as being away.

If nil, timestamping is turned off when away unless `erc-timestamp-format'
is set.

If `erc-timestamp-format' is set, this will not be used."
  :group 'erc-timestamp
  :type 'string)

(defcustom erc-insert-away-timestamp-function 'erc-insert-timestamp-right
  "*Function to use to insert the away timestamp.

See `erc-insert-timestamp-function' for details."
  :group 'erc-timestamp
  :type '(choose (const :tag "Right" erc-insert-timestamp-right)
		 (const :tag "Left" erc-insert-timestamp-left)
		 function))

(defcustom erc-hide-timestamps nil
  "*If non-nil, timestamps will be invisible.

This is useful for logging, because, although timestamps will be
hidden, they will still be present in the logs."
  :group 'erc-timestamp
  :type 'boolean)

(defcustom erc-echo-timestamps nil
  "*If non-nil, print timestamp in the minibuffer when point is moved.
Using this variable, you can turn off normal timestamping,
and simply move point to an irc message to see its timestamp
printed in the minibuffer."
  :group 'erc-timestamp
  :type 'boolean)

(defcustom erc-echo-timestamp-format "Timestamped %A, %H:%M:%S"
  "*Format string to be used when `erc-echo-timestamps' is non-nil.
This string specifies the format of the timestamp being echoed in
the minibuffer."
  :group 'erc-timestamp
  :type 'string)

(defface erc-timestamp-face '((t (:bold t :foreground "green")))
  "ERC timestamp face."
  :group 'erc-faces)

;;;###autoload (autoload 'erc-timestamp-mode "erc-stamp" nil t)
(define-erc-module stamp timestamp
  "This mode timestamps messages in the channel buffers."
  ((add-hook 'erc-mode-hook 'erc-munge-invisibility-spec)
   (add-hook 'erc-insert-modify-hook 'erc-add-timestamp
	     ;; If we use -right, we need to be after filling.
	     (eq erc-insert-timestamp-function 'erc-insert-timestamp-right))
   (add-hook 'erc-send-modify-hook 'erc-add-timestamp
	     (eq erc-insert-timestamp-function 'erc-insert-timestamp-right)))
  ((remove-hook 'erc-mode-hook 'erc-munge-invisibility-spec)
   (remove-hook 'erc-insert-modify-hook 'erc-add-timestamp)
   (remove-hook 'erc-send-modify-hook 'erc-add-timestamp)))

(defun erc-add-timestamp ()
  "Add timestamp and text-properties to message.

This function is ment to be called from `erc-insert-modify-hook'
or `erc-send-modify-hook'."
  (let ((ct (current-time)))
    (if (fboundp erc-insert-timestamp-function)
	(funcall erc-insert-timestamp-function
		 (erc-format-timestamp ct erc-timestamp-format))
      (error "Timestamp function unbound"))
    (when (and (fboundp erc-insert-away-timestamp-function)
	       erc-away-timestamp-format
	       (with-current-buffer (erc-server-buffer) away)
	       (not erc-timestamp-format))
      (funcall erc-insert-away-timestamp-function
	       (erc-format-timestamp ct erc-away-timestamp-format)))

    (add-text-properties (point-min) (point-max)
			 (list 'timestamp ct))
    (add-text-properties (point-min) (point-max)
			 (list 'point-entered 'erc-echo-timestamp))))

(defun erc-insert-timestamp-left (string)
  (goto-char (point-min))
  (insert string))

(defvar erc-timestamp-last-inserted nil
  "Last timestamp inserted into the buffer.")
(make-variable-buffer-local 'erc-timestamp-last-inserted)

(defcustom erc-timestamp-only-if-changed-flag t
  "*Insert only if timestamp actually changed since last insertion.
This only works with `erc-timestamp-rgiht'."
  :group 'erc-timestamp
  :type 'boolean)

(defun erc-insert-timestamp-right (string)
  "Insert timestamp on the right side of the screen.
This function uses the value of `erc-timestamp-only-if-changed-flag'."
  (unless (and erc-timestamp-only-if-changed-flag
	       (string-equal string erc-timestamp-last-inserted))
    (setq erc-timestamp-last-inserted string)
    (goto-char (point-max))
    (forward-char -1) ; Skip the \n
    (let ((slen (length string))
	  (col (- (point) (if (fboundp 'line-beginning-position)
			      (line-beginning-position)
			    (save-excursion (beginning-of-line) (point)))))
	  (wid (window-width)))
      (if (< (+ col slen) wid)
	  (insert (make-string (- wid (+ col slen) 1) ? ) string)
	(insert ?\n (make-string (- wid slen 1) ? ) string)))))
  
(defun erc-format-timestamp (time format)
  "Return TIME formatted as string according to FORMAT.
Return the empty string if FORMAT is nil."
  (if format
      (let ((ts (format-time-string format time)))
	(erc-put-text-property 0 (length ts) 'face 'erc-timestamp-face ts)
	(erc-put-text-property 0 (length ts) 'invisible 'timestamp ts)
	(erc-put-text-property 0 (length ts) 'isearch-open-invisible 'timestamp ts)
	;; N.B. Later use categories instead of this harmless, but
	;; inelegant, hack. -- BPT
	(erc-put-text-property 0 (length ts) 'intangible t ts)
	ts)
    ""))

;; This function is used to munge `buffer-invisibility-spec to an
;; appropriate value. Currently, it only handles timestamps, thus its
;; location.  If you add other features which affect invisibility,
;; please modify this function and move it to a more appropriate
;; location.
(defun erc-munge-invisibility-spec ()
  (if erc-hide-timestamps
      (setq buffer-invisibility-spec
	    (if (listp buffer-invisibility-spec)
		(cons 'timestamp buffer-invisibility-spec)
		(list 't 'timestamp)))
      (setq buffer-invisibility-spec
	    (if (listp buffer-invisibility-spec)
		(remove 'timestamp buffer-invisibility-spec)
	      (list 't)))))

(defun erc-hide-timestamps ()
  "Hide timestamp information from display."
  (interactive)
  (setq erc-hide-timestamps t)
  (erc-munge-invisibility-spec))

(defun erc-show-timestamps ()
  "Show timestamp ifnormation on display.
This function only works if `erc-timestamp-format' was previously
set, and timestamping is already active."
  (interactive)
  (setq erc-hide-timestamps nil)
  (erc-munge-invisibility-spec))

(defun erc-echo-timestamp (before now)
  "Print timestamp text-property of an IRC message.
Argument BEFORE is where point was before it got moved and
NOW is position of point currently."
  (when erc-echo-timestamps
    (let ((stamp (get-text-property now 'timestamp)))
      (when stamp
	(message (format-time-string erc-echo-timestamp-format
				     stamp))))))

(provide 'erc-stamp)

;;; erc-stamp.el ends here
