# GNU Enterprise Application Server - List Object
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: geasList.py 9953 2009-10-11 18:50:17Z reinhard $

import geasInstance
import time
import copy

from gnue.common.datasources import GConditions
from gnue.common.apps import errors, i18n

# =============================================================================
# List class
# =============================================================================

class geasList:

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self, session, classdef, connection, recordset, propertylist,
                condition, dsSort, asSort):
    self.__session    = session
    self.__classdef   = classdef
    self.__connection = connection
    self.__recordset  = recordset
    self.__generator  = recordset.__iter__()
    self.__prefetch   = propertylist      # property names to be prefetched
    self.__condition  = condition
    self.__instances  = []
    self.__unsorted   = []
    self.__isComplete = False
    self.__dsSorting  = dsSort          # sorted by datasource 
    self.__asSorting  = asSort          # additional sortorder of this list
    self.__length     = 0

    # If we have a condition to process, make sure we set a proper callback for
    # exist-conditions
    if self.__condition is not None:
      for item in self.__condition.findChildrenOfType ('GCexist', True, True):
        item.callback = self.__evaluateExist

    # Choose the right cache fillup function, depending on whether appserver
    # has to resort the data from the backend
    if len (self.__asSorting):
      self.__fillupFunc = self.__fillupSorted
    else:
      self.__fillupFunc = self.__fillupUnsorted


  # ---------------------------------------------------------------------------
  # fill up the instance list with up to count instances using sorting/grouping
  # ---------------------------------------------------------------------------

  def __fillupSorted (self, count):
    """
    This function fills the internal list of instances with up to count
    elements. If @count is 0 all available instances are added to the list.
    """

    while self.__wantMore (count):

      # if a pending element is available, start with it; otherwise fetch the
      # next one from the datasource
      if len (self.__unsorted):
        current = self.__unsorted.pop ()
      else:
        current = self.__getInstance ()

      while self.__wantMore (count) and current is not None:
        group = self.__getGroup (current)

        while current is not None and self.__getGroup (current) == group:
          self.__unsorted.append (current)
          current = self.__getInstance ()

        if len (self.__unsorted) > 1:
          self.__sortBatch ()

        self.__instances.extend (self.__unsorted)

        # if no more instances are needed at the moment, but there was a valid
        # element at the end, we hold it as a 'pending but still unsorted' one
        if not self.__wantMore (count) and current is not None:
          self.__unsorted = [current]
        else:
          self.__unsorted = []


      if current is None:
        break

    self.__length = len (self.__instances)


  # ---------------------------------------------------------------------------
  # Fill up instance list without any sorting/grouping
  # ---------------------------------------------------------------------------

  def __fillupUnsorted (self, count):
    """
    This function fills up the internal list of instances with up to @count
    elements.
    """

    while self.__wantMore (count):
      current = self.__getInstance ()
      if current is not None:
        self.__instances.append (current)

    self.__length = len (self.__instances)


  # ---------------------------------------------------------------------------
  # wanna have more instances ?
  # ---------------------------------------------------------------------------

  def __wantMore (self, count):
    """
    This function returns TRUE if more instances are needed to fill up the
    instance cache with @count elements. If @count is zero, an unlimited amount
    of instances is accepted.
    """
    return not self.__isComplete and \
          (not count or len (self.__instances) < count)



  # ---------------------------------------------------------------------------
  # Get the next usable instance and add it to the instance list
  # ---------------------------------------------------------------------------

  def __getInstance (self):
    """
    This function fetches the next usable instance from the recordset according
    to the list's condition. This instance will be returned as function result.
    """

    while True:
      try:
        record = self.__generator.next ()
      except StopIteration:
        # No more records
        self.__isComplete = True
        self.__length = len (self.__instances)
        self.__recordset.close ()
        self.__recordset = None
        return None

      # Found a record - check condition
      instance = geasInstance.geasInstance (self.__session, self.__connection,
                                            record, self.__classdef)
      if self.__condition is None or self.__condition.evaluate (instance):
        return instance


  # ---------------------------------------------------------------------------
  # Get a grouping sequence for an instance
  # ---------------------------------------------------------------------------

  def __getGroup (self, instance):
    """
    This function returns a grouping list according to the datasource-sorting.
    If no such sort order is defined, None is returned.
    """
    if len (self.__dsSorting):
      return instance.get ([item ['name'] for item in self.__dsSorting])
    else:
      return None


  # ---------------------------------------------------------------------------
  # sort the current batch in __unsorted
  # ---------------------------------------------------------------------------

  def __sortBatch (self):
    """
    This function sorts the unsorted batch according to asSorting.
    """

    if len (self.__asSorting) and len (self.__unsorted) > 1:

      for i in self.__unsorted:
        # NOTE: building a sort-sequence this way is not optimal. If there are
        # multiple calculated fields in an order-by all of them will be
        # evaluated.
        order = []

        for element in self.__asSorting:
          
          value = i.get ([element ['name']]) [0]
          if element.get ('ignorecase') and hasattr (value, 'lower'):
            value = value.lower ()

          order.append ((value, element.get ('descending') or False))

        i.setSort (order)


      self.__unsorted.sort ()



  # ---------------------------------------------------------------------------
  # Get the length of the list (the number of entries)
  # ---------------------------------------------------------------------------

  def count (self):
    """
    Get the number of instances available in this list.

    @return: number of instances available in this list
    """

    assert gEnter (4)

    if self.__session.locale:
      i18n.setcurrentlocale (self.__session.locale)

    if self.__condition is not None:
      self.__fillupFunc (0)
      result = len (self.__instances)
    else:
      if self.__recordset is None:
        result = self.__length
      else:
        result = len (self.__recordset)

    assert gLeave (4, result)

    return result


  # ---------------------------------------------------------------------------
  # Fetch data from the database backend
  # ---------------------------------------------------------------------------

  def fetch (self, start, count):
    """
    This function returns @count instances skipping the first @start instances.
    If @start is negative, the starting point is from the end of the list.
    """

    assert gEnter (4)

    if self.__session.locale:
      i18n.setcurrentlocale (self.__session.locale)

    if start < 0:
      self.__fillupFunc (0)
    else:
      self.__fillupFunc (start + count)

    result = []
    for pos in range (start, min (start + count, len (self.__instances))):
      result.append (self.__instances [pos].get (self.__prefetch))

    assert gLeave (4, result)

    return result


  # ---------------------------------------------------------------------------
  # Destroy the list instance
  # ---------------------------------------------------------------------------

  def _destroy (self):
    """
    This function closes the list by freeing all references and resources
    maintained by the list.
    """

    assert gEnter (4)

    if self.__recordset is not None:
      self.__recordset.close ()
      self.__recordset = None

    self.__instances = []
    self.__unsorted  = []
    self.__condition = None

    assert gLeave (4)


  # ---------------------------------------------------------------------------
  # Evaluate an exist condition for an instance
  # ---------------------------------------------------------------------------

  def __evaluateExist (self, caller, lookup):
    """
    This function evaluates an exist-condition for the instance given in
    'lookup'.
    @param caller: GCexist instance defining the condition
    @param lookup: geasInstance instance to be evaluated
    @return: True if there are instance matching the condition, False otherwise
    """

    linkCond = GConditions.buildConditionFromDict ( \
                              {caller.detaillink: lookup [caller.masterlink]})

    parent = linkCond._children [0]

    # If the condition has a subcondition we need to 'copy' this subcondition
    # elements. This is to prevent appserver's changes made to the condition
    # elements later, disturbing the following request.
    if len (caller._children):
      for c in caller._children:
        new = GConditions.buildConditionFromPrefix (c.prefixNotation ())
        new.parent = parent
        parent._children.append (new)

    resList = self.__session.request (caller.table, linkCond, [], [u'gnue_id'])
    result  = resList.fetch (0, 1)

    return len (result) > 0

