/*
     This file is part of GNUnet.
     (C) 2010 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/gnunet-gtk.c
 * @brief Main function of gnunet-gtk
 * @author Christian Grothoff
 */
#include "common.h"
#include "fs_event_handler.h"
#if ENABLE_NLS
#include <locale.h>
#endif
#include "peerinfo.h"

/**
 * Should gnunet-gtk start in tray mode?
 */
static int tray_only;

/**
 * Our tray icon.
 */
static GtkStatusIcon *tray_icon;

/**
 * Initialize GNU gettext for message translation.
 */
static void
setup_nls ()
{
#if ENABLE_NLS
  char *path;

  setlocale (LC_ALL, "");
  GNUNET_asprintf (&path,
		   "%s/%s/locale/",
		   GNUNET_GTK_get_data_dir (),
		   PACKAGE_NAME);
  bindtextdomain ("gnunet-gtk", path);
  textdomain ("gnunet-gtk");
  bind_textdomain_codeset ("GNUnet", "UTF-8");
  bind_textdomain_codeset ("gnunet-gtk", "UTF-8");
  GNUNET_free (path);
#else
  fprintf (stderr,
	   "WARNING: gnunet-gtk was compiled without i18n support (did CFLAGS contain -Werror?).\n");
#endif
}


/**
 * Main context for our event loop.
 */
struct MainContext
{

  /**
   * The scheduler.
   */
  struct GNUNET_SCHEDULER_Handle *sched;

  /**
   * Main loop.
   */
  GMainLoop *gml;

  /**
   * GTK's main context.
   */
  GMainContext *gmc;

  /**
   * Recycled array of polling descriptors.
   */
  GPollFD *cached_poll_array;

  /**
   * The main window.
   */
  GtkWidget *main_window;

  /**
   * Read set.
   */
  struct GNUNET_NETWORK_FDSet *rs;

  /**
   * Write set.
   */
  struct GNUNET_NETWORK_FDSet *ws;

  /**
   * Builder for the main window.
   */
  GtkBuilder *builder; 

  /**
   * Handle for file-sharing operations.
   */
  struct GNUNET_FS_Handle *fs;

  /**
   * List of plugins for meta data extraction.
   */
  struct EXTRACTOR_PluginList *plugins;

  /**
   * Context for peerinfo notifications.
   */
  struct GNUNET_PEERINFO_NotifyContext *pnc;

  /**
   * Size of the 'cached_poll_array'.
   */
  guint cached_poll_array_size;

  /**
   * Return value from last g_main_context_query call.
   */
  guint poll_array_active;

  /**
   * Maximum GTK priority.
   */
  gint max_priority;

};


/**
 * Get the name of the directory where all of our package
 * data is stored ($PREFIX/share/)
 * 
 * @return name of the data directory
 */
const char *
GNUNET_GTK_get_data_dir ()
{
  /* FIXME: this is a hack that does not properly support
     relocation of the binary data... */
  return PACKAGE_DATA;
}


/**
 * Free resources of the given context.
 *
 * @param context what to free, can be NULL
 */
static void
free_context (struct MainContext *context)
{
  if (context == NULL)
    return;
  g_free (context->cached_poll_array);
  GNUNET_NETWORK_fdset_destroy (context->rs);
  GNUNET_NETWORK_fdset_destroy (context->ws);
  g_object_unref (G_OBJECT (context->builder));
  g_main_loop_unref (context->gml);
  GNUNET_free (context);
}


/**
 * Run GTK tasks that are ready.
 *
 * @param cls the 'struct MainContext'
 * @param tc task context
 */
static void
run_ready (void *cls,
	   const struct GNUNET_SCHEDULER_TaskContext *tc);


/**
 * Schedule the main GTK Event loop with the GNUnet scheduler.
 *
 * @param cls the 'struct MainContext'
 * @param tc task context
 */
static void
schedule_main_loop (void *cls,
		    const struct GNUNET_SCHEDULER_TaskContext *tc)
{
  struct MainContext *mc = cls;
  struct GNUNET_TIME_Relative delay;
  gint timeout;
  gint nfds;
  GPollFD *fds;
  guint allocated_nfds;

  if ( (tc != NULL) &&
       (0 != (GNUNET_SCHEDULER_REASON_SHUTDOWN & tc->reason)) )
    {
      g_main_loop_quit (mc->gml);
      GNUNET_SCHEDULER_add_now (mc->sched,
				&schedule_main_loop,
				mc);
      return;
    }
  if (TRUE != g_main_loop_is_running (mc->gml))
    return;
  g_main_context_prepare (mc->gmc, &mc->max_priority);
  allocated_nfds = mc->cached_poll_array_size;
  fds = mc->cached_poll_array;
  while ((nfds = g_main_context_query (mc->gmc, mc->max_priority, 
				       &timeout, fds, 
				       allocated_nfds)) > allocated_nfds)
    {
      g_free (fds);
      mc->cached_poll_array_size = allocated_nfds = nfds;
      mc->cached_poll_array = fds = g_new (GPollFD, nfds);
    }
  mc->poll_array_active = nfds;
  delay.value = (unsigned long long) timeout;
  GNUNET_NETWORK_fdset_zero (mc->rs);
  GNUNET_NETWORK_fdset_zero (mc->ws);
  while (nfds > 0)
    {
      nfds--;
      if (fds[nfds].events & (G_IO_IN | G_IO_HUP | G_IO_ERR))
	GNUNET_NETWORK_fdset_set_native (mc->rs, fds[nfds].fd);
      if (fds[nfds].events & (G_IO_OUT | G_IO_ERR))
	GNUNET_NETWORK_fdset_set_native (mc->ws, fds[nfds].fd);
    }
  GNUNET_SCHEDULER_add_select (mc->sched,
			       GNUNET_SCHEDULER_PRIORITY_UI,
			       GNUNET_SCHEDULER_NO_TASK,
			       delay,
			       mc->rs,
			       mc->ws,
			       &run_ready,
			       mc);
}


/**
 * Run GTK tasks that are ready.
 *
 * @param cls the 'struct MainContext'
 * @param tc task context
 */
static void
run_ready (void *cls,
	   const struct GNUNET_SCHEDULER_TaskContext *tc)
{
  struct MainContext *mc = cls;
  
  g_poll (mc->cached_poll_array,
	  mc->poll_array_active,
	  0);		    
  if (TRUE == 
      g_main_context_check (mc->gmc, 
			    mc->max_priority, 
			    mc->cached_poll_array, 
			    mc->poll_array_active))
    g_main_context_dispatch (mc->gmc);
  GNUNET_SCHEDULER_add_now (mc->sched,
			    &schedule_main_loop,
			    mc);
}


/**
 * Main context (global so we can free it on exit and use
 * it for termination).
 */
static struct MainContext *mc;


/**
 * Task run on shutdown.
 */
static void
shutdown_task (void *cls,
	       const struct GNUNET_SCHEDULER_TaskContext *tc)
{
  if (mc->fs != NULL)
    {
      GNUNET_FS_stop (mc->fs);
      mc->fs = NULL;
    }
  EXTRACTOR_plugin_remove_all (mc->plugins);
  mc->plugins = NULL;
  if (mc->pnc != NULL)
    {
      GNUNET_PEERINFO_notify_cancel (mc->pnc);
      mc->pnc = NULL;
    }
}


/**
 * Return handle for file-sharing operations.
 * @return NULL on error
 */
struct GNUNET_FS_Handle *
GNUNET_GTK_get_fs_handle ()
{
  return mc->fs;
}


/**
 * Get LE plugin list.
 */
struct EXTRACTOR_PluginList *
GNUNET_GTK_get_le_plugins ()
{
  return mc->plugins;
}


/**
 * Callback invoked if the application is supposed to exit.
 */
void 
GNUNET_GTK_quit_cb (GtkObject *object, 
		    gpointer user_data)
{
  if (tray_icon != NULL)
    {
      g_object_unref (G_OBJECT (tray_icon));
      tray_icon = NULL;
    }
  g_main_loop_quit (mc->gml);
  GNUNET_SCHEDULER_add_now (mc->sched,
			    &shutdown_task,
			    NULL);
}


/**
 * Create an initialize a new builder based on the
 * GNUnet-GTK glade file.
 *
 * @param filename name of the resource file to load
 * @return NULL on error
 */
GtkBuilder *
GNUNET_GTK_get_new_builder (const char *filename)
{
  char *glade_path;
  GtkBuilder *ret;
  GError *error;

  ret = gtk_builder_new ();
  gtk_builder_set_translation_domain (ret, "gnunet-gtk");
  GNUNET_asprintf (&glade_path,
		   "%s/%s/%s",
		   GNUNET_GTK_get_data_dir (),
		   PACKAGE_NAME,
		   filename);
  error = NULL;
  if (0 == gtk_builder_add_from_file (ret, glade_path, &error))
    {
      GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
		  _("Failed to load `%s': %s\n"),
		  glade_path,
		  error->message);
      g_error_free (error);
      GNUNET_free (glade_path);
      return NULL;
    }
  gtk_builder_connect_signals (ret, NULL);
  GNUNET_free (glade_path);
  return ret;
}


/**
 * We got a click on our tray icon. Toggle visibility of the main
 * window.
 */
static void 
tray_icon_on_click(GtkStatusIcon *status_icon, 
		   gpointer user_data)
{
  if (gtk_window_is_active (GTK_WINDOW (mc->main_window)))
    gtk_widget_hide (mc->main_window);
  else
    gtk_window_present (GTK_WINDOW (mc->main_window));
}


/**
 * We got a right-click on the tray icon. Display the context
 * menu (which should have a 'quit' button).
 */
static int 
tray_icon_on_menu(GtkWidget *widget, 
		  GdkEvent *event,
		  gpointer user_data)
{
  GtkMenu *tray_menu;
  GdkEventButton *event_button;
  GtkBuilder *builder;
 
  if (event->type == GDK_BUTTON_PRESS)
    {
      event_button = (GdkEventButton *) event;      
      if (event_button->button == 3)
	{
	  builder = GNUNET_GTK_get_new_builder ("status_bar_menu.glade");
	  tray_menu = GTK_MENU (gtk_builder_get_object (builder,
							"GNUNET_GTK_status_bar_popup_menu"));
	  gtk_menu_popup (tray_menu, NULL, NULL, NULL, NULL,
			  event_button->button, event_button->time);
	  g_object_unref (builder);
	}
    }
  return FALSE;
}


/**
 * Create our tray icon.
 */
static GtkStatusIcon *
create_tray_icon() {
  GtkStatusIcon *tray_icon;

  tray_icon = gtk_status_icon_new();
  g_signal_connect(G_OBJECT(tray_icon), "activate", 
		   G_CALLBACK(tray_icon_on_click), NULL);
  g_signal_connect (G_OBJECT(tray_icon), 
		    "button_press_event",
		    G_CALLBACK(tray_icon_on_menu), 
		    tray_icon);
  gtk_status_icon_set_from_icon_name (tray_icon, 
				      "gnunet-gtk");
  gtk_status_icon_set_tooltip(tray_icon, 
			      "gnunet-gtk");
  gtk_status_icon_set_visible(tray_icon, TRUE); 
  return tray_icon;
}


/**
 * Initialize GTK search path for icons.
 */
static void
set_icon_search_path ()
{
  char *buf;

  GNUNET_asprintf (&buf,
		   "%s/icons/",
		   GNUNET_GTK_get_data_dir ());
  gtk_icon_theme_append_search_path (gtk_icon_theme_get_default (), buf);
  GNUNET_free (buf);
}


/**
 * Search selected in 'file' menu. (from main_window_file_search.c)
 */
void
GNUNET_GTK_main_menu_file_search_activate_cb (GtkWidget * dummy, 
					      gpointer data);


/**
 * Add the tab with the 'new' icon for starting a search.
 */ 
static void 
add_new_tab ()
{
  GtkNotebook *notebook;
  GtkWindow *sf;
  gint pages;
  GtkBuilder *builder;
  GtkWidget *label;
  GtkWidget *frame;

  builder = GNUNET_GTK_get_new_builder ("main_tab_new_frame.glade");

  /* load frame */
  sf = GTK_WINDOW (gtk_builder_get_object (builder,
					   "_main_tab_new_frame"));
  label = gtk_bin_get_child (GTK_BIN (sf));
  gtk_widget_ref (label);
  gtk_container_remove (GTK_CONTAINER (sf), label);
  gtk_widget_destroy (GTK_WIDGET (sf));
  g_object_unref (builder);
  g_signal_connect(G_OBJECT(label), "clicked", 
		   G_CALLBACK(GNUNET_GTK_main_menu_file_search_activate_cb), NULL);

  notebook = GTK_NOTEBOOK (GNUNET_GTK_get_main_window_object ("GNUNET_GTK_main_window_notebook"));
  pages = gtk_notebook_get_n_pages (notebook);
  frame = gtk_label_new ("");
  gtk_widget_show (frame);
  gtk_notebook_append_page (notebook, 
			    frame,
			    label);
  gtk_notebook_set_current_page (notebook, 
				 pages);
  gtk_widget_show (GTK_WIDGET (notebook));
}


/**
 * Actual main function run right after GNUnet's scheduler
 * is initialized.  Initializes up GTK and Glade.
 */		      
static void
run (void *cls,
     struct GNUNET_SCHEDULER_Handle *sched,
     char *const *args,
     const char *cfgfile,
     const struct GNUNET_CONFIGURATION_Handle *cfg)
{
  int argc;
 
  /* command-line processing */
  argc = 0;
  while (args[argc] != NULL) argc++;
  gtk_init (&argc, (char ***) &args);
  set_icon_search_path ();
  setup_nls (cfg);

  /* setup main context */
  mc = GNUNET_malloc (sizeof (struct MainContext));
  mc->builder = GNUNET_GTK_get_new_builder ("main-window.glade");
  mc->rs = GNUNET_NETWORK_fdset_create ();
  mc->ws = GNUNET_NETWORK_fdset_create ();
  mc->gml = g_main_loop_new (NULL, TRUE);
  mc->gmc = g_main_loop_get_context (mc->gml);
  mc->sched = sched;  
  if (mc->builder == NULL)
    return;

  /* setup tray icon */
  tray_icon = create_tray_icon();

  /* setup main window */
  mc->main_window = GTK_WIDGET (gtk_builder_get_object (mc->builder, 
							"GNUNET_GTK_main_window"));
  gtk_window_maximize (GTK_WINDOW (mc->main_window));

  /* initialize file-sharing */
  mc->plugins = EXTRACTOR_plugin_add_defaults (EXTRACTOR_OPTION_DEFAULT_POLICY);
  mc->fs = GNUNET_FS_start (sched,
			    cfg,
			    "gnunet-gtk",
			    &GNUNET_GTK_fs_event_handler,
			    NULL,
			    GNUNET_FS_FLAGS_NONE /* fixme later for persistence/probes */,
			    /* set other options here later! */
			    GNUNET_FS_OPTIONS_END);  
  if (mc->fs != NULL)    
    {
      add_new_tab ();
    }
  else
    {
      /* FIXME: hide FS-menu options */
    }
  mc->pnc = GNUNET_PEERINFO_notify (cfg, sched,
				    &GNUNET_GTK_peerinfo_processor,
				    NULL);
  if (mc->pnc == NULL)
    {
      /* FIXME: hide peerinfo widget and menu view option */
    }
  /* make GUI visible */
  if (!tray_only)
    {
      gtk_widget_show (mc->main_window);
      gtk_window_present (GTK_WINDOW (mc->main_window));
    }
  /* start the event loop */
  GNUNET_assert (TRUE == g_main_context_acquire (mc->gmc));
  schedule_main_loop (mc, NULL);
}


/**
 * Get an object from the main window.
 *
 * @param name name of the object
 * @return NULL on error
 */
GObject *
GNUNET_GTK_get_main_window_object (const char *name)
{
  return gtk_builder_get_object (mc->builder, name);
}


/**
 * gnunet-gtk command line options
 */
static struct GNUNET_GETOPT_CommandLineOption options[] = {
  {'t', "tray", NULL,
   gettext_noop ("start in tray mode"), 0,
   &GNUNET_GETOPT_set_one, &tray_only},
  GNUNET_GETOPT_OPTION_END
};


int
main (int argc, char *const *argv)
{
  if (GNUNET_OK !=
      GNUNET_PROGRAM_run (argc, argv,
			  "gnunet-gtk",
			  "GTK GUI for GNUnet",
			  options,
			  &run, NULL))
    {
      free_context (mc);
      mc = NULL;
      return 1;
    }
  free_context (mc);
  mc = NULL;
  return 0;
}


/* end of main.c */
