/*
     This file is part of GNUnet
     (C) 2005, 2006, 2010 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/main_window_adv_pseudonym.c
 * @author Christian Grothoff
 */
#include "common.h"

static GtkBuilder *builder;


/**
 * Canonicalize spelling of currently selected keyword.
 */
void
GNUNET_GTK_advertise_pseudonym_keyword_list_normalize_button_clicked_cb ()
{
  GtkTreeView *tv;
  GtkTreeSelection *sel;
  GtkTreeModel *tm;
  GtkTreeIter iter;
  gchar *value;
  struct GNUNET_FS_Uri *uri;
  struct GNUNET_FS_Uri *nuri;
  char *nvalue;
  char *emsg;

  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_advertise_pseudonym_keyword_list_tree_view"));
  sel = gtk_tree_view_get_selection (tv);
  if (TRUE != gtk_tree_selection_get_selected (sel, &tm, &iter))
    {
      GNUNET_break (0);
      return;
    }
  gtk_tree_model_get (tm, &iter,
		      0, &value,
		      -1);
  emsg = NULL;
  uri = GNUNET_FS_uri_ksk_create (value, &emsg);
  if (uri == NULL)
    {
      GNUNET_break (0);
      GNUNET_free (emsg);
      g_free (value);
      return;
    }
  nuri = GNUNET_FS_uri_ksk_canonicalize (uri);
  nvalue = GNUNET_FS_uri_ksk_to_string_fancy (nuri);
  gtk_list_store_set (GTK_LIST_STORE (tm), &iter,
		      0, nvalue,
		      -1);
  GNUNET_FS_uri_destroy (nuri);
  GNUNET_FS_uri_destroy (uri);
  GNUNET_free (nvalue);
  g_free (value);
}


/**
 * Remove the currently selected keyword from the list.
 */
void
GNUNET_GTK_advertise_pseudonym_keyword_list_del_button_clicked_cb ()
{
  GtkTreeView *tv;
  GtkTreeSelection *sel;
  GtkTreeModel *tm;
  GtkTreeIter iter;

  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_advertise_pseudonym_keyword_list_tree_view"));
  sel = gtk_tree_view_get_selection (tv);
  if (TRUE != gtk_tree_selection_get_selected (sel, &tm, &iter))
    {
      GNUNET_break (0);
      return;
    }
  gtk_list_store_remove (GTK_LIST_STORE (tm),
			 &iter);
}


/**
 * Add keyword from entry line to keyword list.
 */ 
void
GNUNET_GTK_advertise_pseudonym_keyword_list_add_button_clicked_cb ()
{
  const char *keyword;
  GtkEntry *entry;
  GtkListStore *ls;
  GtkTreeIter iter;

  ls = GTK_LIST_STORE (gtk_builder_get_object (builder,
					       "GNUNET_GTK_keyword_list_store"));
  entry = GTK_ENTRY (gtk_builder_get_object (builder,						 
					     "GNUNET_GTK_advertise_pseudonym_keyword_entry"));
  keyword = gtk_entry_get_text (entry);
  if (strlen (keyword) > 0)
    gtk_list_store_insert_with_values (ls, &iter, G_MAXINT, 0, keyword, -1);
  gtk_entry_set_text (entry, "");
}


/**
 * Update sensitivity of keyword add button.
 */
void
GNUNET_GTK_advertise_pseudonym_keyword_entry_changed_cb ()
{
  const char *keyword;
  GtkEntry *entry;
  GtkWidget *button;

  button = GTK_WIDGET (gtk_builder_get_object (builder,						 
					       "GNUNET_GTK_advertise_pseudonym_keyword_list_add_button"));
  entry = GTK_ENTRY (gtk_builder_get_object (builder,						 
					     "GNUNET_GTK_advertise_pseudonym_keyword_entry"));
  keyword = gtk_entry_get_text (entry);
  gtk_widget_set_sensitive (button,
			    (strlen (keyword) > 0) ? TRUE : FALSE);
}

/**
 * Update sensitivity of metadata delete button.
 */
static void
metadata_selection_changed_cb (GtkTreeSelection *ts,
			       gpointer user_data)
{
  GtkTreeView *tv;
  GtkTreeSelection *sel;
  GtkWidget *button;

  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_advertise_pseudonym_metadata_tree_view"));
  sel = gtk_tree_view_get_selection (tv);
  button = GTK_WIDGET (gtk_builder_get_object (builder,						 
					       "GNUNET_GTK_advertise_pseudonym_metadata_delete_button"));
  gtk_widget_set_sensitive (button,
			    gtk_tree_selection_get_selected (sel, NULL, NULL));
}


/**
 * Update sensitivity of keyword delete and normalize button.
 */
static void
keyword_selection_changed_cb (GtkTreeSelection *ts,
			      gpointer user_data)
{
  GtkTreeView *tv;
  GtkTreeSelection *sel;
  GtkWidget *button;

  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_advertise_pseudonym_keyword_list_tree_view"));
  sel = gtk_tree_view_get_selection (tv);
  button = GTK_WIDGET (gtk_builder_get_object (builder,						 
					       "GNUNET_GTK_advertise_pseudonym_keyword_list_del_button"));

  gtk_widget_set_sensitive (button,
			    gtk_tree_selection_get_selected (sel, NULL, NULL));
  button = GTK_WIDGET (gtk_builder_get_object (builder,						 
					       "GNUNET_GTK_advertise_pseudonym_keyword_list_normalize_button"));
  gtk_widget_set_sensitive (button,
			    gtk_tree_selection_get_selected (sel, NULL, NULL));
}


/**
 * Update sensitivity of metadata add button.
 */
void
GNUNET_GTK_advertise_pseudonym_metadata_value_entry_changed_cb ()
{  
  GtkTreeView *tv;
  GtkTreeSelection *sel;
  const char *value;
  GtkEntry *entry;
  GtkWidget *button;

  entry = GTK_ENTRY (gtk_builder_get_object (builder,						 
					     "GNUNET_GTK_advertise_pseudonym_metadata_value_entry"));
  value = gtk_entry_get_text (entry);
  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_advertise_pseudonym_metadata_type_tree_view"));
  sel = gtk_tree_view_get_selection (tv);
  button = GTK_WIDGET (gtk_builder_get_object (builder,						 
					       "GNUNET_GTK_advertise_pseudonym_metadata_add_button"));  
  gtk_widget_set_sensitive (button,
			    (strlen (value) > 0)
			    ? gtk_tree_selection_get_selected (sel, NULL, NULL) 
			    : FALSE);
}


/**
 * Add keyword to keyword list.
 */
void
GNUNET_GTK_advertise_pseudonym_keyword_entry_activate_cb ()
{
  GNUNET_GTK_advertise_pseudonym_keyword_list_add_button_clicked_cb ();
}


void
GNUNET_GTK_advertise_pseudonym_metadata_preview_file_chooser_button_file_set_cb ()
{
  GNUNET_break (0);
}

/**
 * Delete currently selected item from meta data list.
 */
void
GNUNET_GTK_advertise_pseudonym_metadata_delete_button_clicked_cb()
{
  GtkTreeView *tv;
  GtkTreeSelection *sel;
  GtkTreeModel *tm;
  GtkTreeIter iter;

  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_advertise_pseudonym_metadata_tree_view"));
  sel = gtk_tree_view_get_selection (tv);
  if (TRUE != gtk_tree_selection_get_selected (sel, &tm, &iter))
    {
      GNUNET_break (0);
      return;
    }
  gtk_list_store_remove (GTK_LIST_STORE (tm),
			 &iter);
}


/**
 * Add metadata to metadata list.
 */
void
GNUNET_GTK_advertise_pseudonym_metadata_add_button_clicked_cb ()
{
  const char *value;
  char *avalue;
  const char *ivalue;
  GtkEntry *entry;
  GtkListStore *ls;
  GtkTreeModel *tm;
  GtkTreeView *tv;
  GtkTreeSelection *sel;
  GtkTreeIter iter;
  guint type;
  size_t slen;
  char *pos;

  entry = GTK_ENTRY (gtk_builder_get_object (builder,
					     "GNUNET_GTK_advertise_pseudonym_metadata_value_entry"));
  value = gtk_entry_get_text (entry);
  if ((value == NULL) || (strlen (value) == 0))
    {
      GNUNET_break (0);
      return;
    }
  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_advertise_pseudonym_metadata_type_tree_view"));
  tm = gtk_tree_view_get_model (tv);
  sel = gtk_tree_view_get_selection (tv);
  if (TRUE != gtk_tree_selection_get_selected (sel,
					       &tm,
					       &iter))
    {
      GNUNET_break (0);
      return;
    }
  gtk_tree_model_get (tm,
                      &iter, 
		      1, &type, -1);
  ls = GTK_LIST_STORE (gtk_builder_get_object (builder,
					       "GNUNET_GTK_meta_data_list_store"));
  if ( (type == EXTRACTOR_METATYPE_FILENAME) &&
       (value[strlen(value)-1] != '/') )
    {      
      GNUNET_asprintf (&avalue,
		       "%s/",
		       value);
      /* if user typed '\' instead of '/', change it! */
      slen = strlen (avalue);
      while ( (slen > 1) &&
	      (avalue[slen-2] == '\\') )
	{
	  avalue[slen-2] = '/';
	  avalue[slen-1] = '\0';
	  slen--;
	}
      while (NULL != (pos = strstr (avalue, "\\")))
	*pos = '/';
      /* remove '../' everywhere */
      while (NULL != (pos = strstr (avalue, "../")))
	{
	  pos[0] = '_';
	  pos[1] = '_';
	  pos[2] = '_';
	}
      ivalue = avalue;
    }
  else
    {
      ivalue = value;
      avalue = NULL;
    }

  gtk_list_store_insert_with_values (ls, &iter, G_MAXINT,
				     0, type,
				     1, (guint) EXTRACTOR_METAFORMAT_UTF8,
				     2, EXTRACTOR_metatype_to_string (type),
				     3, ivalue,
				     -1);
  GNUNET_free_non_null (avalue);   
  gtk_entry_set_text (GTK_ENTRY (entry), "");
}


/**
 * Callback with information about local (!) namespaces.
 * Contains the names of the local namespace and the global
 * ID.  Adds the to the list store.
 *
 * @param cls closure, the list store.
 * @param name human-readable identifier of the namespace
 * @param id hash identifier for the namespace
 */
static void
add_to_list (void *cls,
	     const char *name,
	     const GNUNET_HashCode *id)
{
  GtkListStore *ls = cls;
  GtkTreeIter iter;

  gtk_list_store_insert_with_values (ls,
				     &iter,
				     -1,
				     0, name,
				     1, GNUNET_FS_namespace_create (GNUNET_GTK_get_fs_handle (),
								    name),
				     -1);
  
}


static void
selection_changed_cb (GtkTreeSelection *ts,
		      gpointer user_data)
{
  GtkTreeIter iter;
  GtkWidget *ok_button;
  
  ok_button = GTK_WIDGET (gtk_builder_get_object (builder,
						  "GNUNET_GTK_select_pseudonym_ok_button"));
  if (TRUE == gtk_tree_selection_get_selected (ts, NULL, &iter))
    gtk_widget_set_sensitive (ok_button, TRUE);
  else
    gtk_widget_set_sensitive (ok_button, FALSE);
}


/**
 * Advertise pseudonym selected.
 */
void
GNUNET_GTK_main_menu_file_advertise_pseudonym_activate_cb (GtkWidget * dummy, 
							   gpointer data)
{
  GtkWidget *ad;
  struct GNUNET_FS_Namespace *ns;
  struct GNUNET_FS_Namespace *nso;
  GtkListStore *ls;
  GtkTreeView *tv;
  GtkTreeIter iter;
  GtkTreeSelection *sel;
  GtkTreeModel *tm;
  GtkSpinButton *sb;
  uint32_t anonymity_level;
  uint32_t priority;
  struct GNUNET_FS_Uri *nxt;
  struct GNUNET_FS_Uri *mrg;
  struct GNUNET_FS_Uri *uri;
  struct GNUNET_CONTAINER_MetaData *nm;
  struct GNUNET_TIME_Absolute expiration;
  const char *root;
  char *value;
  guint ntype;  
  guint nformat;
  guint type;
  guint max_type;
 
  builder = GNUNET_GTK_get_new_builder ("select_pseudonym_dialog.glade");
  if (builder == NULL)
    {
      GNUNET_break (0);     
      return;
    }
  ad = GTK_WIDGET (gtk_builder_get_object (builder,
					   "GNUNET_GTK_select_pseudonym_dialog"));
  ls = GTK_LIST_STORE (gtk_builder_get_object (builder,
					       "GNUNET_GTK_select_pseudonym_liststore"));
  GNUNET_FS_namespace_list (GNUNET_GTK_get_fs_handle (),
			    &add_to_list,
			    ls);
  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_select_pseudonym_tree_view"));
  sel = gtk_tree_view_get_selection (tv);
  g_signal_connect(G_OBJECT(sel), "changed", 
		   G_CALLBACK(selection_changed_cb), NULL); 
  if (GTK_RESPONSE_OK != gtk_dialog_run (GTK_DIALOG (ad)))
    {
      gtk_widget_destroy (ad);
      g_object_unref (G_OBJECT (builder));
      builder = NULL;
      return;
    }
  GNUNET_assert (TRUE == gtk_tree_selection_get_selected (sel, 
							  &tm,
							  &iter));
  gtk_tree_model_get (tm,
		      &iter,
		      1, &ns,
		      -1);
  /* free all namespaces from list store except "ns" */
  if (TRUE ==
      gtk_tree_model_get_iter_first (tm, &iter))
    {
      do
	{
	  gtk_tree_model_get (tm, &iter,
			      1, &nso,
			      -1);
	  if (ns != nso)
	    GNUNET_FS_namespace_delete (nso, GNUNET_NO);
	}
      while (TRUE == gtk_tree_model_iter_next (tm, &iter));
    }
  gtk_widget_destroy (ad);
  g_object_unref (G_OBJECT (builder));
  builder = NULL;


  builder = GNUNET_GTK_get_new_builder ("advertise_pseudonym_dialog.glade");
  if (builder == NULL)
    {
      GNUNET_break (0);
      return;
    }
  GNUNET_GTK_setup_expiration_year_adjustment (builder);
  ad = GTK_WIDGET (gtk_builder_get_object (builder,
					   "GNUNET_GTK_advertise_pseudonym_dialog"));
  ls = GTK_LIST_STORE (gtk_builder_get_object (builder,
					       "GNUNET_GTK_metatype_list_store"));
  max_type = EXTRACTOR_metatype_get_max ();
  type = 1; 
  while (type < max_type - 1)
    {
      gtk_list_store_insert_with_values (ls, &iter, G_MAXINT,
					 0, EXTRACTOR_metatype_to_string (type),
					 1, type,
					 2, EXTRACTOR_metatype_to_description (type),
					 -1);
      type++;
    }
  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_advertise_pseudonym_metadata_type_tree_view"));
  sel = gtk_tree_view_get_selection (tv);
  g_signal_connect(G_OBJECT(sel), "changed", 
		   G_CALLBACK(GNUNET_GTK_advertise_pseudonym_metadata_value_entry_changed_cb), NULL); 

  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_advertise_pseudonym_metadata_tree_view"));
  sel = gtk_tree_view_get_selection (tv);
  g_signal_connect(G_OBJECT(sel), "changed", 
		   G_CALLBACK(metadata_selection_changed_cb), NULL); 

  tv = GTK_TREE_VIEW (gtk_builder_get_object (builder,
					      "GNUNET_GTK_advertise_pseudonym_keyword_list_tree_view"));
  sel = gtk_tree_view_get_selection (tv);
  g_signal_connect(G_OBJECT(sel), "changed", 
		   G_CALLBACK(keyword_selection_changed_cb), NULL); 
  if (GTK_RESPONSE_OK != gtk_dialog_run (GTK_DIALOG (ad)))
    {
      gtk_widget_destroy (ad);
      g_object_unref (G_OBJECT (builder));
      builder = NULL;
      return;
    }
  /* get keywords from list store */
  nxt = NULL;
  mrg = NULL;
  uri = NULL;
  tm = GTK_TREE_MODEL (gtk_builder_get_object (builder,
					       "GNUNET_GTK_keyword_list_store"));
  if (TRUE ==
      gtk_tree_model_get_iter_first (tm, &iter))
    {
      do
	{
	  gtk_tree_model_get (tm, &iter,
			      0, &value,
			      -1);
	  nxt = GNUNET_FS_uri_ksk_create_from_args (1, (const char**) &value);
	  mrg = GNUNET_FS_uri_ksk_merge (nxt, uri);
	  GNUNET_FS_uri_destroy (nxt);
	  if (NULL != uri)
	    GNUNET_FS_uri_destroy (uri);
	  uri = mrg;
	  g_free (value);
	}
      while (TRUE == gtk_tree_model_iter_next (tm, &iter));
    }
  /* get meta */
  nm = GNUNET_CONTAINER_meta_data_create ();
  tm = GTK_TREE_MODEL (gtk_builder_get_object (builder,
					       "GNUNET_GTK_meta_data_list_store"));
  if (TRUE ==
      gtk_tree_model_get_iter_first (tm, &iter))
    {
      do
	{
	  gtk_tree_model_get (tm, &iter,
			      0, &ntype,
			      1, &nformat,
			      3, &value,
			      -1);
	  GNUNET_CONTAINER_meta_data_insert (nm,
					     "<user>",
					     ntype,
					     nformat,
					     "text/plain",
					     value,
					     strlen (value)+1);
	  g_free (value);
	}
      while (TRUE == gtk_tree_model_iter_next (tm, &iter));
    }
  anonymity_level = gtk_spin_button_get_value (GTK_SPIN_BUTTON (gtk_builder_get_object (builder,
											"GNUNET_GTK_advertise_pseudonym_anonymity_spin_button")));
  priority = gtk_spin_button_get_value (GTK_SPIN_BUTTON (gtk_builder_get_object (builder,
										 "GNUNET_GTK_advertise_pseudonym_priority_spin_button")));
  sb = GTK_SPIN_BUTTON (gtk_builder_get_object (builder,
					       "GNUNET_GTK_advertise_pseudonym_expiration_year_spin_button"));
  expiration = GNUNET_GTK_get_expiration_time (sb);

  expiration = GNUNET_TIME_relative_to_absolute (GNUNET_TIME_UNIT_YEARS);
  root = gtk_entry_get_text (GTK_ENTRY (gtk_builder_get_object (builder,
								"GNUNET_GTK_advertise_pseudonym_root_entry")));
  GNUNET_FS_namespace_advertise (GNUNET_GTK_get_fs_handle (),
				 uri,
				 ns,
				 nm,
				 anonymity_level,
				 priority,
				 expiration,
				 root,
				 NULL, NULL);
  GNUNET_FS_namespace_delete (ns, GNUNET_NO);
  GNUNET_CONTAINER_meta_data_destroy (nm);
  if (NULL != uri)
    GNUNET_FS_uri_destroy (uri);
  gtk_widget_destroy (ad);
  g_object_unref (G_OBJECT (builder));
  builder = NULL;
}


/* end of main_window_adv_pseudonym.c */
