/*
     This file is part of GNUnet.
     (C) 2010 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/gnunet-setup.c
 * @brief Main function of gnunet-setup
 * @author Christian Grothoff
 */
#if ENABLE_NLS
#include <locale.h>
#endif
#include "gnunet-setup.h"
#include "gnunet-setup-options.h"
#include <regex.h>

/**
 * Builder for the main window (global!)
 */
GtkBuilder *builder; 

/**
 * Our configuration (global!)
 */
struct GNUNET_CONFIGURATION_Handle *cfg;


/**
 * @brief get the path to a specific GNUnet installation directory or,
 * with GNUNET_IPK_SELF_PREFIX, the current running apps installation directory
 * @author Milan
 * @return a pointer to the dir path (to be freed by the caller)
 */
char *
GNUNET_GTK_installation_get_path (enum GNUNET_OS_InstallationPathKind dirkind);



/**
 * Get the name of the directory where all of our package
 * data is stored ($PREFIX/share/)
 * 
 * @return name of the data directory
 */
const char *
GNUNET_GTK_get_data_dir ()
{
  static char *dd;

  if (dd == NULL)
    dd = GNUNET_GTK_installation_get_path (GNUNET_OS_IPK_DATADIR);
  return dd;
}


/**
 * Create an initialize a new builder based on the
 * GNUnet-GTK glade file.
 *
 * @param filename name of the resource file to load
 * @return NULL on error
 */
GtkBuilder *
GNUNET_GTK_get_new_builder (const char *filename)
{
  char *glade_path;
  GtkBuilder *ret;
  GError *error;

  ret = gtk_builder_new ();
  gtk_builder_set_translation_domain (ret, "gnunet-gtk");
  GNUNET_asprintf (&glade_path,
		   "%s%s",
		   GNUNET_GTK_get_data_dir (),
		   filename);
  error = NULL;
  if (0 == gtk_builder_add_from_file (ret, glade_path, &error))
    {
      GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
		  _("Failed to load `%s': %s\n"),
		  glade_path,
		  error->message);
      g_error_free (error);
      GNUNET_free (glade_path);
      return NULL;
    }
  gtk_builder_connect_signals (ret, NULL);
  GNUNET_free (glade_path);
  return ret;
}



/**
 * Initialize GNU gettext for message translation.
 */
static void
setup_nls ()
{
#if ENABLE_NLS
  char *path;

  setlocale (LC_ALL, "");
  GNUNET_asprintf (&path,
		   "%s/%s/locale/",
		   GNUNET_GTK_get_data_dir (),
		   PACKAGE_NAME);
  bindtextdomain ("gnunet-setup", path);
  textdomain ("gnunet-setup");
  bind_textdomain_codeset ("GNUnet", "UTF-8");
  bind_textdomain_codeset ("gnunet-setup", "UTF-8");
  GNUNET_free (path);
#else
  fprintf (stderr,
	   "WARNING: gnunet-setup was compiled without i18n support (did CFLAGS contain -Werror?).\n");
#endif
}


/**
 * Initialize GTK search path for icons.
 */
static void
set_icon_search_path ()
{
  char *buf;

  buf = GNUNET_GTK_installation_get_path (GNUNET_OS_IPK_ICONDIR);
  gtk_icon_theme_append_search_path (gtk_icon_theme_get_default (), buf);
  GNUNET_free (buf);
}


static gboolean
help_click_callback (GtkWidget *widget,
		     GdkEventButton *event,
		     gpointer user_data)
{
  const struct GNUNET_SETUP_OptionSpecification *os = user_data;
  GtkLinkButton *help;

  if (event->type != GDK_BUTTON_PRESS)
    return FALSE;
  help = GTK_LINK_BUTTON (gtk_builder_get_object (builder, 
						  "GNUNET_setup_help_text"));
  gtk_link_button_set_uri (help, 
			   os->help_url);
  gtk_button_set_label (GTK_BUTTON (help),
			os->help_text);  
  return FALSE;
}


/**
 * Change the visibility of widgets according to the 
 * value and visibility specification given.
 *
 * @param os option specification
 * @param value current value for the given option
 */
static void
update_visibility (const struct GNUNET_SETUP_OptionSpecification *os,
		   const char *value)
{
  unsigned int i;
  const struct GNUNET_SETUP_VisibilitySpecification *vs;
  GtkWidget *widget;
  regex_t r;

  if (os->visibility == NULL)
    return;
  i = 0;
  while (os->visibility[i].widget_name != NULL)
    {
      vs = &os->visibility[i];
      widget = GTK_WIDGET (gtk_builder_get_object (builder, 
						   vs->widget_name));
      if (widget == NULL)
	{
	  GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
		      _("Widget `%s' not found\n"),
		      vs->widget_name);
	}      
      if (NULL != vs->show_value)
	{
	  if (0 != regcomp (&r, vs->show_value, REG_EXTENDED | REG_ICASE | REG_NOSUB))
	    {
	      GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
			  _("Invalid regular expression `%s'\n"),
			  vs->show_value);
	      i++;
	      continue;
	    }
	  if (0 == regexec (&r, value, 0, NULL, 0))
	    gtk_widget_show (widget);
	  else
	    gtk_widget_hide (widget);	    
	  regfree (&r);
	}
      if (NULL != vs->hide_value)
	{
	  if (0 != regcomp (&r, vs->hide_value, REG_ICASE | REG_NOSUB))
	    {
	      GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
			  _("Invalid regular expression `%s'\n"),
			  vs->show_value);
	      i++;
	      continue;
	    }
	  if (0 == regexec (&r, value, 0, NULL, 0))
	    gtk_widget_hide (widget);
	  else
	    gtk_widget_show (widget);
	  regfree (&r);
	}
      i++;
    }
}
		   

/**
 * Function called whenever a widget changes its state.
 */
static void
widget_state_change_callback (const struct GNUNET_SETUP_OptionSpecification *os)
{
  GObject *widget;
  char *value;

  widget = gtk_builder_get_object (builder,
				   os->widget_name);
  GNUNET_assert (NULL != os->save_function);
  if (GNUNET_OK !=
      os->save_function (os->load_save_cls,
			 os->section,
			 os->option,
			 widget,
			 cfg))
    {
      GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
		  _("Failed to obtain option value from widget `%s'\n"),
		  os->widget_name);
      return;
    }
  GNUNET_assert (GNUNET_OK ==
		 GNUNET_CONFIGURATION_get_value_string (cfg,
							os->section,
							os->option,
							&value));
  update_visibility (os, value);
  GNUNET_free (value);
}


int
main (int argc,
      char *const *argv)
{
  char *cfgName = GNUNET_strdup (GNUNET_DEFAULT_USER_CONFIG_FILE);
  char *loglev = GNUNET_strdup ("WARNING");
  struct GNUNET_GETOPT_CommandLineOption options[] = {
    GNUNET_GETOPT_OPTION_CFG_FILE (&cfgName),
    GNUNET_GETOPT_OPTION_HELP (gettext_noop ("Setup tool for GNUnet")),
    GNUNET_GETOPT_OPTION_LOGLEVEL (&loglev),
    GNUNET_GETOPT_OPTION_VERSION (PACKAGE_VERSION),
    GNUNET_GETOPT_OPTION_END
  };
  gint ret;
  GtkWidget *main_window;
  GtkWidget *dialog;
  GObject *widget;
  int arg_off;
  int iret;
  char *const *argp;
  char *value;
  struct GNUNET_CONFIGURATION_Handle *cfgDefault;
  const struct GNUNET_SETUP_OptionSpecification *os;
  unsigned int i;
 
  arg_off = GNUNET_GETOPT_run ("gnunet-setup",
			       options,
			       argc,
			       argv);  
  if (arg_off == GNUNET_SYSERR)
    {
      GNUNET_free (loglev);
      GNUNET_free (cfgName);
      return 1;
    }
  (void) GNUNET_log_setup ("gnunet-setup",
			   loglev,
			   NULL);
  argp = &argv[arg_off];
  argc -= arg_off;
  gtk_init (&argc, (char ***) &argp);
  set_icon_search_path ();
  setup_nls ();
  builder = GNUNET_GTK_get_new_builder ("gnunet-setup.glade");
  if (builder == NULL)
    {
      GNUNET_free (loglev);
      GNUNET_free (cfgName);
      return 1;        
    }
  cfg = GNUNET_CONFIGURATION_create ();
  (void) GNUNET_CONFIGURATION_load (cfg, cfgName);
  main_window = GTK_WIDGET (gtk_builder_get_object (builder, 
						    "GNUNET_setup_dialog"));
  /* load options into dialog */
  i = 0;
  while (option_specifications[i].widget_name != NULL)
    {
      os = &option_specifications[i];
      widget = gtk_builder_get_object (builder,
				       os->widget_name);
      if (NULL == widget)
	{
	  GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
		      _("Widget `%s' not found\n"),
		      os->widget_name);
	  i++;
	  continue;
	}
      if (os->load_function != NULL) 
	{
	  if (GNUNET_OK !=
	      GNUNET_CONFIGURATION_get_value_string (cfg,
						     os->section,
						     os->option,
						     &value))
	    {
	      GNUNET_log (GNUNET_ERROR_TYPE_INFO,
			  _("No default value known for option `%s' in section `%s'\n"),
			  os->option,
			  os->section);
	    }
	  else
	    {
	      if (GNUNET_OK != 
		  os->load_function (os->load_save_cls,
				     os->section, os->option,
				     value,
				     widget,
				     cfg)) 
		{
		  GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
			      _("Failed to initialize widget `%s' with value `%s'\n"),
			      os->widget_name,
			      value);
		}
	      else
		{
		  update_visibility (os, value);
		}
	      GNUNET_free (value);
	    }
	}
      if (os->help_text != NULL)
	{
	  g_signal_connect (widget,
			    "button-press-event",
			    G_CALLBACK (&help_click_callback),
			    (void*) os);
	}
      if (os->change_signal != NULL)
	{
	  GNUNET_assert (NULL != os->save_function);
	  g_signal_connect_swapped (widget,
				    os->change_signal,
				    G_CALLBACK (&widget_state_change_callback),
				    (void*) os);
	}
      i++;
    }

 RESTART:
  ret = gtk_dialog_run (GTK_DIALOG (main_window));
  gtk_widget_hide (main_window);
  cfgDefault = GNUNET_CONFIGURATION_create ();
  (void) GNUNET_CONFIGURATION_load (cfgDefault, NULL); /* load defaults only */
  iret = 0;
  switch (ret)
    {
    case GTK_RESPONSE_CANCEL: /* -6 */
      /* explicit cancel, do not save! */
      break;
    case GTK_RESPONSE_OK: /* save-as, -5 */
      dialog = gtk_file_chooser_dialog_new (_("Save Configuration File"),
					    NULL,
					    GTK_FILE_CHOOSER_ACTION_SAVE,
					    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					    GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
					    NULL);
      gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (dialog), cfgName);
      if (gtk_dialog_run (GTK_DIALOG (dialog)) != GTK_RESPONSE_ACCEPT)
	{
	  gtk_widget_destroy (dialog);
	  goto RESTART;
	}
      GNUNET_free (cfgName);
      cfgName = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      gtk_widget_destroy (dialog);
      /* fall through! */
    case GTK_RESPONSE_DELETE_EVENT: /* window close, -4 */
    case GTK_RESPONSE_ACCEPT: /* save, -3 */
      if (GNUNET_OK !=
	  GNUNET_CONFIGURATION_write_diffs (cfgDefault,
					    cfg,
					    cfgName))
	iret = 1;
      break;
    default:
      GNUNET_break (0);
    }
  gtk_widget_destroy (main_window);
  g_object_unref (G_OBJECT (builder));  
  GNUNET_CONFIGURATION_destroy (cfgDefault);
  GNUNET_CONFIGURATION_destroy (cfg);
  GNUNET_free (cfgName);
  GNUNET_free (loglev);
  return iret;
}


/* end of gnunet-setup.c */
