/* 
   PXKEventFactory.m

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Author:  Scott Christley <scottc@net-community.com>
   Date: March 1996
   
   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include <AppKit/NSApplication.h>
#include <AppKit/NSCStringText.h> /* For definitions of some other keys */

#include <gnustep/xdps/PXKDrawingEngine.h>
#include <gnustep/xdps/PXKEventFactory.h>
#include <gnustep/xdps/PXKWindow.h>
#include <gnustep/xdps/PXKView.h>
#include <gnustep/xdps/PXKDPSContext.h>
#include <gnustep/xdps/PXKApplication.h>
#include <gnustep/xdps/PXKScreen.h>

/*
 * Private functions
 */
static NSEvent*
process_mouse_event(XEvent* xEvent, PXKDPSContext* ctxt,
		    NSEventType eventType, NSTimeInterval timestamp,
		    int eventNumber, int clickCount);

static NSEvent*
process_key_event (XEvent* xEvent, PXKDPSContext* ctxt, NSEventType eventType);

static unsigned short
process_key_code(XEvent *xEvent, KeySym keysym,
		 unsigned int eventModifierFlags);

static unsigned int
process_modifier_flags(unsigned int state);

/* Peek the next X Window event */
NSEvent *
peekEvent(void)
{
  return checkEvent();
}

/* Get the next X/Windows event and translate to a NSEvent */
NSEvent *
checkEvent(void)
{
  PXKDPSContext *cur = (PXKDPSContext*)[NSDPSContext currentContext];
  Display *xDisplay = [cur xDisplay];
  NSEvent* nullEvent = [NSApplication getNullEvent];
  XEvent xEvent;
  NSRect new_frame;
  NSEvent *e;
  PXKWindow *w = nil;

  NSDebugLog(@"Get next XWindows event\n");
  e = nullEvent;

#if 0
{
  long event_mask = ExposureMask | KeyPressMask | KeyReleaseMask |
	       ButtonPressMask | ButtonReleaseMask | 
	       ButtonMotionMask | StructureNotifyMask |
	       PointerMotionMask | EnterWindowMask |
	       LeaveWindowMask | FocusChangeMask |
	       PropertyChangeMask | ColormapChangeMask | KeymapStateMask |
	       VisibilityChangeMask;
  // Use XCheckMaskEvent so that we don't block
  if (!XCheckMaskEvent(xDisplay, event_mask, &xEvent))
    return nil;
}
#else
{
  int count;

  if ((count = XPending (xDisplay)))
    XNextEvent (xDisplay, &xEvent);
  else
    return nil;
}
#endif

  switch  (xEvent.type) {

    /* mouse button events */
  case ButtonPress:
    e = process_mouse_event (&xEvent, cur,
			     (xEvent.xbutton.button == Button1 ?
				  NSLeftMouseDown : NSRightMouseDown),
			     xEvent.xbutton.time,
			     xEvent.xbutton.serial,
			     1);
    break;
  case ButtonRelease:
    e = process_mouse_event (&xEvent, cur,
			     (xEvent.xbutton.button == Button1 ?
				  NSLeftMouseUp : NSRightMouseUp),
			     xEvent.xbutton.time,
			     xEvent.xbutton.serial,
			     1);
    break;

    /* change to stacking order */
  case CirculateNotify:
    NSDebugLog(@"CirculateNotify\n");
    break;
  case CirculateRequest:
    NSDebugLog(@"CirculateRequest\n");
    break;

    /* client events */
  case ClientMessage:
    NSDebugLog(@"ClientMessage\n");
    break;

    /* colormap attribute changes */
  case ColormapNotify:
    NSDebugLog(@"ColormapNotify\n");
    break;

    /* window has been resized, change width and
     * height to send to draw_text and draw_graphics
     * in next Expose */
  case ConfigureNotify:
    NSDebugLog(@"ConfigureNotify\n");
    new_frame.origin.x = (float)xEvent.xconfigure.x;
    new_frame.origin.y = (float)xEvent.xconfigure.y;
    new_frame.size.width = (float)xEvent.xconfigure.width;
    new_frame.size.height = (float)xEvent.xconfigure.height; 
    NSDebugLog(@"New frame %f %f %f %f\n", new_frame.origin.x, new_frame.origin.y,
	  new_frame.size.width, new_frame.size.height);
    NSDebugLog(@"border_width %d\n", xEvent.xconfigure.border_width);
    w = [PXKWindow windowForXWindow:xEvent.xconfigure.window];
    if (xEvent.xconfigure.window == [(PXKWindow *)w xWindow])
      [(PXKWindow *)w setFrameFromXFrame:new_frame];
    NSDebugLog(@"after ConfigureNotify\n");
    break;

    /* same as ConfigureNotify but we get this event before
       the change has actually occurred. */
  case ConfigureRequest:
    NSDebugLog(@"ConfigureRequest\n");
    break;

    /* a window has been created */
  case CreateNotify:
    NSDebugLog(@"CreateNotify\n");
    break;

    /* a window has been destroyed */
  case DestroyNotify:
    NSDebugLog(@"DestroyNotify\n");
    break;

    /* when the pointer enters a window */
  case EnterNotify:
    NSDebugLog(@"EnterNotify\n");
    break;

    /* when the pointer leaves a window */
  case LeaveNotify:
    NSDebugLog(@"LeaveNotify\n");
    break;

    /* portion of window becomes visible and we must redisplay */
  case Expose:
    {
      PXKWindow *window;
      XRectangle rectangle;

      NSDebugLog(@"Expose\n");

      /* find the NSWindow */
      window = [PXKWindow windowForXWindow:xEvent.xexpose.window];
      [window setXExposed: YES];

      rectangle.x = xEvent.xexpose.x;
      rectangle.y = xEvent.xexpose.y;
      rectangle.width = xEvent.xexpose.width;
      rectangle.height = xEvent.xexpose.height;
      [window addExposedRectangle:rectangle];

      if (xEvent.xexpose.count == 0)
	[window processExposedRectangles];
      break;
    }

    /* keyboard focus enters a window */
  case FocusIn:
    {
      PXKWindow *wind = nil;

      NSDebugLog(@"FocusIn\n");
      
      /* Find the NSWindow and make it the key window */
      wind = [PXKWindow windowForXWindow:xEvent.xexpose.window];
      [wind makeKeyWindow];

      break;
    }

    /* keyboard focus leaves a window */
  case FocusOut:
    NSDebugLog(@"FocusOut\n");
    break;

  case GraphicsExpose:
    NSDebugLog(@"GraphicsExpose\n");
    break;

  case NoExpose:
    NSDebugLog(@"NoExpose\n");
    break;

    /* a window is moved because of change in the size of its parent */
  case GravityNotify:
    NSDebugLog(@"GravityNotify\n");
    break;

    /* a key has been pressed */
  case KeyPress:
    e = process_key_event (&xEvent, cur, NSKeyDown);
    //	XRefreshKeyboardMapping(&xEvent);
    break;
	
    /* a key has been released */
  case KeyRelease:
    e = process_key_event (&xEvent, cur, NSKeyUp);
    break;

    /* reports state of keyboard when pointer or focus enters a window */
  case KeymapNotify:
    NSDebugLog(@"KeymapNotify\n");
    break;

    /* when a window changes stage from ummapped to mapped or vice versa */
  case MapNotify:
    NSDebugLog(@"MapNotify\n");
    break;
  case UnmapNotify:
    {
      PXKWindow *wind;

      NSDebugLog(@"UnmapNotify\n");

      // find the NSWindow
      // and tell it that its no longer visible
      wind = [PXKWindow windowForXWindow:xEvent.xunmap.window];
      [(PXKWindow *)wind setXUnmap: NO];

      break;
    }

    /* like MapNotify but before the request is carried out */
  case MapRequest:
    NSDebugLog(@"MapRequest\n");
    break;

    /* when the mappings for keyboard or mouse has been
       change by another client */
  case MappingNotify:
    NSDebugLog(@"MappingNotify\n");
    //	XRefreshKeyboardMapping(&xEvent);
    break;

    /* the mouse has moved */
  case MotionNotify: {
    NSEventType eventType;

    NSDebugLog(@"MotionNotify\n");
    if (xEvent.xbutton.button == Button1)
      eventType = NSLeftMouseDragged;
    else if (xEvent.xbutton.button == Button2)
      eventType = NSRightMouseDragged;
    else
      eventType = NSMouseMoved;
    e = process_mouse_event (&xEvent, cur,
			     eventType,
			     xEvent.xbutton.time,
			     xEvent.xbutton.serial,
			     1);
    break;
  }
	
    /* a property of a window has changed or been deleted */
  case PropertyNotify:
    NSDebugLog(@"PropertyNotify\n");
    break;
	
    /* a client successfully reparents a window */
  case ReparentNotify:
    NSDebugLog(@"ReparentNotify\n");
    NSDebugLog(@"parent offset %f %f\n", xEvent.xreparent.x,
	  xEvent.xreparent.y);
    break;

    /* another client attempts to change the size of a window */
  case ResizeRequest:
    NSDebugLog(@"ResizeRequest\n");
    break;

    /* events dealing with the selection */
  case SelectionClear:
  case SelectionNotify:
  case SelectionRequest:
    NSDebugLog(@"Selection*\n");
    break;

    /* a change in the visibility of a window has occurred */
  case VisibilityNotify:
    NSDebugLog(@"VisibilityNotify\n");
    break;

    /* We shouldn't get here unless we forgot to trap an
       event above. */
  default:
    NSDebugLog(@"Received an untrapped event\n");
    break;
	
  } /* end event type switch */    

  return e;
}

static NSEvent*
process_mouse_event(XEvent* xEvent, PXKDPSContext* context,
		    NSEventType eventType, NSTimeInterval timestamp,
		    int eventNumber, int clickCount)
{
  NSApplication *theApp = [NSApplication sharedApplication];
  NSPoint eventLocation;
  unsigned int eventFlags;
  NSEvent *event;
  PXKWindow *window;
  Window theW;
  NSRect xFrame = NSZeroRect;

  NSDebugLog(@"Process mouse event");

  eventFlags = process_modifier_flags(xEvent->xbutton.state);
    
  // If the pointer is grabbed then use that window instead
  theW = [(PXKApplication *)theApp grabXWindow];
  if (theW == 0)
    theW = xEvent->xbutton.window;

  // Find the window
  window = [PXKWindow windowForXWindow:theW];
  if (window)
    xFrame = [window xFrame];

  eventLocation = [context userPointFromXPoint:
			  NSMakePoint(xEvent->xbutton.x,
				      xFrame.size.height - xEvent->xbutton.y)];
  NSDebugLog (@"xLocation = (%d, %d), userLocation = (%f, %f)",
	xEvent->xbutton.x, (int)(xFrame.size.height - xEvent->xbutton.y),
	eventLocation.x, eventLocation.y);

  event = [NSEvent mouseEventWithType:eventType
		    location:eventLocation
		    modifierFlags:eventFlags
		    timestamp:timestamp
		    windowNumber:[window windowNumber]
		    context:context
		    eventNumber:eventNumber
		    clickCount:clickCount
		    pressure:1.0];

  return event;
}

/* TODO: correctly map key events to NSEvents. This can be made when 16-bit
   character strings will be available. */
static NSEvent*
process_key_event (XEvent* xEvent, PXKDPSContext* context,
		   NSEventType eventType)
{
  char buf[256];
  int count;
  XComposeStatus compose;
  NSString *keys, *ukeys;
  KeySym keysym;
  NSPoint eventLocation;
  unsigned short keyCode;
  unsigned int eventFlags;
  NSEvent *event;
  NSApplication *theApp = [NSApplication sharedApplication];
  PXKWindow *window;
  NSRect xFrame = NSZeroRect;

  NSDebugLog(@"Process key event");

  eventFlags = process_modifier_flags(xEvent->xkey.state);
    
  count = XLookupString((XKeyEvent *)xEvent, buf, 256, &keysym, &compose);

  /* Make sure that the string is properly terminated */
		      
  if (count > 255) {
    buf[255] = '\0';
  } else if (count < 1) {
    buf[0] = '\0';
  } else {
    buf[count] = '\0';
  }

  // Set window to the key window
  window = (PXKWindow*)[theApp keyWindow];
  if (window)
    xFrame = [window xFrame];

  eventLocation = [context userPointFromXPoint:
			  NSMakePoint(xEvent->xbutton.x,
				      xFrame.size.height - xEvent->xbutton.y)];
  NSDebugLog (@"xLocation = (%d, %d), userLocation = (%f, %f)",
	xEvent->xbutton.x, (int)(xFrame.size.height - xEvent->xbutton.y),
	eventLocation.x, eventLocation.y);

  keyCode = process_key_code(xEvent, keysym, eventFlags);
    
  ukeys = [NSString stringWithCString:buf];
  keys = ukeys; /* Stupid implementation (to be replaced) */

  event = [NSEvent keyEventWithType:eventType
		    location:eventLocation
		    modifierFlags:eventFlags
		    timestamp:(NSTimeInterval)xEvent->xkey.time
		    windowNumber:[window windowNumber]
		    context:context
		    characters:keys
		    charactersIgnoringModifiers:ukeys
		    isARepeat:NO
		    keyCode:keyCode];

  return event;
}

static unsigned short
process_key_code(XEvent *xEvent, KeySym keysym,
		 unsigned int eventModifierFlags)
{
  unsigned short keyCode = 0;

  if ((keysym == XK_Return) ||
      (keysym == XK_KP_Enter) ||
      (keysym == XK_Linefeed))
    {
      /* do nothing for now */
	keyCode = 0x0d;
    } else if ((keysym >= XK_Shift_L) && (keysym <= XK_Hyper_R)) {

      /*
       * The shift button is depressed.  This should have already
       * been detected in the process_modifier_flags function.  Therefore,
       * nothing is done here...
       */
    }

  /* if a function key was pressed */
    
  if ((keysym >= XK_F1) && (keysym <= XK_F35)) {

    eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
	
    switch(xEvent->xkey.keycode) {
    case XK_F1: keyCode = NSF1FunctionKey; break;
    case XK_F2: keyCode = NSF2FunctionKey; break;
    case XK_F3: keyCode = NSF3FunctionKey; break;
    case XK_F4: keyCode = NSF4FunctionKey; break;
    case XK_F5: keyCode = NSF5FunctionKey; break;
    case XK_F6: keyCode = NSF6FunctionKey; break;
    case XK_F7: keyCode = NSF7FunctionKey; break;
    case XK_F8: keyCode = NSF8FunctionKey; break;
    case XK_F9: keyCode = NSF9FunctionKey; break;
    case XK_F10: keyCode = NSF10FunctionKey; break;
    case XK_F11: keyCode = NSF11FunctionKey; break;
    case XK_F12: keyCode = NSF12FunctionKey; break;
    case XK_F13: keyCode = NSF13FunctionKey; break;
    case XK_F14: keyCode = NSF14FunctionKey; break;
    case XK_F15: keyCode = NSF15FunctionKey; break;
    case XK_F16: keyCode = NSF16FunctionKey; break;
    case XK_F17: keyCode = NSF17FunctionKey; break;
    case XK_F18: keyCode = NSF18FunctionKey; break;
    case XK_F19: keyCode = NSF19FunctionKey; break;
    case XK_F20: keyCode = NSF20FunctionKey; break;
    case XK_F21: keyCode = NSF21FunctionKey; break;
    case XK_F22: keyCode = NSF22FunctionKey; break;
    case XK_F23: keyCode = NSF23FunctionKey; break;
    case XK_F24: keyCode = NSF24FunctionKey; break;
    case XK_F25: keyCode = NSF25FunctionKey; break;
    case XK_F26: keyCode = NSF26FunctionKey; break;
    case XK_F27: keyCode = NSF27FunctionKey; break;
    case XK_F28: keyCode = NSF28FunctionKey; break;
    case XK_F29: keyCode = NSF29FunctionKey; break;
    case XK_F30: keyCode = NSF30FunctionKey; break;
    case XK_F31: keyCode = NSF31FunctionKey; break;
    case XK_F32: keyCode = NSF32FunctionKey; break;
    case XK_F33: keyCode = NSF33FunctionKey; break;
    case XK_F34: keyCode = NSF34FunctionKey; break;
    case XK_F35: keyCode = NSF35FunctionKey; break;

    default: /* do nothing */
    }
  } else {
    if (keysym == XK_BackSpace) {
      keyCode = NSBackspaceKey;
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Delete) {
      keyCode = NSDeleteFunctionKey;
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Home) {
      keyCode = NSHomeFunctionKey;
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Left) {
      keyCode = NSLeftArrowFunctionKey;
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Up) {
      keyCode = NSUpArrowFunctionKey;
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Right) {
      keyCode = NSRightArrowFunctionKey;
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Down) {
      keyCode = NSDownArrowFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Prior) {
      keyCode = NSPrevFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
#ifndef NeXT
    } else if (keysym == XK_Page_Up) {
      keyCode = NSPageUpFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
#endif
    } else if (keysym == XK_Next) {
      keyCode = NSNextFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
#ifndef NeXT
    } else if (keysym == XK_Page_Down) {
      keyCode = NSPageDownFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
#endif
    } else if (keysym == XK_End) {
      keyCode = NSEndFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Begin) {
      keyCode = NSBeginFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Select) {
      keyCode = NSSelectFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Print) {
      keyCode = NSPrintFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Execute) {
      keyCode = NSExecuteFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Insert) {
      keyCode = NSInsertFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Undo) {
      keyCode = NSUndoFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Redo) {
      keyCode = NSRedoFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Menu) {
      keyCode = NSMenuFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Find) {
      keyCode = NSFindFunctionKey;
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 

#if 0
    } else if (keysym == XK_Cancel) {
      keyCode = NSSystemFunctionKey; /* ??? */		
#endif

    } else if (keysym == XK_Help) {
      keyCode = NSHelpFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Break) {
      keyCode = NSBreakFunctionKey;
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;    
    } else if (keysym == XK_Mode_switch) {
      keyCode = NSModeSwitchFunctionKey;
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 

#if 0	
    } else if (keysym == XK_script_switch) {
      keyCode = NSSystemFunctionKey; /* ??? */
    } else if (keysym == XK_Num_Lock) {
      keyCode = NSSystemFunctionKey; /* ??? */
#endif
#ifndef NeXT
    } else if (keysym == XK_Sys_Req) {
      keyCode = NSSysReqFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
#endif
    } else if (keysym == XK_Scroll_Lock) {
      keyCode = NSScrollLockFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Pause) {
      keyCode = NSPauseFunctionKey;		
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if (keysym == XK_Clear) {
      keyCode = NSClearDisplayFunctionKey;
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask; 
    } else if ((keysym == XK_Shift_L) || (keysym == XK_Shift_R)) {
      eventModifierFlags = eventModifierFlags |
	NSFunctionKeyMask |
	NSShiftKeyMask; 
    } else if ((keysym == XK_Control_L) || (keysym == XK_Control_R)) {
      eventModifierFlags = eventModifierFlags |
	NSFunctionKeyMask |
	NSControlKeyMask; 
    } else if (keysym == XK_Alt_R) {
      eventModifierFlags = eventModifierFlags |
	NSFunctionKeyMask |
	NSAlternateKeyMask; 
    } else if (keysym == XK_Alt_L) {
      eventModifierFlags = eventModifierFlags |
	NSFunctionKeyMask |
	NSCommandKeyMask; 
    } else if (keysym == XK_Tab) {
      keyCode = 0x09;
    }
  }
    
  /* If the key press originated from the key pad */
    
  if ((keysym >= XK_KP_Space) && (keysym <= XK_KP_9)) {
    eventModifierFlags = eventModifierFlags | NSNumericPadKeyMask;

    switch(keysym) {
    case XK_KP_F1:        keyCode = NSF1FunctionKey;         break;
    case XK_KP_F2:        keyCode = NSF2FunctionKey;         break;
    case XK_KP_F3:        keyCode = NSF3FunctionKey;         break;
    case XK_KP_F4:        keyCode = NSF4FunctionKey;         break;
#ifndef NeXT
    case XK_KP_Home:      keyCode = NSHomeFunctionKey;       break;
    case XK_KP_Left:      keyCode = NSLeftArrowFunctionKey;  break;
    case XK_KP_Up:        keyCode = NSUpArrowFunctionKey;    break;
    case XK_KP_Right:     keyCode = NSRightArrowFunctionKey; break;
    case XK_KP_Down:      keyCode = NSDownArrowFunctionKey;  break;
    case XK_KP_Prior:     keyCode = NSPrevFunctionKey;       break;
      //case XK_KP_Page_Up:   keyCode = NSPageUpFunctionKey;     break;
    case XK_KP_Next:      keyCode = NSNextFunctionKey;       break;
      //case XK_KP_Page_Down: keyCode = NSPageDownFunctionKey;   break;
    case XK_KP_End:       keyCode = NSEndFunctionKey;        break;
    case XK_KP_Begin:     keyCode = NSBeginFunctionKey;      break;
    case XK_KP_Insert:    keyCode = NSInsertFunctionKey;     break;
    case XK_KP_Delete:    keyCode = NSDeleteFunctionKey;     break;
#endif
	    
    default:  break;  /* Nothing to do */
    }
  } /* End of keypad stuff */

  if (((keysym >= XK_KP_Space) && (keysym <= XK_KP_9)) ||
      ((keysym >= XK_space) && (keysym <= XK_asciitilde)))
    {	
      /* Not processed */
    }

  return keyCode;
}

/*
 * process_modifier_flags determines which modifier keys (Command, Control,
 * Shift, and so forth) were held down while the event occured and
 * sets the eventModifierFlags instance variable accordingly.
 */
    
static unsigned int
process_modifier_flags(unsigned int state)
{
  unsigned int eventModifierFlags = 0; /* Clear the mask */

  if (state & ControlMask) {
    eventModifierFlags = eventModifierFlags | NSControlKeyMask;
  }

  if (state & ShiftMask) {
    eventModifierFlags = eventModifierFlags | NSShiftKeyMask;
  }

  if (state & Mod1Mask) {
    eventModifierFlags = eventModifierFlags | NSCommandKeyMask; /* ? */
    NSDebugLog (@"setButtonModifierFlags(): Mod1Mask\n");
  }

  if (state & Mod2Mask) {
    eventModifierFlags = eventModifierFlags | NSAlternateKeyMask;/* ? */
    NSDebugLog (@"setButtonModifierFlags(): Mod2Mask\n");
  }

  if (state & Mod3Mask) {
    eventModifierFlags = eventModifierFlags | NSAlphaShiftKeyMask;/* ? */
    NSDebugLog (@"setButtonModifierFlags(): Mod3Mask\n");
  }

  if (state & Mod4Mask) {
    eventModifierFlags = eventModifierFlags | NSHelpKeyMask; /* ? */
    NSDebugLog (@"setButtonModifierFlags(): Mod4Mask\n");
  }

  if (state & Mod5Mask) {
    eventModifierFlags = eventModifierFlags | NSControlKeyMask; /* ? */
    NSDebugLog (@"setButtonModifierFlags(): Mod5Mask\n");
  }

  return eventModifierFlags;
}
